/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftGroup;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.StateMachineException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.ConfigurationManager;
import org.apache.ratis.server.impl.RaftConfiguration;
import org.apache.ratis.server.impl.RaftServerConstants;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.impl.StateMachineUpdater;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.MemoryRaftLog;
import org.apache.ratis.server.storage.RaftLog;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.SegmentedRaftLog;
import org.apache.ratis.server.storage.SnapshotManager;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.util.Timestamp;

public class ServerState
implements Closeable {
    private final RaftPeerId selfId;
    private final RaftGroupId groupId;
    private final RaftServerImpl server;
    private final RaftLog log;
    private final ConfigurationManager configurationManager;
    private final StateMachineUpdater stateMachineUpdater;
    private final RaftStorage storage;
    private final SnapshotManager snapshotManager;
    private volatile Timestamp lastNoLeaderTime;
    private final long leaderElectionTimeoutMs;
    private final AtomicLong currentTerm = new AtomicLong();
    private volatile RaftPeerId leaderId;
    private volatile RaftPeerId votedFor;
    private volatile TermIndex latestInstalledSnapshot;

    ServerState(RaftPeerId id, RaftGroup group, RaftProperties prop, RaftServerImpl server, StateMachine stateMachine) throws IOException {
        this.selfId = id;
        this.groupId = group.getGroupId();
        this.server = server;
        RaftConfiguration initialConf = RaftConfiguration.newBuilder().setConf(group.getPeers()).build();
        this.configurationManager = new ConfigurationManager(initialConf);
        RaftServerImpl.LOG.info("{}:{} {}", new Object[]{id, this.groupId, this.configurationManager});
        File dir = ServerState.chooseStorageDir(RaftServerConfigKeys.storageDirs(prop), group.getGroupId().getUuid().toString());
        this.storage = new RaftStorage(dir, RaftServerConstants.StartupOption.REGULAR);
        this.snapshotManager = new SnapshotManager(this.storage, id);
        long lastApplied = this.initStatemachine(stateMachine, group.getGroupId());
        this.leaderId = null;
        this.lastNoLeaderTime = Timestamp.currentTime();
        this.leaderElectionTimeoutMs = RaftServerConfigKeys.leaderElectionTimeout(prop).toIntExact(TimeUnit.MILLISECONDS);
        this.log = this.initLog(id, prop, lastApplied, this::setRaftConf);
        RaftLog.Metadata metadata = this.log.loadMetadata();
        this.currentTerm.set(metadata.getTerm());
        this.votedFor = metadata.getVotedFor();
        this.stateMachineUpdater = new StateMachineUpdater(stateMachine, server, this.log, lastApplied, prop);
    }

    static File chooseStorageDir(List<File> volumes, String targetSubDir) throws IOException {
        HashMap numberOfStorageDirPerVolume = new HashMap();
        File[] empty = new File[]{};
        ArrayList resultList = new ArrayList();
        volumes.stream().flatMap(volume -> {
            File[] dirs = Optional.ofNullable(volume.listFiles()).orElse(empty);
            numberOfStorageDirPerVolume.put(volume, dirs.length);
            return Arrays.stream(dirs);
        }).filter(dir -> targetSubDir.equals(dir.getName())).forEach(resultList::add);
        if (resultList.size() > 1) {
            throw new IOException("More than one directories found for " + targetSubDir + ": " + resultList);
        }
        if (resultList.size() == 1) {
            return (File)resultList.get(0);
        }
        return numberOfStorageDirPerVolume.entrySet().stream().min(Comparator.comparing(Map.Entry::getValue)).map(Map.Entry::getKey).map(v -> new File((File)v, targetSubDir)).orElseThrow(() -> new IOException("No storage directory found."));
    }

    private long initStatemachine(StateMachine sm, RaftGroupId gid) throws IOException {
        sm.initialize(this.server.getProxy(), gid, this.storage);
        this.storage.setStateMachineStorage(sm.getStateMachineStorage());
        SnapshotInfo snapshot = sm.getLatestSnapshot();
        if (snapshot == null || snapshot.getTermIndex().getIndex() < 0L) {
            return -1L;
        }
        RaftConfiguration raftConf = this.storage.readRaftConfiguration();
        if (raftConf != null) {
            this.setRaftConf(raftConf.getLogEntryIndex(), raftConf);
        }
        return snapshot.getIndex();
    }

    void writeRaftConfiguration(RaftProtos.LogEntryProto conf) {
        this.storage.writeRaftConfiguration(conf);
    }

    void start() {
        this.stateMachineUpdater.start();
    }

    private RaftLog initLog(RaftPeerId id, RaftProperties prop, long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        RaftLog log;
        if (RaftServerConfigKeys.Log.useMemory(prop)) {
            int maxBufferSize = RaftServerConfigKeys.Log.Appender.bufferByteLimit(prop).getSizeInt();
            log = new MemoryRaftLog(id, lastIndexInSnapshot, maxBufferSize);
        } else {
            log = new SegmentedRaftLog(id, this.server, this.storage, lastIndexInSnapshot, prop);
        }
        log.open(lastIndexInSnapshot, logConsumer);
        return log;
    }

    public RaftConfiguration getRaftConf() {
        return this.configurationManager.getCurrent();
    }

    public RaftPeerId getSelfId() {
        return this.selfId;
    }

    public long getCurrentTerm() {
        return this.currentTerm.get();
    }

    boolean updateCurrentTerm(long newTerm) {
        long current = this.currentTerm.getAndUpdate(curTerm -> Math.max(curTerm, newTerm));
        if (newTerm > current) {
            this.votedFor = null;
            this.setLeader(null, "updateCurrentTerm");
            return true;
        }
        return false;
    }

    RaftPeerId getLeaderId() {
        return this.leaderId;
    }

    boolean hasLeader() {
        return this.leaderId != null;
    }

    long initElection() {
        this.votedFor = this.selfId;
        this.setLeader(null, "initElection");
        return this.currentTerm.incrementAndGet();
    }

    void persistMetadata() throws IOException {
        this.log.writeMetadata(this.currentTerm.get(), this.votedFor);
    }

    void grantVote(RaftPeerId candidateId) {
        this.votedFor = candidateId;
        this.setLeader(null, "grantVote");
    }

    void setLeader(RaftPeerId newLeaderId, String op) {
        if (!Objects.equals(this.leaderId, newLeaderId)) {
            String suffix;
            if (newLeaderId == null) {
                this.lastNoLeaderTime = Timestamp.currentTime();
                suffix = "";
            } else {
                Timestamp previous = this.lastNoLeaderTime;
                this.lastNoLeaderTime = null;
                suffix = ", leader elected after " + previous.elapsedTimeMs() + "ms";
            }
            RaftServerImpl.LOG.info("{}:{} change Leader from {} to {} at term {} for {}{}", new Object[]{this.selfId, this.groupId, this.leaderId, newLeaderId, this.getCurrentTerm(), op, suffix});
            this.leaderId = newLeaderId;
        }
    }

    boolean checkForExtendedNoLeader() {
        return this.getLastLeaderElapsedTimeMs() > this.leaderElectionTimeoutMs;
    }

    long getLastLeaderElapsedTimeMs() {
        Timestamp t = this.lastNoLeaderTime;
        return t == null ? 0L : t.elapsedTimeMs();
    }

    void becomeLeader() {
        this.setLeader(this.selfId, "becomeLeader");
    }

    public RaftLog getLog() {
        return this.log;
    }

    void appendLog(TransactionContext operation) throws StateMachineException {
        this.log.append(this.currentTerm.get(), operation);
        Objects.requireNonNull(operation.getLogEntry());
    }

    boolean recognizeLeader(RaftPeerId leaderId, long leaderTerm) {
        long current = this.currentTerm.get();
        if (leaderTerm < current) {
            return false;
        }
        if (leaderTerm > current || this.leaderId == null) {
            return true;
        }
        return this.leaderId.equals((Object)leaderId);
    }

    boolean recognizeCandidate(RaftPeerId candidateId, long candidateTerm) {
        if (!this.getRaftConf().containsInConf(candidateId)) {
            return false;
        }
        long current = this.currentTerm.get();
        if (candidateTerm > current) {
            return true;
        }
        if (candidateTerm == current) {
            return this.votedFor == null || this.votedFor.equals((Object)candidateId);
        }
        return false;
    }

    boolean isLogUpToDate(TermIndex candidateLastEntry) {
        TermIndex local = this.log.getLastEntryTermIndex();
        SnapshotInfo snapshot = this.server.getStateMachine().getLatestSnapshot();
        if (local == null && snapshot == null) {
            return true;
        }
        if (candidateLastEntry == null) {
            return false;
        }
        if (local == null || snapshot != null && snapshot.getIndex() > local.getIndex()) {
            local = snapshot.getTermIndex();
        }
        return local.compareTo(candidateLastEntry) <= 0;
    }

    public String toString() {
        return this.selfId + ":t" + this.currentTerm + ", leader=" + this.leaderId + ", voted=" + this.votedFor + ", raftlog=" + this.log + ", conf=" + this.getRaftConf();
    }

    boolean isConfCommitted() {
        return this.getLog().getLastCommittedIndex() >= this.getRaftConf().getLogEntryIndex();
    }

    void setRaftConf(RaftProtos.LogEntryProto entry) {
        if (entry.hasConfigurationEntry()) {
            this.setRaftConf(entry.getIndex(), ServerProtoUtils.toRaftConfiguration(entry));
        }
    }

    void setRaftConf(long logIndex, RaftConfiguration conf) {
        this.configurationManager.addConfiguration(logIndex, conf);
        this.server.getServerRpc().addPeers(conf.getPeers());
        RaftServerImpl.LOG.info("{}:{} set configuration {} at {}", new Object[]{this.getSelfId(), this.groupId, conf, logIndex});
        RaftServerImpl.LOG.trace("{}: {}", (Object)this.getSelfId(), (Object)this.configurationManager);
    }

    void updateConfiguration(RaftProtos.LogEntryProto[] entries) {
        if (entries != null && entries.length > 0) {
            this.configurationManager.removeConfigurations(entries[0].getIndex());
            Arrays.stream(entries).forEach(this::setRaftConf);
        }
    }

    boolean updateStatemachine(long majorityIndex, long currentTerm) {
        if (this.log.updateLastCommitted(majorityIndex, currentTerm)) {
            this.stateMachineUpdater.notifyUpdater();
            return true;
        }
        return false;
    }

    void reloadStateMachine(long lastIndexInSnapshot, long currentTerm) {
        this.log.updateLastCommitted(lastIndexInSnapshot, currentTerm);
        this.stateMachineUpdater.reloadStateMachine();
    }

    @Override
    public void close() throws IOException {
        try {
            this.stateMachineUpdater.stopAndJoin();
        }
        catch (InterruptedException e) {
            RaftServerImpl.LOG.warn(this.getSelfId() + ": Interrupted when joining stateMachineUpdater", (Throwable)e);
        }
        RaftServerImpl.LOG.info("{}:{} closes. The last applied log index is {}", new Object[]{this.getSelfId(), this.groupId, this.getLastAppliedIndex()});
        this.log.close();
        this.storage.close();
    }

    public RaftStorage getStorage() {
        return this.storage;
    }

    void installSnapshot(RaftProtos.InstallSnapshotRequestProto request) throws IOException {
        StateMachine sm = this.server.getStateMachine();
        sm.pause();
        this.snapshotManager.installSnapshot(sm, request);
        this.log.syncWithSnapshot(request.getTermIndex().getIndex());
        this.latestInstalledSnapshot = ServerProtoUtils.toTermIndex(request.getTermIndex());
    }

    SnapshotInfo getLatestSnapshot() {
        return this.server.getStateMachine().getStateMachineStorage().getLatestSnapshot();
    }

    public TermIndex getLatestInstalledSnapshot() {
        return this.latestInstalledSnapshot;
    }

    public long getLastAppliedIndex() {
        return this.stateMachineUpdater.getLastAppliedIndex();
    }
}

