/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.management.ManagementFactory;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.ratis.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RaftStorageDirectory {
    static final Logger LOG = LoggerFactory.getLogger(RaftStorageDirectory.class);
    static final String STORAGE_DIR_CURRENT = "current";
    static final String STORAGE_FILE_LOCK = "in_use.lock";
    static final String META_FILE_NAME = "raft-meta";
    static final String LOG_FILE_INPROGRESS = "inprogress";
    static final String LOG_FILE_PREFIX = "log";
    static final String STATE_MACHINE = "sm";
    static final String TEMP = "tmp";
    static final Pattern CLOSED_SEGMENT_REGEX = Pattern.compile("log_(\\d+)-(\\d+)");
    static final Pattern OPEN_SEGMENT_REGEX = Pattern.compile("log_inprogress_(\\d+)(?:\\..*)?");
    private final File root;
    private FileLock lock;

    RaftStorageDirectory(File dir) {
        this.root = dir;
        this.lock = null;
    }

    public File getRoot() {
        return this.root;
    }

    void clearDirectory() throws IOException {
        File curDir = this.getCurrentDir();
        this.clearDirectory(curDir);
        this.clearDirectory(this.getStateMachineDir());
    }

    void clearDirectory(File dir) throws IOException {
        if (dir.exists()) {
            LOG.info(dir + " already exists.  Deleting it ...");
            FileUtils.deleteFully((File)dir);
        }
        FileUtils.createDirectories((File)dir);
    }

    File getCurrentDir() {
        return new File(this.root, STORAGE_DIR_CURRENT);
    }

    File getMetaFile() {
        return new File(this.getCurrentDir(), META_FILE_NAME);
    }

    File getMetaTmpFile() {
        return new File(this.getCurrentDir(), "raft-meta.tmp");
    }

    File getOpenLogFile(long startIndex) {
        return new File(this.getCurrentDir(), RaftStorageDirectory.getOpenLogFileName(startIndex));
    }

    static String getOpenLogFileName(long startIndex) {
        return "log_inprogress_" + startIndex;
    }

    File getClosedLogFile(long startIndex, long endIndex) {
        return new File(this.getCurrentDir(), RaftStorageDirectory.getClosedLogFileName(startIndex, endIndex));
    }

    static String getClosedLogFileName(long startIndex, long endIndex) {
        return "log_" + startIndex + "-" + endIndex;
    }

    public File getStateMachineDir() {
        return new File(this.getRoot(), STATE_MACHINE);
    }

    public File getNewTempDir() {
        return new File(new File(this.getRoot(), TEMP), UUID.randomUUID().toString());
    }

    public Path relativizeToRoot(Path p) {
        if (p.isAbsolute()) {
            return this.getRoot().toPath().relativize(p);
        }
        return p;
    }

    public List<LogPathAndIndex> getLogSegmentFiles() throws IOException {
        ArrayList<LogPathAndIndex> list = new ArrayList<LogPathAndIndex>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.getCurrentDir().toPath());){
            block9: for (Path path : stream) {
                for (Pattern pattern : Arrays.asList(CLOSED_SEGMENT_REGEX, OPEN_SEGMENT_REGEX)) {
                    Matcher matcher = pattern.matcher(path.getFileName().toString());
                    if (!matcher.matches()) continue;
                    if (pattern == OPEN_SEGMENT_REGEX && Files.size(path) == 0L) {
                        Files.delete(path);
                        LOG.info("Delete zero size file " + path);
                        continue block9;
                    }
                    long startIndex = Long.parseLong(matcher.group(1));
                    long endIndex = matcher.groupCount() == 2 ? Long.parseLong(matcher.group(2)) : -1L;
                    list.add(new LogPathAndIndex(path, startIndex, endIndex));
                    continue block9;
                }
            }
        }
        Collections.sort(list, (o1, o2) -> o1.startIndex == o2.startIndex ? 0 : (o1.startIndex < o2.startIndex ? -1 : 1));
        return list;
    }

    boolean isCurrentEmpty() throws IOException {
        File currentDir = this.getCurrentDir();
        if (!currentDir.exists()) {
            return true;
        }
        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(currentDir.toPath());){
            if (dirStream.iterator().hasNext()) {
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    StorageState analyzeStorage(boolean toLock) throws IOException {
        Objects.requireNonNull(this.root, "root directory is null");
        String rootPath = this.root.getCanonicalPath();
        try {
            if (!this.root.exists()) {
                LOG.info("The storage directory " + rootPath + " does not exist. Creating ...");
                FileUtils.createDirectories((File)this.root);
            }
            if (!this.root.isDirectory()) {
                LOG.warn(rootPath + " is not a directory");
                return StorageState.NON_EXISTENT;
            }
            if (!Files.isWritable(this.root.toPath())) {
                LOG.warn("The storage directory " + rootPath + " is not writable.");
                return StorageState.NON_EXISTENT;
            }
        }
        catch (SecurityException ex) {
            LOG.warn("Cannot access storage directory " + rootPath, (Throwable)ex);
            return StorageState.NON_EXISTENT;
        }
        if (toLock) {
            this.lock();
        }
        if (this.hasMetaFile()) {
            return StorageState.NORMAL;
        }
        return StorageState.NOT_FORMATTED;
    }

    boolean hasMetaFile() throws IOException {
        return this.getMetaFile().exists();
    }

    public void lock() throws IOException {
        FileLock newLock = this.tryLock();
        if (newLock == null) {
            String msg = "Cannot lock storage " + this.root + ". The directory is already locked";
            LOG.info(msg);
            throw new IOException(msg);
        }
        this.lock = newLock;
    }

    private FileLock tryLock() throws IOException {
        FileLock res;
        boolean deletionHookAdded = false;
        File lockF = new File(this.root, STORAGE_FILE_LOCK);
        if (!lockF.exists()) {
            lockF.deleteOnExit();
            deletionHookAdded = true;
        }
        RandomAccessFile file = new RandomAccessFile(lockF, "rws");
        String jvmName = ManagementFactory.getRuntimeMXBean().getName();
        try {
            res = file.getChannel().tryLock();
            if (null == res) {
                LOG.error("Unable to acquire file lock on path " + lockF.toString());
                throw new OverlappingFileLockException();
            }
            file.write(jvmName.getBytes(StandardCharsets.UTF_8));
            LOG.info("Lock on " + lockF + " acquired by nodename " + jvmName);
        }
        catch (OverlappingFileLockException oe) {
            LOG.error("It appears that another process has already locked the storage directory: " + this.root, (Throwable)oe);
            file.close();
            throw new IOException("Failed to lock storage " + this.root + ". The directory is already locked", oe);
        }
        catch (IOException e) {
            LOG.error("Failed to acquire lock on " + lockF + ". If this storage directory is mounted via NFS, ensure that the appropriate nfs lock services are running.", (Throwable)e);
            file.close();
            throw e;
        }
        if (!deletionHookAdded) {
            lockF.deleteOnExit();
        }
        return res;
    }

    public void unlock() throws IOException {
        if (this.lock == null) {
            return;
        }
        this.lock.release();
        this.lock.channel().close();
        this.lock = null;
    }

    public String toString() {
        return "Storage Directory " + this.root;
    }

    public static class LogPathAndIndex {
        public final Path path;
        public final long startIndex;
        public final long endIndex;

        LogPathAndIndex(Path path, long startIndex, long endIndex) {
            this.path = path;
            this.startIndex = startIndex;
            this.endIndex = endIndex;
        }

        public String toString() {
            return this.path + "-" + this.startIndex + "-" + this.endIndex;
        }
    }

    static enum StorageState {
        NON_EXISTENT,
        NOT_FORMATTED,
        NORMAL;

    }
}

