/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.Consumer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.CacheInvalidationPolicy;
import org.apache.ratis.server.storage.LogSegment;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.RaftStorageDirectory;
import org.apache.ratis.shaded.proto.RaftProtos;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RaftLogCache {
    public static final Logger LOG = LoggerFactory.getLogger(RaftLogCache.class);
    private final String name;
    private volatile LogSegment openSegment;
    private final List<LogSegment> closedSegments;
    private final RaftStorage storage;
    private final int maxCachedSegments;
    private final CacheInvalidationPolicy evictionPolicy = new CacheInvalidationPolicy.CacheInvalidationPolicyDefault();

    RaftLogCache(RaftPeerId selfId, RaftStorage storage, RaftProperties properties) {
        this.name = selfId + "-" + this.getClass().getSimpleName();
        this.storage = storage;
        this.maxCachedSegments = RaftServerConfigKeys.Log.maxCachedSegmentNum(properties);
        this.closedSegments = new ArrayList<LogSegment>();
    }

    int getMaxCachedSegments() {
        return this.maxCachedSegments;
    }

    void loadSegment(RaftStorageDirectory.LogPathAndIndex pi, boolean isOpen, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        LogSegment logSegment = LogSegment.loadSegment(this.storage, pi.path.toFile(), pi.startIndex, pi.endIndex, isOpen, keepEntryInCache, logConsumer);
        this.addSegment(logSegment);
    }

    long getCachedSegmentNum() {
        return this.closedSegments.stream().filter(LogSegment::hasCache).count();
    }

    boolean shouldEvict() {
        return this.getCachedSegmentNum() > (long)this.maxCachedSegments;
    }

    void evictCache(long[] followerIndices, long flushedIndex, long lastAppliedIndex) {
        List<LogSegment> toEvict = this.evictionPolicy.evict(followerIndices, flushedIndex, lastAppliedIndex, this.closedSegments, this.maxCachedSegments);
        for (LogSegment s : toEvict) {
            s.evictCache();
        }
    }

    private LogSegment getLastClosedSegment() {
        return this.closedSegments.isEmpty() ? null : this.closedSegments.get(this.closedSegments.size() - 1);
    }

    private void validateAdding(LogSegment segment) {
        LogSegment lastClosed = this.getLastClosedSegment();
        if (lastClosed != null) {
            Preconditions.assertTrue((!lastClosed.isOpen() ? 1 : 0) != 0);
            Preconditions.assertTrue((lastClosed.getEndIndex() + 1L == segment.getStartIndex() ? 1 : 0) != 0);
        }
    }

    void addSegment(LogSegment segment) {
        this.validateAdding(segment);
        if (segment.isOpen()) {
            this.setOpenSegment(segment);
        } else {
            this.closedSegments.add(segment);
        }
    }

    void addOpenSegment(long startIndex) {
        this.setOpenSegment(LogSegment.newOpenSegment(this.storage, startIndex));
    }

    private void setOpenSegment(LogSegment openSegment) {
        LOG.trace("{}: setOpenSegment to {}", (Object)this.name, (Object)openSegment);
        Preconditions.assertTrue((this.openSegment == null ? 1 : 0) != 0);
        this.openSegment = Objects.requireNonNull(openSegment);
    }

    private void clearOpenSegment() {
        LOG.trace("{}: clearOpenSegment {}", (Object)this.name, (Object)this.openSegment);
        Objects.requireNonNull(this.openSegment);
        this.openSegment = null;
    }

    LogSegment getOpenSegment() {
        return this.openSegment;
    }

    void rollOpenSegment(boolean createNewOpen) {
        Preconditions.assertTrue((this.openSegment != null && this.openSegment.numOfEntries() > 0 ? 1 : 0) != 0);
        long nextIndex = this.openSegment.getEndIndex() + 1L;
        this.openSegment.close();
        this.closedSegments.add(this.openSegment);
        this.clearOpenSegment();
        if (createNewOpen) {
            this.addOpenSegment(nextIndex);
        }
    }

    LogSegment getSegment(long index) {
        if (this.openSegment != null && index >= this.openSegment.getStartIndex()) {
            return this.openSegment;
        }
        int segmentIndex = Collections.binarySearch(this.closedSegments, index);
        return segmentIndex < 0 ? null : this.closedSegments.get(segmentIndex);
    }

    LogSegment.LogRecord getLogRecord(long index) {
        LogSegment segment = this.getSegment(index);
        return segment == null ? null : segment.getLogRecord(index);
    }

    TermIndex[] getTermIndices(long startIndex, long endIndex) {
        if (startIndex < 0L || startIndex < this.getStartIndex()) {
            throw new IndexOutOfBoundsException("startIndex = " + startIndex + ", log cache starts from index " + this.getStartIndex());
        }
        if (startIndex > endIndex) {
            throw new IndexOutOfBoundsException("startIndex(" + startIndex + ") > endIndex(" + endIndex + ")");
        }
        long realEnd = Math.min(this.getEndIndex() + 1L, endIndex);
        if (startIndex >= realEnd) {
            return TermIndex.EMPTY_TERMINDEX_ARRAY;
        }
        TermIndex[] entries = new TermIndex[Math.toIntExact(realEnd - startIndex)];
        int segmentIndex = Collections.binarySearch(this.closedSegments, startIndex);
        if (segmentIndex < 0) {
            this.getFromSegment(this.openSegment, startIndex, entries, 0, entries.length);
        } else {
            int numberFromSegment;
            long index = startIndex;
            for (int i = segmentIndex; i < this.closedSegments.size() && index < realEnd; index += (long)numberFromSegment, ++i) {
                LogSegment s = this.closedSegments.get(i);
                numberFromSegment = Math.toIntExact(Math.min(realEnd - index, s.getEndIndex() - index + 1L));
                this.getFromSegment(s, index, entries, Math.toIntExact(index - startIndex), numberFromSegment);
            }
            if (index < realEnd) {
                this.getFromSegment(this.openSegment, index, entries, Math.toIntExact(index - startIndex), Math.toIntExact(realEnd - index));
            }
        }
        return entries;
    }

    private void getFromSegment(LogSegment segment, long startIndex, TermIndex[] entries, int offset, int size) {
        long endIndex = segment.getEndIndex();
        endIndex = Math.min(endIndex, startIndex + (long)size - 1L);
        int index = offset;
        for (long i = startIndex; i <= endIndex; ++i) {
            LogSegment.LogRecord r = segment.getLogRecord(i);
            entries[index++] = r == null ? null : r.getTermIndex();
        }
    }

    boolean isConfigEntry(TermIndex ti) {
        LogSegment segment = this.getSegment(ti.getIndex());
        return segment != null && segment.isConfigEntry(ti);
    }

    long getStartIndex() {
        if (this.closedSegments.isEmpty()) {
            return this.openSegment != null ? this.openSegment.getStartIndex() : -1L;
        }
        return this.closedSegments.get(0).getStartIndex();
    }

    long getEndIndex() {
        return this.openSegment != null ? this.openSegment.getEndIndex() : (this.closedSegments.isEmpty() ? -1L : this.closedSegments.get(this.closedSegments.size() - 1).getEndIndex());
    }

    TermIndex getLastTermIndex() {
        return this.openSegment != null && this.openSegment.numOfEntries() > 0 ? this.openSegment.getLastTermIndex() : (this.closedSegments.isEmpty() ? null : this.closedSegments.get(this.closedSegments.size() - 1).getLastTermIndex());
    }

    void appendEntry(RaftProtos.LogEntryProto entry) {
        Preconditions.assertTrue((this.openSegment != null ? 1 : 0) != 0);
        this.openSegment.appendToOpenSegment(entry);
    }

    private SegmentFileInfo deleteOpenSegment() {
        long oldEnd = this.openSegment.getEndIndex();
        this.openSegment.clear();
        SegmentFileInfo info = new SegmentFileInfo(this.openSegment.getStartIndex(), oldEnd, true, 0L, this.openSegment.getEndIndex());
        this.clearOpenSegment();
        return info;
    }

    TruncationSegments truncate(long index) {
        int segmentIndex = Collections.binarySearch(this.closedSegments, index);
        if (segmentIndex == -this.closedSegments.size() - 1) {
            if (this.openSegment != null && this.openSegment.getEndIndex() >= index) {
                long oldEnd = this.openSegment.getEndIndex();
                if (index == this.openSegment.getStartIndex()) {
                    return new TruncationSegments(null, Collections.singletonList(this.deleteOpenSegment()));
                }
                this.openSegment.truncate(index);
                Preconditions.assertTrue((!this.openSegment.isOpen() ? 1 : 0) != 0);
                SegmentFileInfo info = new SegmentFileInfo(this.openSegment.getStartIndex(), oldEnd, true, this.openSegment.getTotalSize(), this.openSegment.getEndIndex());
                this.closedSegments.add(this.openSegment);
                this.clearOpenSegment();
                return new TruncationSegments(info, Collections.emptyList());
            }
        } else if (segmentIndex >= 0) {
            LogSegment ts = this.closedSegments.get(segmentIndex);
            long oldEnd = ts.getEndIndex();
            ArrayList<SegmentFileInfo> list = new ArrayList<SegmentFileInfo>();
            ts.truncate(index);
            int size = this.closedSegments.size();
            for (int i = size - 1; i >= (ts.numOfEntries() == 0 ? segmentIndex : segmentIndex + 1); --i) {
                LogSegment s = this.closedSegments.remove(i);
                long endOfS = i == segmentIndex ? oldEnd : s.getEndIndex();
                s.clear();
                list.add(new SegmentFileInfo(s.getStartIndex(), endOfS, false, 0L, s.getEndIndex()));
            }
            if (this.openSegment != null) {
                list.add(this.deleteOpenSegment());
            }
            SegmentFileInfo t = ts.numOfEntries() == 0 ? null : new SegmentFileInfo(ts.getStartIndex(), oldEnd, false, ts.getTotalSize(), ts.getEndIndex());
            return new TruncationSegments(t, list);
        }
        return null;
    }

    Iterator<TermIndex> iterator(long startIndex) {
        return new EntryIterator(startIndex);
    }

    int getNumOfSegments() {
        return this.closedSegments.size() + (this.openSegment == null ? 0 : 1);
    }

    boolean isEmpty() {
        return this.closedSegments.isEmpty() && this.openSegment == null;
    }

    void clear() {
        if (this.openSegment != null) {
            this.openSegment.clear();
            this.clearOpenSegment();
        }
        this.closedSegments.forEach(LogSegment::clear);
        this.closedSegments.clear();
    }

    private class EntryIterator
    implements Iterator<TermIndex> {
        private long nextIndex;
        private LogSegment currentSegment;
        private int segmentIndex;

        EntryIterator(long start) {
            this.nextIndex = start;
            this.segmentIndex = Collections.binarySearch(RaftLogCache.this.closedSegments, this.nextIndex);
            if (this.segmentIndex >= 0) {
                this.currentSegment = (LogSegment)RaftLogCache.this.closedSegments.get(this.segmentIndex);
            } else {
                this.segmentIndex = -this.segmentIndex - 1;
                if (this.segmentIndex == RaftLogCache.this.closedSegments.size()) {
                    this.currentSegment = RaftLogCache.this.openSegment;
                } else {
                    Preconditions.assertTrue((this.segmentIndex == 0 ? 1 : 0) != 0);
                    throw new IndexOutOfBoundsException();
                }
            }
        }

        @Override
        public boolean hasNext() {
            return this.currentSegment != null && this.currentSegment.getLogRecord(this.nextIndex) != null;
        }

        @Override
        public TermIndex next() {
            LogSegment.LogRecord record;
            if (this.currentSegment == null || (record = this.currentSegment.getLogRecord(this.nextIndex)) == null) {
                throw new NoSuchElementException();
            }
            if (++this.nextIndex > this.currentSegment.getEndIndex() && this.currentSegment != RaftLogCache.this.openSegment) {
                ++this.segmentIndex;
                this.currentSegment = this.segmentIndex == RaftLogCache.this.closedSegments.size() ? RaftLogCache.this.openSegment : (LogSegment)RaftLogCache.this.closedSegments.get(this.segmentIndex);
            }
            return record.getTermIndex();
        }
    }

    static class TruncationSegments {
        final SegmentFileInfo toTruncate;
        final SegmentFileInfo[] toDelete;

        TruncationSegments(SegmentFileInfo toTruncate, List<SegmentFileInfo> toDelete) {
            this.toDelete = toDelete == null ? null : toDelete.toArray(new SegmentFileInfo[toDelete.size()]);
            this.toTruncate = toTruncate;
        }

        public String toString() {
            return "toTruncate: " + this.toTruncate + "\n  toDelete: " + Arrays.toString(this.toDelete);
        }
    }

    static class SegmentFileInfo {
        final long startIndex;
        final long endIndex;
        final boolean isOpen;
        final long targetLength;
        final long newEndIndex;

        SegmentFileInfo(long start, long end, boolean isOpen, long targetLength, long newEndIndex) {
            this.startIndex = start;
            this.endIndex = end;
            this.isOpen = isOpen;
            this.targetLength = targetLength;
            this.newEndIndex = newEndIndex;
        }

        public String toString() {
            return "(" + this.startIndex + ", " + this.endIndex + ") isOpen? " + this.isOpen + ", length=" + this.targetLength + ", newEndIndex=" + this.newEndIndex;
        }
    }
}

