/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.impl.RaftConfiguration;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerState;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.shaded.com.google.protobuf.ByteString;
import org.apache.ratis.shaded.proto.RaftProtos;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ProtoUtils;
import org.apache.ratis.util.Timestamp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LeaderElection
extends Daemon {
    public static final Logger LOG = LoggerFactory.getLogger(LeaderElection.class);
    private final RaftServerImpl server;
    private ExecutorCompletionService<RaftProtos.RequestVoteReplyProto> service;
    private ExecutorService executor;
    private volatile boolean running;
    private final RaftConfiguration conf;
    private final Collection<RaftPeer> others;

    private ResultAndTerm logAndReturn(Result result, List<RaftProtos.RequestVoteReplyProto> responses, List<Exception> exceptions, long newTerm) {
        LOG.info(this.server.getId() + ": Election " + (Object)((Object)result) + "; received " + responses.size() + " response(s) " + responses.stream().map(ProtoUtils::toString).collect(Collectors.toList()) + " and " + exceptions.size() + " exception(s); " + this.server.getState());
        int i = 0;
        for (Exception e : exceptions) {
            LOG.info("  " + i++ + ": " + e);
            LOG.trace("TRACE", (Throwable)e);
        }
        return new ResultAndTerm(result, newTerm);
    }

    LeaderElection(RaftServerImpl server) {
        this.server = server;
        this.conf = server.getRaftConf();
        this.others = this.conf.getOtherPeers(server.getId());
        this.running = true;
    }

    void stopRunning() {
        this.running = false;
    }

    private void initExecutor() {
        Preconditions.assertTrue((!this.others.isEmpty() ? 1 : 0) != 0);
        this.executor = Executors.newFixedThreadPool(this.others.size(), Daemon::new);
        this.service = new ExecutorCompletionService(this.executor);
    }

    public void run() {
        try {
            this.askForVotes();
        }
        catch (InterruptedException e) {
            LOG.info(this.server.getId() + " " + ((Object)((Object)this)).getClass().getSimpleName() + " thread is interrupted gracefully; server=" + this.server);
        }
        catch (IOException e) {
            LOG.warn("Failed to persist votedFor/term. Exit the leader election.", (Throwable)e);
            this.stopRunning();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void askForVotes() throws InterruptedException, IOException {
        ServerState state = this.server.getState();
        while (this.running && this.server.isCandidate()) {
            ResultAndTerm r;
            SnapshotInfo snapshot;
            long electionTerm;
            RaftServerImpl raftServerImpl = this.server;
            synchronized (raftServerImpl) {
                electionTerm = state.initElection();
                this.server.getState().persistMetadata();
            }
            LOG.info(state.getSelfId() + ": begin an election in Term " + electionTerm);
            TermIndex lastEntry = state.getLog().getLastEntryTermIndex();
            if (lastEntry == null && (snapshot = state.getLatestSnapshot()) != null) {
                lastEntry = snapshot.getTermIndex();
            }
            if (this.others.isEmpty()) {
                r = new ResultAndTerm(Result.PASSED, electionTerm);
            } else {
                try {
                    this.initExecutor();
                    int submitted = this.submitRequests(electionTerm, lastEntry);
                    r = this.waitForResults(electionTerm, submitted);
                }
                finally {
                    if (this.executor != null) {
                        this.executor.shutdown();
                    }
                }
            }
            RaftServerImpl raftServerImpl2 = this.server;
            synchronized (raftServerImpl2) {
                if (electionTerm != state.getCurrentTerm() || !this.running || !this.server.isCandidate()) {
                    return;
                }
                switch (r.result) {
                    case PASSED: {
                        this.server.changeToLeader();
                        return;
                    }
                    case SHUTDOWN: {
                        LOG.info("{} received shutdown response when requesting votes.", (Object)this.server.getId());
                        this.server.getProxy().close();
                        return;
                    }
                    case REJECTED: 
                    case DISCOVERED_A_NEW_TERM: {
                        long term = r.term > this.server.getState().getCurrentTerm() ? r.term : this.server.getState().getCurrentTerm();
                        this.server.changeToFollower(term, true);
                        return;
                    }
                }
            }
        }
    }

    private int submitRequests(long electionTerm, TermIndex lastEntry) {
        int submitted = 0;
        for (RaftPeer peer : this.others) {
            RaftProtos.RequestVoteRequestProto r = this.server.createRequestVoteRequest(peer.getId(), electionTerm, lastEntry);
            this.service.submit(() -> this.server.getServerRpc().requestVote(r));
            ++submitted;
        }
        return submitted;
    }

    private ResultAndTerm waitForResults(long electionTerm, int submitted) throws InterruptedException {
        Timestamp timeout = new Timestamp().addTimeMs((long)this.server.getRandomTimeoutMs());
        ArrayList<RaftProtos.RequestVoteReplyProto> responses = new ArrayList<RaftProtos.RequestVoteReplyProto>();
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        int waitForNum = submitted;
        ArrayList<RaftPeerId> votedPeers = new ArrayList<RaftPeerId>();
        while (waitForNum > 0 && this.running && this.server.isCandidate()) {
            long waitTime = -timeout.elapsedTimeMs();
            if (waitTime <= 0L) {
                return this.logAndReturn(Result.TIMEOUT, responses, exceptions, -1L);
            }
            try {
                Future<RaftProtos.RequestVoteReplyProto> future = this.service.poll(waitTime, TimeUnit.MILLISECONDS);
                if (future == null) continue;
                RaftProtos.RequestVoteReplyProto r = future.get();
                responses.add(r);
                if (r.getShouldShutdown()) {
                    return this.logAndReturn(Result.SHUTDOWN, responses, exceptions, -1L);
                }
                if (r.getTerm() > electionTerm) {
                    return this.logAndReturn(Result.DISCOVERED_A_NEW_TERM, responses, exceptions, r.getTerm());
                }
                if (r.getServerReply().getSuccess()) {
                    votedPeers.add(RaftPeerId.valueOf((ByteString)r.getServerReply().getReplyId()));
                    if (this.conf.hasMajority(votedPeers, this.server.getId())) {
                        return this.logAndReturn(Result.PASSED, responses, exceptions, -1L);
                    }
                }
            }
            catch (ExecutionException e) {
                LOG.info("{} got exception when requesting votes: {}", (Object)this.server.getId(), (Object)e);
                LOG.trace("TRACE", (Throwable)e);
                exceptions.add(e);
            }
            --waitForNum;
        }
        return this.logAndReturn(Result.REJECTED, responses, exceptions, -1L);
    }

    private static class ResultAndTerm {
        final Result result;
        final long term;

        ResultAndTerm(Result result, long term) {
            this.result = result;
            this.term = term;
        }
    }

    static enum Result {
        PASSED,
        REJECTED,
        TIMEOUT,
        DISCOVERED_A_NEW_TERM,
        SHUTDOWN;

    }
}

