/*
 * Decompiled with CFR 0.152.
 */
package org.apache.qpid.server.security;

import com.google.common.collect.Sets;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import org.apache.qpid.server.configuration.IllegalConfigurationException;
import org.apache.qpid.server.logging.EventLogger;
import org.apache.qpid.server.logging.messages.TrustStoreMessages;
import org.apache.qpid.server.model.AbstractConfigurationChangeListener;
import org.apache.qpid.server.model.AbstractConfiguredObject;
import org.apache.qpid.server.model.Broker;
import org.apache.qpid.server.model.ConfiguredObject;
import org.apache.qpid.server.model.ManagedAttributeField;
import org.apache.qpid.server.model.State;
import org.apache.qpid.server.model.TrustStore;
import org.apache.qpid.server.model.VirtualHostNode;
import org.apache.qpid.server.security.CertificateDetails;
import org.apache.qpid.server.security.CertificateDetailsImpl;
import org.apache.qpid.server.security.TrustAnchorValidatingTrustManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractTrustStore<X extends AbstractTrustStore<X>>
extends AbstractConfiguredObject<X>
implements TrustStore<X> {
    private static Logger LOGGER = LoggerFactory.getLogger(AbstractTrustStore.class);
    protected static final long ONE_DAY = 86400000L;
    private final Broker<?> _broker;
    private final EventLogger _eventLogger;
    @ManagedAttributeField
    private boolean _exposedAsMessageSource;
    @ManagedAttributeField
    private List<VirtualHostNode<?>> _includedVirtualHostNodeMessageSources;
    @ManagedAttributeField
    private List<VirtualHostNode<?>> _excludedVirtualHostNodeMessageSources;
    @ManagedAttributeField
    private boolean _trustAnchorValidityEnforced;
    private ScheduledFuture<?> _checkExpiryTaskFuture;

    AbstractTrustStore(Map<String, Object> attributes, Broker<?> broker) {
        super(broker, attributes);
        this._broker = broker;
        this._eventLogger = broker.getEventLogger();
        this._eventLogger.message(TrustStoreMessages.CREATE(this.getName()));
    }

    public final Broker<?> getBroker() {
        return this._broker;
    }

    final EventLogger getEventLogger() {
        return this._eventLogger;
    }

    @Override
    protected ListenableFuture<Void> onClose() {
        this.onCloseOrDelete();
        return Futures.immediateFuture(null);
    }

    private void onCloseOrDelete() {
        if (this._checkExpiryTaskFuture != null) {
            this._checkExpiryTaskFuture.cancel(false);
            this._checkExpiryTaskFuture = null;
        }
    }

    @Override
    protected void logOperation(String operation) {
        this._broker.getEventLogger().message(TrustStoreMessages.OPERATION(operation));
    }

    void initializeExpiryChecking() {
        int checkFrequency = this.getCertificateExpiryCheckFrequency();
        if (this.getBroker().getState() == State.ACTIVE) {
            this._checkExpiryTaskFuture = this.getBroker().scheduleHouseKeepingTask(checkFrequency, TimeUnit.DAYS, this::checkCertificateExpiry);
        } else {
            final int frequency = checkFrequency;
            this.getBroker().addChangeListener(new AbstractConfigurationChangeListener(){

                @Override
                public void stateChanged(ConfiguredObject<?> object, State oldState, State newState) {
                    if (newState == State.ACTIVE) {
                        AbstractTrustStore.this._checkExpiryTaskFuture = AbstractTrustStore.this.getBroker().scheduleHouseKeepingTask(frequency, TimeUnit.DAYS, () -> AbstractTrustStore.this.checkCertificateExpiry());
                        AbstractTrustStore.this.getBroker().removeChangeListener(this);
                    }
                }
            });
        }
    }

    @Override
    protected ListenableFuture<Void> onDelete() {
        this.onCloseOrDelete();
        this._eventLogger.message(TrustStoreMessages.DELETE(this.getName()));
        return super.onDelete();
    }

    private void checkCertificateExpiry() {
        int expiryWarning = this.getCertificateExpiryWarnPeriod();
        if (expiryWarning > 0) {
            long currentTime = System.currentTimeMillis();
            Date expiryTestDate = new Date(currentTime + 86400000L * (long)expiryWarning);
            try {
                Certificate[] certificatesInternal = this.getCertificates();
                if (certificatesInternal.length > 0) {
                    Arrays.stream(certificatesInternal).filter(cert -> cert instanceof X509Certificate).forEach(x509cert -> this.checkCertificateExpiry(currentTime, expiryTestDate, (X509Certificate)x509cert));
                }
            }
            catch (GeneralSecurityException e) {
                LOGGER.debug("Unexpected exception whilst checking certificate expiry", (Throwable)e);
            }
        }
    }

    private void checkCertificateExpiry(long currentTime, Date expiryTestDate, X509Certificate cert) {
        try {
            cert.checkValidity(expiryTestDate);
        }
        catch (CertificateExpiredException e) {
            long timeToExpiry = cert.getNotAfter().getTime() - currentTime;
            int days = Math.max(0, (int)(timeToExpiry / 86400000L));
            this.getEventLogger().message(TrustStoreMessages.EXPIRING(this.getName(), String.valueOf(days), cert.getSubjectDN().toString()));
        }
        catch (CertificateNotYetValidException certificateNotYetValidException) {
            // empty catch block
        }
    }

    @Override
    public final TrustManager[] getTrustManagers() throws GeneralSecurityException {
        if (this.isTrustAnchorValidityEnforced()) {
            HashSet trustManagerCerts = Sets.newHashSet((Object[])this.getCertificates());
            HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
            HashSet<Certificate> otherCerts = new HashSet<Certificate>();
            for (Certificate certs : trustManagerCerts) {
                if (certs instanceof X509Certificate && this.isSelfSigned((X509Certificate)certs)) {
                    trustAnchors.add(new TrustAnchor((X509Certificate)certs, null));
                    continue;
                }
                otherCerts.add(certs);
            }
            TrustManager[] trustManagers = this.getTrustManagersInternal();
            TrustManager[] wrappedTrustManagers = new TrustManager[trustManagers.length];
            for (int i = 0; i < trustManagers.length; ++i) {
                TrustManager trustManager = trustManagers[i];
                wrappedTrustManagers[i] = trustManager instanceof X509TrustManager ? new TrustAnchorValidatingTrustManager(this.getName(), (X509TrustManager)trustManager, trustAnchors, otherCerts) : trustManager;
            }
            return wrappedTrustManagers;
        }
        return this.getTrustManagersInternal();
    }

    protected abstract TrustManager[] getTrustManagersInternal() throws GeneralSecurityException;

    @Override
    public final int getCertificateExpiryWarnPeriod() {
        try {
            return this.getContextValue(Integer.class, "qpid.truststore.certificateExpiryWarnPeriod");
        }
        catch (IllegalArgumentException | NullPointerException e) {
            LOGGER.warn("The value of the context variable '{}' for truststore {} cannot be converted to an integer. The value {} will be used as a default", new Object[]{"qpid.truststore.certificateExpiryWarnPeriod", this.getName(), 30});
            return 30;
        }
    }

    @Override
    public int getCertificateExpiryCheckFrequency() {
        int checkFrequency;
        try {
            checkFrequency = this.getContextValue(Integer.class, "qpid.truststore.certificateExpiryCheckFrequency");
        }
        catch (IllegalArgumentException | NullPointerException e) {
            LOGGER.warn("Cannot parse the context variable {} ", (Object)"qpid.truststore.certificateExpiryCheckFrequency", (Object)e);
            checkFrequency = 1;
        }
        return checkFrequency;
    }

    @Override
    public boolean isTrustAnchorValidityEnforced() {
        return this._trustAnchorValidityEnforced;
    }

    @Override
    public boolean isExposedAsMessageSource() {
        return this._exposedAsMessageSource;
    }

    @Override
    public List<VirtualHostNode<?>> getIncludedVirtualHostNodeMessageSources() {
        return this._includedVirtualHostNodeMessageSources;
    }

    @Override
    public List<VirtualHostNode<?>> getExcludedVirtualHostNodeMessageSources() {
        return this._excludedVirtualHostNodeMessageSources;
    }

    @Override
    public List<CertificateDetails> getCertificateDetails() {
        try {
            Certificate[] certificatesInternal = this.getCertificates();
            if (certificatesInternal.length > 0) {
                return Arrays.stream(certificatesInternal).filter(cert -> cert instanceof X509Certificate).map(x509cert -> new CertificateDetailsImpl((X509Certificate)x509cert)).collect(Collectors.toList());
            }
            return Collections.emptyList();
        }
        catch (GeneralSecurityException e) {
            throw new IllegalConfigurationException("Failed to extract certificate details", e);
        }
    }

    private boolean isSelfSigned(X509Certificate cert) throws GeneralSecurityException {
        try {
            PublicKey key = cert.getPublicKey();
            cert.verify(key);
            return true;
        }
        catch (InvalidKeyException | SignatureException e) {
            return false;
        }
    }
}

