/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.authentication.oidc;

import com.auth0.jwk.Jwk;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.github.benmanes.caffeine.cache.AsyncCacheLoader;
import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.kubernetes.client.openapi.ApiCallback;
import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.openapi.apis.OpenidApi;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import javax.naming.AuthenticationException;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.authentication.AuthenticationProvider;
import org.apache.pulsar.broker.authentication.oidc.AuthenticationExceptionCode;
import org.apache.pulsar.broker.authentication.oidc.ConfigUtils;
import org.asynchttpclient.AsyncHttpClient;

public class JwksCache {
    private final AsyncLoadingCache<Optional<String>, List<Jwk>> cache;
    private final ConcurrentHashMap<Optional<String>, Long> jwksLastRefreshTime = new ConcurrentHashMap();
    private final long keyIdCacheMissRefreshNanos;
    private final ObjectReader reader = new ObjectMapper().readerFor(HashMap.class);
    private final AsyncHttpClient httpClient;
    private final OpenidApi openidApi;
    private final AuthenticationProvider authenticationProvider;

    JwksCache(AuthenticationProvider authenticationProvider, ServiceConfiguration config, AsyncHttpClient httpClient, ApiClient apiClient) throws IOException {
        this.authenticationProvider = authenticationProvider;
        this.httpClient = httpClient;
        this.openidApi = apiClient != null ? new OpenidApi(apiClient) : null;
        this.keyIdCacheMissRefreshNanos = TimeUnit.SECONDS.toNanos(ConfigUtils.getConfigValueAsInt(config, "openIDKeyIdCacheMissRefreshSeconds", 300));
        int maxSize = ConfigUtils.getConfigValueAsInt(config, "openIDCacheSize", 5);
        int refreshAfterWriteSeconds = ConfigUtils.getConfigValueAsInt(config, "openIDCacheRefreshAfterWriteSeconds", 64800);
        int expireAfterSeconds = ConfigUtils.getConfigValueAsInt(config, "openIDCacheExpirationSeconds", 86400);
        AsyncCacheLoader loader = (jwksUri, executor) -> {
            this.jwksLastRefreshTime.put((Optional<String>)jwksUri, System.nanoTime());
            if (jwksUri.isPresent()) {
                return this.getJwksFromJwksUri((String)jwksUri.get());
            }
            return this.getJwksFromKubernetesApiServer();
        };
        this.cache = Caffeine.newBuilder().maximumSize((long)maxSize).refreshAfterWrite((long)refreshAfterWriteSeconds, TimeUnit.SECONDS).expireAfterWrite((long)expireAfterSeconds, TimeUnit.SECONDS).buildAsync(loader);
    }

    CompletableFuture<Jwk> getJwk(String jwksUri, String keyId) {
        if (jwksUri == null) {
            this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
            return CompletableFuture.failedFuture(new IllegalArgumentException("jwksUri must not be null."));
        }
        return this.getJwkAndMaybeReload(Optional.of(jwksUri), keyId, false);
    }

    private CompletableFuture<Jwk> getJwkAndMaybeReload(Optional<String> maybeJwksUri, String keyId, boolean failOnMissingKeyId) {
        return this.cache.get(maybeJwksUri).thenCompose(jwks -> {
            try {
                return CompletableFuture.completedFuture(this.getJwkForKID(maybeJwksUri, (List<Jwk>)jwks, keyId));
            }
            catch (IllegalArgumentException e) {
                if (failOnMissingKeyId) {
                    throw e;
                }
                Long lastRefresh = this.jwksLastRefreshTime.get(maybeJwksUri);
                if (lastRefresh == null || System.nanoTime() - lastRefresh > this.keyIdCacheMissRefreshNanos) {
                    this.cache.synchronous().invalidate((Object)maybeJwksUri);
                }
                return this.getJwkAndMaybeReload(maybeJwksUri, keyId, true);
            }
        });
    }

    private CompletableFuture<List<Jwk>> getJwksFromJwksUri(String jwksUri) {
        return this.httpClient.prepareGet(jwksUri).execute().toCompletableFuture().thenCompose(result -> {
            CompletableFuture<List<Jwk>> future = new CompletableFuture<List<Jwk>>();
            try {
                HashMap jwks = (HashMap)this.reader.readValue(result.getResponseBodyAsBytes());
                future.complete(this.convertToJwks(jwksUri, jwks));
            }
            catch (AuthenticationException e) {
                this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
                future.completeExceptionally(e);
            }
            catch (Exception e) {
                this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
                future.completeExceptionally(new AuthenticationException("Error retrieving public key at " + jwksUri + ": " + e.getMessage()));
            }
            return future;
        });
    }

    CompletableFuture<Jwk> getJwkFromKubernetesApiServer(String keyId) {
        if (this.openidApi == null) {
            this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
            return CompletableFuture.failedFuture(new AuthenticationException("Failed to retrieve public key from Kubernetes API server: Kubernetes fallback is not enabled."));
        }
        return this.getJwkAndMaybeReload(Optional.empty(), keyId, false);
    }

    private CompletableFuture<List<Jwk>> getJwksFromKubernetesApiServer() {
        final CompletableFuture<List<Jwk>> future = new CompletableFuture<List<Jwk>>();
        try {
            this.openidApi.getServiceAccountIssuerOpenIDKeysetAsync((ApiCallback)new ApiCallback<String>(){

                public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                    JwksCache.this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
                    future.completeExceptionally(new AuthenticationException("Failed to retrieve public key from Kubernetes API server. Message: " + e.getMessage() + " Response body: " + e.getResponseBody()));
                }

                public void onSuccess(String result, int statusCode, Map<String, List<String>> responseHeaders) {
                    try {
                        HashMap jwks = (HashMap)JwksCache.this.reader.readValue(result);
                        future.complete(JwksCache.this.convertToJwks("Kubernetes API server", jwks));
                    }
                    catch (AuthenticationException e) {
                        JwksCache.this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
                        future.completeExceptionally(e);
                    }
                    catch (Exception e) {
                        JwksCache.this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
                        future.completeExceptionally(new AuthenticationException("Error retrieving public key at Kubernetes API server: " + e.getMessage()));
                    }
                }

                public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {
                }

                public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {
                }
            });
        }
        catch (ApiException e) {
            this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
            future.completeExceptionally(new AuthenticationException("Failed to retrieve public key from Kubernetes API server: " + e.getMessage()));
        }
        return future;
    }

    private Jwk getJwkForKID(Optional<String> maybeJwksUri, List<Jwk> jwks, String keyId) {
        for (Jwk jwk : jwks) {
            if (!jwk.getId().equals(keyId)) continue;
            return jwk;
        }
        this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PUBLIC_KEY);
        throw new IllegalArgumentException("No JWK found for Key ID " + keyId);
    }

    private List<Jwk> convertToJwks(String jwksUri, Map<String, Object> jwks) throws AuthenticationException {
        try {
            List jwkList = (List)jwks.get("keys");
            ArrayList<Jwk> result = new ArrayList<Jwk>();
            for (Map jwk : jwkList) {
                result.add(Jwk.fromValues((Map)jwk));
            }
            return result;
        }
        catch (ClassCastException e) {
            throw new AuthenticationException("Malformed JWKS returned by: " + jwksUri);
        }
    }
}

