/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.function.BiConsumer;
import java.util.regex.Pattern;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.namespace.NamespaceService;
import org.apache.pulsar.broker.resources.TopicResources;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.ServerCnx;
import org.apache.pulsar.common.api.proto.CommandWatchTopicListClose;
import org.apache.pulsar.common.api.proto.ServerError;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.topics.TopicList;
import org.apache.pulsar.common.util.collections.ConcurrentLongHashMap;
import org.apache.pulsar.metadata.api.NotificationType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TopicListService {
    private static final Logger log = LoggerFactory.getLogger(TopicListService.class);
    private final NamespaceService namespaceService;
    private final TopicResources topicResources;
    private final ServerCnx connection;
    private final boolean enableSubscriptionPatternEvaluation;
    private final int maxSubscriptionPatternLength;
    private final ConcurrentLongHashMap<CompletableFuture<TopicListWatcher>> watchers;

    public TopicListService(PulsarService pulsar, ServerCnx connection, boolean enableSubscriptionPatternEvaluation, int maxSubscriptionPatternLength) {
        this.namespaceService = pulsar.getNamespaceService();
        this.connection = connection;
        this.enableSubscriptionPatternEvaluation = enableSubscriptionPatternEvaluation;
        this.maxSubscriptionPatternLength = maxSubscriptionPatternLength;
        this.watchers = ConcurrentLongHashMap.newBuilder().expectedItems(8).concurrencyLevel(1).build();
        this.topicResources = pulsar.getPulsarResources().getTopicResources();
    }

    public void inactivate() {
        for (Long watcherId : new HashSet(this.watchers.keys())) {
            this.deleteTopicListWatcher(watcherId);
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    public void handleWatchTopicList(NamespaceName namespaceName, long watcherId, long requestId, Pattern topicsPattern, String topicsHash, Semaphore lookupSemaphore) {
        CompletableFuture watcherFuture;
        block6: {
            if (!this.enableSubscriptionPatternEvaluation || topicsPattern.pattern().length() > this.maxSubscriptionPatternLength) {
                Object msg = "Unable to create topic list watcher: ";
                msg = !this.enableSubscriptionPatternEvaluation ? (String)msg + "Evaluating subscription patterns is disabled." : (String)msg + "Pattern longer than maximum: " + this.maxSubscriptionPatternLength;
                log.warn("[{}] {} on namespace {}", new Object[]{this.connection.toString(), msg, namespaceName});
                this.connection.getCommandSender().sendErrorResponse(requestId, ServerError.NotAllowedError, (String)msg);
                lookupSemaphore.release();
                return;
            }
            watcherFuture = new CompletableFuture();
            CompletableFuture existingWatcherFuture = (CompletableFuture)this.watchers.putIfAbsent(watcherId, watcherFuture);
            if (existingWatcherFuture != null) {
                if (existingWatcherFuture.isDone() && !existingWatcherFuture.isCompletedExceptionally()) {
                    TopicListWatcher watcher2 = existingWatcherFuture.getNow(null);
                    log.info("[{}] Watcher with the same id is already created: watcherId={}, watcher={}", new Object[]{this.connection.toString(), watcherId, watcher2});
                    watcherFuture = existingWatcherFuture;
                    break block6;
                } else {
                    ServerError error;
                    log.warn("[{}] Watcher with id is already present on the connection, consumerId={}", (Object)this.connection.toString(), (Object)watcherId);
                    if (!existingWatcherFuture.isDone()) {
                        error = ServerError.ServiceNotReady;
                    } else {
                        error = ServerError.UnknownError;
                        this.watchers.remove(watcherId, (Object)existingWatcherFuture);
                    }
                    this.connection.getCommandSender().sendErrorResponse(requestId, error, "Topic list watcher is already present on the connection");
                    lookupSemaphore.release();
                    return;
                }
            }
            this.initializeTopicsListWatcher(watcherFuture, namespaceName, watcherId, topicsPattern);
        }
        CompletableFuture finalWatcherFuture = watcherFuture;
        ((CompletableFuture)finalWatcherFuture.thenAccept(watcher -> {
            List<String> topicList = watcher.getMatchingTopics();
            String hash = TopicList.calculateHash(topicList);
            if (hash.equals(topicsHash)) {
                topicList = Collections.emptyList();
            }
            if (log.isDebugEnabled()) {
                log.debug("[{}] Received WatchTopicList for namespace [//{}] by {}", new Object[]{this.connection.toString(), namespaceName, requestId});
            }
            this.connection.getCommandSender().sendWatchTopicListSuccess(requestId, watcherId, hash, topicList);
            lookupSemaphore.release();
        })).exceptionally(ex -> {
            log.warn("[{}] Error WatchTopicList for namespace [//{}] by {}", new Object[]{this.connection.toString(), namespaceName, requestId});
            this.connection.getCommandSender().sendErrorResponse(requestId, BrokerServiceException.getClientErrorCode(new BrokerServiceException.ServerMetadataException((Throwable)ex)), ex.getMessage());
            this.watchers.remove(watcherId, (Object)finalWatcherFuture);
            lookupSemaphore.release();
            return null;
        });
    }

    public void initializeTopicsListWatcher(CompletableFuture<TopicListWatcher> watcherFuture, NamespaceName namespace, long watcherId, Pattern topicsPattern) {
        ((CompletableFuture)this.namespaceService.getListOfPersistentTopics(namespace).thenApply(topics -> {
            TopicListWatcher watcher = new TopicListWatcher(this, watcherId, topicsPattern, (List<String>)topics);
            this.topicResources.registerPersistentTopicListener(namespace, (BiConsumer)watcher);
            return watcher;
        })).whenComplete((watcher, exception) -> {
            if (exception != null) {
                watcherFuture.completeExceptionally((Throwable)exception);
            } else if (!watcherFuture.complete((TopicListWatcher)watcher)) {
                log.warn("[{}] Watcher future was already completed. Deregistering watcherId={}.", (Object)this.connection.toString(), (Object)watcherId);
                this.topicResources.deregisterPersistentTopicListener((BiConsumer)watcher);
            }
        });
    }

    public void handleWatchTopicListClose(CommandWatchTopicListClose commandWatchTopicListClose) {
        long requestId = commandWatchTopicListClose.getRequestId();
        long watcherId = commandWatchTopicListClose.getWatcherId();
        this.deleteTopicListWatcher(watcherId);
        this.connection.getCommandSender().sendSuccessResponse(requestId);
    }

    public void deleteTopicListWatcher(Long watcherId) {
        CompletableFuture watcherFuture = (CompletableFuture)this.watchers.get(watcherId.longValue());
        if (watcherFuture == null) {
            log.info("[{}] TopicListWatcher was not registered on the connection: {}", (Object)watcherId, (Object)this.connection.toString());
            return;
        }
        if (!watcherFuture.isDone() && watcherFuture.completeExceptionally(new IllegalStateException("Closed watcher before creation was complete"))) {
            log.info("[{}] Closed watcher before its creation was completed. watcherId={}", (Object)this.connection.toString(), (Object)watcherId);
            this.watchers.remove(watcherId.longValue());
            return;
        }
        if (watcherFuture.isCompletedExceptionally()) {
            log.info("[{}] Closed watcher that already failed to be created. watcherId={}", (Object)this.connection.toString(), (Object)watcherId);
            this.watchers.remove(watcherId.longValue());
            return;
        }
        this.topicResources.deregisterPersistentTopicListener((BiConsumer)watcherFuture.getNow(null));
        this.watchers.remove(watcherId.longValue());
        log.info("[{}] Closed watcher, watcherId={}", (Object)this.connection.toString(), (Object)watcherId);
    }

    public void sendTopicListUpdate(long watcherId, String topicsHash, List<String> deletedTopics, List<String> newTopics) {
        this.connection.getCommandSender().sendWatchTopicListUpdate(watcherId, newTopics, deletedTopics, topicsHash);
    }

    public static class TopicListWatcher
    implements BiConsumer<String, NotificationType> {
        private final List<String> matchingTopics;
        private final TopicListService topicListService;
        private final long id;
        private final Pattern topicsPattern;

        public TopicListWatcher(TopicListService topicListService, long id, Pattern topicsPattern, List<String> topics) {
            this.topicListService = topicListService;
            this.id = id;
            this.topicsPattern = topicsPattern;
            this.matchingTopics = TopicList.filterTopics(topics, (Pattern)topicsPattern);
        }

        public List<String> getMatchingTopics() {
            return this.matchingTopics;
        }

        @Override
        public void accept(String topicName, NotificationType notificationType) {
            if (this.topicsPattern.matcher(TopicName.get((String)topicName).getPartitionedTopicName()).matches()) {
                List<String> deletedTopics;
                List<String> newTopics;
                if (notificationType == NotificationType.Deleted) {
                    newTopics = Collections.emptyList();
                    deletedTopics = Collections.singletonList(topicName);
                    this.matchingTopics.remove(topicName);
                } else {
                    deletedTopics = Collections.emptyList();
                    newTopics = Collections.singletonList(topicName);
                    this.matchingTopics.add(topicName);
                }
                String hash = TopicList.calculateHash(this.matchingTopics);
                this.topicListService.sendTopicListUpdate(this.id, hash, deletedTopics, newTopics);
            }
        }
    }
}

