/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.loadbalance.extensions;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import org.apache.pulsar.broker.PulsarServerException;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.loadbalance.extensions.BrokerRegistry;
import org.apache.pulsar.broker.loadbalance.extensions.data.BrokerLookupData;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.metadata.api.MetadataCache;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.apache.pulsar.metadata.api.Notification;
import org.apache.pulsar.metadata.api.NotificationType;
import org.apache.pulsar.metadata.api.extended.CreateOption;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BrokerRegistryImpl
implements BrokerRegistry {
    private static final Logger log = LoggerFactory.getLogger(BrokerRegistryImpl.class);
    private static final int MAX_REGISTER_RETRY_DELAY_IN_MILLIS = 1000;
    private final PulsarService pulsar;
    private final ServiceConfiguration conf;
    private final BrokerLookupData brokerLookupData;
    private final MetadataCache<BrokerLookupData> brokerLookupDataMetadataCache;
    private final String brokerIdKeyPath;
    private final ScheduledExecutorService scheduler;
    private final List<BiConsumer<String, NotificationType>> listeners;
    @VisibleForTesting
    final AtomicReference<State> state = new AtomicReference<State>(State.Init);

    @VisibleForTesting
    BrokerRegistryImpl(PulsarService pulsar, MetadataCache<BrokerLookupData> brokerLookupDataMetadataCache) {
        this.pulsar = pulsar;
        this.conf = pulsar.getConfiguration();
        this.brokerLookupDataMetadataCache = brokerLookupDataMetadataCache;
        this.scheduler = pulsar.getLoadManagerExecutor();
        this.listeners = new ArrayList<BiConsumer<String, NotificationType>>();
        this.brokerIdKeyPath = BrokerRegistryImpl.keyPath(pulsar.getBrokerId());
        this.brokerLookupData = new BrokerLookupData(pulsar.getWebServiceAddress(), pulsar.getWebServiceAddressTls(), pulsar.getBrokerServiceUrl(), pulsar.getBrokerServiceUrlTls(), pulsar.getAdvertisedListeners(), pulsar.getProtocolDataToAdvertise(), pulsar.getConfiguration().isEnablePersistentTopics(), pulsar.getConfiguration().isEnableNonPersistentTopics(), this.conf.getLoadManagerClassName(), System.currentTimeMillis(), pulsar.getBrokerVersion());
    }

    public BrokerRegistryImpl(PulsarService pulsar) {
        this(pulsar, (MetadataCache<BrokerLookupData>)pulsar.getLocalMetadataStore().getMetadataCache(BrokerLookupData.class));
    }

    @Override
    public synchronized void start() throws PulsarServerException {
        if (!this.state.compareAndSet(State.Init, State.Started)) {
            throw new PulsarServerException("Cannot start the broker registry in state " + String.valueOf((Object)this.state.get()));
        }
        this.pulsar.getLocalMetadataStore().registerListener(this::handleMetadataStoreNotification);
        try {
            this.registerAsync().get(this.conf.getMetadataStoreOperationTimeoutSeconds(), TimeUnit.SECONDS);
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw PulsarServerException.from((Throwable)e);
        }
    }

    @Override
    public boolean isStarted() {
        State state = this.state.get();
        return state == State.Started || state == State.Registered;
    }

    @Override
    public boolean isRegistered() {
        State state = this.state.get();
        return state == State.Registered;
    }

    @Override
    public CompletableFuture<Void> registerAsync() {
        State state = this.state.get();
        if (state != State.Started && state != State.Registered) {
            log.info("[{}] Skip registering self because the state is {}", (Object)this.getBrokerId(), (Object)state);
            return CompletableFuture.completedFuture(null);
        }
        log.info("[{}] Started registering self to {} (state: {})", new Object[]{this.getBrokerId(), this.brokerIdKeyPath, state});
        return this.brokerLookupDataMetadataCache.put(this.brokerIdKeyPath, (Object)this.brokerLookupData, EnumSet.of(CreateOption.Ephemeral)).orTimeout(this.pulsar.getConfiguration().getMetadataStoreOperationTimeoutSeconds(), TimeUnit.SECONDS).whenComplete((__, ex) -> {
            if (ex == null) {
                this.state.set(State.Registered);
                log.info("[{}] Finished registering self", (Object)this.getBrokerId());
            } else {
                log.error("[{}] Failed registering self", (Object)this.getBrokerId(), ex);
            }
        });
    }

    private void doRegisterAsyncWithRetries(int retry, CompletableFuture<Void> future) {
        this.pulsar.getExecutor().schedule(() -> this.registerAsync().whenComplete((__, e) -> {
            if (e != null) {
                this.doRegisterAsyncWithRetries(retry + 1, future);
            } else {
                future.complete(null);
            }
        }), (long)Math.min(1000, retry * retry * 50), TimeUnit.MILLISECONDS);
    }

    private CompletableFuture<Void> registerAsyncWithRetries() {
        CompletableFuture<Void> retryFuture = new CompletableFuture<Void>();
        this.doRegisterAsyncWithRetries(0, retryFuture);
        return retryFuture;
    }

    @Override
    public synchronized void unregister() throws MetadataStoreException {
        block9: {
            if (this.state.compareAndSet(State.Registered, State.Unregistering)) {
                try {
                    this.brokerLookupDataMetadataCache.delete(this.brokerIdKeyPath).get(this.conf.getMetadataStoreOperationTimeoutSeconds(), TimeUnit.SECONDS);
                }
                catch (ExecutionException e) {
                    if (e.getCause() instanceof MetadataStoreException.NotFoundException) {
                        log.warn("{} has already been unregistered", (Object)this.brokerIdKeyPath);
                        break block9;
                    }
                    throw MetadataStoreException.unwrap((Throwable)e);
                }
                catch (InterruptedException | TimeoutException e) {
                    throw MetadataStoreException.unwrap((Throwable)e);
                }
                finally {
                    this.state.set(State.Started);
                }
            }
        }
    }

    @Override
    public String getBrokerId() {
        return this.pulsar.getBrokerId();
    }

    @Override
    public CompletableFuture<List<String>> getAvailableBrokersAsync() {
        this.checkState();
        return this.brokerLookupDataMetadataCache.getChildren("/loadbalance/brokers");
    }

    @Override
    public CompletableFuture<Optional<BrokerLookupData>> lookupAsync(String broker) {
        this.checkState();
        return this.brokerLookupDataMetadataCache.get(BrokerRegistryImpl.keyPath(broker));
    }

    @Override
    public CompletableFuture<Map<String, BrokerLookupData>> getAvailableBrokerLookupDataAsync() {
        this.checkState();
        return this.getAvailableBrokersAsync().thenCompose(availableBrokers -> {
            ConcurrentHashMap map = new ConcurrentHashMap();
            ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
            for (String brokerId : availableBrokers) {
                futures.add(this.lookupAsync(brokerId).thenAccept(lookupDataOpt -> {
                    if (lookupDataOpt.isPresent()) {
                        map.put(brokerId, (BrokerLookupData)lookupDataOpt.get());
                    } else {
                        log.warn("Got an empty lookup data, brokerId: {}", (Object)brokerId);
                    }
                }));
            }
            return FutureUtil.waitForAll(futures).thenApply(__ -> map);
        });
    }

    @Override
    public synchronized void addListener(BiConsumer<String, NotificationType> listener) {
        this.checkState();
        this.listeners.add(listener);
    }

    @Override
    public synchronized void close() throws PulsarServerException {
        if (this.state.get() == State.Closed) {
            return;
        }
        try {
            this.listeners.clear();
            this.unregister();
        }
        catch (Exception ex) {
            log.error("Unexpected error when unregistering the broker registry", (Throwable)ex);
        }
        finally {
            this.state.set(State.Closed);
        }
    }

    private void handleMetadataStoreNotification(Notification t) {
        if (!this.isStarted() || !BrokerRegistryImpl.isVerifiedNotification(t)) {
            return;
        }
        try {
            CompletableFuture<Object> register;
            if (log.isDebugEnabled()) {
                log.debug("Handle notification: [{}]", (Object)t);
            }
            String brokerId = t.getPath().substring("/loadbalance/brokers".length() + 1);
            if (t.getType() == NotificationType.Deleted && this.getBrokerId().equals(brokerId)) {
                this.state.set(State.Started);
                register = this.registerAsyncWithRetries();
            } else {
                register = CompletableFuture.completedFuture(null);
            }
            register.thenAccept(__ -> {
                if (this.listeners.isEmpty()) {
                    return;
                }
                this.scheduler.submit(() -> {
                    for (BiConsumer<String, NotificationType> listener : this.listeners) {
                        listener.accept(brokerId, t.getType());
                    }
                });
            });
        }
        catch (RejectedExecutionException rejectedExecutionException) {
            // empty catch block
        }
    }

    @VisibleForTesting
    protected static boolean isVerifiedNotification(Notification t) {
        return t.getPath().startsWith("/loadbalance/brokers/") && t.getPath().length() > "/loadbalance/brokers".length() + 1;
    }

    @VisibleForTesting
    protected static String keyPath(String brokerId) {
        return String.format("%s/%s", "/loadbalance/brokers", brokerId);
    }

    private void checkState() throws IllegalStateException {
        if (this.state.get() == State.Closed) {
            throw new IllegalStateException("The registry already closed.");
        }
    }

    protected static enum State {
        Init,
        Started,
        Registered,
        Unregistering,
        Closed;

    }
}

