/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pinot.spi.eventlistener.query;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.apache.pinot.spi.utils.CommonConstants.CONFIG_OF_BROKER_EVENT_LISTENER_CLASS_NAME;
import static org.apache.pinot.spi.utils.CommonConstants.CONFIG_OF_REQUEST_CONTEXT_TRACKED_HEADER_KEYS;
import static org.apache.pinot.spi.utils.CommonConstants.DEFAULT_BROKER_EVENT_LISTENER_CLASS_NAME;


public class PinotBrokerQueryEventListenerFactory {
  private static final Logger LOGGER = LoggerFactory.getLogger(PinotBrokerQueryEventListenerFactory.class);
  private static BrokerQueryEventListener _brokerQueryEventListener = null;
  private static List<String> _allowlistQueryRequestHeaders = new ArrayList<>();

  private PinotBrokerQueryEventListenerFactory() {
  }

  /**
   * Initialize the BrokerQueryEventListener and registers the eventListener
   */
  @VisibleForTesting
  public synchronized static void init(PinotConfiguration eventListenerConfiguration) {
    // Initializes BrokerQueryEventListener.
    initializeBrokerQueryEventListener(eventListenerConfiguration);
    // Initializes request headers
    initializeAllowlistQueryRequestHeaders(eventListenerConfiguration);
  }

  /**
   * Initializes PinotBrokerQueryEventListener with event-listener configurations.
   * @param eventListenerConfiguration The subset of the configuration containing the event-listener-related keys
   */
  private static void initializeBrokerQueryEventListener(PinotConfiguration eventListenerConfiguration) {
    String brokerQueryEventListenerClassName =
        eventListenerConfiguration.getProperty(CONFIG_OF_BROKER_EVENT_LISTENER_CLASS_NAME,
            DEFAULT_BROKER_EVENT_LISTENER_CLASS_NAME);
    LOGGER.info("{} will be initialized as the PinotBrokerQueryEventListener", brokerQueryEventListenerClassName);

    Optional<Class<?>> clazzFound;
    try {
      clazzFound = Optional.of(Class.forName(brokerQueryEventListenerClassName));
    } catch (ClassNotFoundException e) {
      throw new RuntimeException("Failed to initialize BrokerQueryEventListener. "
          + "Please check if any pinot-event-listener related jar is actually added to the classpath.");
    }

    clazzFound.ifPresent(clazz -> {
      try {
        BrokerQueryEventListener brokerQueryEventListener = (BrokerQueryEventListener) clazz.newInstance();
        brokerQueryEventListener.init(eventListenerConfiguration);
        registerBrokerEventListener(brokerQueryEventListener);
      } catch (Exception e) {
        LOGGER.error("Caught exception while initializing event listener registry: {}, skipping it", clazz, e);
      }
    });

    Preconditions.checkState(_brokerQueryEventListener != null, "Failed to initialize BrokerQueryEventListener. "
        + "Please check if any pinot-event-listener related jar is actually added to the classpath.");
  }

  /**
   * Initializes allowlist request-headers to extract from query request.
   * @param eventListenerConfiguration The subset of the configuration containing the event-listener-related keys
   */
  private static void initializeAllowlistQueryRequestHeaders(PinotConfiguration eventListenerConfiguration) {
    List<String> allowlistQueryRequestHeaders =
        Splitter.on(",").omitEmptyStrings().trimResults()
            .splitToList(eventListenerConfiguration.getProperty(CONFIG_OF_REQUEST_CONTEXT_TRACKED_HEADER_KEYS, ""));

    LOGGER.info("{}: allowlist headers will be used for PinotBrokerQueryEventListener", allowlistQueryRequestHeaders);
    registerAllowlistQueryRequestHeaders(allowlistQueryRequestHeaders);
  }

  /**
   * Registers a broker event listener.
   */
  private static void registerBrokerEventListener(BrokerQueryEventListener brokerQueryEventListener) {
    LOGGER.info("Registering broker event listener : {}", brokerQueryEventListener.getClass().getName());
    _brokerQueryEventListener = brokerQueryEventListener;
  }

  /**
   * Registers allowlist http headers for query-requests.
   */
  private static void registerAllowlistQueryRequestHeaders(List<String> allowlistQueryRequestHeaders) {
    LOGGER.info("Registering query request headers allowlist : {}", allowlistQueryRequestHeaders);
    _allowlistQueryRequestHeaders = ImmutableList.copyOf(allowlistQueryRequestHeaders);
  }

  /**
   * Returns the brokerQueryEventListener. If the BrokerQueryEventListener is null,
   * first creates and initializes the BrokerQueryEventListener.
   * @param eventListenerConfiguration event-listener configs
   */
  public static synchronized BrokerQueryEventListener getBrokerQueryEventListener(
      PinotConfiguration eventListenerConfiguration) {
    if (_brokerQueryEventListener == null) {
      init(eventListenerConfiguration);
    }
    return _brokerQueryEventListener;
  }

  @VisibleForTesting
  public static BrokerQueryEventListener getBrokerQueryEventListener() {
    return getBrokerQueryEventListener(new PinotConfiguration(Collections.emptyMap()));
  }

  @VisibleForTesting
  public static List<String> getAllowlistQueryRequestHeaders() {
    return _allowlistQueryRequestHeaders;
  }
}
