/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import {
  Box,
  Chip,
  CircularProgress,
  IconButton,
  makeStyles,
  Tooltip,
} from "@material-ui/core";
import { HelpOutlineOutlined } from "@material-ui/icons";
import { getSegmentLevelDebugDetails } from "../requests";
import CustomCodemirror from "./CustomCodemirror";
import CustomDialog from "./CustomDialog";
import { NotificationContext } from "./Notification/NotificationContext";
import clsx from "clsx";
import { DISPLAY_SEGMENT_STATUS, SegmentDebugDetails } from "Models";
import React, { useContext, useEffect, useState } from "react";

const useStyles = makeStyles((theme) => ({
  error: {
    color: theme.palette.error.main,
    border: `1px solid ${theme.palette.error.main}`,
  },
  success: {
    color: theme.palette.success.main,
    border: `1px solid ${theme.palette.success.main}`,
  },
  warning: {
    color: theme.palette.warning.main,
    border: `1px solid ${theme.palette.warning.main}`,
  },
  segmentDebugDetails: {
    "& .CodeMirror": { fontSize: 14, height: "100%" },
    maxHeight: 500,
  },
}));

interface SegmentStatusRendererProps {
  status: DISPLAY_SEGMENT_STATUS;
  segmentName: string;
  tableName: string;
}

export enum StatusColor {
  Error = "error",
  Warning = "warning",
  Success = "success",
}

export const SegmentStatusRenderer = ({
  status,
  segmentName,
  tableName,
}: SegmentStatusRendererProps) => {
  const { dispatch: notify } = useContext(NotificationContext);
  const [statusTooltipTitle, setStatusTooltipTitle] = useState<string>("");
  const [statusColor, setStatusColor] = useState<StatusColor | null>(
    null
  );
  const [errorDetailsVisible, setErrorDetailsVisible] =
    useState<boolean>(false);
  const [segmentDebugDetails, setSegmentDebugDetails] =
    useState<SegmentDebugDetails | null>(null);
  const segmentStatusRendererClasses = useStyles();

  useEffect(() => {
    initializeValues();
  }, [status]);

  const initializeValues = () => {
    switch (status) {
      case DISPLAY_SEGMENT_STATUS.GOOD: {
        setStatusColor(StatusColor.Success);
        setStatusTooltipTitle("All the servers of this segment are ONLINE/CONSUMING");

        break;
      }
      case DISPLAY_SEGMENT_STATUS.BAD: {
        setStatusColor(StatusColor.Error);
        setStatusTooltipTitle("One or more servers of this segment are in ERROR state");

        break;
      }
      case DISPLAY_SEGMENT_STATUS.PARTIAL: {
        setStatusColor(StatusColor.Warning);
        setStatusTooltipTitle("External view is OFFLINE or missing for one or more servers of this segment");

        break;
      }
    }
  };

  const fetchSegmentErrorDetails = async () => {
    setSegmentDebugDetails(null);
    try {
      const segmentDebugDetails = await getSegmentLevelDebugDetails(
        tableName,
        segmentName
      );
      setSegmentDebugDetails(segmentDebugDetails);
    } catch (error) {
      notify({
        type: "error",
        message: "Error occurred while fetching segment debug details.",
        show: true,
      });
      setSegmentDebugDetails({} as SegmentDebugDetails);
    }
  };

  const handleShowErrorDetailsClick = () => {
    setErrorDetailsVisible(true);
    fetchSegmentErrorDetails();
  };

  const handleHideErrorDetails = () => {
    setErrorDetailsVisible(false);
  };

  return (
    <>
      <Tooltip arrow title={statusTooltipTitle} placement="top">
        <Chip
          className={clsx([segmentStatusRendererClasses[statusColor]])}
          label={status}
          variant="outlined"
        />
      </Tooltip>

      {/* Only show when segment status is bad */}
      {status === DISPLAY_SEGMENT_STATUS.BAD && (
        <>
          <Tooltip
            title="Click to show segment error details"
            arrow
            placement="top"
          >
            <IconButton onClick={handleShowErrorDetailsClick}>
              <HelpOutlineOutlined fontSize="small" />
            </IconButton>
          </Tooltip>
          <CustomDialog
            title="Segment Debug Details"
            open={errorDetailsVisible}
            handleClose={handleHideErrorDetails}
            showOkBtn={false}
          >
            {/* Loading */}
            {!segmentDebugDetails && (
              <Box display="flex" justifyContent="center"><CircularProgress /></Box>
            )}
            {segmentDebugDetails && (
              <CustomCodemirror
                showLineWrapToggle
                customClass={segmentStatusRendererClasses.segmentDebugDetails}
                data={segmentDebugDetails}
              />
            )}
          </CustomDialog>
        </>
      )}
    </>
  );
};
