/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import React, { useEffect, useState, useContext } from 'react';
import { get, keys, last } from 'lodash';
import moment from 'moment';
import { UnControlled as CodeMirror } from 'react-codemirror2';
import { Grid, makeStyles, Box } from '@material-ui/core';
import { NotificationContext } from '../components/Notification/NotificationContext';
import SimpleAccordion from '../components/SimpleAccordion';
import CustomButton from '../components/CustomButton';
import { useConfirm } from '../components/Confirm';
import PinotMethodUtils from '../utils/PinotMethodUtils';
import useScheduleAdhocModal from '../components/useScheduleAdhocModal';
import useMinionMetadata from '../components/useMinionMetaData';
import useTaskListing from '../components/useTaskListing';

const jsonoptions = {
  lineNumbers: true,
  mode: 'application/json',
  styleActiveLine: true,
  gutters: ['CodeMirror-lint-markers'],
  theme: 'default',
  readOnly: true
};

const useStyles = makeStyles(() => ({
  gridContainer: {
    padding: 20,
    backgroundColor: 'white',
    maxHeight: 'calc(100vh - 70px)',
    overflowY: 'auto'
  },
  operationDiv: {
    border: '1px #BDCCD9 solid',
    borderRadius: 4,
    marginBottom: 20
  },
  body: {
    borderTop: '1px solid #BDCCD9',
    fontSize: '16px',
    lineHeight: '3rem',
    paddingLeft: '15px',
  },
  highlightBackground: {
    border: '1px #4285f4 solid',
    backgroundColor: 'rgba(66, 133, 244, 0.05)',
    borderRadius: 4,
    marginBottom: '20px',
  },
  sqlDiv: {
    border: '1px #BDCCD9 solid',
    borderRadius: 4,
    marginBottom: '20px',
  },
  queryOutput: {
    border: '1px solid #BDCCD9',
    '& .CodeMirror': { height: 532 },
  },
}));

const TaskQueueTable = (props) => {
  const classes = useStyles();
  const { taskType, queueTableName: tableName } = props.match.params;

  const {dispatch} = useContext(NotificationContext);
  const [fetching, setFetching] = useState(true);
  const [jobDetail, setJobDetail] = useState({});
  const [tableDetail, setTableDetail] = useState({});
  const [mostRecentErrorRunMessage, setMostRecentErrorRunMessage] = useState('');
  const scheduleAdhocModal = useScheduleAdhocModal();
  const minionMetadata = useMinionMetadata({ taskType, tableName });
  const taskListing = useTaskListing({ taskType, tableName });

  const fetchData = async () => {
    setFetching(true);
    const detail = await PinotMethodUtils.getScheduleJobDetail(tableName, taskType);
    const tableDetailRes = await PinotMethodUtils.getTableDetails(tableName);
    const taskGeneratorDebugData = await PinotMethodUtils.getTaskGeneratorDebugData(tableName, taskType);
    const mostRecentErrorRunMessagesTS = get(taskGeneratorDebugData, 'data.0.mostRecentErrorRunMessages', {});
    const mostRecentErrorRunMessagesTSLastTime = last(keys(mostRecentErrorRunMessagesTS).sort());
    setMostRecentErrorRunMessage(get(mostRecentErrorRunMessagesTS, mostRecentErrorRunMessagesTSLastTime, ''));
    setTableDetail(tableDetailRes);
    setJobDetail(detail);
    setFetching(false);
  };

  useEffect(() => {
    fetchData();
  }, []);

  const handleScheduleNow = async () => {
    const res = await PinotMethodUtils.scheduleTaskAction(tableName, taskType);
    if (get(res, `data.${taskType}`, null) === null) {
      dispatch({
        type: 'error',
        message: `Could not schedule task`,
        show: true
      });
    } else {
      dispatch({
        type: 'success',
        message: `${get(res, `data.${taskType}`, null)} scheduled successfully`,
        show: true
      });
    }
    
    await fetchData();
    scheduleNowConfirm.setConfirmDialog(false);
  };

  const scheduleNowConfirm = useConfirm({
    dialogTitle: 'Schedule now',
    dialogContent: 'Are you sure want to schedule now?',
    successCallback: handleScheduleNow
  });

  const cronExpression = get(jobDetail, 'Triggers.0.CronExpression', '');
  const previousFireTime = get(jobDetail, 'Triggers.0.PreviousFireTime', 0);
  const nextFireTime = get(jobDetail, 'Triggers.0.NextFireTime', 0);

  return (
    <Grid item xs className={classes.gridContainer}>
      <div className={classes.operationDiv}>
        <SimpleAccordion
          headerTitle="Operations"
          showSearchBox={false}
        >
          <div>
            <CustomButton
              onClick={() => scheduleNowConfirm.setConfirmDialog(true)}
              tooltipTitle="Schedule now"
              enableTooltip={true}
            >
              Schedule Now
            </CustomButton>
            <CustomButton
              onClick={() => scheduleAdhocModal.handleOpen()}
              tooltipTitle="Execute ADHOC"
              enableTooltip={true}
            >
              Schedule ADHOC
            </CustomButton>
          </div>
        </SimpleAccordion>
      </div>
      <div className={classes.highlightBackground}>
        {/* <TableToolbar name="Summary" showSearchBox={false} /> */}
        <Grid container className={classes.body}>
          <Grid item xs={12}>
            <strong>Table Name:</strong> {tableName}
          </Grid>
          <Grid item xs={12}>
            <strong>Cron Schedule :</strong> {cronExpression}
          </Grid>
          <Grid item xs={12}>
            <strong>Previous Fire Time: </strong> {previousFireTime ? moment(previousFireTime).toString() : '-'}
          </Grid>
          <Grid item xs={12}>
            <strong>Next Fire Time: </strong> {nextFireTime ? moment(nextFireTime).toString() : '-'}
          </Grid>
        </Grid>
      </div>
      <Grid container spacing={2}>
        <Grid item xs={6}>
          <div className={classes.sqlDiv}>
            <SimpleAccordion
              headerTitle="Task Config"
              showSearchBox={false}
            >
              <CodeMirror
                options={jsonoptions}
                value={JSON.stringify(get(tableDetail, `OFFLINE.task.taskTypeConfigsMap.${taskType}`, {}), null, '  ')}
                className={classes.queryOutput}
                autoCursor={false}
              />
            </SimpleAccordion>
          </div>
        </Grid>
        <Grid item xs={6}>
          <div className={classes.sqlDiv}>
            {minionMetadata.content}
          </div>
          <div className={classes.sqlDiv}>
            <SimpleAccordion
              headerTitle="Scheduling Errors"
              showSearchBox={false}
            >
              <Box p={3} style={{ overflow: 'auto' }}>
                {mostRecentErrorRunMessage}
              </Box>
            </SimpleAccordion>
          </div>
        </Grid>
        <Grid item xs={12}>
          {taskListing.content}
        </Grid>
      </Grid>
      {scheduleAdhocModal.dialog}
      {scheduleNowConfirm.confirmComponent}
    </Grid>
  );
};

export default TaskQueueTable;