/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.plugin.filesystem;

import com.azure.core.credential.TokenCredential;
import com.azure.core.http.ProxyOptions;
import com.azure.core.http.netty.NettyAsyncHttpClientBuilder;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.util.Context;
import com.azure.identity.ClientSecretCredential;
import com.azure.identity.ClientSecretCredentialBuilder;
import com.azure.identity.DefaultAzureCredentialBuilder;
import com.azure.storage.common.StorageSharedKeyCredential;
import com.azure.storage.common.Utility;
import com.azure.storage.file.datalake.DataLakeDirectoryClient;
import com.azure.storage.file.datalake.DataLakeFileClient;
import com.azure.storage.file.datalake.DataLakeFileSystemClient;
import com.azure.storage.file.datalake.DataLakeServiceClient;
import com.azure.storage.file.datalake.DataLakeServiceClientBuilder;
import com.azure.storage.file.datalake.models.DataLakeRequestConditions;
import com.azure.storage.file.datalake.models.DataLakeStorageException;
import com.azure.storage.file.datalake.models.ListPathsOptions;
import com.azure.storage.file.datalake.models.PathHttpHeaders;
import com.azure.storage.file.datalake.models.PathItem;
import com.azure.storage.file.datalake.models.PathProperties;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.OffsetDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.pinot.plugin.filesystem.AzurePinotFSUtil;
import org.apache.pinot.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.pinot.shaded.com.google.common.base.Preconditions;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.filesystem.BasePinotFS;
import org.apache.pinot.spi.filesystem.FileMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ADLSGen2PinotFS
extends BasePinotFS {
    private static final Logger LOGGER = LoggerFactory.getLogger(ADLSGen2PinotFS.class);
    private static final String AUTHENTICATION_TYPE = "authenticationType";
    private static final String ACCOUNT_NAME = "accountName";
    private static final String ACCESS_KEY = "accessKey";
    private static final String FILE_SYSTEM_NAME = "fileSystemName";
    private static final String ENABLE_CHECKSUM = "enableChecksum";
    private static final String CLIENT_ID = "clientId";
    private static final String CLIENT_SECRET = "clientSecret";
    private static final String TENANT_ID = "tenantId";
    private static final String MANAGED_IDENTITY_CLIENT_ID = "managedIdentityClientId";
    private static final String AUTHORITY_HOST = "authorityHost";
    private static final String PROXY_HOST = "proxyHost";
    private static final String PROXY_PORT = "proxyPort";
    private static final String PROXY_USERNAME = "proxyUsername";
    private static final String PROXY_PASSWORD = "proxyPassword";
    private static final String HTTPS_URL_PREFIX = "https://";
    private static final String AZURE_STORAGE_DNS_SUFFIX = ".dfs.core.windows.net";
    private static final String AZURE_BLOB_DNS_SUFFIX = ".blob.core.windows.net";
    private static final String PATH_ALREADY_EXISTS_ERROR_CODE = "PathAlreadyExists";
    private static final String CONTAINER_NOT_FOUND_ERROR_CODE = "ContainerNotFound";
    private static final String IS_DIRECTORY_KEY = "hdi_isfolder";
    private static final int NOT_FOUND_STATUS_CODE = 404;
    private static final int ALREADY_EXISTS_STATUS_CODE = 409;
    private static final int BUFFER_SIZE = 0x400000;
    private DataLakeFileSystemClient _fileSystemClient;
    private boolean _enableChecksum;

    public ADLSGen2PinotFS() {
    }

    public ADLSGen2PinotFS(DataLakeFileSystemClient fileSystemClient) {
        this._fileSystemClient = fileSystemClient;
    }

    public void init(PinotConfiguration config) {
        this._enableChecksum = config.getProperty(ENABLE_CHECKSUM, false);
        String accountName = config.getProperty(ACCOUNT_NAME);
        String authTypeStr = config.getProperty(AUTHENTICATION_TYPE, AuthenticationType.ACCESS_KEY.name());
        AuthenticationType authType = AuthenticationType.valueOf(authTypeStr.toUpperCase());
        String accessKey = config.getProperty(ACCESS_KEY);
        String fileSystemName = config.getProperty(FILE_SYSTEM_NAME);
        String clientId = config.getProperty(CLIENT_ID);
        String clientSecret = config.getProperty(CLIENT_SECRET);
        String tenantId = config.getProperty(TENANT_ID);
        String managedIdentityClientId = config.getProperty(MANAGED_IDENTITY_CLIENT_ID);
        String authorityHost = config.getProperty(AUTHORITY_HOST);
        String proxyHost = config.getProperty(PROXY_HOST);
        String proxyUsername = config.getProperty(PROXY_USERNAME);
        String proxyPassword = config.getProperty(PROXY_PASSWORD);
        String proxyPort = config.getProperty(PROXY_PORT);
        String dfsServiceEndpointUrl = HTTPS_URL_PREFIX + accountName + AZURE_STORAGE_DNS_SUFFIX;
        DataLakeServiceClientBuilder dataLakeServiceClientBuilder = new DataLakeServiceClientBuilder().endpoint(dfsServiceEndpointUrl);
        switch (authType) {
            case ACCESS_KEY: {
                LOGGER.info("Authenticating using the access key to the account.");
                Preconditions.checkNotNull((Object)accountName, (Object)"Account Name cannot be null");
                Preconditions.checkNotNull((Object)accessKey, (Object)"Access Key cannot be null");
                StorageSharedKeyCredential sharedKeyCredential = new StorageSharedKeyCredential(accountName, accessKey);
                dataLakeServiceClientBuilder.credential(sharedKeyCredential);
                break;
            }
            case AZURE_AD: {
                LOGGER.info("Authenticating using Azure Active Directory");
                Preconditions.checkNotNull((Object)clientId, (Object)"Client ID cannot be null");
                Preconditions.checkNotNull((Object)clientSecret, (Object)"ClientSecret cannot be null");
                Preconditions.checkNotNull((Object)tenantId, (Object)"TenantId cannot be null");
                ClientSecretCredential clientSecretCredential = ((ClientSecretCredentialBuilder)((ClientSecretCredentialBuilder)new ClientSecretCredentialBuilder().clientId(clientId)).clientSecret(clientSecret).tenantId(tenantId)).build();
                dataLakeServiceClientBuilder.credential((TokenCredential)clientSecretCredential);
                break;
            }
            case AZURE_AD_WITH_PROXY: {
                LOGGER.info("Authenticating using Azure Active Directory with proxy");
                Preconditions.checkNotNull((Object)clientId, (Object)"Client Id cannot be null");
                Preconditions.checkNotNull((Object)clientSecret, (Object)"ClientSecret cannot be null");
                Preconditions.checkNotNull((Object)tenantId, (Object)"Tenant Id cannot be null");
                Preconditions.checkNotNull((Object)proxyHost, (Object)"Proxy Host cannot be null");
                Preconditions.checkNotNull((Object)proxyPort, (Object)"Proxy Port cannot be null");
                Preconditions.checkNotNull((Object)proxyUsername, (Object)"Proxy Username cannot be null");
                Preconditions.checkNotNull((Object)proxyPassword, (Object)"Proxy Password cannot be null");
                NettyAsyncHttpClientBuilder builder = new NettyAsyncHttpClientBuilder();
                builder.proxy(new ProxyOptions(ProxyOptions.Type.HTTP, new InetSocketAddress(proxyHost, Integer.parseInt(proxyPort))).setCredentials(proxyUsername, proxyPassword));
                ClientSecretCredentialBuilder clientSecretCredentialBuilder = (ClientSecretCredentialBuilder)((ClientSecretCredentialBuilder)new ClientSecretCredentialBuilder().clientId(clientId)).clientSecret(clientSecret).tenantId(tenantId);
                clientSecretCredentialBuilder.httpClient(builder.build());
                dataLakeServiceClientBuilder.credential((TokenCredential)clientSecretCredentialBuilder.build());
                break;
            }
            case DEFAULT: {
                LOGGER.info("Authenticating using Azure default credential");
                DefaultAzureCredentialBuilder defaultAzureCredentialBuilder = new DefaultAzureCredentialBuilder();
                if (tenantId != null) {
                    LOGGER.info("Set tenant ID to {}", (Object)tenantId);
                    defaultAzureCredentialBuilder.tenantId(tenantId);
                }
                if (managedIdentityClientId != null) {
                    LOGGER.info("Set managed identity client ID to {}", (Object)managedIdentityClientId);
                    defaultAzureCredentialBuilder.managedIdentityClientId(managedIdentityClientId);
                }
                if (authorityHost != null) {
                    LOGGER.info("Set authority host to {}", (Object)authorityHost);
                    defaultAzureCredentialBuilder.authorityHost(authorityHost);
                }
                dataLakeServiceClientBuilder.credential((TokenCredential)defaultAzureCredentialBuilder.build());
                break;
            }
            case ANONYMOUS_ACCESS: {
                LOGGER.info("Authenticating using anonymous access");
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected authType: " + authType);
            }
        }
        DataLakeServiceClient serviceClient = dataLakeServiceClientBuilder.buildClient();
        this._fileSystemClient = this.getOrCreateClientWithFileSystem(serviceClient, fileSystemName);
        LOGGER.info("ADLSGen2PinotFS is initialized (accountName={}, fileSystemName={}, dfsServiceEndpointUrl={}, enableChecksum={})", accountName, fileSystemName, dfsServiceEndpointUrl, this._enableChecksum);
    }

    @VisibleForTesting
    public DataLakeFileSystemClient getOrCreateClientWithFileSystem(DataLakeServiceClient serviceClient, String fileSystemName) {
        try {
            DataLakeFileSystemClient fileSystemClient = serviceClient.getFileSystemClient(fileSystemName);
            fileSystemClient.getProperties();
            return fileSystemClient;
        }
        catch (DataLakeStorageException e) {
            if (e.getStatusCode() == 404 && e.getErrorCode().equals(CONTAINER_NOT_FOUND_ERROR_CODE)) {
                LOGGER.info("FileSystem with name {} does not exist. Creating one with the same name.", (Object)fileSystemName);
                return serviceClient.createFileSystem(fileSystemName);
            }
            throw e;
        }
    }

    public boolean mkdir(URI uri) throws IOException {
        LOGGER.debug("mkdir is called with uri='{}'", (Object)uri);
        try {
            DataLakeRequestConditions requestConditions = new DataLakeRequestConditions().setIfNoneMatch("*");
            this._fileSystemClient.createDirectoryWithResponse(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(uri), null, null, null, null, requestConditions, null, null);
            return true;
        }
        catch (DataLakeStorageException e) {
            if (e.getStatusCode() == 409 && e.getErrorCode().equals(PATH_ALREADY_EXISTS_ERROR_CODE)) {
                return true;
            }
            LOGGER.error("Exception thrown while calling mkdir (uri={}, errorStatus ={})", uri, e.getStatusCode(), e);
            throw new IOException(e);
        }
    }

    public boolean delete(URI segmentUri, boolean forceDelete) throws IOException {
        LOGGER.debug("delete is called with segmentUri='{}', forceDelete='{}'", (Object)segmentUri, (Object)forceDelete);
        try {
            boolean isDirectory = this.isDirectory(segmentUri);
            if (isDirectory && this.listFiles(segmentUri, false).length > 0 && !forceDelete) {
                return false;
            }
            String path = AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(segmentUri);
            if (isDirectory) {
                this._fileSystemClient.deleteDirectoryWithResponse(path, true, null, null, Context.NONE).getValue();
            } else {
                this._fileSystemClient.deleteFile(path);
            }
            return true;
        }
        catch (DataLakeStorageException e) {
            throw new IOException(e);
        }
    }

    public boolean doMove(URI srcUri, URI dstUri) throws IOException {
        LOGGER.debug("doMove is called with srcUri='{}', dstUri='{}'", (Object)srcUri, (Object)dstUri);
        try {
            DataLakeDirectoryClient directoryClient = this._fileSystemClient.getDirectoryClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(srcUri));
            directoryClient.rename(null, AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(dstUri));
            return true;
        }
        catch (DataLakeStorageException e) {
            throw new IOException(e);
        }
    }

    public boolean copyDir(URI srcUri, URI dstUri) throws IOException {
        LOGGER.debug("copy is called with srcUri='{}', dstUri='{}'", (Object)srcUri, (Object)dstUri);
        if (srcUri.equals(dstUri)) {
            return true;
        }
        if (this.exists(dstUri)) {
            this.delete(dstUri, true);
        }
        if (!this.isDirectory(srcUri)) {
            return this.copySrcToDst(srcUri, dstUri);
        }
        try {
            boolean copySucceeded = true;
            Path srcPath = Paths.get(srcUri.getPath(), new String[0]);
            for (String path : this.listFiles(srcUri, true)) {
                URI currentSrc = new URI(srcUri.getScheme(), srcUri.getHost(), path, null);
                String relativeSrcPath = srcPath.relativize(Paths.get(path, new String[0])).toString();
                String newDstPath = Paths.get(dstUri.getPath(), relativeSrcPath).toString();
                URI newDst = new URI(dstUri.getScheme(), dstUri.getHost(), newDstPath, null);
                if (this.isDirectory(currentSrc)) {
                    this.mkdir(newDst);
                    continue;
                }
                copySucceeded &= this.copySrcToDst(currentSrc, newDst);
            }
            return copySucceeded;
        }
        catch (DataLakeStorageException | URISyntaxException e) {
            throw new IOException(e);
        }
    }

    public boolean exists(URI fileUri) throws IOException {
        try {
            this._fileSystemClient.getDirectoryClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(fileUri)).getProperties();
            return true;
        }
        catch (DataLakeStorageException e) {
            if (e.getStatusCode() == 404) {
                return false;
            }
            throw new IOException(e);
        }
    }

    public long length(URI fileUri) throws IOException {
        try {
            PathProperties pathProperties = this._fileSystemClient.getDirectoryClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(fileUri)).getProperties();
            return pathProperties.getFileSize();
        }
        catch (DataLakeStorageException e) {
            throw new IOException(e);
        }
    }

    public String[] listFiles(URI fileUri, boolean recursive) throws IOException {
        LOGGER.debug("listFiles is called with fileUri='{}', recursive='{}'", (Object)fileUri, (Object)recursive);
        try {
            PagedIterable<PathItem> iter = this.listPathItems(fileUri, recursive);
            return (String[])iter.stream().map(p -> AzurePinotFSUtil.convertAzureStylePathToUriStylePath(p.getName())).toArray(String[]::new);
        }
        catch (DataLakeStorageException e) {
            throw new IOException(e);
        }
    }

    public List<FileMetadata> listFilesWithMetadata(URI fileUri, boolean recursive) throws IOException {
        LOGGER.debug("listFilesWithMetadata is called with fileUri='{}', recursive='{}'", (Object)fileUri, (Object)recursive);
        try {
            PagedIterable<PathItem> iter = this.listPathItems(fileUri, recursive);
            return iter.stream().map(ADLSGen2PinotFS::getFileMetadata).collect(Collectors.toList());
        }
        catch (DataLakeStorageException e) {
            throw new IOException(e);
        }
    }

    private PagedIterable<PathItem> listPathItems(URI fileUri, boolean recursive) throws IOException {
        String pathForListPathsOptions = Utility.urlDecode(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(fileUri));
        ListPathsOptions options = new ListPathsOptions().setPath(pathForListPathsOptions).setRecursive(recursive);
        return this._fileSystemClient.listPaths(options, null);
    }

    private static FileMetadata getFileMetadata(PathItem file) {
        String path = AzurePinotFSUtil.convertAzureStylePathToUriStylePath(file.getName());
        return new FileMetadata.Builder().setFilePath(path).setLastModifiedTime(file.getLastModified().toInstant().toEpochMilli()).setLength(file.getContentLength()).setIsDirectory(file.isDirectory()).build();
    }

    public void copyToLocalFile(URI srcUri, File dstFile) throws Exception {
        byte[] md5FromLocalFile;
        DataLakeFileClient fileClient;
        byte[] md5ContentFromMetadata;
        LOGGER.debug("copyToLocalFile is called with srcUri='{}', dstFile='{}'", (Object)srcUri, (Object)dstFile);
        if (dstFile.exists()) {
            if (dstFile.isDirectory()) {
                FileUtils.deleteDirectory((File)dstFile);
            } else {
                FileUtils.deleteQuietly((File)dstFile);
            }
        }
        byte[] buffer = new byte[0x400000];
        try (InputStream inputStream = this.open(srcUri);
             FileOutputStream outputStream = new FileOutputStream(dstFile);){
            int bytesRead;
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                ((OutputStream)outputStream).write(buffer, 0, bytesRead);
            }
        }
        if (this._enableChecksum && (md5ContentFromMetadata = (fileClient = this._fileSystemClient.getFileClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(srcUri))).getProperties().getContentMd5()) != null && md5ContentFromMetadata.length > 0 && !Arrays.equals(md5FromLocalFile = this.computeContentMd5(dstFile), md5ContentFromMetadata)) {
            FileUtils.deleteQuietly((File)dstFile);
            throw new IOException("Computed MD5 and MD5 from metadata do not match");
        }
    }

    public void copyFromLocalFile(File srcFile, URI dstUri) throws Exception {
        LOGGER.debug("copyFromLocalFile is called with srcFile='{}', dstUri='{}'", (Object)srcFile, (Object)dstUri);
        byte[] contentMd5 = this.computeContentMd5(srcFile);
        try (FileInputStream fileInputStream = new FileInputStream(srcFile);){
            this.copyInputStreamToDst(fileInputStream, dstUri, contentMd5);
        }
    }

    public boolean isDirectory(URI uri) throws IOException {
        try {
            PathProperties pathProperties = this.getPathProperties(uri);
            Map<String, String> metadata = pathProperties.getMetadata();
            return Boolean.valueOf(metadata.get(IS_DIRECTORY_KEY));
        }
        catch (DataLakeStorageException e) {
            throw new IOException("Failed while checking isDirectory for : " + uri, e);
        }
    }

    public long lastModified(URI uri) throws IOException {
        try {
            PathProperties pathProperties = this.getPathProperties(uri);
            OffsetDateTime offsetDateTime = pathProperties.getLastModified();
            return offsetDateTime.toInstant().toEpochMilli();
        }
        catch (DataLakeStorageException e) {
            throw new IOException("Failed while checking lastModified time for : " + uri, e);
        }
    }

    public boolean touch(URI uri) throws IOException {
        try {
            DataLakeFileClient fileClient = this._fileSystemClient.getFileClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(uri));
            PathProperties pathProperties = fileClient.getProperties();
            fileClient.setHttpHeaders(this.getPathHttpHeaders(pathProperties));
            return true;
        }
        catch (DataLakeStorageException e) {
            throw new IOException(e);
        }
    }

    public InputStream open(URI uri) throws IOException {
        return this._fileSystemClient.getFileClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(uri)).openInputStream().getInputStream();
    }

    private boolean copySrcToDst(URI srcUri, URI dstUri) throws IOException {
        PathProperties pathProperties = this._fileSystemClient.getFileClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(srcUri)).getProperties();
        try (InputStream inputStream = this.open(srcUri);){
            boolean bl = this.copyInputStreamToDst(inputStream, dstUri, pathProperties.getContentMd5());
            return bl;
        }
    }

    private boolean copyInputStreamToDst(InputStream inputStream, URI dstUri, byte[] contentMd5) throws IOException {
        long totalBytesRead = 0L;
        byte[] buffer = new byte[0x400000];
        DataLakeFileClient fileClient = this._fileSystemClient.createFile(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(dstUri));
        if (contentMd5 != null) {
            PathHttpHeaders pathHttpHeaders = this.getPathHttpHeaders(fileClient.getProperties());
            pathHttpHeaders.setContentMd5(contentMd5);
            fileClient.setHttpHeaders(pathHttpHeaders);
        }
        try {
            int bytesRead;
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                byte[] md5BlockHash = null;
                if (this._enableChecksum) {
                    MessageDigest md5Block = MessageDigest.getInstance("MD5");
                    md5Block.update(buffer, 0, bytesRead);
                    md5BlockHash = md5Block.digest();
                }
                ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(buffer, 0, bytesRead);
                fileClient.appendWithResponse(byteArrayInputStream, totalBytesRead, bytesRead, md5BlockHash, null, null, Context.NONE);
                byteArrayInputStream.close();
                totalBytesRead += (long)bytesRead;
            }
            fileClient.flush(totalBytesRead);
            return true;
        }
        catch (DataLakeStorageException | NoSuchAlgorithmException e) {
            throw new IOException(e);
        }
    }

    private byte[] computeContentMd5(File file) throws Exception {
        MessageDigest messageDigest = MessageDigest.getInstance("MD5");
        byte[] buffer = new byte[0x400000];
        try (FileInputStream inputStream = new FileInputStream(file);){
            int bytesRead;
            while ((bytesRead = ((InputStream)inputStream).read(buffer)) != -1) {
                messageDigest.update(buffer, 0, bytesRead);
            }
        }
        return messageDigest.digest();
    }

    private PathProperties getPathProperties(URI uri) throws IOException {
        return this._fileSystemClient.getDirectoryClient(AzurePinotFSUtil.convertUriToUrlEncodedAzureStylePath(uri)).getProperties();
    }

    private PathHttpHeaders getPathHttpHeaders(PathProperties pathProperties) {
        return new PathHttpHeaders().setCacheControl(pathProperties.getCacheControl()).setContentDisposition(pathProperties.getContentDisposition()).setContentEncoding(pathProperties.getContentEncoding()).setContentMd5(pathProperties.getContentMd5()).setContentLanguage(pathProperties.getContentLanguage()).setContentType(pathProperties.getContentType());
    }

    private static enum AuthenticationType {
        ACCESS_KEY,
        AZURE_AD,
        AZURE_AD_WITH_PROXY,
        ANONYMOUS_ACCESS,
        DEFAULT;

    }
}

