/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.management.ObjectName;
import org.apache.hadoop.hdds.client.RatisReplicationConfig;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.client.StandaloneReplicationConfig;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.SCMHAManager;
import org.apache.hadoop.hdds.scm.ha.SCMServiceManager;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.pipeline.BackgroundPipelineCreatorV2;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineFactory;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineProvider;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManagerImpl;
import org.apache.hadoop.hdds.scm.pipeline.RatisPipelineUtils;
import org.apache.hadoop.hdds.scm.pipeline.SCMPipelineMetrics;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.util.Time;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipelineManagerImpl
implements PipelineManager {
    private static final Logger LOG = LoggerFactory.getLogger(PipelineManagerImpl.class);
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private PipelineFactory pipelineFactory;
    private PipelineStateManager stateManager;
    private BackgroundPipelineCreatorV2 backgroundPipelineCreator;
    private final ConfigurationSource conf;
    private final EventPublisher eventPublisher;
    private ObjectName pmInfoBean;
    private final SCMPipelineMetrics metrics;
    private final long pipelineWaitDefaultTimeout;
    private final SCMHAManager scmhaManager;
    private final SCMContext scmContext;
    private final NodeManager nodeManager;
    private AtomicBoolean freezePipelineCreation;

    protected PipelineManagerImpl(ConfigurationSource conf, SCMHAManager scmhaManager, NodeManager nodeManager, PipelineStateManager pipelineStateManager, PipelineFactory pipelineFactory, EventPublisher eventPublisher, SCMContext scmContext) {
        this.pipelineFactory = pipelineFactory;
        this.stateManager = pipelineStateManager;
        this.conf = conf;
        this.scmhaManager = scmhaManager;
        this.nodeManager = nodeManager;
        this.eventPublisher = eventPublisher;
        this.scmContext = scmContext;
        this.pmInfoBean = MBeans.register((String)"SCMPipelineManager", (String)"SCMPipelineManagerInfo", (Object)this);
        this.metrics = SCMPipelineMetrics.create();
        this.pipelineWaitDefaultTimeout = conf.getTimeDuration("hdds.pipeline.report.interval", "60s", TimeUnit.MILLISECONDS);
        this.freezePipelineCreation = new AtomicBoolean();
    }

    public static PipelineManagerImpl newPipelineManager(ConfigurationSource conf, SCMHAManager scmhaManager, NodeManager nodeManager, Table<PipelineID, Pipeline> pipelineStore, EventPublisher eventPublisher, SCMContext scmContext, SCMServiceManager serviceManager) throws IOException {
        PipelineStateManager stateManager = PipelineStateManagerImpl.newBuilder().setPipelineStore(pipelineStore).setRatisServer(scmhaManager.getRatisServer()).setNodeManager(nodeManager).setSCMDBTransactionBuffer(scmhaManager.getDBTransactionBuffer()).build();
        PipelineFactory pipelineFactory = new PipelineFactory(nodeManager, stateManager, conf, eventPublisher, scmContext);
        PipelineManagerImpl pipelineManager = new PipelineManagerImpl(conf, scmhaManager, nodeManager, stateManager, pipelineFactory, eventPublisher, scmContext);
        BackgroundPipelineCreatorV2 backgroundPipelineCreator = new BackgroundPipelineCreatorV2(pipelineManager, conf, serviceManager, scmContext);
        pipelineManager.setBackgroundPipelineCreator(backgroundPipelineCreator);
        return pipelineManager;
    }

    @Override
    public Pipeline createPipeline(ReplicationConfig replicationConfig) throws IOException {
        if (!this.isPipelineCreationAllowed() && !this.factorOne(replicationConfig)) {
            LOG.debug("Pipeline creation is not allowed until safe mode prechecks complete");
            throw new IOException("Pipeline creation is not allowed as safe mode prechecks have not yet passed");
        }
        if (this.freezePipelineCreation.get()) {
            String message = "Cannot create new pipelines while pipeline creation is frozen.";
            LOG.info(message);
            throw new IOException(message);
        }
        this.acquireWriteLock();
        try {
            Pipeline pipeline = this.pipelineFactory.create(replicationConfig);
            this.stateManager.addPipeline(pipeline.getProtobufMessage(1));
            this.recordMetricsForPipeline(pipeline);
            Pipeline pipeline2 = pipeline;
            return pipeline2;
        }
        catch (IOException ex) {
            LOG.debug("Failed to create pipeline with replicationConfig {}.", (Object)replicationConfig, (Object)ex);
            this.metrics.incNumPipelineCreationFailed();
            throw ex;
        }
        finally {
            this.releaseWriteLock();
        }
    }

    private boolean factorOne(ReplicationConfig replicationConfig) {
        if (replicationConfig.getReplicationType() == HddsProtos.ReplicationType.RATIS) {
            return ((RatisReplicationConfig)replicationConfig).getReplicationFactor() == HddsProtos.ReplicationFactor.ONE;
        }
        if (replicationConfig.getReplicationType() == HddsProtos.ReplicationType.STAND_ALONE) {
            return ((StandaloneReplicationConfig)replicationConfig).getReplicationFactor() == HddsProtos.ReplicationFactor.ONE;
        }
        return false;
    }

    @Override
    public Pipeline createPipeline(ReplicationConfig replicationConfig, List<DatanodeDetails> nodes) {
        return this.pipelineFactory.create(replicationConfig, nodes);
    }

    @Override
    public Pipeline getPipeline(PipelineID pipelineID) throws PipelineNotFoundException {
        return this.stateManager.getPipeline(pipelineID);
    }

    @Override
    public boolean containsPipeline(PipelineID pipelineID) {
        try {
            this.getPipeline(pipelineID);
            return true;
        }
        catch (PipelineNotFoundException e) {
            return false;
        }
    }

    @Override
    public List<Pipeline> getPipelines() {
        return this.stateManager.getPipelines();
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig) {
        return this.stateManager.getPipelines(replicationConfig);
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig config, Pipeline.PipelineState state) {
        return this.stateManager.getPipelines(config, state);
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig, Pipeline.PipelineState state, Collection<DatanodeDetails> excludeDns, Collection<PipelineID> excludePipelines) {
        return this.stateManager.getPipelines(replicationConfig, state, excludeDns, excludePipelines);
    }

    @Override
    public void addContainerToPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        this.stateManager.addContainerToPipeline(pipelineID, containerID);
    }

    @Override
    public void addContainerToPipelineSCMStart(PipelineID pipelineID, ContainerID containerID) throws IOException {
        this.stateManager.addContainerToPipelineSCMStart(pipelineID, containerID);
    }

    @Override
    public void removeContainerFromPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        this.stateManager.removeContainerFromPipeline(pipelineID, containerID);
    }

    @Override
    public NavigableSet<ContainerID> getContainersInPipeline(PipelineID pipelineID) throws IOException {
        return this.stateManager.getContainers(pipelineID);
    }

    @Override
    public int getNumberOfContainers(PipelineID pipelineID) throws IOException {
        return this.stateManager.getNumberOfContainers(pipelineID);
    }

    @Override
    public void openPipeline(PipelineID pipelineId) throws IOException {
        this.acquireWriteLock();
        try {
            Pipeline pipeline = this.stateManager.getPipeline(pipelineId);
            if (pipeline.isClosed()) {
                throw new IOException("Closed pipeline can not be opened");
            }
            if (pipeline.getPipelineState() == Pipeline.PipelineState.ALLOCATED) {
                LOG.info("Pipeline {} moved to OPEN state", (Object)pipeline);
                this.stateManager.updatePipelineState(pipelineId.getProtobuf(), HddsProtos.PipelineState.PIPELINE_OPEN);
            }
            this.metrics.incNumPipelineCreated();
            this.metrics.createPerPipelineMetrics(pipeline);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    protected void removePipeline(Pipeline pipeline) throws IOException {
        this.pipelineFactory.close(pipeline.getType(), pipeline);
        PipelineID pipelineID = pipeline.getId();
        this.acquireWriteLock();
        try {
            this.stateManager.removePipeline(pipelineID.getProtobuf());
            this.metrics.incNumPipelineDestroyed();
        }
        catch (IOException ex) {
            this.metrics.incNumPipelineDestroyFailed();
            throw ex;
        }
        finally {
            this.releaseWriteLock();
        }
    }

    protected void closeContainersForPipeline(PipelineID pipelineId) throws IOException {
        NavigableSet<ContainerID> containerIDs = this.stateManager.getContainers(pipelineId);
        for (ContainerID containerID : containerIDs) {
            this.eventPublisher.fireEvent(SCMEvents.CLOSE_CONTAINER, (Object)containerID);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void closePipeline(Pipeline pipeline, boolean onTimeout) throws IOException {
        PipelineID pipelineID = pipeline.getId();
        this.acquireWriteLock();
        try {
            if (!pipeline.isClosed()) {
                this.stateManager.updatePipelineState(pipelineID.getProtobuf(), HddsProtos.PipelineState.PIPELINE_CLOSED);
                LOG.info("Pipeline {} moved to CLOSED state", (Object)pipeline);
            }
            this.metrics.removePipelineMetrics(pipelineID);
        }
        finally {
            this.releaseWriteLock();
        }
        this.closeContainersForPipeline(pipelineID);
        if (!onTimeout) {
            this.removePipeline(pipeline);
        }
    }

    @Override
    public void scrubPipeline(ReplicationConfig config) throws IOException {
        Instant currentTime = Instant.now();
        Long pipelineScrubTimeoutInMills = this.conf.getTimeDuration("ozone.scm.pipeline.allocated.timeout", "5m", TimeUnit.MILLISECONDS);
        List<Pipeline> candidates = this.stateManager.getPipelines(config);
        for (Pipeline p : candidates) {
            if (p.getPipelineState() == Pipeline.PipelineState.ALLOCATED && currentTime.toEpochMilli() - p.getCreationTimestamp().toEpochMilli() >= pipelineScrubTimeoutInMills) {
                LOG.info("Scrubbing pipeline: id: " + p.getId().toString() + " since it stays at ALLOCATED stage for " + Duration.between(currentTime, p.getCreationTimestamp()).toMinutes() + " mins.");
                this.closePipeline(p, false);
            }
            if (p.getPipelineState() != Pipeline.PipelineState.CLOSED) continue;
            LOG.info("Scrubbing pipeline: id: " + p.getId().toString() + " since it stays at CLOSED stage.");
            this.closeContainersForPipeline(p.getId());
            this.removePipeline(p);
        }
    }

    @Override
    public void startPipelineCreator() {
        throw new RuntimeException("Not supported in HA code.");
    }

    @Override
    public void triggerPipelineCreation() {
        throw new RuntimeException("Not supported in HA code.");
    }

    @Override
    public void incNumBlocksAllocatedMetric(PipelineID id) {
        this.metrics.incNumBlocksAllocated(id);
    }

    @Override
    public int minHealthyVolumeNum(Pipeline pipeline) {
        return this.nodeManager.minHealthyVolumeNum(pipeline.getNodes());
    }

    @Override
    public int minPipelineLimit(Pipeline pipeline) {
        return this.nodeManager.minPipelineLimit(pipeline.getNodes());
    }

    @Override
    public void activatePipeline(PipelineID pipelineID) throws IOException {
        this.acquireWriteLock();
        try {
            this.stateManager.updatePipelineState(pipelineID.getProtobuf(), HddsProtos.PipelineState.PIPELINE_OPEN);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    @Override
    public void deactivatePipeline(PipelineID pipelineID) throws IOException {
        this.acquireWriteLock();
        try {
            this.stateManager.updatePipelineState(pipelineID.getProtobuf(), HddsProtos.PipelineState.PIPELINE_DORMANT);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    @Override
    public void waitPipelineReady(PipelineID pipelineID, long timeout) throws IOException {
        boolean ready;
        long st = Time.monotonicNow();
        if (timeout == 0L) {
            timeout = this.pipelineWaitDefaultTimeout;
        }
        do {
            Pipeline pipeline;
            try {
                pipeline = this.stateManager.getPipeline(pipelineID);
            }
            catch (PipelineNotFoundException e) {
                throw new PipelineNotFoundException(String.format("Pipeline %s cannot be found", pipelineID));
            }
            ready = pipeline.isOpen();
            if (ready) continue;
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        } while (!ready && Time.monotonicNow() - st < timeout);
        if (!ready) {
            throw new IOException(String.format("Pipeline %s is not ready in %d ms", pipelineID, timeout));
        }
    }

    @Override
    public Map<String, Integer> getPipelineInfo() throws NotLeaderException {
        HashMap<String, Integer> pipelineInfo = new HashMap<String, Integer>();
        for (Pipeline.PipelineState state : Pipeline.PipelineState.values()) {
            pipelineInfo.put(state.toString(), 0);
        }
        this.stateManager.getPipelines().forEach(pipeline -> pipelineInfo.computeIfPresent(pipeline.getPipelineState().toString(), (k, v) -> v + 1));
        return pipelineInfo;
    }

    @Override
    public boolean getSafeModeStatus() {
        return this.scmContext.isInSafeMode();
    }

    @Override
    public void reinitialize(Table<PipelineID, Pipeline> pipelineStore) throws IOException {
        this.stateManager.reinitialize(pipelineStore);
    }

    @Override
    public void freezePipelineCreation() {
        this.freezePipelineCreation.set(true);
        this.backgroundPipelineCreator.stop();
    }

    @Override
    public void resumePipelineCreation() {
        this.freezePipelineCreation.set(false);
        this.backgroundPipelineCreator.start();
    }

    @Override
    public void close() throws IOException {
        if (this.backgroundPipelineCreator != null) {
            this.backgroundPipelineCreator.stop();
        }
        if (this.pmInfoBean != null) {
            MBeans.unregister((ObjectName)this.pmInfoBean);
            this.pmInfoBean = null;
        }
        SCMPipelineMetrics.unRegister();
        this.pipelineFactory.shutdown();
        try {
            this.stateManager.close();
        }
        catch (Exception ex) {
            LOG.error("PipelineStateManagerImpl close failed", (Throwable)ex);
        }
    }

    @VisibleForTesting
    public boolean isPipelineCreationAllowed() {
        return this.scmContext.isLeader() && this.scmContext.isPreCheckComplete();
    }

    @VisibleForTesting
    public void setPipelineProvider(HddsProtos.ReplicationType replicationType, PipelineProvider provider) {
        this.pipelineFactory.setProvider(replicationType, provider);
    }

    @VisibleForTesting
    public PipelineStateManager getStateManager() {
        return this.stateManager;
    }

    @VisibleForTesting
    public SCMHAManager getScmhaManager() {
        return this.scmhaManager;
    }

    private void setBackgroundPipelineCreator(BackgroundPipelineCreatorV2 backgroundPipelineCreator) {
        this.backgroundPipelineCreator = backgroundPipelineCreator;
    }

    @VisibleForTesting
    public BackgroundPipelineCreatorV2 getBackgroundPipelineCreator() {
        return this.backgroundPipelineCreator;
    }

    @VisibleForTesting
    public PipelineFactory getPipelineFactory() {
        return this.pipelineFactory;
    }

    private void recordMetricsForPipeline(Pipeline pipeline) {
        this.metrics.incNumPipelineAllocated();
        if (pipeline.isOpen()) {
            this.metrics.incNumPipelineCreated();
            this.metrics.createPerPipelineMetrics(pipeline);
        }
        switch (pipeline.getType()) {
            case STAND_ALONE: {
                return;
            }
            case RATIS: {
                List<Pipeline> overlapPipelines = RatisPipelineUtils.checkPipelineContainSameDatanodes(this.stateManager, pipeline);
                if (!overlapPipelines.isEmpty()) {
                    this.metrics.incNumPipelineContainSameDatanodes();
                    for (Pipeline overlapPipeline : overlapPipelines) {
                        LOG.info("Pipeline: " + pipeline.getId().toString() + " contains same datanodes as previous pipelines: " + overlapPipeline.getId().toString() + " nodeIds: " + ((DatanodeDetails)pipeline.getNodes().get(0)).getUuid().toString() + ", " + ((DatanodeDetails)pipeline.getNodes().get(1)).getUuid().toString() + ", " + ((DatanodeDetails)pipeline.getNodes().get(2)).getUuid().toString());
                    }
                }
                return;
            }
        }
    }

    @Override
    public void acquireReadLock() {
        this.lock.readLock().lock();
    }

    @Override
    public void releaseReadLock() {
        this.lock.readLock().unlock();
    }

    @Override
    public void acquireWriteLock() {
        this.lock.writeLock().lock();
    }

    @Override
    public void releaseWriteLock() {
        this.lock.writeLock().unlock();
    }
}

