/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.stream.Collectors;
import javax.management.ObjectName;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.VersionInfo;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeMetric;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.net.Node;
import org.apache.hadoop.hdds.scm.node.CommandQueue;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.DatanodeUsageInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStateManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.SCMNodeMetrics;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.server.SCMStorageConfig;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.upgrade.HDDSLayoutVersionManager;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.net.CachedDNSToSwitchMapping;
import org.apache.hadoop.net.DNSToSwitchMapping;
import org.apache.hadoop.net.TableMapping;
import org.apache.hadoop.ozone.protocol.VersionResponse;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.FinalizeNewLayoutVersionCommand;
import org.apache.hadoop.ozone.protocol.commands.RegisteredCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.ozone.protocol.commands.SetNodeOperationalStateCommand;
import org.apache.hadoop.ozone.upgrade.LayoutVersionManager;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.Time;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMNodeManager
implements NodeManager {
    public static final Logger LOG = LoggerFactory.getLogger(SCMNodeManager.class);
    private final NodeStateManager nodeStateManager;
    private final VersionInfo version;
    private final CommandQueue commandQueue;
    private final SCMNodeMetrics metrics;
    private ObjectName nmInfoBean;
    private final SCMStorageConfig scmStorageConfig;
    private final NetworkTopology clusterMap;
    private final DNSToSwitchMapping dnsToSwitchMapping;
    private final boolean useHostname;
    private final ConcurrentHashMap<String, Set<String>> dnsToUuidMap = new ConcurrentHashMap();
    private final int numPipelinesPerMetadataVolume;
    private final int heavyNodeCriteria;
    private final HDDSLayoutVersionManager scmLayoutVersionManager;
    private final EventPublisher scmNodeEventPublisher;
    private final SCMContext scmContext;

    public SCMNodeManager(OzoneConfiguration conf, SCMStorageConfig scmStorageConfig, EventPublisher eventPublisher, NetworkTopology networkTopology, SCMContext scmContext, HDDSLayoutVersionManager layoutVersionManager) {
        this.scmNodeEventPublisher = eventPublisher;
        this.nodeStateManager = new NodeStateManager((ConfigurationSource)conf, eventPublisher, (LayoutVersionManager)layoutVersionManager);
        this.version = VersionInfo.getLatestVersion();
        this.commandQueue = new CommandQueue();
        this.scmStorageConfig = scmStorageConfig;
        this.scmLayoutVersionManager = layoutVersionManager;
        LOG.info("Entering startup safe mode.");
        this.registerMXBean();
        this.metrics = SCMNodeMetrics.create(this);
        this.clusterMap = networkTopology;
        Class dnsToSwitchMappingClass = conf.getClass("net.topology.node.switch.mapping.impl", TableMapping.class, DNSToSwitchMapping.class);
        DNSToSwitchMapping newInstance = (DNSToSwitchMapping)ReflectionUtils.newInstance((Class)dnsToSwitchMappingClass, (Configuration)conf);
        this.dnsToSwitchMapping = newInstance instanceof CachedDNSToSwitchMapping ? newInstance : new CachedDNSToSwitchMapping(newInstance);
        this.useHostname = conf.getBoolean("dfs.datanode.use.datanode.hostname", false);
        this.numPipelinesPerMetadataVolume = conf.getInt("ozone.scm.pipeline.per.metadata.disk", 2);
        String dnLimit = conf.get("ozone.scm.datanode.pipeline.limit");
        this.heavyNodeCriteria = dnLimit == null ? 0 : Integer.parseInt(dnLimit);
        this.scmContext = scmContext;
    }

    private void registerMXBean() {
        this.nmInfoBean = MBeans.register((String)"SCMNodeManager", (String)"SCMNodeManagerInfo", (Object)this);
    }

    private void unregisterMXBean() {
        if (this.nmInfoBean != null) {
            MBeans.unregister((ObjectName)this.nmInfoBean);
            this.nmInfoBean = null;
        }
    }

    protected NodeStateManager getNodeStateManager() {
        return this.nodeStateManager;
    }

    @Override
    public List<DatanodeDetails> getNodes(NodeStatus nodeStatus) {
        return this.nodeStateManager.getNodes(nodeStatus).stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public List<DatanodeDetails> getNodes(HddsProtos.NodeOperationalState opState, HddsProtos.NodeState health) {
        return this.nodeStateManager.getNodes(opState, health).stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public List<DatanodeDetails> getAllNodes() {
        return this.nodeStateManager.getAllNodes().stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public int getNodeCount(NodeStatus nodeStatus) {
        return this.nodeStateManager.getNodeCount(nodeStatus);
    }

    @Override
    public int getNodeCount(HddsProtos.NodeOperationalState nodeOpState, HddsProtos.NodeState health) {
        return this.nodeStateManager.getNodeCount(nodeOpState, health);
    }

    @Override
    public NodeStatus getNodeStatus(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getNodeStatus(datanodeDetails);
    }

    @Override
    public void setNodeOperationalState(DatanodeDetails datanodeDetails, HddsProtos.NodeOperationalState newState) throws NodeNotFoundException {
        this.setNodeOperationalState(datanodeDetails, newState, 0L);
    }

    @Override
    public void setNodeOperationalState(DatanodeDetails datanodeDetails, HddsProtos.NodeOperationalState newState, long opStateExpiryEpocSec) throws NodeNotFoundException {
        this.nodeStateManager.setNodeOperationalState(datanodeDetails, newState, opStateExpiryEpocSec);
    }

    @Override
    public void close() throws IOException {
        this.unregisterMXBean();
        this.metrics.unRegister();
        this.nodeStateManager.close();
    }

    public VersionResponse getVersion(StorageContainerDatanodeProtocolProtos.SCMVersionRequestProto versionRequest) {
        return VersionResponse.newBuilder().setVersion(this.version.getVersion()).addValue("scmUuid", this.scmStorageConfig.getScmId()).addValue("clusterID", this.scmStorageConfig.getClusterID()).build();
    }

    @Override
    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto) {
        return this.register(datanodeDetails, nodeReport, pipelineReportsProto, StorageContainerDatanodeProtocolProtos.LayoutVersionProto.newBuilder().setMetadataLayoutVersion(this.scmLayoutVersionManager.getMetadataLayoutVersion()).setSoftwareLayoutVersion(this.scmLayoutVersionManager.getSoftwareLayoutVersion()).build());
    }

    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto, StorageContainerDatanodeProtocolProtos.LayoutVersionProto layoutInfo) {
        if (layoutInfo.getSoftwareLayoutVersion() != this.scmLayoutVersionManager.getSoftwareLayoutVersion()) {
            return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.errorNodeNotPermitted).setDatanode(datanodeDetails).setClusterID(this.scmStorageConfig.getClusterID()).build();
        }
        if (!this.isNodeRegistered(datanodeDetails).booleanValue()) {
            InetAddress dnAddress = Server.getRemoteIp();
            if (dnAddress != null) {
                datanodeDetails.setHostName(dnAddress.getHostName());
                datanodeDetails.setIpAddress(dnAddress.getHostAddress());
            }
            try {
                datanodeDetails.setNetworkName(datanodeDetails.getUuidString());
                String dnsName = this.useHostname ? datanodeDetails.getHostName() : datanodeDetails.getIpAddress();
                String networkLocation = this.nodeResolve(dnsName);
                if (networkLocation != null) {
                    datanodeDetails.setNetworkLocation(networkLocation);
                }
                this.clusterMap.add((Node)datanodeDetails);
                this.nodeStateManager.addNode(datanodeDetails, layoutInfo);
                DatanodeInfo dn = this.nodeStateManager.getNode(datanodeDetails);
                Preconditions.checkState((dn.getParent() != null ? 1 : 0) != 0);
                this.addEntryTodnsToUuidMap(dnsName, datanodeDetails.getUuidString());
                this.processNodeReport(datanodeDetails, nodeReport);
                LOG.info("Registered Data node : {}", (Object)datanodeDetails);
                this.scmNodeEventPublisher.fireEvent(SCMEvents.NEW_NODE, (Object)datanodeDetails);
            }
            catch (NodeAlreadyExistsException e) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Datanode is already registered. Datanode: {}", (Object)datanodeDetails.toString());
                }
            }
            catch (NodeNotFoundException e) {
                LOG.error("Cannot find datanode {} from nodeStateManager", (Object)datanodeDetails.toString());
            }
        }
        return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.success).setDatanode(datanodeDetails).setClusterID(this.scmStorageConfig.getClusterID()).build();
    }

    @SuppressFBWarnings(value={"AT_OPERATION_SEQUENCE_ON_CONCURRENT_ABSTRACTION"}, justification="The method is synchronized and this is the only place dnsToUuidMap is modified")
    private synchronized void addEntryTodnsToUuidMap(String dnsName, String uuid) {
        Set<String> dnList = this.dnsToUuidMap.get(dnsName);
        if (dnList == null) {
            dnList = ConcurrentHashMap.newKeySet();
            this.dnsToUuidMap.put(dnsName, dnList);
        }
        dnList.add(uuid);
    }

    public List<SCMCommand> processHeartbeat(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.LayoutVersionProto layoutInfo) {
        Preconditions.checkNotNull((Object)datanodeDetails, (Object)"Heartbeat is missing DatanodeDetails.");
        try {
            this.nodeStateManager.updateLastHeartbeatTime(datanodeDetails);
            this.nodeStateManager.updateLastKnownLayoutVersion(datanodeDetails, layoutInfo);
            this.metrics.incNumHBProcessed();
            this.updateDatanodeOpState(datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumHBProcessingFailed();
            LOG.error("SCM trying to process heartbeat from an unregistered node {}. Ignoring the heartbeat.", (Object)datanodeDetails);
        }
        return this.commandQueue.getCommand(datanodeDetails.getUuid());
    }

    boolean opStateDiffers(DatanodeDetails dnDetails, NodeStatus nodeStatus) {
        return nodeStatus.getOperationalState() != dnDetails.getPersistedOpState() || nodeStatus.getOpStateExpiryEpochSeconds() != dnDetails.getPersistedOpStateExpiryEpochSec();
    }

    protected void updateDatanodeOpState(DatanodeDetails reportedDn) throws NodeNotFoundException {
        NodeStatus scmStatus = this.getNodeStatus(reportedDn);
        if (this.opStateDiffers(reportedDn, scmStatus)) {
            if (this.scmContext.isLeader()) {
                LOG.info("Scheduling a command to update the operationalState persisted on {} as the reported value does not match the value stored in SCM ({}, {})", new Object[]{reportedDn, scmStatus.getOperationalState(), scmStatus.getOpStateExpiryEpochSeconds()});
                try {
                    SetNodeOperationalStateCommand command = new SetNodeOperationalStateCommand(Time.monotonicNow(), scmStatus.getOperationalState(), scmStatus.getOpStateExpiryEpochSeconds());
                    command.setTerm(this.scmContext.getTermOfLeader());
                    this.addDatanodeCommand(reportedDn.getUuid(), (SCMCommand)command);
                }
                catch (NotLeaderException nle) {
                    LOG.warn("Skip sending SetNodeOperationalStateCommand, since current SCM is not leader.", (Throwable)nle);
                    return;
                }
            } else {
                LOG.info("Update the operationalState saved in follower SCM for {} as the reported value does not match the value stored in SCM ({}, {})", new Object[]{reportedDn, scmStatus.getOperationalState(), scmStatus.getOpStateExpiryEpochSeconds()});
                this.setNodeOperationalState(reportedDn, reportedDn.getPersistedOpState(), reportedDn.getPersistedOpStateExpiryEpochSec());
            }
        }
        DatanodeInfo scmDnd = this.nodeStateManager.getNode(reportedDn);
        scmDnd.setPersistedOpStateExpiryEpochSec(reportedDn.getPersistedOpStateExpiryEpochSec());
        scmDnd.setPersistedOpState(reportedDn.getPersistedOpState());
    }

    public Boolean isNodeRegistered(DatanodeDetails datanodeDetails) {
        try {
            this.nodeStateManager.getNode(datanodeDetails);
            return true;
        }
        catch (NodeNotFoundException e) {
            return false;
        }
    }

    @Override
    public void processNodeReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing node report from [datanode={}]", (Object)datanodeDetails.getHostName());
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("HB is received from [datanode={}]: <json>{}</json>", (Object)datanodeDetails.getHostName(), (Object)nodeReport.toString().replaceAll("\n", "\\\\n"));
        }
        try {
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            if (nodeReport != null) {
                datanodeInfo.updateStorageReports(nodeReport.getStorageReportList());
                datanodeInfo.updateMetaDataStorageReports(nodeReport.getMetadataStorageReportList());
                this.metrics.incNumNodeReportProcessed();
            }
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumNodeReportProcessingFailed();
            LOG.warn("Got node report from unregistered datanode {}", (Object)datanodeDetails);
        }
    }

    @Override
    public void processLayoutVersionReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.LayoutVersionProto layoutVersionReport) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing Layout Version report from [datanode={}]", (Object)datanodeDetails.getHostName());
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("HB is received from [datanode={}]: <json>{}</json>", (Object)datanodeDetails.getHostName(), (Object)layoutVersionReport.toString().replaceAll("\n", "\\\\n"));
        }
        int scmSlv = this.scmLayoutVersionManager.getSoftwareLayoutVersion();
        int scmMlv = this.scmLayoutVersionManager.getMetadataLayoutVersion();
        int dnSlv = layoutVersionReport.getSoftwareLayoutVersion();
        int dnMlv = layoutVersionReport.getMetadataLayoutVersion();
        if (dnSlv > scmSlv) {
            LOG.error("Invalid data node in the cluster : {}. DataNode SoftwareLayoutVersion = {}, SCM SoftwareLayoutVersion = {}", new Object[]{datanodeDetails.getHostName(), dnSlv, scmSlv});
        }
        if (dnMlv < scmMlv) {
            LOG.warn("Data node {} can not be used in any pipeline in the cluster. DataNode MetadataLayoutVersion = {}, SCM MetadataLayoutVersion = {}", new Object[]{datanodeDetails.getHostName(), dnMlv, scmMlv});
            FinalizeNewLayoutVersionCommand finalizeCmd = new FinalizeNewLayoutVersionCommand(true, StorageContainerDatanodeProtocolProtos.LayoutVersionProto.newBuilder().setSoftwareLayoutVersion(dnSlv).setMetadataLayoutVersion(dnSlv).build());
            try {
                finalizeCmd.setTerm(this.scmContext.getTermOfLeader());
                this.scmNodeEventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)new CommandForDatanode(datanodeDetails.getUuid(), (SCMCommand)finalizeCmd));
            }
            catch (NotLeaderException ex) {
                LOG.warn("Skip sending finalize upgrade command since current SCM isnot leader.", (Throwable)ex);
            }
        }
    }

    @Override
    public SCMNodeStat getStats() {
        long capacity = 0L;
        long used = 0L;
        long remaining = 0L;
        for (SCMNodeStat stat : this.getNodeStats().values()) {
            capacity += stat.getCapacity().get().longValue();
            used += stat.getScmUsed().get().longValue();
            remaining += stat.getRemaining().get().longValue();
        }
        return new SCMNodeStat(capacity, used, remaining);
    }

    @Override
    public Map<DatanodeDetails, SCMNodeStat> getNodeStats() {
        HashMap<DatanodeDetails, SCMNodeStat> nodeStats = new HashMap<DatanodeDetails, SCMNodeStat>();
        List<DatanodeInfo> healthyNodes = this.nodeStateManager.getNodes(null, HddsProtos.NodeState.HEALTHY);
        List<DatanodeInfo> healthyReadOnlyNodes = this.nodeStateManager.getNodes(null, HddsProtos.NodeState.HEALTHY_READONLY);
        List<DatanodeInfo> staleNodes = this.nodeStateManager.getStaleNodes();
        ArrayList<DatanodeInfo> datanodes = new ArrayList<DatanodeInfo>(healthyNodes);
        datanodes.addAll(healthyReadOnlyNodes);
        datanodes.addAll(staleNodes);
        for (DatanodeInfo dnInfo : datanodes) {
            SCMNodeStat nodeStat = this.getNodeStatInternal(dnInfo);
            if (nodeStat == null) continue;
            nodeStats.put(dnInfo, nodeStat);
        }
        return nodeStats;
    }

    @Override
    public List<DatanodeUsageInfo> getMostOrLeastUsedDatanodes(boolean mostUsed) {
        List<DatanodeDetails> healthyNodes = this.getNodes(HddsProtos.NodeOperationalState.IN_SERVICE, HddsProtos.NodeState.HEALTHY);
        ArrayList<DatanodeUsageInfo> datanodeUsageInfoList = new ArrayList<DatanodeUsageInfo>(healthyNodes.size());
        for (DatanodeDetails node : healthyNodes) {
            SCMNodeStat stat = this.getNodeStatInternal(node);
            datanodeUsageInfoList.add(new DatanodeUsageInfo(node, stat));
        }
        if (mostUsed) {
            datanodeUsageInfoList.sort(DatanodeUsageInfo.getMostUtilized().reversed());
        } else {
            datanodeUsageInfoList.sort(DatanodeUsageInfo.getMostUtilized());
        }
        return datanodeUsageInfoList;
    }

    @Override
    public DatanodeUsageInfo getUsageInfo(DatanodeDetails dn) {
        SCMNodeStat stat = this.getNodeStatInternal(dn);
        return new DatanodeUsageInfo(dn, stat);
    }

    @Override
    public SCMNodeMetric getNodeStat(DatanodeDetails datanodeDetails) {
        SCMNodeStat nodeStat = this.getNodeStatInternal(datanodeDetails);
        return nodeStat != null ? new SCMNodeMetric(nodeStat) : null;
    }

    private SCMNodeStat getNodeStatInternal(DatanodeDetails datanodeDetails) {
        try {
            long capacity = 0L;
            long used = 0L;
            long remaining = 0L;
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = datanodeInfo.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                capacity += reportProto.getCapacity();
                used += reportProto.getScmUsed();
                remaining += reportProto.getRemaining();
            }
            return new SCMNodeStat(capacity, used, remaining);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)datanodeDetails.getUuid());
            return null;
        }
    }

    @Override
    public Map<String, Map<String, Integer>> getNodeCount() {
        HashMap<String, Map<String, Integer>> nodes = new HashMap<String, Map<String, Integer>>();
        for (HddsProtos.NodeOperationalState opState : HddsProtos.NodeOperationalState.values()) {
            HashMap<String, Integer> states = new HashMap<String, Integer>();
            for (HddsProtos.NodeState health : HddsProtos.NodeState.values()) {
                states.put(health.name(), 0);
            }
            nodes.put(opState.name(), states);
        }
        for (DatanodeInfo dni : this.nodeStateManager.getAllNodes()) {
            NodeStatus status = dni.getNodeStatus();
            ((Map)nodes.get(status.getOperationalState().name())).compute(status.getHealth().name(), (k, v) -> v + 1);
        }
        return nodes;
    }

    @Override
    public Map<String, Long> getNodeInfo() {
        HashMap<String, Long> nodeInfo = new HashMap<String, Long>();
        for (UsageStates s : UsageStates.values()) {
            for (UsageMetrics stat : UsageMetrics.values()) {
                nodeInfo.put(s.label + stat.name(), 0L);
            }
        }
        for (DatanodeInfo node : this.nodeStateManager.getAllNodes()) {
            String keyPrefix = "";
            NodeStatus status = node.getNodeStatus();
            if (status.isMaintenance()) {
                keyPrefix = UsageStates.MAINT.getLabel();
            } else if (status.isDecommission()) {
                keyPrefix = UsageStates.DECOM.getLabel();
            } else {
                if (!status.isAlive()) continue;
                keyPrefix = UsageStates.ONLINE.getLabel();
            }
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = node.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                if (reportProto.getStorageType() == StorageContainerDatanodeProtocolProtos.StorageTypeProto.DISK) {
                    nodeInfo.compute(keyPrefix + UsageMetrics.DiskCapacity.name(), (k, v) -> v + reportProto.getCapacity());
                    nodeInfo.compute(keyPrefix + UsageMetrics.DiskRemaining.name(), (k, v) -> v + reportProto.getRemaining());
                    nodeInfo.compute(keyPrefix + UsageMetrics.DiskUsed.name(), (k, v) -> v + reportProto.getScmUsed());
                    continue;
                }
                if (reportProto.getStorageType() != StorageContainerDatanodeProtocolProtos.StorageTypeProto.SSD) continue;
                nodeInfo.compute(keyPrefix + UsageMetrics.SSDCapacity.name(), (k, v) -> v + reportProto.getCapacity());
                nodeInfo.compute(keyPrefix + UsageMetrics.SSDRemaining.name(), (k, v) -> v + reportProto.getRemaining());
                nodeInfo.compute(keyPrefix + UsageMetrics.SSDUsed.name(), (k, v) -> v + reportProto.getScmUsed());
            }
        }
        return nodeInfo;
    }

    @Override
    public int minHealthyVolumeNum(List<DatanodeDetails> dnList) {
        ArrayList<Integer> volumeCountList = new ArrayList<Integer>(dnList.size());
        for (DatanodeDetails dn : dnList) {
            try {
                volumeCountList.add(this.nodeStateManager.getNode(dn).getHealthyVolumeCount());
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)dn.getUuid());
            }
        }
        Preconditions.checkArgument((!volumeCountList.isEmpty() ? 1 : 0) != 0);
        return (Integer)Collections.min(volumeCountList);
    }

    @Override
    public int pipelineLimit(DatanodeDetails dn) {
        try {
            if (this.heavyNodeCriteria > 0) {
                return this.heavyNodeCriteria;
            }
            if (this.nodeStateManager.getNode(dn).getHealthyVolumeCount() > 0) {
                return this.numPipelinesPerMetadataVolume * this.nodeStateManager.getNode(dn).getMetaDataVolumeCount();
            }
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)dn.getUuid());
        }
        return 0;
    }

    @Override
    public int minPipelineLimit(List<DatanodeDetails> dnList) {
        ArrayList<Integer> pipelineCountList = new ArrayList<Integer>(dnList.size());
        for (DatanodeDetails dn : dnList) {
            pipelineCountList.add(this.pipelineLimit(dn));
        }
        Preconditions.checkArgument((!pipelineCountList.isEmpty() ? 1 : 0) != 0);
        return (Integer)Collections.min(pipelineCountList);
    }

    @Override
    public Collection<DatanodeDetails> getPeerList(DatanodeDetails dn) {
        HashSet<DatanodeDetails> dns = new HashSet<DatanodeDetails>();
        Preconditions.checkNotNull((Object)dn);
        Set<PipelineID> pipelines = this.nodeStateManager.getPipelineByDnID(dn.getUuid());
        PipelineManager pipelineManager = this.scmContext.getScm().getPipelineManager();
        if (!pipelines.isEmpty()) {
            pipelines.forEach(id -> {
                try {
                    Pipeline pipeline = pipelineManager.getPipeline((PipelineID)id);
                    List peers = pipeline.getNodes();
                    dns.addAll(peers);
                }
                catch (PipelineNotFoundException pipelineNotFoundException) {
                    // empty catch block
                }
            });
        }
        dns.remove(dn);
        return dns;
    }

    @Override
    public Set<PipelineID> getPipelines(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelineByDnID(datanodeDetails.getUuid());
    }

    @Override
    public int getPipelinesCount(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelinesCount(datanodeDetails);
    }

    @Override
    public void addPipeline(Pipeline pipeline) {
        this.nodeStateManager.addPipeline(pipeline);
    }

    @Override
    public void removePipeline(Pipeline pipeline) {
        this.nodeStateManager.removePipeline(pipeline);
    }

    @Override
    public void addContainer(DatanodeDetails datanodeDetails, ContainerID containerId) throws NodeNotFoundException {
        this.nodeStateManager.addContainer(datanodeDetails.getUuid(), containerId);
    }

    @Override
    public void setContainers(DatanodeDetails datanodeDetails, Set<ContainerID> containerIds) throws NodeNotFoundException {
        this.nodeStateManager.setContainers(datanodeDetails.getUuid(), containerIds);
    }

    @Override
    public Set<ContainerID> getContainers(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getContainers(datanodeDetails.getUuid());
    }

    @Override
    public void addDatanodeCommand(UUID dnId, SCMCommand command) {
        this.commandQueue.addCommand(dnId, command);
    }

    public void onMessage(CommandForDatanode commandForDatanode, EventPublisher ignored) {
        this.addDatanodeCommand(commandForDatanode.getDatanodeId(), commandForDatanode.getCommand());
    }

    @Override
    public List<SCMCommand> getCommandQueue(UUID dnID) {
        return this.commandQueue.getCommand(dnID);
    }

    @Override
    public DatanodeDetails getNodeByUuid(String uuid) {
        if (Strings.isNullOrEmpty((String)uuid)) {
            LOG.warn("uuid is null");
            return null;
        }
        DatanodeDetails temp = DatanodeDetails.newBuilder().setUuid(UUID.fromString(uuid)).build();
        try {
            return this.nodeStateManager.getNode(temp);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot find node for uuid {}", (Object)uuid);
            return null;
        }
    }

    @Override
    public List<DatanodeDetails> getNodesByAddress(String address) {
        LinkedList<DatanodeDetails> results = new LinkedList<DatanodeDetails>();
        if (Strings.isNullOrEmpty((String)address)) {
            LOG.warn("address is null");
            return results;
        }
        Set<String> uuids = this.dnsToUuidMap.get(address);
        if (uuids == null) {
            LOG.warn("Cannot find node for address {}", (Object)address);
            return results;
        }
        for (String uuid : uuids) {
            DatanodeDetails temp = DatanodeDetails.newBuilder().setUuid(UUID.fromString(uuid)).build();
            try {
                results.add(this.nodeStateManager.getNode(temp));
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Cannot find node for uuid {}", (Object)uuid);
            }
        }
        return results;
    }

    @Override
    public NetworkTopology getClusterNetworkTopologyMap() {
        return this.clusterMap;
    }

    private String nodeResolve(String hostname) {
        ArrayList<String> hosts = new ArrayList<String>(1);
        hosts.add(hostname);
        List resolvedHosts = this.dnsToSwitchMapping.resolve(hosts);
        if (resolvedHosts != null && !resolvedHosts.isEmpty()) {
            String location = (String)resolvedHosts.get(0);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Resolve datanode {} return location {}", (Object)hostname, (Object)location);
            }
            return location;
        }
        LOG.error("Node {} Resolution failed. Please make sure that DNS table mapping or configured mapping is functional.", (Object)hostname);
        return null;
    }

    @VisibleForTesting
    ScheduledFuture pauseHealthCheck() {
        return this.nodeStateManager.pause();
    }

    @VisibleForTesting
    ScheduledFuture unpauseHealthCheck() {
        return this.nodeStateManager.unpause();
    }

    @VisibleForTesting
    long getSkippedHealthChecks() {
        return this.nodeStateManager.getSkippedHealthChecks();
    }

    @Override
    @VisibleForTesting
    public HDDSLayoutVersionManager getLayoutVersionManager() {
        return this.scmLayoutVersionManager;
    }

    @Override
    @VisibleForTesting
    public void forceNodesToHealthyReadOnly() {
        this.nodeStateManager.forceNodesToHealthyReadOnly();
    }

    private static enum UsageStates {
        ONLINE(""),
        MAINT("Maintenance"),
        DECOM("Decommissioned");

        private final String label;

        public String getLabel() {
            return this.label;
        }

        private UsageStates(String label) {
            this.label = label;
        }
    }

    private static enum UsageMetrics {
        DiskCapacity,
        DiskUsed,
        DiskRemaining,
        SSDCapacity,
        SSDUsed,
        SSDRemaining;

    }
}

