/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.DatanodeAdminError;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ReplicationManager;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.node.DatanodeAdminMonitor;
import org.apache.hadoop.hdds.scm.node.DatanodeAdminMonitorImpl;
import org.apache.hadoop.hdds.scm.node.InvalidHostStringException;
import org.apache.hadoop.hdds.scm.node.InvalidNodeStateException;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeDecommissionManager {
    private ScheduledExecutorService executor;
    private DatanodeAdminMonitor monitor;
    private NodeManager nodeManager;
    private SCMContext scmContext;
    private EventPublisher eventQueue;
    private ReplicationManager replicationManager;
    private OzoneConfiguration conf;
    private boolean useHostnames;
    private long monitorInterval;
    private static final Logger LOG = LoggerFactory.getLogger(NodeDecommissionManager.class);

    private List<DatanodeDetails> mapHostnamesToDatanodes(List<String> hosts) throws InvalidHostStringException {
        LinkedList<DatanodeDetails> results = new LinkedList<DatanodeDetails>();
        for (String hostString : hosts) {
            InetAddress addr;
            HostDefinition host = new HostDefinition(hostString);
            try {
                addr = InetAddress.getByName(host.getHostname());
            }
            catch (UnknownHostException e) {
                throw new InvalidHostStringException("Unable to resolve host " + host.getRawHostname(), e);
            }
            String dnsName = this.useHostnames ? addr.getHostName() : addr.getHostAddress();
            List<DatanodeDetails> found = this.nodeManager.getNodesByAddress(dnsName);
            if (found.size() == 0) {
                throw new InvalidHostStringException("Host " + host.getRawHostname() + " (" + dnsName + ") is not running any datanodes registered with SCM. Please check the host name.");
            }
            if (found.size() == 1) {
                if (host.getPort() != -1 && !this.validateDNPortMatch(host.getPort(), found.get(0))) {
                    throw new InvalidHostStringException("Host " + host.getRawHostname() + " is running a datanode registered with SCM, but the port number doesn't match. Please check the port number.");
                }
                results.add(found.get(0));
                continue;
            }
            if (found.size() <= 1) continue;
            DatanodeDetails match = null;
            for (DatanodeDetails dn : found) {
                if (!this.validateDNPortMatch(host.getPort(), dn)) continue;
                match = dn;
                break;
            }
            if (match == null) {
                throw new InvalidHostStringException("Host " + host.getRawHostname() + " is running multiple datanodes registered with SCM, but no port numbers match. Please check the port number.");
            }
            results.add(match);
        }
        return results;
    }

    private boolean validateDNPortMatch(int port, DatanodeDetails dn) {
        for (DatanodeDetails.Port p : dn.getPorts()) {
            if (p.getValue() != port) continue;
            return true;
        }
        return false;
    }

    public NodeDecommissionManager(OzoneConfiguration config, NodeManager nm, ContainerManager containerManager, SCMContext scmContext, EventPublisher eventQueue, ReplicationManager rm) {
        this.nodeManager = nm;
        this.conf = config;
        this.scmContext = scmContext;
        this.eventQueue = eventQueue;
        this.replicationManager = rm;
        this.executor = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("DatanodeAdminManager-%d").setDaemon(true).build());
        this.useHostnames = this.conf.getBoolean("dfs.datanode.use.datanode.hostname", false);
        this.monitorInterval = this.conf.getTimeDuration("ozone.scm.datanode.admin.monitor.interval", "30s", TimeUnit.SECONDS);
        if (this.monitorInterval <= 0L) {
            LOG.warn("{} must be greater than zero, defaulting to {}", (Object)"ozone.scm.datanode.admin.monitor.interval", (Object)"30s");
            this.conf.set("ozone.scm.datanode.admin.monitor.interval", "30s");
            this.monitorInterval = this.conf.getTimeDuration("ozone.scm.datanode.admin.monitor.interval", "30s", TimeUnit.SECONDS);
        }
        this.monitor = new DatanodeAdminMonitorImpl(this.conf, eventQueue, this.nodeManager, this.replicationManager);
        this.executor.scheduleAtFixedRate(this.monitor, this.monitorInterval, this.monitorInterval, TimeUnit.SECONDS);
    }

    @VisibleForTesting
    public DatanodeAdminMonitor getMonitor() {
        return this.monitor;
    }

    public synchronized List<DatanodeAdminError> decommissionNodes(List<String> nodes) throws InvalidHostStringException {
        List<DatanodeDetails> dns = this.mapHostnamesToDatanodes(nodes);
        ArrayList<DatanodeAdminError> errors = new ArrayList<DatanodeAdminError>();
        for (DatanodeDetails dn : dns) {
            try {
                this.startDecommission(dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("The host {} was not found in SCM. Ignoring the request to decommission it", (Object)dn.getHostName());
                errors.add(new DatanodeAdminError(dn.getHostName(), "The host was not found in SCM"));
            }
            catch (InvalidNodeStateException e) {
                errors.add(new DatanodeAdminError(dn.getHostName(), e.getMessage()));
            }
        }
        return errors;
    }

    public synchronized void continueAdminForNode(DatanodeDetails dn) throws NodeNotFoundException {
        if (!this.scmContext.isLeader()) {
            LOG.info("follower SCM ignored continue admin for datanode {}", (Object)dn);
            return;
        }
        HddsProtos.NodeOperationalState opState = this.getNodeStatus(dn).getOperationalState();
        if (opState == HddsProtos.NodeOperationalState.DECOMMISSIONING || opState == HddsProtos.NodeOperationalState.ENTERING_MAINTENANCE || opState == HddsProtos.NodeOperationalState.IN_MAINTENANCE) {
            LOG.info("Continue admin for datanode {}", (Object)dn);
            this.monitor.startMonitoring(dn);
        }
    }

    public synchronized void startDecommission(DatanodeDetails dn) throws NodeNotFoundException, InvalidNodeStateException {
        NodeStatus nodeStatus = this.getNodeStatus(dn);
        HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
        if (opState == HddsProtos.NodeOperationalState.IN_SERVICE) {
            LOG.info("Starting Decommission for node {}", (Object)dn);
            this.nodeManager.setNodeOperationalState(dn, HddsProtos.NodeOperationalState.DECOMMISSIONING);
            this.monitor.startMonitoring(dn);
        } else if (nodeStatus.isDecommission()) {
            LOG.info("Start Decommission called on node {} in state {}. Nothing to do.", (Object)dn, (Object)opState);
        } else {
            LOG.error("Cannot decommission node {} in state {}", (Object)dn, (Object)opState);
            throw new InvalidNodeStateException("Cannot decommission node " + dn + " in state " + opState);
        }
    }

    public synchronized List<DatanodeAdminError> recommissionNodes(List<String> nodes) throws InvalidHostStringException {
        List<DatanodeDetails> dns = this.mapHostnamesToDatanodes(nodes);
        ArrayList<DatanodeAdminError> errors = new ArrayList<DatanodeAdminError>();
        for (DatanodeDetails dn : dns) {
            try {
                this.recommission(dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Host {} was not found in SCM. Ignoring the request to recommission it.", (Object)dn.getHostName());
                errors.add(new DatanodeAdminError(dn.getHostName(), "The host was not found in SCM"));
            }
        }
        return errors;
    }

    public synchronized void recommission(DatanodeDetails dn) throws NodeNotFoundException {
        NodeStatus nodeStatus = this.getNodeStatus(dn);
        HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
        if (opState != HddsProtos.NodeOperationalState.IN_SERVICE) {
            this.monitor.stopMonitoring(dn);
            LOG.info("Queued node {} for recommission", (Object)dn);
        } else {
            LOG.info("Recommission called on node {} with state {}. Nothing to do.", (Object)dn, (Object)opState);
        }
    }

    public synchronized List<DatanodeAdminError> startMaintenanceNodes(List<String> nodes, int endInHours) throws InvalidHostStringException {
        List<DatanodeDetails> dns = this.mapHostnamesToDatanodes(nodes);
        ArrayList<DatanodeAdminError> errors = new ArrayList<DatanodeAdminError>();
        for (DatanodeDetails dn : dns) {
            try {
                this.startMaintenance(dn, endInHours);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("The host {} was not found in SCM. Ignoring the request to start maintenance on it", (Object)dn.getHostName());
            }
            catch (InvalidNodeStateException e) {
                errors.add(new DatanodeAdminError(dn.getHostName(), e.getMessage()));
            }
        }
        return errors;
    }

    public synchronized void startMaintenance(DatanodeDetails dn, int endInHours) throws NodeNotFoundException, InvalidNodeStateException {
        NodeStatus nodeStatus = this.getNodeStatus(dn);
        HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
        long maintenanceEnd = 0L;
        if (endInHours != 0) {
            maintenanceEnd = System.currentTimeMillis() / 1000L + (long)endInHours * 60L * 60L;
        }
        if (opState == HddsProtos.NodeOperationalState.IN_SERVICE) {
            this.nodeManager.setNodeOperationalState(dn, HddsProtos.NodeOperationalState.ENTERING_MAINTENANCE, maintenanceEnd);
            this.monitor.startMonitoring(dn);
            LOG.info("Starting Maintenance for node {}", (Object)dn);
        } else if (nodeStatus.isMaintenance()) {
            LOG.info("Starting Maintenance called on node {} with state {}. Nothing to do.", (Object)dn, (Object)opState);
        } else {
            LOG.error("Cannot start maintenance on node {} in state {}", (Object)dn, (Object)opState);
            throw new InvalidNodeStateException("Cannot start maintenance on node " + dn + " in state " + opState);
        }
    }

    public void stop() {
        if (this.executor != null) {
            this.executor.shutdown();
        }
    }

    private NodeStatus getNodeStatus(DatanodeDetails dn) throws NodeNotFoundException {
        return this.nodeManager.getNodeStatus(dn);
    }

    public void onBecomeLeader() {
        this.nodeManager.getAllNodes().forEach(datanodeDetails -> {
            try {
                this.continueAdminForNode((DatanodeDetails)datanodeDetails);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("NodeNotFound when adding the node to the decommissionManager", (Throwable)e);
            }
        });
    }

    static class HostDefinition {
        private String rawHostname;
        private String hostname;
        private int port;

        HostDefinition(String hostname) throws InvalidHostStringException {
            this.rawHostname = hostname;
            this.parseHostname();
        }

        public String getRawHostname() {
            return this.rawHostname;
        }

        public String getHostname() {
            return this.hostname;
        }

        public int getPort() {
            return this.port;
        }

        private void parseHostname() throws InvalidHostStringException {
            try {
                URI uri = new URI("empty://" + this.rawHostname.trim());
                this.hostname = uri.getHost();
                this.port = uri.getPort();
                if (this.hostname == null) {
                    throw new InvalidHostStringException("The string " + this.rawHostname + " does not contain a value hostname or hostname:port definition");
                }
            }
            catch (URISyntaxException e) {
                throw new InvalidHostStringException("Unable to parse the hoststring " + this.rawHostname, e);
            }
        }
    }
}

