/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Queue;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.ReplicationManager;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.node.DatanodeAdminMonitor;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatanodeAdminMonitorImpl
implements DatanodeAdminMonitor {
    private OzoneConfiguration conf;
    private EventPublisher eventQueue;
    private NodeManager nodeManager;
    private ReplicationManager replicationManager;
    private Queue<DatanodeDetails> pendingNodes = new ArrayDeque<DatanodeDetails>();
    private Queue<DatanodeDetails> cancelledNodes = new ArrayDeque<DatanodeDetails>();
    private Set<DatanodeDetails> trackedNodes = new HashSet<DatanodeDetails>();
    private static final Logger LOG = LoggerFactory.getLogger(DatanodeAdminMonitorImpl.class);

    public DatanodeAdminMonitorImpl(OzoneConfiguration conf, EventPublisher eventQueue, NodeManager nodeManager, ReplicationManager replicationManager) {
        this.conf = conf;
        this.eventQueue = eventQueue;
        this.nodeManager = nodeManager;
        this.replicationManager = replicationManager;
    }

    @Override
    public synchronized void startMonitoring(DatanodeDetails dn) {
        this.cancelledNodes.remove(dn);
        this.pendingNodes.add(dn);
    }

    @Override
    public synchronized void stopMonitoring(DatanodeDetails dn) {
        this.pendingNodes.remove(dn);
        this.cancelledNodes.add(dn);
    }

    @Override
    public synchronized Set<DatanodeDetails> getTrackedNodes() {
        return Collections.unmodifiableSet(this.trackedNodes);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            DatanodeAdminMonitorImpl datanodeAdminMonitorImpl = this;
            synchronized (datanodeAdminMonitorImpl) {
                this.processCancelledNodes();
                this.processPendingNodes();
            }
            this.processTransitioningNodes();
            if (this.trackedNodes.size() > 0 || this.pendingNodes.size() > 0) {
                LOG.info("There are {} nodes tracked for decommission and maintenance. {} pending nodes.", (Object)this.trackedNodes.size(), (Object)this.pendingNodes.size());
            }
        }
        catch (Exception e) {
            LOG.error("Caught an error in the DatanodeAdminMonitor", (Throwable)e);
        }
    }

    public int getPendingCount() {
        return this.pendingNodes.size();
    }

    public int getCancelledCount() {
        return this.cancelledNodes.size();
    }

    public int getTrackedNodeCount() {
        return this.trackedNodes.size();
    }

    private void processCancelledNodes() {
        while (!this.cancelledNodes.isEmpty()) {
            DatanodeDetails dn = this.cancelledNodes.poll();
            try {
                this.stopTrackingNode(dn);
                this.putNodeBackInService(dn);
                LOG.info("Recommissioned node {}", (Object)dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Failed processing the cancel admin request for {}", (Object)dn, (Object)e);
            }
        }
    }

    private void processPendingNodes() {
        while (!this.pendingNodes.isEmpty()) {
            this.startTrackingNode(this.pendingNodes.poll());
        }
    }

    private void processTransitioningNodes() {
        Iterator<DatanodeDetails> iterator = this.trackedNodes.iterator();
        while (iterator.hasNext()) {
            DatanodeDetails dn = iterator.next();
            try {
                NodeStatus status = this.getNodeStatus(dn);
                if (!this.shouldContinueWorkflow(dn, status)) {
                    this.abortWorkflow(dn);
                    iterator.remove();
                    continue;
                }
                if (status.isMaintenance() && status.operationalStateExpired()) {
                    this.completeMaintenance(dn);
                    iterator.remove();
                    continue;
                }
                if (!status.isDecommissioning() && !status.isEnteringMaintenance() || !this.checkPipelinesClosedOnNode(dn) || status.getOperationalState() != dn.getPersistedOpState() || !this.checkContainersReplicatedOnNode(dn)) continue;
                status = this.getNodeStatus(dn);
                if (status.isDead()) {
                    LOG.warn("Datanode {} is dead and the admin workflow cannot continue. The node will be put back to IN_SERVICE and handled as a dead node", (Object)dn);
                    this.putNodeBackInService(dn);
                    iterator.remove();
                    continue;
                }
                if (status.isDecommissioning()) {
                    this.completeDecommission(dn);
                    iterator.remove();
                    continue;
                }
                if (!status.isEnteringMaintenance()) continue;
                this.putIntoMaintenance(dn);
            }
            catch (NodeNotFoundException e) {
                LOG.error("An unexpected error occurred processing datanode {}. Aborting the admin workflow", (Object)dn, (Object)e);
                this.abortWorkflow(dn);
                iterator.remove();
            }
        }
    }

    private boolean shouldContinueWorkflow(DatanodeDetails dn, NodeStatus nodeStatus) {
        if (!nodeStatus.isDecommission() && !nodeStatus.isMaintenance()) {
            LOG.warn("Datanode {} has an operational state of {} when it should be undergoing decommission or maintenance. Aborting admin for this node.", (Object)dn, (Object)nodeStatus.getOperationalState());
            return false;
        }
        if (nodeStatus.isDead() && !nodeStatus.isInMaintenance()) {
            LOG.error("Datanode {} is dead but is not IN_MAINTENANCE. Aborting the admin workflow for this node", (Object)dn);
            return false;
        }
        return true;
    }

    private boolean checkPipelinesClosedOnNode(DatanodeDetails dn) throws NodeNotFoundException {
        Set<PipelineID> pipelines = this.nodeManager.getPipelines(dn);
        NodeStatus status = this.nodeManager.getNodeStatus(dn);
        if (pipelines == null || pipelines.size() == 0 || status.operationalStateExpired()) {
            return true;
        }
        LOG.info("Waiting for pipelines to close for {}. There are {} pipelines", (Object)dn, (Object)pipelines.size());
        return false;
    }

    private boolean checkContainersReplicatedOnNode(DatanodeDetails dn) throws NodeNotFoundException {
        int sufficientlyReplicated = 0;
        int underReplicated = 0;
        int unhealthy = 0;
        ArrayList<ContainerID> underReplicatedIDs = new ArrayList<ContainerID>();
        ArrayList<ContainerID> unhealthyIDs = new ArrayList<ContainerID>();
        Set<ContainerID> containers = this.nodeManager.getContainers(dn);
        for (ContainerID cid : containers) {
            try {
                ContainerReplicaCount replicaSet = this.replicationManager.getContainerReplicaCount(cid);
                if (replicaSet.isSufficientlyReplicated()) {
                    ++sufficientlyReplicated;
                } else {
                    if (LOG.isDebugEnabled()) {
                        underReplicatedIDs.add(cid);
                    }
                    ++underReplicated;
                }
                if (replicaSet.isHealthy()) continue;
                if (LOG.isDebugEnabled()) {
                    unhealthyIDs.add(cid);
                }
                ++unhealthy;
            }
            catch (ContainerNotFoundException e) {
                LOG.warn("ContainerID {} present in node list for {} but not found in containerManager", (Object)cid, (Object)dn);
            }
        }
        LOG.info("{} has {} sufficientlyReplicated, {} underReplicated and {} unhealthy containers", new Object[]{dn, sufficientlyReplicated, underReplicated, unhealthy});
        if (LOG.isDebugEnabled() && underReplicatedIDs.size() < 10000 && unhealthyIDs.size() < 10000) {
            LOG.debug("{} has {} underReplicated [{}] and {} unhealthy [{}] containers", new Object[]{dn, underReplicated, underReplicatedIDs.stream().map(Object::toString).collect(Collectors.joining(", ")), unhealthy, unhealthyIDs.stream().map(Object::toString).collect(Collectors.joining(", "))});
        }
        return underReplicated == 0 && unhealthy == 0;
    }

    private void completeDecommission(DatanodeDetails dn) throws NodeNotFoundException {
        this.setNodeOpState(dn, HddsProtos.NodeOperationalState.DECOMMISSIONED);
        LOG.info("Datanode {} has completed the admin workflow. The operational state has been set to {}", (Object)dn, (Object)HddsProtos.NodeOperationalState.DECOMMISSIONED);
    }

    private void putIntoMaintenance(DatanodeDetails dn) throws NodeNotFoundException {
        LOG.info("Datanode {} has entered maintenance", (Object)dn);
        this.setNodeOpState(dn, HddsProtos.NodeOperationalState.IN_MAINTENANCE);
    }

    private void completeMaintenance(DatanodeDetails dn) throws NodeNotFoundException {
        LOG.info("Datanode {} has ended maintenance automatically", (Object)dn);
        this.putNodeBackInService(dn);
    }

    private void startTrackingNode(DatanodeDetails dn) {
        this.eventQueue.fireEvent(SCMEvents.START_ADMIN_ON_NODE, (Object)dn);
        this.trackedNodes.add(dn);
    }

    private void stopTrackingNode(DatanodeDetails dn) {
        this.trackedNodes.remove(dn);
    }

    private void abortWorkflow(DatanodeDetails dn) {
        try {
            this.putNodeBackInService(dn);
        }
        catch (NodeNotFoundException e) {
            LOG.error("Unable to set the node OperationalState for {} while aborting the datanode admin workflow", (Object)dn);
        }
    }

    private void putNodeBackInService(DatanodeDetails dn) throws NodeNotFoundException {
        this.setNodeOpState(dn, HddsProtos.NodeOperationalState.IN_SERVICE);
    }

    private void setNodeOpState(DatanodeDetails dn, HddsProtos.NodeOperationalState state) throws NodeNotFoundException {
        long expiry = 0L;
        if (state == HddsProtos.NodeOperationalState.IN_MAINTENANCE || state == HddsProtos.NodeOperationalState.ENTERING_MAINTENANCE) {
            NodeStatus status = this.nodeManager.getNodeStatus(dn);
            expiry = status.getOpStateExpiryEpochSeconds();
        }
        this.nodeManager.setNodeOperationalState(dn, state, expiry);
    }

    private NodeStatus getNodeStatus(DatanodeDetails dnd) throws NodeNotFoundException {
        return this.nodeManager.getNodeStatus(dnd);
    }
}

