/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.balancer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.scm.PlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ReplicationManager;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerConfiguration;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerMetrics;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerSelectionCriteria;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerMoveSelection;
import org.apache.hadoop.hdds.scm.container.balancer.FindTargetGreedy;
import org.apache.hadoop.hdds.scm.container.balancer.FindTargetStrategy;
import org.apache.hadoop.hdds.scm.container.placement.metrics.LongMetric;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.node.DatanodeUsageInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerBalancer {
    public static final Logger LOG = LoggerFactory.getLogger(ContainerBalancer.class);
    private NodeManager nodeManager;
    private ContainerManager containerManager;
    private ReplicationManager replicationManager;
    private OzoneConfiguration ozoneConfiguration;
    private final SCMContext scmContext;
    private double threshold;
    private int totalNodesInCluster;
    private double maxDatanodesRatioToInvolvePerIteration;
    private long maxSizeToMovePerIteration;
    private int countDatanodesInvolvedPerIteration;
    private long sizeMovedPerIteration;
    private int idleIteration;
    private List<DatanodeUsageInfo> unBalancedNodes;
    private List<DatanodeUsageInfo> overUtilizedNodes;
    private List<DatanodeUsageInfo> underUtilizedNodes;
    private List<DatanodeUsageInfo> withinThresholdUtilizedNodes;
    private ContainerBalancerConfiguration config;
    private ContainerBalancerMetrics metrics;
    private long clusterCapacity;
    private long clusterUsed;
    private long clusterRemaining;
    private double clusterAvgUtilisation;
    private double upperLimit;
    private volatile boolean balancerRunning;
    private Thread currentBalancingThread;
    private Lock lock;
    private ContainerBalancerSelectionCriteria selectionCriteria;
    private Map<DatanodeDetails, ContainerMoveSelection> sourceToTargetMap;
    private Map<DatanodeDetails, Long> sizeLeavingNode;
    private Map<DatanodeDetails, Long> sizeEnteringNode;
    private Set<ContainerID> selectedContainers;
    private FindTargetStrategy findTargetStrategy;
    private Map<ContainerMoveSelection, CompletableFuture<ReplicationManager.MoveResult>> moveSelectionToFutureMap;

    public ContainerBalancer(NodeManager nodeManager, ContainerManager containerManager, ReplicationManager replicationManager, OzoneConfiguration ozoneConfiguration, SCMContext scmContext, PlacementPolicy placementPolicy) {
        this.nodeManager = nodeManager;
        this.containerManager = containerManager;
        this.replicationManager = replicationManager;
        this.ozoneConfiguration = ozoneConfiguration;
        this.config = new ContainerBalancerConfiguration(ozoneConfiguration);
        this.metrics = new ContainerBalancerMetrics();
        this.scmContext = scmContext;
        this.selectedContainers = new HashSet<ContainerID>();
        this.overUtilizedNodes = new ArrayList<DatanodeUsageInfo>();
        this.underUtilizedNodes = new ArrayList<DatanodeUsageInfo>();
        this.withinThresholdUtilizedNodes = new ArrayList<DatanodeUsageInfo>();
        this.unBalancedNodes = new ArrayList<DatanodeUsageInfo>();
        this.lock = new ReentrantLock();
        this.findTargetStrategy = new FindTargetGreedy(containerManager, placementPolicy);
    }

    public boolean start(ContainerBalancerConfiguration balancerConfiguration) {
        this.lock.lock();
        try {
            if (this.balancerRunning) {
                LOG.error("Container Balancer is already running.");
                boolean bl = false;
                return bl;
            }
            this.balancerRunning = true;
            this.config = balancerConfiguration;
            this.ozoneConfiguration = this.config.getOzoneConfiguration();
            LOG.info("Starting Container Balancer...{}", (Object)this);
            this.currentBalancingThread = new Thread(this::balance);
            this.currentBalancingThread.setName("ContainerBalancer");
            this.currentBalancingThread.setDaemon(true);
            this.currentBalancingThread.start();
        }
        finally {
            this.lock.unlock();
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void balance() {
        this.idleIteration = this.config.getIdleIteration();
        if (this.idleIteration == -1) {
            this.idleIteration = Integer.MAX_VALUE;
        }
        this.threshold = this.config.getThreshold();
        this.maxDatanodesRatioToInvolvePerIteration = this.config.getMaxDatanodesRatioToInvolvePerIteration();
        this.maxSizeToMovePerIteration = this.config.getMaxSizeToMovePerIteration();
        for (int i = 0; i < this.idleIteration && this.balancerRunning; ++i) {
            if (!this.initializeIteration()) {
                this.stop();
                return;
            }
            IterationResult iR = this.doIteration();
            if (iR == IterationResult.CAN_NOT_BALANCE_ANY_MORE) {
                this.stop();
                return;
            }
            if (!this.isBalancerRunning()) {
                return;
            }
            if (i == this.idleIteration - 1) continue;
            ContainerBalancer containerBalancer = this;
            synchronized (containerBalancer) {
                try {
                    this.wait(this.config.getBalancingInterval().toMillis());
                }
                catch (InterruptedException e) {
                    LOG.info("Container Balancer was interrupted while waiting for next iteration.");
                    this.stop();
                    return;
                }
            }
        }
        this.stop();
    }

    private boolean initializeIteration() {
        if (this.scmContext.isInSafeMode()) {
            LOG.error("Container Balancer cannot operate while SCM is in Safe Mode.");
            return false;
        }
        if (!this.scmContext.isLeader()) {
            LOG.warn("Current SCM is not the leader.");
            return false;
        }
        List<DatanodeUsageInfo> datanodeUsageInfos = this.nodeManager.getMostOrLeastUsedDatanodes(true);
        if (datanodeUsageInfos.isEmpty()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Container Balancer could not retrieve nodes from Node Manager.");
            }
            return false;
        }
        this.totalNodesInCluster = datanodeUsageInfos.size();
        this.clusterCapacity = 0L;
        this.clusterUsed = 0L;
        this.clusterRemaining = 0L;
        this.selectedContainers.clear();
        this.overUtilizedNodes.clear();
        this.underUtilizedNodes.clear();
        this.withinThresholdUtilizedNodes.clear();
        this.unBalancedNodes.clear();
        this.countDatanodesInvolvedPerIteration = 0;
        this.sizeMovedPerIteration = 0L;
        this.clusterAvgUtilisation = this.calculateAvgUtilization(datanodeUsageInfos);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Average utilization of the cluster is {}", (Object)this.clusterAvgUtilisation);
        }
        double lowerLimit = this.clusterAvgUtilisation - this.threshold;
        this.upperLimit = this.clusterAvgUtilisation + this.threshold;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Lower limit for utilization is {} and Upper limit for utilization is {}", (Object)lowerLimit, (Object)this.upperLimit);
        }
        long countDatanodesToBalance = 0L;
        double overLoadedBytes = 0.0;
        double underLoadedBytes = 0.0;
        for (DatanodeUsageInfo datanodeUsageInfo2 : datanodeUsageInfos) {
            double utilization = datanodeUsageInfo2.calculateUtilization();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Utilization for node {} is {}", (Object)datanodeUsageInfo2.getDatanodeDetails().getUuidString(), (Object)utilization);
            }
            if (utilization > this.upperLimit) {
                this.overUtilizedNodes.add(datanodeUsageInfo2);
                ++countDatanodesToBalance;
                overLoadedBytes += this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), utilization) - this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), this.upperLimit);
                continue;
            }
            if (utilization < lowerLimit) {
                this.underUtilizedNodes.add(datanodeUsageInfo2);
                ++countDatanodesToBalance;
                underLoadedBytes += this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), lowerLimit) - this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), utilization);
                continue;
            }
            this.withinThresholdUtilizedNodes.add(datanodeUsageInfo2);
        }
        this.metrics.setDatanodesNumToBalance(new LongMetric(countDatanodesToBalance));
        Collections.reverse(this.underUtilizedNodes);
        this.unBalancedNodes = new ArrayList<DatanodeUsageInfo>(this.overUtilizedNodes.size() + this.underUtilizedNodes.size());
        this.unBalancedNodes.addAll(this.overUtilizedNodes);
        this.unBalancedNodes.addAll(this.underUtilizedNodes);
        if (this.unBalancedNodes.isEmpty()) {
            LOG.info("Did not find any unbalanced Datanodes.");
            return false;
        }
        LOG.info("Container Balancer has identified {} Over-Utilized and {} Under-Utilized Datanodes that need to be balanced.", (Object)this.overUtilizedNodes.size(), (Object)this.underUtilizedNodes.size());
        this.selectionCriteria = new ContainerBalancerSelectionCriteria(this.config, this.nodeManager, this.replicationManager, this.containerManager);
        this.sourceToTargetMap = new HashMap<DatanodeDetails, ContainerMoveSelection>(this.overUtilizedNodes.size() + this.withinThresholdUtilizedNodes.size());
        this.sizeLeavingNode = new HashMap<DatanodeDetails, Long>(this.overUtilizedNodes.size() + this.withinThresholdUtilizedNodes.size());
        this.overUtilizedNodes.forEach(datanodeUsageInfo -> this.sizeLeavingNode.put(datanodeUsageInfo.getDatanodeDetails(), 0L));
        this.withinThresholdUtilizedNodes.forEach(datanodeUsageInfo -> this.sizeLeavingNode.put(datanodeUsageInfo.getDatanodeDetails(), 0L));
        this.sizeEnteringNode = new HashMap<DatanodeDetails, Long>(this.underUtilizedNodes.size() + this.withinThresholdUtilizedNodes.size());
        this.underUtilizedNodes.forEach(datanodeUsageInfo -> this.sizeEnteringNode.put(datanodeUsageInfo.getDatanodeDetails(), 0L));
        this.withinThresholdUtilizedNodes.forEach(datanodeUsageInfo -> this.sizeEnteringNode.put(datanodeUsageInfo.getDatanodeDetails(), 0L));
        return true;
    }

    private IterationResult doIteration() {
        ContainerMoveSelection moveSelection;
        IterationResult result;
        DatanodeDetails source;
        List<DatanodeDetails> potentialTargets = this.getPotentialTargets();
        HashSet<DatanodeDetails> selectedTargets = new HashSet<DatanodeDetails>(potentialTargets.size());
        this.moveSelectionToFutureMap = new HashMap<ContainerMoveSelection, CompletableFuture<ReplicationManager.MoveResult>>(this.unBalancedNodes.size());
        boolean isMoveGenerated = false;
        for (DatanodeUsageInfo datanodeUsageInfo : this.overUtilizedNodes) {
            source = datanodeUsageInfo.getDatanodeDetails();
            result = this.checkConditionsForBalancing();
            if (result != null) {
                LOG.info("Exiting current iteration: {}", (Object)result);
                return result;
            }
            moveSelection = this.matchSourceWithTarget(source, potentialTargets);
            if (moveSelection == null) continue;
            isMoveGenerated = true;
            LOG.info("ContainerBalancer is trying to move container {} from source datanode {} to target datanode {}", new Object[]{moveSelection.getContainerID().toString(), source.getUuidString(), moveSelection.getTargetNode().getUuidString()});
            if (!this.moveContainer(source, moveSelection)) continue;
            potentialTargets = this.updateTargetsAndSelectionCriteria(potentialTargets, selectedTargets, moveSelection, source);
        }
        if (selectedTargets.size() < this.underUtilizedNodes.size()) {
            potentialTargets.removeAll(selectedTargets);
            Collections.reverse(this.withinThresholdUtilizedNodes);
            for (DatanodeUsageInfo datanodeUsageInfo : this.withinThresholdUtilizedNodes) {
                source = datanodeUsageInfo.getDatanodeDetails();
                result = this.checkConditionsForBalancing();
                if (result != null) {
                    LOG.info("Exiting current iteration: {}", (Object)result);
                    return result;
                }
                moveSelection = this.matchSourceWithTarget(source, potentialTargets);
                if (moveSelection == null) continue;
                isMoveGenerated = true;
                LOG.info("ContainerBalancer is trying to move container {} from source datanode {} to target datanode {}", new Object[]{moveSelection.getContainerID().toString(), source.getUuidString(), moveSelection.getTargetNode().getUuidString()});
                if (!this.moveContainer(source, moveSelection)) continue;
                potentialTargets = this.updateTargetsAndSelectionCriteria(potentialTargets, selectedTargets, moveSelection, source);
            }
        }
        if (!isMoveGenerated) {
            return IterationResult.CAN_NOT_BALANCE_ANY_MORE;
        }
        this.countDatanodesInvolvedPerIteration = 0;
        this.sizeMovedPerIteration = 0L;
        for (Map.Entry entry : this.moveSelectionToFutureMap.entrySet()) {
            ContainerMoveSelection moveSelection2 = (ContainerMoveSelection)entry.getKey();
            CompletableFuture future = (CompletableFuture)entry.getValue();
            try {
                ReplicationManager.MoveResult result2 = (ReplicationManager.MoveResult)((Object)future.get(this.config.getMoveTimeout().toMillis(), TimeUnit.MILLISECONDS));
                if (result2 != ReplicationManager.MoveResult.COMPLETED) continue;
                try {
                    ContainerInfo container = this.containerManager.getContainer(moveSelection2.getContainerID());
                    this.sizeMovedPerIteration += container.getUsedBytes();
                    this.countDatanodesInvolvedPerIteration += 2;
                }
                catch (ContainerNotFoundException e) {
                    LOG.warn("Could not find Container {} while checking move results in ContainerBalancer", (Object)moveSelection2.getContainerID(), (Object)e);
                }
                this.metrics.incrementMovedContainersNum(1L);
                this.metrics.incrementDataSizeBalancedGB(this.sizeMovedPerIteration);
            }
            catch (InterruptedException e) {
                LOG.warn("Container move for container {} was interrupted.", (Object)moveSelection2.getContainerID(), (Object)e);
                Thread.currentThread().interrupt();
            }
            catch (ExecutionException e) {
                LOG.warn("Container move for container {} completed exceptionally.", (Object)moveSelection2.getContainerID(), (Object)e);
            }
            catch (TimeoutException e) {
                LOG.warn("Container move for container {} timed out.", (Object)moveSelection2.getContainerID(), (Object)e);
            }
        }
        LOG.info("Number of datanodes involved in this iteration: {}. Size moved in this iteration: {}B.", (Object)this.countDatanodesInvolvedPerIteration, (Object)this.sizeMovedPerIteration);
        return IterationResult.ITERATION_COMPLETED;
    }

    private ContainerMoveSelection matchSourceWithTarget(DatanodeDetails source, Collection<DatanodeDetails> potentialTargets) {
        ContainerMoveSelection moveSelection;
        NavigableSet<ContainerID> candidateContainers = this.selectionCriteria.getCandidateContainers(source);
        if (candidateContainers.isEmpty()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ContainerBalancer could not find any candidate containers for datanode {}", (Object)source.getUuidString());
            }
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("ContainerBalancer is finding suitable target for source datanode {}", (Object)source.getUuidString());
        }
        if ((moveSelection = this.findTargetStrategy.findTargetForContainerMove(source, potentialTargets, candidateContainers, this::canSizeEnterTarget)) == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ContainerBalancer could not find a suitable target for source node {}.", (Object)source.getUuidString());
            }
            return null;
        }
        LOG.info("ContainerBalancer matched source datanode {} with target datanode {} for container move.", (Object)source.getUuidString(), (Object)moveSelection.getTargetNode().getUuidString());
        return moveSelection;
    }

    private IterationResult checkConditionsForBalancing() {
        if ((double)(this.countDatanodesInvolvedPerIteration + 2) > this.maxDatanodesRatioToInvolvePerIteration * (double)this.totalNodesInCluster) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Hit max datanodes to involve limit. {} datanodes have already been involved and the limit is {}.", (Object)this.countDatanodesInvolvedPerIteration, (Object)(this.maxDatanodesRatioToInvolvePerIteration * (double)this.totalNodesInCluster));
            }
            return IterationResult.MAX_DATANODES_TO_INVOLVE_REACHED;
        }
        if (this.sizeMovedPerIteration + (long)this.ozoneConfiguration.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES) > this.maxSizeToMovePerIteration) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Hit max size to move limit. {} bytes have already been moved and the limit is {} bytes.", (Object)this.sizeMovedPerIteration, (Object)this.maxSizeToMovePerIteration);
            }
            return IterationResult.MAX_SIZE_TO_MOVE_REACHED;
        }
        return null;
    }

    private boolean moveContainer(DatanodeDetails source, ContainerMoveSelection moveSelection) {
        CompletableFuture<ReplicationManager.MoveResult> future;
        ContainerID container = moveSelection.getContainerID();
        try {
            future = this.replicationManager.move(container, source, moveSelection.getTargetNode());
        }
        catch (ContainerNotFoundException e) {
            LOG.warn("Could not find Container {} for container move", (Object)container, (Object)e);
            return false;
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Container move failed for container {}", (Object)container, (Object)e);
            return false;
        }
        if (future.isDone()) {
            if (future.isCompletedExceptionally()) {
                LOG.info("Container move for container {} from source {} to target {}completed exceptionally", new Object[]{container.toString(), source.getUuidString(), moveSelection.getTargetNode().getUuidString()});
                return false;
            }
            ReplicationManager.MoveResult result = future.join();
            this.moveSelectionToFutureMap.put(moveSelection, future);
            return result == ReplicationManager.MoveResult.COMPLETED;
        }
        this.moveSelectionToFutureMap.put(moveSelection, future);
        return true;
    }

    private List<DatanodeDetails> updateTargetsAndSelectionCriteria(Collection<DatanodeDetails> potentialTargets, Set<DatanodeDetails> selectedTargets, ContainerMoveSelection moveSelection, DatanodeDetails source) {
        this.countDatanodesInvolvedPerIteration += 2;
        this.incSizeSelectedForMoving(source, moveSelection);
        this.sourceToTargetMap.put(source, moveSelection);
        selectedTargets.add(moveSelection.getTargetNode());
        this.selectedContainers.add(moveSelection.getContainerID());
        this.selectionCriteria.setSelectedContainers(this.selectedContainers);
        return potentialTargets.stream().filter(node -> this.sizeEnteringNode.get(node) < this.config.getMaxSizeEnteringTarget()).collect(Collectors.toList());
    }

    private double ratioToBytes(Long nodeCapacity, double utilizationRatio) {
        return (double)nodeCapacity.longValue() * utilizationRatio;
    }

    double calculateAvgUtilization(List<DatanodeUsageInfo> nodes) {
        if (nodes.size() == 0) {
            LOG.warn("No nodes to calculate average utilization for in ContainerBalancer.");
            return 0.0;
        }
        SCMNodeStat aggregatedStats = new SCMNodeStat(0L, 0L, 0L);
        for (DatanodeUsageInfo node : nodes) {
            aggregatedStats.add(node.getScmNodeStat());
        }
        this.clusterCapacity = aggregatedStats.getCapacity().get();
        this.clusterUsed = aggregatedStats.getScmUsed().get();
        this.clusterRemaining = aggregatedStats.getRemaining().get();
        return (double)(this.clusterCapacity - this.clusterRemaining) / (double)this.clusterCapacity;
    }

    boolean canSizeEnterTarget(DatanodeDetails target, long size) {
        if (this.sizeEnteringNode.containsKey(target)) {
            long sizeEnteringAfterMove = this.sizeEnteringNode.get(target) + size;
            return sizeEnteringAfterMove <= this.config.getMaxSizeEnteringTarget() && this.nodeManager.getUsageInfo(target).calculateUtilization(sizeEnteringAfterMove) <= this.upperLimit;
        }
        return false;
    }

    private List<DatanodeDetails> getPotentialTargets() {
        ArrayList<DatanodeDetails> potentialTargets = new ArrayList<DatanodeDetails>(this.underUtilizedNodes.size() + this.withinThresholdUtilizedNodes.size());
        this.underUtilizedNodes.forEach(node -> potentialTargets.add(node.getDatanodeDetails()));
        this.withinThresholdUtilizedNodes.forEach(node -> potentialTargets.add(node.getDatanodeDetails()));
        return potentialTargets;
    }

    private void incSizeSelectedForMoving(DatanodeDetails source, ContainerMoveSelection moveSelection) {
        ContainerInfo container;
        DatanodeDetails target = moveSelection.getTargetNode();
        try {
            container = this.containerManager.getContainer(moveSelection.getContainerID());
        }
        catch (ContainerNotFoundException e) {
            LOG.warn("Could not find Container {} while matching source and target nodes in ContainerBalancer", (Object)moveSelection.getContainerID(), (Object)e);
            return;
        }
        long size = container.getUsedBytes();
        this.sizeMovedPerIteration += size;
        this.sizeLeavingNode.put(source, this.sizeLeavingNode.get(source) + size);
        this.sizeEnteringNode.put(target, this.sizeEnteringNode.get(target) + size);
    }

    public void stop() {
        this.lock.lock();
        try {
            if (!this.balancerRunning) {
                LOG.info("Container Balancer is not running.");
                return;
            }
            this.balancerRunning = false;
            this.currentBalancingThread.interrupt();
            this.currentBalancingThread.join(1000L);
            this.currentBalancingThread = null;
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while waiting for balancing thread to join.");
            Thread.currentThread().interrupt();
        }
        finally {
            this.lock.unlock();
        }
        LOG.info("Container Balancer stopped successfully.");
    }

    public void setNodeManager(NodeManager nodeManager) {
        this.nodeManager = nodeManager;
    }

    public void setContainerManager(ContainerManager containerManager) {
        this.containerManager = containerManager;
    }

    public void setReplicationManager(ReplicationManager replicationManager) {
        this.replicationManager = replicationManager;
    }

    public void setOzoneConfiguration(OzoneConfiguration ozoneConfiguration) {
        this.ozoneConfiguration = ozoneConfiguration;
    }

    List<DatanodeUsageInfo> getUnBalancedNodes() {
        return this.unBalancedNodes;
    }

    public void setUnBalancedNodes(List<DatanodeUsageInfo> unBalancedNodes) {
        this.unBalancedNodes = unBalancedNodes;
    }

    public void setFindTargetStrategy(FindTargetStrategy findTargetStrategy) {
        this.findTargetStrategy = findTargetStrategy;
    }

    public Map<DatanodeDetails, ContainerMoveSelection> getSourceToTargetMap() {
        return this.sourceToTargetMap;
    }

    public boolean isBalancerRunning() {
        return this.balancerRunning;
    }

    int getCountDatanodesInvolvedPerIteration() {
        return this.countDatanodesInvolvedPerIteration;
    }

    long getSizeMovedPerIteration() {
        return this.sizeMovedPerIteration;
    }

    public String toString() {
        String status = String.format("%nContainer Balancer status:%n%-30s %s%n%-30s %b%n", "Key", "Value", "Running", this.balancerRunning);
        return status + this.config.toString();
    }

    static enum IterationResult {
        ITERATION_COMPLETED,
        MAX_DATANODES_TO_INVOLVE_REACHED,
        MAX_SIZE_TO_MOVE_REACHED,
        CAN_NOT_BALANCE_ANY_MORE;

    }
}

