/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.GeneratedMessage;
import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.Config;
import org.apache.hadoop.hdds.conf.ConfigGroup;
import org.apache.hadoop.hdds.conf.ConfigTag;
import org.apache.hadoop.hdds.conf.ConfigType;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.SCMRatisProtocol;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.ContainerPlacementStatus;
import org.apache.hadoop.hdds.scm.PlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.common.helpers.MoveDataNodePair;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManagerMetrics;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.SCMHAInvocationHandler;
import org.apache.hadoop.hdds.scm.ha.SCMHAManager;
import org.apache.hadoop.hdds.scm.ha.SCMRatisServer;
import org.apache.hadoop.hdds.scm.ha.SCMService;
import org.apache.hadoop.hdds.scm.ha.SCMServiceManager;
import org.apache.hadoop.hdds.scm.metadata.DBTransactionBuffer;
import org.apache.hadoop.hdds.scm.metadata.Replicate;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.protocol.commands.CloseContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.DeleteContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.ReplicateContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.util.ExitUtil;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicationManager
implements SCMService {
    public static final Logger LOG = LoggerFactory.getLogger(ReplicationManager.class);
    public static final String METRICS_SOURCE_NAME = "SCMReplicationManager";
    private final ContainerManager containerManager;
    private final PlacementPolicy containerPlacement;
    private final EventPublisher eventPublisher;
    private final SCMContext scmContext;
    private final NodeManager nodeManager;
    private final Map<ContainerID, List<InflightAction>> inflightReplication;
    private final Map<ContainerID, List<InflightAction>> inflightDeletion;
    private final Map<ContainerID, CompletableFuture<MoveResult>> inflightMoveFuture;
    private final ReplicationManagerConfiguration rmConf;
    private Thread replicationMonitor;
    private volatile boolean running;
    private int minHealthyForMaintenance;
    private long currentContainerSize;
    private final Lock serviceLock = new ReentrantLock();
    private SCMService.ServiceStatus serviceStatus = SCMService.ServiceStatus.PAUSING;
    private final long waitTimeInMillis;
    private long lastTimeToBeReadyInMillis = 0L;
    private final Clock clock;
    private ReplicationManagerMetrics metrics;
    private final MoveScheduler moveScheduler;

    public ReplicationManager(ConfigurationSource conf, ContainerManager containerManager, PlacementPolicy containerPlacement, EventPublisher eventPublisher, SCMContext scmContext, SCMServiceManager serviceManager, NodeManager nodeManager, Clock clock, SCMHAManager scmhaManager, Table<ContainerID, MoveDataNodePair> moveTable) throws IOException {
        this.containerManager = containerManager;
        this.containerPlacement = containerPlacement;
        this.eventPublisher = eventPublisher;
        this.scmContext = scmContext;
        this.nodeManager = nodeManager;
        this.rmConf = (ReplicationManagerConfiguration)conf.getObject(ReplicationManagerConfiguration.class);
        this.running = false;
        this.inflightReplication = new ConcurrentHashMap<ContainerID, List<InflightAction>>();
        this.inflightDeletion = new ConcurrentHashMap<ContainerID, List<InflightAction>>();
        this.inflightMoveFuture = new ConcurrentHashMap<ContainerID, CompletableFuture<MoveResult>>();
        this.minHealthyForMaintenance = this.rmConf.getMaintenanceReplicaMinimum();
        this.clock = clock;
        this.waitTimeInMillis = conf.getTimeDuration("hdds.scm.wait.time.after.safemode.exit", "5m", TimeUnit.MILLISECONDS);
        this.currentContainerSize = (long)conf.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
        this.metrics = null;
        this.moveScheduler = new MoveSchedulerImpl.Builder().setDBTransactionBuffer(scmhaManager.getDBTransactionBuffer()).setRatisServer(scmhaManager.getRatisServer()).setMoveTable(moveTable).build();
        serviceManager.register(this);
        this.start();
    }

    @Override
    public synchronized void start() {
        if (!this.isRunning()) {
            this.metrics = ReplicationManagerMetrics.create(this);
            LOG.info("Starting Replication Monitor Thread.");
            this.running = true;
            this.replicationMonitor = new Thread(this::run);
            this.replicationMonitor.setName("ReplicationMonitor");
            this.replicationMonitor.setDaemon(true);
            this.replicationMonitor.start();
        } else {
            LOG.info("Replication Monitor Thread is already running.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isRunning() {
        if (!this.running) {
            ReplicationManager replicationManager = this;
            synchronized (replicationManager) {
                return this.replicationMonitor != null && this.replicationMonitor.isAlive();
            }
        }
        return true;
    }

    @Override
    public synchronized void stop() {
        if (this.running) {
            LOG.info("Stopping Replication Monitor Thread.");
            this.inflightReplication.clear();
            this.inflightDeletion.clear();
            this.running = false;
            this.metrics.unRegister();
            this.notifyAll();
        } else {
            LOG.info("Replication Monitor Thread is not running.");
        }
    }

    public synchronized void processAll() {
        long start = this.clock.millis();
        List<ContainerInfo> containers = this.containerManager.getContainers();
        containers.forEach(this::processContainer);
        LOG.info("Replication Monitor Thread took {} milliseconds for processing {} containers.", (Object)(this.clock.millis() - start), (Object)containers.size());
    }

    private synchronized void run() {
        try {
            while (this.running) {
                this.processAll();
                this.wait(this.rmConf.getInterval());
            }
        }
        catch (Throwable t) {
            LOG.error("Exception in Replication Monitor Thread.", t);
            ExitUtil.terminate((int)1, (Throwable)t);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processContainer(ContainerInfo container) {
        if (!this.shouldRun()) {
            return;
        }
        ContainerID id = container.containerID();
        try {
            ContainerInfo containerInfo = container;
            synchronized (containerInfo) {
                Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(id);
                HddsProtos.LifeCycleState state = container.getState();
                if (state == HddsProtos.LifeCycleState.OPEN) {
                    if (!this.isOpenContainerHealthy(container, replicas)) {
                        this.eventPublisher.fireEvent(SCMEvents.CLOSE_CONTAINER, (Object)id);
                    }
                    return;
                }
                if (state == HddsProtos.LifeCycleState.CLOSING) {
                    for (ContainerReplica replica : replicas) {
                        if (replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.UNHEALTHY) continue;
                        this.sendCloseCommand(container, replica.getDatanodeDetails(), false);
                    }
                    return;
                }
                if (state == HddsProtos.LifeCycleState.QUASI_CLOSED && this.canForceCloseContainer(container, replicas)) {
                    this.forceCloseContainer(container, replicas);
                    return;
                }
                this.updateInflightAction(container, this.inflightReplication, action -> replicas.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)((InflightAction)action).datanode)), () -> this.metrics.incrNumReplicationCmdsTimeout(), () -> {
                    this.metrics.incrNumReplicationCmdsCompleted();
                    this.metrics.incrNumReplicationBytesCompleted(container.getUsedBytes());
                });
                this.updateInflightAction(container, this.inflightDeletion, action -> replicas.stream().noneMatch(r -> r.getDatanodeDetails().equals((Object)((InflightAction)action).datanode)), () -> this.metrics.incrNumDeletionCmdsTimeout(), () -> this.metrics.incrNumDeletionCmdsCompleted());
                if (state == HddsProtos.LifeCycleState.DELETING) {
                    this.handleContainerUnderDelete(container, replicas);
                    return;
                }
                if (state == HddsProtos.LifeCycleState.DELETED) {
                    return;
                }
                ContainerReplicaCount replicaSet = this.getContainerReplicaCount(container, replicas);
                ContainerPlacementStatus placementStatus = this.getPlacementStatus(replicas, container.getReplicationConfig().getRequiredNodes());
                if (this.isContainerEmpty(container, replicas)) {
                    this.deleteContainerReplicas(container, replicas);
                    return;
                }
                if (!replicaSet.isSufficientlyReplicated() || !placementStatus.isPolicySatisfied()) {
                    this.handleUnderReplicatedContainer(container, replicaSet, placementStatus);
                    return;
                }
                if (replicaSet.isOverReplicated()) {
                    this.handleOverReplicatedContainer(container, replicaSet);
                    return;
                }
                if (!replicaSet.isHealthy()) {
                    this.handleUnstableContainer(container, replicas);
                }
            }
        }
        catch (ContainerNotFoundException ex) {
            LOG.warn("Missing container {}.", (Object)id);
        }
        catch (Exception ex) {
            LOG.warn("Process container {} error: ", (Object)id, (Object)ex);
        }
    }

    private void updateInflightAction(ContainerInfo container, Map<ContainerID, List<InflightAction>> inflightActions, Predicate<InflightAction> filter, Runnable timeoutCounter, Runnable completedCounter) {
        ContainerID id = container.containerID();
        long deadline = this.clock.millis() - this.rmConf.getEventTimeout();
        if (inflightActions.containsKey(id)) {
            List<InflightAction> actions = inflightActions.get(id);
            Iterator<InflightAction> iter = actions.iterator();
            while (iter.hasNext()) {
                try {
                    boolean isNotInService;
                    InflightAction a = iter.next();
                    NodeStatus status = this.nodeManager.getNodeStatus(a.datanode);
                    boolean isUnhealthy = status.getHealth() != HddsProtos.NodeState.HEALTHY;
                    boolean isCompleted = filter.test(a);
                    boolean isTimeout = a.time < deadline;
                    boolean bl = isNotInService = status.getOperationalState() != HddsProtos.NodeOperationalState.IN_SERVICE;
                    if (!isCompleted && !isUnhealthy && !isTimeout && !isNotInService) continue;
                    iter.remove();
                    if (isTimeout) {
                        timeoutCounter.run();
                    } else if (isCompleted) {
                        completedCounter.run();
                    }
                    this.updateMoveIfNeeded(isUnhealthy, isCompleted, isTimeout, isNotInService, container, a.datanode, inflightActions);
                }
                catch (ContainerNotFoundException | NodeNotFoundException e) {
                    iter.remove();
                }
            }
            if (actions.isEmpty()) {
                inflightActions.remove(id);
            }
        }
    }

    private void updateMoveIfNeeded(boolean isUnhealthy, boolean isCompleted, boolean isTimeout, boolean isNotInService, ContainerInfo container, DatanodeDetails dn, Map<ContainerID, List<InflightAction>> inflightActions) throws ContainerNotFoundException {
        ContainerID id = container.containerID();
        MoveDataNodePair kv = this.moveScheduler.getMoveDataNodePair(id);
        if (kv == null) {
            return;
        }
        boolean isSource = kv.getSrc().equals((Object)dn);
        boolean isTarget = kv.getTgt().equals((Object)dn);
        if (!isSource && !isTarget) {
            return;
        }
        boolean isInflightReplication = inflightActions.equals(this.inflightReplication);
        if (isSource && isInflightReplication) {
            this.compleleteMoveFutureWithResult(id, MoveResult.UNEXPECTED_REMOVE_SOURCE_AT_INFLIGHT_REPLICATION);
            this.moveScheduler.completeMove(id.getProtobuf());
            return;
        }
        if (isTarget && !isInflightReplication) {
            this.compleleteMoveFutureWithResult(id, MoveResult.UNEXPECTED_REMOVE_TARGET_AT_INFLIGHT_DELETION);
            this.moveScheduler.completeMove(id.getProtobuf());
            return;
        }
        if (!isInflightReplication || !isCompleted) {
            if (isInflightReplication) {
                if (isUnhealthy) {
                    this.compleleteMoveFutureWithResult(id, MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
                } else if (isNotInService) {
                    this.compleleteMoveFutureWithResult(id, MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
                } else {
                    this.compleleteMoveFutureWithResult(id, MoveResult.REPLICATION_FAIL_TIME_OUT);
                }
            } else if (isUnhealthy) {
                this.compleleteMoveFutureWithResult(id, MoveResult.DELETION_FAIL_NODE_UNHEALTHY);
            } else if (isTimeout) {
                this.compleleteMoveFutureWithResult(id, MoveResult.DELETION_FAIL_TIME_OUT);
            } else if (isNotInService) {
                this.compleleteMoveFutureWithResult(id, MoveResult.DELETION_FAIL_NODE_NOT_IN_SERVICE);
            } else {
                this.compleleteMoveFutureWithResult(id, MoveResult.COMPLETED);
            }
            this.moveScheduler.completeMove(id.getProtobuf());
        } else {
            this.deleteSrcDnForMove(container, this.containerManager.getContainerReplicas(id));
        }
    }

    public CompletableFuture<MoveResult> move(ContainerID cid, DatanodeDetails src, DatanodeDetails tgt) throws ContainerNotFoundException, NodeNotFoundException {
        return this.move(cid, new MoveDataNodePair(src, tgt));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<MoveResult> move(ContainerID cid, MoveDataNodePair mp) throws ContainerNotFoundException, NodeNotFoundException {
        ContainerInfo cif;
        CompletableFuture<MoveResult> ret = new CompletableFuture<MoveResult>();
        if (!this.isRunning()) {
            ret.complete(MoveResult.FAIL_NOT_RUNNING);
            return ret;
        }
        if (!this.scmContext.isLeader()) {
            ret.complete(MoveResult.FAIL_NOT_LEADER);
            return ret;
        }
        DatanodeDetails srcDn = mp.getSrc();
        DatanodeDetails targetDn = mp.getTgt();
        NodeStatus currentNodeStat = this.nodeManager.getNodeStatus(srcDn);
        HddsProtos.NodeState healthStat = currentNodeStat.getHealth();
        HddsProtos.NodeOperationalState operationalState = currentNodeStat.getOperationalState();
        if (healthStat != HddsProtos.NodeState.HEALTHY) {
            ret.complete(MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
            return ret;
        }
        if (operationalState != HddsProtos.NodeOperationalState.IN_SERVICE) {
            ret.complete(MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
            return ret;
        }
        currentNodeStat = this.nodeManager.getNodeStatus(targetDn);
        healthStat = currentNodeStat.getHealth();
        operationalState = currentNodeStat.getOperationalState();
        if (healthStat != HddsProtos.NodeState.HEALTHY) {
            ret.complete(MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
            return ret;
        }
        if (operationalState != HddsProtos.NodeOperationalState.IN_SERVICE) {
            ret.complete(MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
            return ret;
        }
        ContainerInfo containerInfo = cif = this.containerManager.getContainer(cid);
        synchronized (containerInfo) {
            Set<ContainerReplica> currentReplicas = this.containerManager.getContainerReplicas(cid);
            Set replicas = currentReplicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toSet());
            if (replicas.contains(targetDn)) {
                ret.complete(MoveResult.REPLICATION_FAIL_EXIST_IN_TARGET);
                return ret;
            }
            if (!replicas.contains(srcDn)) {
                ret.complete(MoveResult.REPLICATION_FAIL_NOT_EXIST_IN_SOURCE);
                return ret;
            }
            if (this.inflightReplication.containsKey(cid)) {
                ret.complete(MoveResult.REPLICATION_FAIL_INFLIGHT_REPLICATION);
                return ret;
            }
            if (this.inflightDeletion.containsKey(cid)) {
                ret.complete(MoveResult.REPLICATION_FAIL_INFLIGHT_DELETION);
                return ret;
            }
            HddsProtos.LifeCycleState currentContainerStat = cif.getState();
            if (currentContainerStat != HddsProtos.LifeCycleState.CLOSED) {
                ret.complete(MoveResult.REPLICATION_FAIL_CONTAINER_NOT_CLOSED);
                return ret;
            }
            if (!this.isPolicySatisfiedAfterMove(cif, srcDn, targetDn, currentReplicas.stream().collect(Collectors.toList()))) {
                ret.complete(MoveResult.PLACEMENT_POLICY_NOT_SATISFIED);
                return ret;
            }
            try {
                this.moveScheduler.startMove(cid.getProtobuf(), mp.getProtobufMessage(1));
            }
            catch (IOException e) {
                LOG.warn("Exception while starting move {}", (Object)cid);
                ret.complete(MoveResult.FAIL_CAN_NOT_RECORD_TO_DB);
                return ret;
            }
            this.inflightMoveFuture.putIfAbsent(cid, ret);
            this.sendReplicateCommand(cif, targetDn, Collections.singletonList(srcDn));
        }
        LOG.info("receive a move request about container {} , from {} to {}", new Object[]{cid, srcDn.getUuid(), targetDn.getUuid()});
        return ret;
    }

    private boolean isPolicySatisfiedAfterMove(ContainerInfo cif, DatanodeDetails srcDn, DatanodeDetails targetDn, List<ContainerReplica> replicas) {
        Set<ContainerReplica> movedReplicas = replicas.stream().collect(Collectors.toSet());
        movedReplicas.removeIf(r -> r.getDatanodeDetails().equals((Object)srcDn));
        movedReplicas.add(ContainerReplica.newBuilder().setDatanodeDetails(targetDn).setContainerID(cif.containerID()).setContainerState(StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED).build());
        ContainerPlacementStatus placementStatus = this.getPlacementStatus(movedReplicas, cif.getReplicationConfig().getRequiredNodes());
        return placementStatus.isPolicySatisfied();
    }

    private int getInflightAdd(ContainerID id) {
        return this.inflightReplication.getOrDefault(id, Collections.emptyList()).size();
    }

    private int getInflightDel(ContainerID id) {
        return this.inflightDeletion.getOrDefault(id, Collections.emptyList()).size();
    }

    private boolean isContainerEmpty(ContainerInfo container, Set<ContainerReplica> replicas) {
        return container.getState() == HddsProtos.LifeCycleState.CLOSED && container.getUsedBytes() == 0L && container.getNumberOfKeys() == 0L && replicas.stream().allMatch(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED && r.getBytesUsed() == 0L && r.getKeyCount() == 0L);
    }

    public ContainerReplicaCount getContainerReplicaCount(ContainerID containerID) throws ContainerNotFoundException {
        ContainerInfo container = this.containerManager.getContainer(containerID);
        return this.getContainerReplicaCount(container);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ContainerReplicaCount getContainerReplicaCount(ContainerInfo container) throws ContainerNotFoundException {
        ContainerInfo containerInfo = container;
        synchronized (containerInfo) {
            Set<ContainerReplica> replica = this.containerManager.getContainerReplicas(container.containerID());
            return this.getContainerReplicaCount(container, replica);
        }
    }

    private ContainerReplicaCount getContainerReplicaCount(ContainerInfo container, Set<ContainerReplica> replica) {
        return new ContainerReplicaCount(container, replica, this.getInflightAdd(container.containerID()), this.getInflightDel(container.containerID()), container.getReplicationConfig().getRequiredNodes(), this.minHealthyForMaintenance);
    }

    private boolean canForceCloseContainer(ContainerInfo container, Set<ContainerReplica> replicas) {
        Preconditions.assertTrue((container.getState() == HddsProtos.LifeCycleState.QUASI_CLOSED ? 1 : 0) != 0);
        int replicationFactor = container.getReplicationConfig().getRequiredNodes();
        long uniqueQuasiClosedReplicaCount = replicas.stream().filter(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED).map(ContainerReplica::getOriginDatanodeId).distinct().count();
        return uniqueQuasiClosedReplicaCount > (long)(replicationFactor / 2);
    }

    private void deleteContainerReplicas(ContainerInfo container, Set<ContainerReplica> replicas) throws IOException, InvalidStateTransitionException {
        Preconditions.assertTrue((container.getState() == HddsProtos.LifeCycleState.CLOSED ? 1 : 0) != 0);
        Preconditions.assertTrue((container.getNumberOfKeys() == 0L ? 1 : 0) != 0);
        Preconditions.assertTrue((container.getUsedBytes() == 0L ? 1 : 0) != 0);
        replicas.stream().forEach(rp -> {
            Preconditions.assertTrue((rp.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED ? 1 : 0) != 0);
            Preconditions.assertTrue((rp.getBytesUsed() == 0L ? 1 : 0) != 0);
            Preconditions.assertTrue((rp.getKeyCount() == 0L ? 1 : 0) != 0);
            this.sendDeleteCommand(container, rp.getDatanodeDetails(), false);
        });
        this.containerManager.updateContainerState(container.containerID(), HddsProtos.LifeCycleEvent.DELETE);
        LOG.debug("Deleting empty container replicas for {},", (Object)container);
    }

    private void handleContainerUnderDelete(ContainerInfo container, Set<ContainerReplica> replicas) throws IOException, InvalidStateTransitionException {
        if (replicas.size() == 0) {
            this.containerManager.updateContainerState(container.containerID(), HddsProtos.LifeCycleEvent.CLEANUP);
            LOG.debug("Container {} state changes to DELETED", (Object)container);
        } else {
            List deletionInFlight = this.inflightDeletion.getOrDefault(container.containerID(), Collections.emptyList()).stream().map(action -> ((InflightAction)action).datanode).collect(Collectors.toList());
            Set filteredReplicas = replicas.stream().filter(r -> !deletionInFlight.contains(r.getDatanodeDetails())).collect(Collectors.toSet());
            if (filteredReplicas.size() > 0) {
                filteredReplicas.stream().forEach(rp -> this.sendDeleteCommand(container, rp.getDatanodeDetails(), false));
                LOG.debug("Resend delete Container command for {}", (Object)container);
            }
        }
    }

    private void forceCloseContainer(ContainerInfo container, Set<ContainerReplica> replicas) {
        Preconditions.assertTrue((container.getState() == HddsProtos.LifeCycleState.QUASI_CLOSED ? 1 : 0) != 0);
        List quasiClosedReplicas = replicas.stream().filter(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED).collect(Collectors.toList());
        Long sequenceId = quasiClosedReplicas.stream().map(ContainerReplica::getSequenceId).max(Long::compare).orElse(-1L);
        LOG.info("Force closing container {} with BCSID {}, which is in QUASI_CLOSED state.", (Object)container.containerID(), (Object)sequenceId);
        quasiClosedReplicas.stream().filter(r -> sequenceId != -1L).filter(replica -> replica.getSequenceId().equals(sequenceId)).forEach(replica -> this.sendCloseCommand(container, replica.getDatanodeDetails(), true));
    }

    private void handleUnderReplicatedContainer(ContainerInfo container, ContainerReplicaCount replicaSet, ContainerPlacementStatus placementStatus) {
        LOG.debug("Handling under-replicated container: {}", (Object)container);
        Set<ContainerReplica> replicas = replicaSet.getReplica();
        try {
            if (replicaSet.isSufficientlyReplicated() && placementStatus.isPolicySatisfied()) {
                LOG.info("The container {} with replicas {} is sufficiently replicated and is not mis-replicated", (Object)container.getContainerID(), (Object)replicaSet);
                return;
            }
            int repDelta = replicaSet.additionalReplicaNeeded();
            ContainerID id = container.containerID();
            List deletionInFlight = this.inflightDeletion.getOrDefault(id, Collections.emptyList()).stream().map(action -> ((InflightAction)action).datanode).collect(Collectors.toList());
            List replicationInFlight = this.inflightReplication.getOrDefault(id, Collections.emptyList()).stream().map(action -> ((InflightAction)action).datanode).collect(Collectors.toList());
            List<DatanodeDetails> source = replicas.stream().filter(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED || r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED).filter(r -> this.getNodeStatus(r.getDatanodeDetails()).isHealthy()).filter(r -> !deletionInFlight.contains(r.getDatanodeDetails())).sorted((r1, r2) -> r2.getSequenceId().compareTo(r1.getSequenceId())).map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
            if (source.size() > 0) {
                int replicasNeeded;
                int replicationFactor = container.getReplicationConfig().getRequiredNodes();
                ArrayList targetReplicas = new ArrayList(source);
                targetReplicas.addAll(replicationInFlight);
                ContainerPlacementStatus inFlightplacementStatus = this.containerPlacement.validateContainerPlacement(targetReplicas, replicationFactor);
                int misRepDelta = inFlightplacementStatus.misReplicationCount();
                int n = replicasNeeded = repDelta < misRepDelta ? misRepDelta : repDelta;
                if (replicasNeeded <= 0) {
                    LOG.debug("Container {} meets replication requirement with inflight replicas", (Object)id);
                    return;
                }
                long dataSizeRequired = Math.max(container.getUsedBytes(), this.currentContainerSize);
                List excludeList = replicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
                excludeList.addAll(replicationInFlight);
                List selectedDatanodes = this.containerPlacement.chooseDatanodes(excludeList, null, replicasNeeded, 0L, dataSizeRequired);
                if (repDelta > 0) {
                    LOG.info("Container {} is under replicated. Expected replica count is {}, but found {}.", new Object[]{id, replicationFactor, replicationFactor - repDelta});
                }
                int newMisRepDelta = misRepDelta;
                if (misRepDelta > 0) {
                    LOG.info("Container: {}. {}", (Object)id, (Object)placementStatus.misReplicatedReason());
                    targetReplicas.addAll(selectedDatanodes);
                    newMisRepDelta = this.containerPlacement.validateContainerPlacement(targetReplicas, replicationFactor).misReplicationCount();
                }
                if (repDelta > 0 || newMisRepDelta < misRepDelta) {
                    for (DatanodeDetails datanode : selectedDatanodes) {
                        this.sendReplicateCommand(container, datanode, source);
                    }
                } else {
                    LOG.warn("Container {} is mis-replicated, requiring {} additional replicas. After selecting new nodes, mis-replication has not improved. No additional replicas will be scheduled", (Object)id, (Object)misRepDelta);
                }
            } else {
                LOG.warn("Cannot replicate container {}, no healthy replica found.", (Object)container.containerID());
            }
        }
        catch (IOException | IllegalStateException ex) {
            LOG.warn("Exception while replicating container {}.", (Object)container.getContainerID(), (Object)ex);
        }
    }

    private void handleOverReplicatedContainer(ContainerInfo container, ContainerReplicaCount replicaSet) {
        Set<ContainerReplica> replicas = replicaSet.getReplica();
        ContainerID id = container.containerID();
        int replicationFactor = container.getReplicationConfig().getRequiredNodes();
        int excess = replicaSet.additionalReplicaNeeded() * -1;
        if (excess > 0) {
            LOG.info("Container {} is over replicated. Expected replica count is {}, but found {}.", new Object[]{id, replicationFactor, replicationFactor + excess});
            ArrayList<ContainerReplica> eligibleReplicas = new ArrayList<ContainerReplica>(replicas);
            eligibleReplicas.sort(Comparator.comparingLong(ContainerReplica::hashCode));
            LinkedHashMap uniqueReplicas = new LinkedHashMap();
            if (container.getState() != HddsProtos.LifeCycleState.CLOSED) {
                replicas.stream().filter(r -> ReplicationManager.compareState(container.getState(), r.getState())).forEach(r -> uniqueReplicas.putIfAbsent(r.getOriginDatanodeId(), r));
                eligibleReplicas.removeAll(uniqueReplicas.values());
            }
            eligibleReplicas.removeIf(r -> r.getDatanodeDetails().getPersistedOpState() != HddsProtos.NodeOperationalState.IN_SERVICE);
            List unhealthyReplicas = eligibleReplicas.stream().filter(r -> !ReplicationManager.compareState(container.getState(), r.getState())).collect(Collectors.toList());
            for (ContainerReplica r2 : unhealthyReplicas) {
                if (excess <= 0) break;
                this.sendDeleteCommand(container, r2.getDatanodeDetails(), true);
                --excess;
            }
            eligibleReplicas.removeAll(unhealthyReplicas);
            this.removeExcessReplicasIfNeeded(excess, container, eligibleReplicas);
        }
    }

    private void deleteSrcDnForMove(ContainerInfo cif, Set<ContainerReplica> replicaSet) {
        ContainerID cid = cif.containerID();
        MoveDataNodePair movePair = this.moveScheduler.getMoveDataNodePair(cid);
        if (movePair == null) {
            return;
        }
        DatanodeDetails srcDn = movePair.getSrc();
        ContainerReplicaCount replicaCount = this.getContainerReplicaCount(cif, replicaSet);
        if (!replicaSet.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)srcDn))) {
            this.compleleteMoveFutureWithResult(cid, MoveResult.COMPLETED);
            this.moveScheduler.completeMove(cid.getProtobuf());
            return;
        }
        int replicationFactor = cif.getReplicationConfig().getRequiredNodes();
        ContainerPlacementStatus currentCPS = this.getPlacementStatus(replicaSet, replicationFactor);
        Set<ContainerReplica> newReplicaSet = replicaSet.stream().collect(Collectors.toSet());
        newReplicaSet.removeIf(r -> r.getDatanodeDetails().equals((Object)srcDn));
        ContainerPlacementStatus newCPS = this.getPlacementStatus(newReplicaSet, replicationFactor);
        if (replicaCount.isOverReplicated() && this.isPlacementStatusActuallyEqual(currentCPS, newCPS)) {
            this.sendDeleteCommand(cif, srcDn, true);
        } else {
            LOG.info("can not remove source replica after successfully replicated to target datanode");
            this.compleleteMoveFutureWithResult(cid, MoveResult.DELETE_FAIL_POLICY);
            this.moveScheduler.completeMove(cid.getProtobuf());
        }
    }

    private void removeExcessReplicasIfNeeded(int excess, ContainerInfo container, List<ContainerReplica> eligibleReplicas) {
        if (excess > 0) {
            HashSet<ContainerReplica> eligibleSet = new HashSet<ContainerReplica>(eligibleReplicas);
            int replicationFactor = container.getReplicationConfig().getRequiredNodes();
            ContainerPlacementStatus ps = this.getPlacementStatus(eligibleSet, replicationFactor);
            for (ContainerReplica r : eligibleReplicas) {
                if (excess <= 0) break;
                eligibleSet.remove(r);
                ContainerPlacementStatus nowPS = this.getPlacementStatus(eligibleSet, replicationFactor);
                if (this.isPlacementStatusActuallyEqual(ps, nowPS)) {
                    this.sendDeleteCommand(container, r.getDatanodeDetails(), true);
                    --excess;
                    continue;
                }
                eligibleSet.add(r);
            }
            if (excess > 0) {
                LOG.info("The container {} is over replicated with {} excess replica. The excess replicas cannot be removed without violating the placement policy", (Object)container, (Object)excess);
            }
        }
    }

    private boolean isPlacementStatusActuallyEqual(ContainerPlacementStatus cps1, ContainerPlacementStatus cps2) {
        return !cps1.isPolicySatisfied() && cps1.actualPlacementCount() == cps2.actualPlacementCount() || cps1.isPolicySatisfied() && cps2.isPolicySatisfied();
    }

    private ContainerPlacementStatus getPlacementStatus(Set<ContainerReplica> replicas, int replicationFactor) {
        List replicaDns = replicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
        return this.containerPlacement.validateContainerPlacement(replicaDns, replicationFactor);
    }

    private void handleUnstableContainer(ContainerInfo container, Set<ContainerReplica> replicas) {
        List unhealthyReplicas = replicas.stream().filter(r -> !ReplicationManager.compareState(container.getState(), r.getState())).collect(Collectors.toList());
        Iterator iterator = unhealthyReplicas.iterator();
        while (iterator.hasNext()) {
            ContainerReplica replica2 = (ContainerReplica)iterator.next();
            StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State state = replica2.getState();
            if (state == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN || state == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSING) {
                this.sendCloseCommand(container, replica2.getDatanodeDetails(), false);
                iterator.remove();
            }
            if (state != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED || container.getSequenceId() != replica2.getSequenceId().longValue()) continue;
            this.sendCloseCommand(container, replica2.getDatanodeDetails(), true);
            iterator.remove();
        }
        unhealthyReplicas.stream().findFirst().ifPresent(replica -> this.sendDeleteCommand(container, replica.getDatanodeDetails(), false));
    }

    private void sendCloseCommand(ContainerInfo container, DatanodeDetails datanode, boolean force) {
        ContainerID containerID = container.containerID();
        LOG.info("Sending close container command for container {} to datanode {}.", (Object)containerID, (Object)datanode);
        CloseContainerCommand closeContainerCommand = new CloseContainerCommand(container.getContainerID(), container.getPipelineID(), force);
        try {
            closeContainerCommand.setTerm(this.scmContext.getTermOfLeader());
        }
        catch (NotLeaderException nle) {
            LOG.warn("Skip sending close container command, since current SCM is not leader.", (Throwable)nle);
            return;
        }
        closeContainerCommand.setEncodedToken(this.getContainerToken(containerID));
        this.eventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)new CommandForDatanode(datanode.getUuid(), (SCMCommand)closeContainerCommand));
    }

    private String getContainerToken(ContainerID containerID) {
        StorageContainerManager scm = this.scmContext.getScm();
        return scm != null ? scm.getContainerTokenGenerator().generateEncodedToken(containerID) : "";
    }

    private void sendReplicateCommand(ContainerInfo container, DatanodeDetails datanode, List<DatanodeDetails> sources) {
        LOG.info("Sending replicate container command for container {} to datanode {} from datanodes {}", new Object[]{container.containerID(), datanode, sources});
        ContainerID id = container.containerID();
        ReplicateContainerCommand replicateCommand = new ReplicateContainerCommand(id.getId(), sources);
        this.inflightReplication.computeIfAbsent(id, k -> new ArrayList());
        this.sendAndTrackDatanodeCommand(datanode, (SCMCommand)replicateCommand, action -> this.inflightReplication.get(id).add((InflightAction)action));
        this.metrics.incrNumReplicationCmdsSent();
        this.metrics.incrNumReplicationBytesTotal(container.getUsedBytes());
    }

    private void sendDeleteCommand(ContainerInfo container, DatanodeDetails datanode, boolean force) {
        LOG.info("Sending delete container command for container {} to datanode {}", (Object)container.containerID(), (Object)datanode);
        ContainerID id = container.containerID();
        DeleteContainerCommand deleteCommand = new DeleteContainerCommand(id.getId(), force);
        this.inflightDeletion.computeIfAbsent(id, k -> new ArrayList());
        this.sendAndTrackDatanodeCommand(datanode, (SCMCommand)deleteCommand, action -> this.inflightDeletion.get(id).add((InflightAction)action));
        this.metrics.incrNumDeletionCmdsSent();
    }

    private <T extends GeneratedMessage> void sendAndTrackDatanodeCommand(DatanodeDetails datanode, SCMCommand<T> command, Consumer<InflightAction> tracker) {
        try {
            command.setTerm(this.scmContext.getTermOfLeader());
        }
        catch (NotLeaderException nle) {
            LOG.warn("Skip sending datanode command, since current SCM is not leader.", (Throwable)nle);
            return;
        }
        CommandForDatanode datanodeCommand = new CommandForDatanode(datanode.getUuid(), command);
        this.eventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)datanodeCommand);
        tracker.accept(new InflightAction(datanode, this.clock.millis()));
    }

    private NodeStatus getNodeStatus(DatanodeDetails dn) {
        try {
            return this.nodeManager.getNodeStatus(dn);
        }
        catch (NodeNotFoundException e) {
            throw new IllegalStateException("Unable to find NodeStatus for " + dn, e);
        }
    }

    public static boolean compareState(HddsProtos.LifeCycleState containerState, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State replicaState) {
        switch (containerState) {
            case OPEN: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN;
            }
            case CLOSING: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSING;
            }
            case QUASI_CLOSED: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED;
            }
            case CLOSED: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED;
            }
            case DELETING: {
                return false;
            }
            case DELETED: {
                return false;
            }
        }
        return false;
    }

    private boolean isOpenContainerHealthy(ContainerInfo container, Set<ContainerReplica> replicas) {
        HddsProtos.LifeCycleState state = container.getState();
        return replicas.stream().allMatch(r -> ReplicationManager.compareState(state, r.getState()));
    }

    public boolean isContainerReplicatingOrDeleting(ContainerID containerID) {
        return this.inflightReplication.containsKey(containerID) || this.inflightDeletion.containsKey(containerID);
    }

    @Override
    public void notifyStatusChanged() {
        this.serviceLock.lock();
        try {
            if (this.scmContext.isLeaderReady() && !this.scmContext.isInSafeMode()) {
                if (this.serviceStatus != SCMService.ServiceStatus.RUNNING) {
                    LOG.info("Service {} transitions to RUNNING.", (Object)this.getServiceName());
                    this.lastTimeToBeReadyInMillis = this.clock.millis();
                    this.serviceStatus = SCMService.ServiceStatus.RUNNING;
                }
                this.onLeaderReadyAndOutOfSafeMode();
            } else {
                this.serviceStatus = SCMService.ServiceStatus.PAUSING;
            }
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    @Override
    public boolean shouldRun() {
        this.serviceLock.lock();
        try {
            boolean bl = this.serviceStatus == SCMService.ServiceStatus.RUNNING && this.clock.millis() - this.lastTimeToBeReadyInMillis >= this.waitTimeInMillis;
            return bl;
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    @Override
    public String getServiceName() {
        return ReplicationManager.class.getSimpleName();
    }

    public ReplicationManagerMetrics getMetrics() {
        return this.metrics;
    }

    public Map<ContainerID, List<InflightAction>> getInflightReplication() {
        return this.inflightReplication;
    }

    public Map<ContainerID, List<InflightAction>> getInflightDeletion() {
        return this.inflightDeletion;
    }

    public Map<ContainerID, CompletableFuture<MoveResult>> getInflightMove() {
        return this.inflightMoveFuture;
    }

    public MoveScheduler getMoveScheduler() {
        return this.moveScheduler;
    }

    private void onLeaderReadyAndOutOfSafeMode() {
        LinkedList needToRemove = new LinkedList();
        this.moveScheduler.getInflightMove().forEach((k, v) -> {
            ContainerInfo cif;
            Set<ContainerReplica> replicas;
            try {
                replicas = this.containerManager.getContainerReplicas((ContainerID)k);
                cif = this.containerManager.getContainer((ContainerID)k);
            }
            catch (ContainerNotFoundException e) {
                needToRemove.add(k.getProtobuf());
                LOG.error("can not find container {} while processing replicated move", k);
                return;
            }
            boolean isSrcExist = replicas.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)v.getSrc()));
            boolean isTgtExist = replicas.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)v.getTgt()));
            if (isSrcExist) {
                if (isTgtExist) {
                    this.deleteSrcDnForMove(cif, replicas);
                } else {
                    this.sendReplicateCommand(cif, v.getTgt(), Collections.singletonList(v.getSrc()));
                }
            } else {
                needToRemove.add(k.getProtobuf());
            }
        });
        needToRemove.forEach(this.moveScheduler::completeMove);
    }

    private void compleleteMoveFutureWithResult(ContainerID cid, MoveResult mr) {
        if (this.inflightMoveFuture.containsKey(cid)) {
            this.inflightMoveFuture.get(cid).complete(mr);
            this.inflightMoveFuture.remove(cid);
        }
    }

    public static final class MoveSchedulerImpl
    implements MoveScheduler {
        private Table<ContainerID, MoveDataNodePair> moveTable;
        private final DBTransactionBuffer transactionBuffer;
        private final Map<ContainerID, MoveDataNodePair> inflightMove;

        private MoveSchedulerImpl(Table<ContainerID, MoveDataNodePair> moveTable, DBTransactionBuffer transactionBuffer) throws IOException {
            this.moveTable = moveTable;
            this.transactionBuffer = transactionBuffer;
            this.inflightMove = new ConcurrentHashMap<ContainerID, MoveDataNodePair>();
            this.initialize();
        }

        @Override
        public void completeMove(HddsProtos.ContainerID contianerIDProto) {
            ContainerID cid = null;
            try {
                cid = ContainerID.getFromProtobuf((HddsProtos.ContainerID)contianerIDProto);
                this.transactionBuffer.removeFromBuffer(this.moveTable, (Object)cid);
            }
            catch (IOException e) {
                LOG.warn("Exception while completing move {}", (Object)cid);
            }
            this.inflightMove.remove(cid);
        }

        @Override
        public void startMove(HddsProtos.ContainerID contianerIDProto, HddsProtos.MoveDataNodePairProto mdnpp) throws IOException {
            ContainerID cid = null;
            MoveDataNodePair mp = null;
            try {
                cid = ContainerID.getFromProtobuf((HddsProtos.ContainerID)contianerIDProto);
                mp = MoveDataNodePair.getFromProtobuf((HddsProtos.MoveDataNodePairProto)mdnpp);
                if (!this.inflightMove.containsKey(cid)) {
                    this.transactionBuffer.addToBuffer(this.moveTable, (Object)cid, (Object)mp);
                    this.inflightMove.putIfAbsent(cid, mp);
                }
            }
            catch (IOException e) {
                LOG.warn("Exception while completing move {}", (Object)cid);
            }
        }

        @Override
        public MoveDataNodePair getMoveDataNodePair(ContainerID cid) {
            return this.inflightMove.get(cid);
        }

        @Override
        public void reinitialize(Table<ContainerID, MoveDataNodePair> mt) throws IOException {
            this.moveTable = mt;
            this.inflightMove.clear();
            this.initialize();
        }

        private void initialize() throws IOException {
            for (Table.KeyValue kv : this.moveTable) {
                ContainerID cid = (ContainerID)kv.getKey();
                MoveDataNodePair mp = (MoveDataNodePair)kv.getValue();
                Preconditions.assertNotNull((Object)cid, (String)"moved container id should not be null");
                Preconditions.assertNotNull((Object)mp, (String)"MoveDataNodePair container id should not be null");
                this.inflightMove.put(cid, mp);
            }
        }

        @Override
        public Map<ContainerID, MoveDataNodePair> getInflightMove() {
            return this.inflightMove;
        }

        public static class Builder {
            private Table<ContainerID, MoveDataNodePair> moveTable;
            private DBTransactionBuffer transactionBuffer;
            private SCMRatisServer ratisServer;

            public Builder setRatisServer(SCMRatisServer scmRatisServer) {
                this.ratisServer = scmRatisServer;
                return this;
            }

            public Builder setMoveTable(Table<ContainerID, MoveDataNodePair> mt) {
                this.moveTable = mt;
                return this;
            }

            public Builder setDBTransactionBuffer(DBTransactionBuffer trxBuffer) {
                this.transactionBuffer = trxBuffer;
                return this;
            }

            public MoveScheduler build() throws IOException {
                Preconditions.assertNotNull(this.moveTable, (String)"moveTable is null");
                Preconditions.assertNotNull((Object)this.transactionBuffer, (String)"transactionBuffer is null");
                MoveSchedulerImpl impl = new MoveSchedulerImpl(this.moveTable, this.transactionBuffer);
                SCMHAInvocationHandler invocationHandler = new SCMHAInvocationHandler(SCMRatisProtocol.RequestType.MOVE, impl, this.ratisServer);
                return (MoveScheduler)Proxy.newProxyInstance(SCMHAInvocationHandler.class.getClassLoader(), new Class[]{MoveScheduler.class}, (InvocationHandler)invocationHandler);
            }
        }
    }

    public static interface MoveScheduler {
        @Replicate
        public void completeMove(HddsProtos.ContainerID var1);

        @Replicate
        public void startMove(HddsProtos.ContainerID var1, HddsProtos.MoveDataNodePairProto var2) throws IOException;

        public MoveDataNodePair getMoveDataNodePair(ContainerID var1);

        public void reinitialize(Table<ContainerID, MoveDataNodePair> var1) throws IOException;

        public Map<ContainerID, MoveDataNodePair> getInflightMove();
    }

    @ConfigGroup(prefix="hdds.scm.replication")
    public static class ReplicationManagerConfiguration {
        @Config(key="thread.interval", type=ConfigType.TIME, defaultValue="300s", tags={ConfigTag.SCM, ConfigTag.OZONE}, description="There is a replication monitor thread running inside SCM which takes care of replicating the containers in the cluster. This property is used to configure the interval in which that thread runs.")
        private long interval = Duration.ofSeconds(300L).toMillis();
        @Config(key="event.timeout", type=ConfigType.TIME, defaultValue="30m", tags={ConfigTag.SCM, ConfigTag.OZONE}, description="Timeout for the container replication/deletion commands sent  to datanodes. After this timeout the command will be retried.")
        private long eventTimeout = Duration.ofMinutes(30L).toMillis();
        @Config(key="maintenance.replica.minimum", type=ConfigType.INT, defaultValue="2", tags={ConfigTag.SCM, ConfigTag.OZONE}, description="The minimum number of container replicas which must  be available for a node to enter maintenance. If putting a  node into maintenance reduces the available replicas for any  container below this level, the node will remain in the  entering maintenance state until a new replica is created.")
        private int maintenanceReplicaMinimum = 2;

        public void setInterval(Duration interval) {
            this.interval = interval.toMillis();
        }

        public void setEventTimeout(Duration timeout) {
            this.eventTimeout = timeout.toMillis();
        }

        public void setMaintenanceReplicaMinimum(int replicaCount) {
            this.maintenanceReplicaMinimum = replicaCount;
        }

        public long getInterval() {
            return this.interval;
        }

        public long getEventTimeout() {
            return this.eventTimeout;
        }

        public int getMaintenanceReplicaMinimum() {
            return this.maintenanceReplicaMinimum;
        }
    }

    static final class InflightAction {
        private final DatanodeDetails datanode;
        private final long time;

        private InflightAction(DatanodeDetails datanode, long time) {
            this.datanode = datanode;
            this.time = time;
        }

        @VisibleForTesting
        public DatanodeDetails getDatanode() {
            return this.datanode;
        }
    }

    public static enum MoveResult {
        COMPLETED,
        FAIL_NOT_RUNNING,
        FAIL_NOT_LEADER,
        REPLICATION_FAIL_NOT_EXIST_IN_SOURCE,
        REPLICATION_FAIL_EXIST_IN_TARGET,
        REPLICATION_FAIL_CONTAINER_NOT_CLOSED,
        REPLICATION_FAIL_INFLIGHT_DELETION,
        REPLICATION_FAIL_INFLIGHT_REPLICATION,
        REPLICATION_FAIL_TIME_OUT,
        REPLICATION_FAIL_NODE_NOT_IN_SERVICE,
        REPLICATION_FAIL_NODE_UNHEALTHY,
        DELETION_FAIL_NODE_NOT_IN_SERVICE,
        DELETION_FAIL_TIME_OUT,
        DELETION_FAIL_NODE_UNHEALTHY,
        DELETE_FAIL_POLICY,
        PLACEMENT_POLICY_NOT_SATISFIED,
        UNEXPECTED_REMOVE_SOURCE_AT_INFLIGHT_REPLICATION,
        UNEXPECTED_REMOVE_TARGET_AT_INFLIGHT_DELETION,
        FAIL_CAN_NOT_RECORD_TO_DB;

    }
}

