/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.SCMRatisProtocol;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ContainerStateManagerV2;
import org.apache.hadoop.hdds.scm.container.states.ContainerState;
import org.apache.hadoop.hdds.scm.container.states.ContainerStateMap;
import org.apache.hadoop.hdds.scm.ha.CheckedConsumer;
import org.apache.hadoop.hdds.scm.ha.ExecutionUtil;
import org.apache.hadoop.hdds.scm.ha.SCMHAInvocationHandler;
import org.apache.hadoop.hdds.scm.ha.SCMRatisServer;
import org.apache.hadoop.hdds.scm.metadata.DBTransactionBuffer;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.common.statemachine.StateMachine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ContainerStateManagerImpl
implements ContainerStateManagerV2 {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerStateManagerImpl.class);
    private final long containerSize;
    private ContainerStateMap containers;
    private Table<ContainerID, ContainerInfo> containerStore;
    private final DBTransactionBuffer transactionBuffer;
    private final PipelineManager pipelineManager;
    private final StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent> stateMachine;
    private ConcurrentHashMap<ContainerState, ContainerID> lastUsedMap;
    private final Map<HddsProtos.LifeCycleEvent, CheckedConsumer<ContainerInfo, IOException>> containerStateChangeActions;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();

    private ContainerStateManagerImpl(Configuration conf, PipelineManager pipelineManager, Table<ContainerID, ContainerInfo> containerStore, DBTransactionBuffer buffer) throws IOException {
        this.pipelineManager = pipelineManager;
        this.containerStore = containerStore;
        this.stateMachine = this.newStateMachine();
        this.containerSize = this.getConfiguredContainerSize(conf);
        this.containers = new ContainerStateMap();
        this.lastUsedMap = new ConcurrentHashMap();
        this.containerStateChangeActions = this.getContainerStateChangeActions();
        this.transactionBuffer = buffer;
        this.initialize();
    }

    private StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent> newStateMachine() {
        HashSet<HddsProtos.LifeCycleState> finalStates = new HashSet<HddsProtos.LifeCycleState>();
        finalStates.add(HddsProtos.LifeCycleState.CLOSED);
        finalStates.add(HddsProtos.LifeCycleState.DELETED);
        StateMachine containerLifecycleSM = new StateMachine((Enum)HddsProtos.LifeCycleState.OPEN, finalStates);
        containerLifecycleSM.addTransition((Enum)HddsProtos.LifeCycleState.OPEN, (Enum)HddsProtos.LifeCycleState.CLOSING, (Enum)HddsProtos.LifeCycleEvent.FINALIZE);
        containerLifecycleSM.addTransition((Enum)HddsProtos.LifeCycleState.CLOSING, (Enum)HddsProtos.LifeCycleState.QUASI_CLOSED, (Enum)HddsProtos.LifeCycleEvent.QUASI_CLOSE);
        containerLifecycleSM.addTransition((Enum)HddsProtos.LifeCycleState.CLOSING, (Enum)HddsProtos.LifeCycleState.CLOSED, (Enum)HddsProtos.LifeCycleEvent.CLOSE);
        containerLifecycleSM.addTransition((Enum)HddsProtos.LifeCycleState.QUASI_CLOSED, (Enum)HddsProtos.LifeCycleState.CLOSED, (Enum)HddsProtos.LifeCycleEvent.FORCE_CLOSE);
        containerLifecycleSM.addTransition((Enum)HddsProtos.LifeCycleState.CLOSED, (Enum)HddsProtos.LifeCycleState.DELETING, (Enum)HddsProtos.LifeCycleEvent.DELETE);
        containerLifecycleSM.addTransition((Enum)HddsProtos.LifeCycleState.DELETING, (Enum)HddsProtos.LifeCycleState.DELETED, (Enum)HddsProtos.LifeCycleEvent.CLEANUP);
        this.makeStateTransitionIdempotent((StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent>)containerLifecycleSM, HddsProtos.LifeCycleEvent.FINALIZE, HddsProtos.LifeCycleState.CLOSING, HddsProtos.LifeCycleState.QUASI_CLOSED, HddsProtos.LifeCycleState.CLOSED, HddsProtos.LifeCycleState.DELETING, HddsProtos.LifeCycleState.DELETED);
        this.makeStateTransitionIdempotent((StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent>)containerLifecycleSM, HddsProtos.LifeCycleEvent.QUASI_CLOSE, HddsProtos.LifeCycleState.QUASI_CLOSED, HddsProtos.LifeCycleState.CLOSED, HddsProtos.LifeCycleState.DELETING, HddsProtos.LifeCycleState.DELETED);
        this.makeStateTransitionIdempotent((StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent>)containerLifecycleSM, HddsProtos.LifeCycleEvent.CLOSE, HddsProtos.LifeCycleState.CLOSED, HddsProtos.LifeCycleState.DELETING, HddsProtos.LifeCycleState.DELETED);
        this.makeStateTransitionIdempotent((StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent>)containerLifecycleSM, HddsProtos.LifeCycleEvent.FORCE_CLOSE, HddsProtos.LifeCycleState.CLOSED, HddsProtos.LifeCycleState.DELETING, HddsProtos.LifeCycleState.DELETED);
        this.makeStateTransitionIdempotent((StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent>)containerLifecycleSM, HddsProtos.LifeCycleEvent.DELETE, HddsProtos.LifeCycleState.DELETING, HddsProtos.LifeCycleState.DELETED);
        this.makeStateTransitionIdempotent((StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent>)containerLifecycleSM, HddsProtos.LifeCycleEvent.CLEANUP, HddsProtos.LifeCycleState.DELETED);
        return containerLifecycleSM;
    }

    private void makeStateTransitionIdempotent(StateMachine<HddsProtos.LifeCycleState, HddsProtos.LifeCycleEvent> sm, HddsProtos.LifeCycleEvent event, HddsProtos.LifeCycleState ... states) {
        for (HddsProtos.LifeCycleState state : states) {
            sm.addTransition((Enum)state, (Enum)state, (Enum)event);
        }
    }

    private long getConfiguredContainerSize(Configuration conf) {
        return (long)conf.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
    }

    private void initialize() throws IOException {
        TableIterator iterator = this.containerStore.iterator();
        while (iterator.hasNext()) {
            ContainerInfo container = (ContainerInfo)((Table.KeyValue)iterator.next()).getValue();
            Preconditions.checkNotNull((Object)container);
            this.containers.addContainer(container);
            if (container.getState() != HddsProtos.LifeCycleState.OPEN) continue;
            try {
                this.pipelineManager.addContainerToPipelineSCMStart(container.getPipelineID(), container.containerID());
            }
            catch (PipelineNotFoundException ex) {
                LOG.warn("Found container {} which is in OPEN state with pipeline {} that does not exist. Marking container for closing.", (Object)container, (Object)container.getPipelineID());
                try {
                    this.updateContainerState(container.containerID().getProtobuf(), HddsProtos.LifeCycleEvent.FINALIZE);
                }
                catch (InvalidStateTransitionException e) {
                    LOG.warn("Unable to finalize Container {}.", (Object)container);
                }
            }
        }
    }

    private Map<HddsProtos.LifeCycleEvent, CheckedConsumer<ContainerInfo, IOException>> getContainerStateChangeActions() {
        EnumMap<HddsProtos.LifeCycleEvent, CheckedConsumer<ContainerInfo, IOException>> actions = new EnumMap<HddsProtos.LifeCycleEvent, CheckedConsumer<ContainerInfo, IOException>>(HddsProtos.LifeCycleEvent.class);
        actions.put(HddsProtos.LifeCycleEvent.FINALIZE, info -> this.pipelineManager.removeContainerFromPipeline(info.getPipelineID(), info.containerID()));
        return actions;
    }

    @Override
    public Set<ContainerID> getContainerIDs() {
        this.lock.readLock().lock();
        try {
            Set<ContainerID> set = this.containers.getAllContainerIDs();
            return set;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public Set<ContainerID> getContainerIDs(HddsProtos.LifeCycleState state) {
        this.lock.readLock().lock();
        try {
            NavigableSet<ContainerID> navigableSet = this.containers.getContainerIDsByState(state);
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public ContainerInfo getContainer(HddsProtos.ContainerID id) {
        this.lock.readLock().lock();
        try {
            ContainerInfo containerInfo = this.containers.getContainerInfo(ContainerID.getFromProtobuf((HddsProtos.ContainerID)id));
            return containerInfo;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void addContainer(HddsProtos.ContainerInfoProto containerInfo) throws IOException {
        Preconditions.checkNotNull((Object)containerInfo);
        ContainerInfo container = ContainerInfo.fromProtobuf((HddsProtos.ContainerInfoProto)containerInfo);
        ContainerID containerID = container.containerID();
        PipelineID pipelineID = container.getPipelineID();
        this.lock.writeLock().lock();
        try {
            if (!this.containers.contains(containerID)) {
                ExecutionUtil.create(() -> {
                    this.transactionBuffer.addToBuffer(this.containerStore, (Object)containerID, (Object)container);
                    this.containers.addContainer(container);
                    if (this.pipelineManager.containsPipeline(pipelineID)) {
                        this.pipelineManager.addContainerToPipeline(pipelineID, containerID);
                    } else if (containerInfo.getState().equals((Object)HddsProtos.LifeCycleState.OPEN)) {
                        throw new PipelineNotFoundException();
                    }
                }).onException(() -> {
                    this.containers.removeContainer(containerID);
                    this.transactionBuffer.removeFromBuffer(this.containerStore, (Object)containerID);
                }).execute();
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public boolean contains(HddsProtos.ContainerID id) {
        this.lock.readLock().lock();
        try {
            boolean bl = this.containers.contains(ContainerID.getFromProtobuf((HddsProtos.ContainerID)id));
            return bl;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateContainerState(HddsProtos.ContainerID containerID, HddsProtos.LifeCycleEvent event) throws IOException, InvalidStateTransitionException {
        ContainerID id = ContainerID.getFromProtobuf((HddsProtos.ContainerID)containerID);
        this.lock.writeLock().lock();
        try {
            if (this.containers.contains(id)) {
                ContainerInfo oldInfo = this.containers.getContainerInfo(id);
                HddsProtos.LifeCycleState oldState = oldInfo.getState();
                HddsProtos.LifeCycleState newState = (HddsProtos.LifeCycleState)this.stateMachine.getNextState((Enum)oldInfo.getState(), (Enum)event);
                if (newState.getNumber() > oldState.getNumber()) {
                    ExecutionUtil.create(() -> {
                        this.containers.updateState(id, oldState, newState);
                        this.transactionBuffer.addToBuffer(this.containerStore, (Object)id, (Object)this.containers.getContainerInfo(id));
                    }).onException(() -> {
                        this.transactionBuffer.addToBuffer(this.containerStore, (Object)id, (Object)oldInfo);
                        this.containers.updateState(id, newState, oldState);
                    }).execute();
                    this.containerStateChangeActions.getOrDefault(event, info -> {}).execute(oldInfo);
                }
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public Set<ContainerReplica> getContainerReplicas(HddsProtos.ContainerID id) {
        this.lock.readLock().lock();
        try {
            Set<ContainerReplica> set = this.containers.getContainerReplicas(ContainerID.getFromProtobuf((HddsProtos.ContainerID)id));
            return set;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public void updateContainerReplica(HddsProtos.ContainerID id, ContainerReplica replica) {
        this.lock.writeLock().lock();
        try {
            this.containers.updateContainerReplica(ContainerID.getFromProtobuf((HddsProtos.ContainerID)id), replica);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void removeContainerReplica(HddsProtos.ContainerID id, ContainerReplica replica) {
        this.lock.writeLock().lock();
        try {
            this.containers.removeContainerReplica(ContainerID.getFromProtobuf((HddsProtos.ContainerID)id), replica);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateDeleteTransactionId(Map<ContainerID, Long> deleteTransactionMap) throws IOException {
        this.lock.writeLock().lock();
        try {
            for (Map.Entry<ContainerID, Long> transaction : deleteTransactionMap.entrySet()) {
                ContainerInfo info = this.containers.getContainerInfo(transaction.getKey());
                info.updateDeleteTransactionId(transaction.getValue().longValue());
                this.transactionBuffer.addToBuffer(this.containerStore, (Object)info.containerID(), (Object)info);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerInfo getMatchingContainer(long size, String owner, PipelineID pipelineID, NavigableSet<ContainerID> containerIDs) {
        if (containerIDs.isEmpty()) {
            return null;
        }
        ContainerState key = new ContainerState(owner, pipelineID);
        ContainerID lastID = this.lastUsedMap.getOrDefault(key, (ContainerID)containerIDs.first());
        NavigableSet<ContainerID> resultSet = containerIDs.tailSet(lastID, false);
        if (resultSet.isEmpty()) {
            resultSet = containerIDs;
        }
        this.lock.readLock().lock();
        try {
            ContainerInfo selectedContainer = this.findContainerWithSpace(size, resultSet);
            if (selectedContainer == null) {
                resultSet = containerIDs.headSet(lastID, true);
                selectedContainer = this.findContainerWithSpace(size, resultSet);
            }
            if (selectedContainer != null) {
                this.lastUsedMap.put(key, selectedContainer.containerID());
            }
            ContainerInfo containerInfo = selectedContainer;
            return containerInfo;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private ContainerInfo findContainerWithSpace(long size, NavigableSet<ContainerID> searchSet) {
        for (ContainerID id : searchSet) {
            ContainerInfo containerInfo = this.containers.getContainerInfo(id);
            if (containerInfo.getUsedBytes() + size > this.containerSize) continue;
            containerInfo.updateLastUsedTime();
            return containerInfo;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void removeContainer(HddsProtos.ContainerID id) throws IOException {
        this.lock.writeLock().lock();
        try {
            ContainerID cid = ContainerID.getFromProtobuf((HddsProtos.ContainerID)id);
            ContainerInfo containerInfo = this.containers.getContainerInfo(cid);
            ExecutionUtil.create(() -> {
                this.transactionBuffer.removeFromBuffer(this.containerStore, (Object)cid);
                this.containers.removeContainer(cid);
            }).onException(() -> this.containerStore.put((Object)cid, (Object)containerInfo)).execute();
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void reinitialize(Table<ContainerID, ContainerInfo> store) throws IOException {
        this.lock.writeLock().lock();
        try {
            this.close();
            this.containerStore = store;
            this.containers = new ContainerStateMap();
            this.lastUsedMap = new ConcurrentHashMap();
            this.initialize();
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.containerStore.close();
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private Configuration conf;
        private PipelineManager pipelineMgr;
        private SCMRatisServer scmRatisServer;
        private Table<ContainerID, ContainerInfo> table;
        private DBTransactionBuffer transactionBuffer;

        public Builder setSCMDBTransactionBuffer(DBTransactionBuffer buffer) {
            this.transactionBuffer = buffer;
            return this;
        }

        public Builder setConfiguration(Configuration config) {
            this.conf = config;
            return this;
        }

        public Builder setPipelineManager(PipelineManager pipelineManager) {
            this.pipelineMgr = pipelineManager;
            return this;
        }

        public Builder setRatisServer(SCMRatisServer ratisServer) {
            this.scmRatisServer = ratisServer;
            return this;
        }

        public Builder setContainerStore(Table<ContainerID, ContainerInfo> containerStore) {
            this.table = containerStore;
            return this;
        }

        public ContainerStateManagerV2 build() throws IOException {
            Preconditions.checkNotNull((Object)this.conf);
            Preconditions.checkNotNull((Object)this.pipelineMgr);
            Preconditions.checkNotNull(this.table);
            ContainerStateManagerImpl csm = new ContainerStateManagerImpl(this.conf, this.pipelineMgr, this.table, this.transactionBuffer);
            SCMHAInvocationHandler invocationHandler = new SCMHAInvocationHandler(SCMRatisProtocol.RequestType.CONTAINER, csm, this.scmRatisServer);
            return (ContainerStateManagerV2)Proxy.newProxyInstance(SCMHAInvocationHandler.class.getClassLoader(), new Class[]{ContainerStateManagerV2.class}, (InvocationHandler)invocationHandler);
        }
    }
}

