/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.common.utils.BufferUtils;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.impl.ChunkLayOutVersion;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.ChunkUtils;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.BlockManager;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.ChunkManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilePerChunkStrategy
implements ChunkManager {
    private static final Logger LOG = LoggerFactory.getLogger(FilePerChunkStrategy.class);
    private final boolean doSyncWrite;
    private final BlockManager blockManager;
    private final long defaultReadBufferCapacity;
    private final VolumeSet volumeSet;

    public FilePerChunkStrategy(boolean sync, BlockManager manager, VolumeSet volSet) {
        this.doSyncWrite = sync;
        this.blockManager = manager;
        this.defaultReadBufferCapacity = manager == null ? 0L : manager.getDefaultReadBufferCapacity();
        this.volumeSet = volSet;
    }

    private static void checkLayoutVersion(Container container) {
        Preconditions.checkArgument((((ContainerData)container.getContainerData()).getLayOutVersion() == ChunkLayOutVersion.FILE_PER_CHUNK ? 1 : 0) != 0);
    }

    @Override
    public void writeChunk(Container container, BlockID blockID, ChunkInfo info, ChunkBuffer data, DispatcherContext dispatcherContext) throws StorageContainerException {
        FilePerChunkStrategy.checkLayoutVersion(container);
        Preconditions.checkNotNull((Object)dispatcherContext);
        DispatcherContext.WriteChunkStage stage = dispatcherContext.getStage();
        try {
            KeyValueContainer kvContainer = (KeyValueContainer)container;
            KeyValueContainerData containerData = kvContainer.getContainerData();
            HddsVolume volume = containerData.getVolume();
            File chunkFile = FilePerChunkStrategy.getChunkFile(kvContainer, blockID, info);
            boolean isOverwrite = ChunkUtils.validateChunkForOverwrite(chunkFile, info);
            File tmpChunkFile = this.getTmpChunkFile(chunkFile, dispatcherContext);
            if (LOG.isDebugEnabled()) {
                LOG.debug("writing chunk:{} chunk stage:{} chunk file:{} tmp chunk file:{}", new Object[]{info.getChunkName(), stage, chunkFile, tmpChunkFile});
            }
            long len = info.getLen();
            long offset = 0L;
            switch (stage) {
                case WRITE_DATA: {
                    if (isOverwrite) {
                        LOG.warn("ChunkFile already exists {}. Deleting it.", (Object)chunkFile);
                        FileUtil.fullyDelete((File)chunkFile);
                    }
                    if (tmpChunkFile.exists()) {
                        LOG.warn("tmpChunkFile already exists {}. Overwriting it.", (Object)tmpChunkFile);
                    }
                    ChunkUtils.writeData(tmpChunkFile, data, offset, len, volume, this.doSyncWrite);
                    break;
                }
                case COMMIT_DATA: {
                    if (isOverwrite) {
                        LOG.warn("ChunkFile already exists {}", (Object)chunkFile);
                        return;
                    }
                    this.commitChunk(tmpChunkFile, chunkFile);
                    containerData.updateWriteStats(len, isOverwrite);
                    break;
                }
                case COMBINED: {
                    ChunkUtils.writeData(chunkFile, data, offset, len, volume, this.doSyncWrite);
                    containerData.updateWriteStats(len, isOverwrite);
                    break;
                }
                default: {
                    throw new IOException("Can not identify write operation.");
                }
            }
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (IOException ex) {
            throw new StorageContainerException("Internal error: ", (Throwable)ex, ContainerProtos.Result.IO_EXCEPTION);
        }
    }

    @Override
    public ChunkBuffer readChunk(Container container, BlockID blockID, ChunkInfo info, DispatcherContext dispatcherContext) throws StorageContainerException {
        FilePerChunkStrategy.checkLayoutVersion(container);
        ChunkUtils.limitReadSize(info.getLen());
        KeyValueContainer kvContainer = (KeyValueContainer)container;
        KeyValueContainerData containerData = kvContainer.getContainerData();
        HddsVolume volume = containerData.getVolume();
        File finalChunkFile = FilePerChunkStrategy.getChunkFile(kvContainer, blockID, info);
        ArrayList<File> possibleFiles = new ArrayList<File>();
        possibleFiles.add(finalChunkFile);
        if (dispatcherContext != null && dispatcherContext.isReadFromTmpFile()) {
            possibleFiles.add(this.getTmpChunkFile(finalChunkFile, dispatcherContext));
            possibleFiles.add(finalChunkFile);
        }
        long len = info.getLen();
        long bufferCapacity = ChunkManager.getBufferCapacityForChunkRead(info, this.defaultReadBufferCapacity);
        Object[] dataBuffers = BufferUtils.assignByteBuffers((long)len, (long)bufferCapacity);
        long chunkFileOffset = 0L;
        if (info.getOffset() != 0L) {
            try {
                BlockData blockData = this.blockManager.getBlock(kvContainer, blockID);
                List chunks = blockData.getChunks();
                String chunkName = info.getChunkName();
                boolean found = false;
                for (ContainerProtos.ChunkInfo chunk : chunks) {
                    if (!chunk.getChunkName().equals(chunkName)) continue;
                    chunkFileOffset = chunk.getOffset();
                    found = true;
                    break;
                }
                if (!found) {
                    throw new StorageContainerException("Cannot find chunk " + chunkName + " in block " + blockID.toString(), ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
                }
            }
            catch (IOException e) {
                throw new StorageContainerException("Cannot find block " + blockID.toString() + " for chunk " + info.getChunkName(), ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
            }
        }
        for (File file : possibleFiles) {
            try {
                if (!file.exists()) continue;
                long offset = info.getOffset() - chunkFileOffset;
                Preconditions.checkState((offset >= 0L ? 1 : 0) != 0);
                ChunkUtils.readData(file, (ByteBuffer[])dataBuffers, offset, len, volume);
                return ChunkBuffer.wrap((List)Lists.newArrayList((Object[])dataBuffers));
            }
            catch (StorageContainerException ex) {
                if (ex.getResult() != ContainerProtos.Result.UNABLE_TO_FIND_CHUNK) {
                    throw ex;
                }
                BufferUtils.clearBuffers((ByteBuffer[])dataBuffers);
            }
        }
        throw new StorageContainerException("Chunk file can't be found " + ((Object)possibleFiles).toString(), ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
    }

    @Override
    public void deleteChunk(Container container, BlockID blockID, ChunkInfo info) throws StorageContainerException {
        boolean allowed;
        FilePerChunkStrategy.checkLayoutVersion(container);
        Preconditions.checkNotNull((Object)blockID, (Object)"Block ID cannot be null.");
        KeyValueContainer kvContainer = (KeyValueContainer)container;
        File chunkFile = FilePerChunkStrategy.getChunkFile(kvContainer, blockID, info);
        if (!chunkFile.exists()) {
            LOG.warn("Chunk file not found for chunk {}", (Object)info);
            return;
        }
        long chunkFileSize = chunkFile.length();
        boolean bl = allowed = info.getLen() == chunkFileSize || info.getLen() + info.getOffset() == chunkFileSize;
        if (!allowed) {
            LOG.error("Not Supported Operation. Trying to delete a chunk that is in shared file. chunk info : {}", (Object)info);
            throw new StorageContainerException("Not Supported Operation. Trying to delete a chunk that is in shared file. chunk info : " + info, ContainerProtos.Result.UNSUPPORTED_REQUEST);
        }
        FileUtil.fullyDelete((File)chunkFile);
        LOG.info("Deleted chunk file {} (size {}) for chunk {}", new Object[]{chunkFile, chunkFileSize, info});
    }

    @Override
    public void deleteChunks(Container container, BlockData blockData) throws StorageContainerException {
        for (ContainerProtos.ChunkInfo chunk : blockData.getChunks()) {
            try {
                ChunkInfo chunkInfo = ChunkInfo.getFromProtoBuf((ContainerProtos.ChunkInfo)chunk);
                this.deleteChunk(container, blockData.getBlockID(), chunkInfo);
            }
            catch (IOException e) {
                throw new StorageContainerException((Throwable)e, ContainerProtos.Result.INVALID_ARGUMENT);
            }
        }
    }

    private static File getChunkFile(KeyValueContainer container, BlockID blockID, ChunkInfo info) throws StorageContainerException {
        return ChunkLayOutVersion.FILE_PER_CHUNK.getChunkFile(container.getContainerData(), blockID, info);
    }

    private File getTmpChunkFile(File chunkFile, DispatcherContext dispatcherContext) {
        return new File(chunkFile.getParent(), chunkFile.getName() + "." + "tmp" + "." + dispatcherContext.getTerm() + "." + dispatcherContext.getLogIndex());
    }

    private void commitChunk(File tmpChunkFile, File chunkFile) throws IOException {
        Files.move(tmpChunkFile.toPath(), chunkFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }
}

