/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.OzoneChecksumException;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ChunkLayOutVersion;
import org.apache.hadoop.ozone.container.common.impl.ContainerDataYaml;
import org.apache.hadoop.ozone.container.common.interfaces.BlockIterator;
import org.apache.hadoop.ozone.container.common.utils.ReferenceCountedDB;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.ChunkUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerLocationUtil;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyValueContainerCheck {
    private static final Logger LOG = LoggerFactory.getLogger(KeyValueContainerCheck.class);
    private long containerID;
    private KeyValueContainerData onDiskContainerData;
    private ConfigurationSource checkConfig;
    private String metadataPath;

    public KeyValueContainerCheck(String metadataPath, ConfigurationSource conf, long containerID) {
        Preconditions.checkArgument((metadataPath != null ? 1 : 0) != 0);
        this.checkConfig = conf;
        this.containerID = containerID;
        this.onDiskContainerData = null;
        this.metadataPath = metadataPath;
    }

    public boolean fastCheck() {
        LOG.info("Running basic checks for container {};", (Object)this.containerID);
        boolean valid = false;
        try {
            this.loadContainerData();
            this.checkLayout();
            this.checkContainerFile();
            valid = true;
        }
        catch (IOException e) {
            LOG.info("Marking Container {} UNHEALTHY as it failed metadata check", (Object)this.containerID);
            this.handleCorruption(e);
        }
        return valid;
    }

    public boolean fullCheck(DataTransferThrottler throttler, Canceler canceler) {
        boolean valid;
        try {
            valid = this.fastCheck();
            if (valid) {
                this.scanData(throttler, canceler);
            }
        }
        catch (IOException e) {
            this.handleCorruption(e);
            valid = false;
        }
        return valid;
    }

    private void checkLayout() throws IOException {
        this.checkDirPath(this.metadataPath);
        String chunksPath = this.onDiskContainerData.getChunksPath();
        this.checkDirPath(chunksPath);
    }

    private void checkDirPath(String path) throws IOException {
        File dirPath = new File(path);
        try {
            if (!dirPath.isDirectory()) {
                String errStr = "Not a directory [" + path + "]";
                throw new IOException(errStr);
            }
        }
        catch (SecurityException se) {
            throw new IOException("Security exception checking dir [" + path + "]", se);
        }
        String[] ls = dirPath.list();
        if (ls == null) {
            String errStr = "null listing for directory [" + path + "]";
            throw new IOException(errStr);
        }
    }

    private void checkContainerFile() throws IOException {
        Preconditions.checkState((this.onDiskContainerData != null ? 1 : 0) != 0, (Object)"Container File not loaded");
        ContainerUtils.verifyChecksum(this.onDiskContainerData, this.checkConfig);
        if (this.onDiskContainerData.getContainerType() != ContainerProtos.ContainerType.KeyValueContainer) {
            String errStr = "Bad Container type in Containerdata for " + this.containerID;
            throw new IOException(errStr);
        }
        if (this.onDiskContainerData.getContainerID() != this.containerID) {
            String errStr = "Bad ContainerID field in Containerdata for " + this.containerID;
            throw new IOException(errStr);
        }
        String dbType = this.onDiskContainerData.getContainerDBType();
        if (!dbType.equals("RocksDB")) {
            String errStr = "Unknown DBType [" + dbType + "] in Container File for  [" + this.containerID + "]";
            throw new IOException(errStr);
        }
        KeyValueContainerData kvData = this.onDiskContainerData;
        if (!this.metadataPath.equals(kvData.getMetadataPath())) {
            String errStr = "Bad metadata path in Containerdata for " + this.containerID + "Expected [" + this.metadataPath + "] Got [" + kvData.getMetadataPath() + "]";
            throw new IOException(errStr);
        }
    }

    private void scanData(DataTransferThrottler throttler, Canceler canceler) throws IOException {
        Preconditions.checkState((this.onDiskContainerData != null ? 1 : 0) != 0, (Object)"invoke loadContainerData prior to calling this function");
        File metaDir = new File(this.metadataPath);
        File dbFile = KeyValueContainerLocationUtil.getContainerDBFile(metaDir, this.containerID);
        if (!dbFile.exists() || !dbFile.canRead()) {
            String dbFileErrorMsg = "Unable to access DB File [" + dbFile.toString() + "] for Container [" + this.containerID + "] metadata path [" + this.metadataPath + "]";
            throw new IOException(dbFileErrorMsg);
        }
        this.onDiskContainerData.setDbFile(dbFile);
        ChunkLayOutVersion layout = this.onDiskContainerData.getLayOutVersion();
        try (ReferenceCountedDB db = BlockUtils.getDB(this.onDiskContainerData, this.checkConfig);
             BlockIterator<BlockData> kvIter = db.getStore().getBlockIterator();){
            while (kvIter.hasNext()) {
                BlockData block = kvIter.nextBlock();
                for (ContainerProtos.ChunkInfo chunk : block.getChunks()) {
                    File chunkFile = layout.getChunkFile(this.onDiskContainerData, block.getBlockID(), ChunkInfo.getFromProtoBuf((ContainerProtos.ChunkInfo)chunk));
                    if (!chunkFile.exists()) {
                        String localBlockID = Long.toString(block.getBlockID().getLocalID());
                        BlockData bdata = (BlockData)db.getStore().getBlockDataTable().get((Object)localBlockID);
                        if (bdata == null) continue;
                        throw new IOException("Missing chunk file " + chunkFile.getAbsolutePath());
                    }
                    if (chunk.getChecksumData().getType() == ContainerProtos.ChecksumType.NONE) continue;
                    KeyValueContainerCheck.verifyChecksum(block, chunk, chunkFile, layout, throttler, canceler);
                }
            }
        }
    }

    private static void verifyChecksum(BlockData block, ContainerProtos.ChunkInfo chunk, File chunkFile, ChunkLayOutVersion layout, DataTransferThrottler throttler, Canceler canceler) throws IOException {
        ChecksumData checksumData = ChecksumData.getFromProtoBuf((ContainerProtos.ChecksumData)chunk.getChecksumData());
        int checksumCount = checksumData.getChecksums().size();
        int bytesPerChecksum = checksumData.getBytesPerChecksum();
        Checksum cal = new Checksum(checksumData.getChecksumType(), bytesPerChecksum);
        ByteBuffer buffer = ByteBuffer.allocate(bytesPerChecksum);
        long bytesRead = 0L;
        try (FileChannel channel = FileChannel.open(chunkFile.toPath(), ChunkUtils.READ_OPTIONS, ChunkUtils.NO_ATTRIBUTES);){
            if (layout == ChunkLayOutVersion.FILE_PER_BLOCK) {
                channel.position(chunk.getOffset());
            }
            for (int i = 0; i < checksumCount; ++i) {
                int v;
                if (layout == ChunkLayOutVersion.FILE_PER_BLOCK && i == checksumCount - 1 && chunk.getLen() % (long)bytesPerChecksum != 0L) {
                    buffer.limit((int)(chunk.getLen() % (long)bytesPerChecksum));
                }
                if ((v = channel.read(buffer)) == -1) break;
                bytesRead += (long)v;
                buffer.flip();
                throttler.throttle((long)v, canceler);
                ByteString expected = (ByteString)checksumData.getChecksums().get(i);
                ByteString actual = (ByteString)cal.computeChecksum(buffer).getChecksums().get(0);
                if (expected.equals((Object)actual)) continue;
                throw new OzoneChecksumException(String.format("Inconsistent read for chunk=%s checksum item %d expected checksum %s actual checksum %s for block %s", ChunkInfo.getFromProtoBuf((ContainerProtos.ChunkInfo)chunk), i, Arrays.toString(expected.toByteArray()), Arrays.toString(actual.toByteArray()), block.getBlockID()));
            }
            if (bytesRead != chunk.getLen()) {
                throw new OzoneChecksumException(String.format("Inconsistent read for chunk=%s expected length=%d actual length=%d for block %s", chunk.getChunkName(), chunk.getLen(), bytesRead, block.getBlockID()));
            }
        }
    }

    private void loadContainerData() throws IOException {
        File containerFile = KeyValueContainer.getContainerFile(this.metadataPath, this.containerID);
        this.onDiskContainerData = (KeyValueContainerData)ContainerDataYaml.readContainerFile(containerFile);
    }

    private void handleCorruption(IOException e) {
        String errStr = "Corruption detected in container: [" + this.containerID + "] ";
        String logMessage = errStr + "Exception: [" + e.getMessage() + "]";
        LOG.error(logMessage);
    }
}

