/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.ozone.common.InconsistentStorageStateException;
import org.apache.hadoop.ozone.container.common.HDDSVolumeLayoutVersion;
import org.apache.hadoop.ozone.container.common.helpers.DatanodeVersionFile;
import org.apache.hadoop.ozone.container.common.utils.HddsVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.StorageVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class HddsVolume
extends StorageVolume {
    private static final Logger LOG = LoggerFactory.getLogger(HddsVolume.class);
    public static final String HDDS_VOLUME_DIR = "hdds";
    private VolumeState state;
    private final VolumeIOStats volumeIOStats;
    private String storageID;
    private String clusterID;
    private String datanodeUuid;
    private long cTime;
    private int layoutVersion;
    private final AtomicLong committedBytes;

    private HddsVolume(Builder b) throws IOException {
        super(b);
        if (!b.getFailedVolume()) {
            this.state = VolumeState.NOT_INITIALIZED;
            this.clusterID = b.clusterID;
            this.datanodeUuid = b.datanodeUuid;
            this.volumeIOStats = new VolumeIOStats(b.getVolumeRootStr());
            this.committedBytes = new AtomicLong(0L);
            LOG.info("Creating HddsVolume: {} of storage type : {} capacity : {}", new Object[]{this.getStorageDir(), b.getStorageType(), this.getVolumeInfo().getCapacity()});
            this.initialize();
        } else {
            this.volumeIOStats = null;
            this.storageID = UUID.randomUUID().toString();
            this.state = VolumeState.FAILED;
            this.committedBytes = null;
        }
    }

    private void initialize() throws IOException {
        VolumeState intialVolumeState = this.analyzeVolumeState();
        switch (intialVolumeState) {
            case NON_EXISTENT: {
                if (!this.getStorageDir().mkdirs()) {
                    throw new IOException("Cannot create directory " + this.getStorageDir());
                }
                this.setState(VolumeState.NOT_FORMATTED);
                this.createVersionFile();
                break;
            }
            case NOT_FORMATTED: {
                this.createVersionFile();
                break;
            }
            case NOT_INITIALIZED: {
                this.readVersionFile();
                this.setState(VolumeState.NORMAL);
                break;
            }
            case INCONSISTENT: {
                throw new IOException("Volume is in an " + (Object)((Object)VolumeState.INCONSISTENT) + " state. Skipped loading volume: " + this.getStorageDir().getPath());
            }
            default: {
                throw new IOException("Unrecognized initial state : " + (Object)((Object)intialVolumeState) + "of volume : " + this.getStorageDir());
            }
        }
    }

    private VolumeState analyzeVolumeState() {
        if (!this.getStorageDir().exists()) {
            return VolumeState.NON_EXISTENT;
        }
        if (!this.getStorageDir().isDirectory()) {
            LOG.warn("Volume {} exists but is not a directory, current volume state: {}.", (Object)this.getStorageDir().getPath(), (Object)VolumeState.INCONSISTENT);
            return VolumeState.INCONSISTENT;
        }
        File[] files = this.getStorageDir().listFiles();
        if (files == null || files.length == 0) {
            return VolumeState.NOT_FORMATTED;
        }
        if (!this.getVersionFile().exists()) {
            LOG.warn("VERSION file does not exist in volume {}, current volume state: {}.", (Object)this.getStorageDir().getPath(), (Object)VolumeState.INCONSISTENT);
            return VolumeState.INCONSISTENT;
        }
        return VolumeState.NOT_INITIALIZED;
    }

    public void format(String cid) throws IOException {
        Preconditions.checkNotNull((Object)cid, (Object)"clusterID cannot be null while formatting Volume");
        this.clusterID = cid;
        this.initialize();
    }

    private void createVersionFile() throws IOException {
        this.storageID = HddsVolumeUtil.generateUuid();
        this.cTime = Time.now();
        this.layoutVersion = HDDSVolumeLayoutVersion.getLatestVersion().getVersion();
        if (this.clusterID == null || this.datanodeUuid == null) {
            LOG.debug("ClusterID not available. Cannot format the volume {}", (Object)this.getStorageDir().getPath());
            this.setState(VolumeState.NOT_FORMATTED);
        } else {
            this.writeVersionFile();
            this.setState(VolumeState.NORMAL);
        }
    }

    private void writeVersionFile() throws IOException {
        Preconditions.checkNotNull((Object)this.storageID, (Object)"StorageID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.clusterID, (Object)"ClusterID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.datanodeUuid, (Object)"DatanodeUUID cannot be null in Version File");
        Preconditions.checkArgument((this.cTime > 0L ? 1 : 0) != 0, (Object)"Creation Time should be positive");
        Preconditions.checkArgument((this.layoutVersion == HDDSVolumeLayoutVersion.getLatestVersion().getVersion() ? 1 : 0) != 0, (Object)"Version File should have the latest LayOutVersion");
        File versionFile = this.getVersionFile();
        LOG.debug("Writing Version file to disk, {}", (Object)versionFile);
        DatanodeVersionFile dnVersionFile = new DatanodeVersionFile(this.storageID, this.clusterID, this.datanodeUuid, this.cTime, this.layoutVersion);
        dnVersionFile.createVersionFile(versionFile);
    }

    private void readVersionFile() throws IOException {
        File versionFile = this.getVersionFile();
        Properties props = DatanodeVersionFile.readFrom(versionFile);
        if (props.isEmpty()) {
            throw new InconsistentStorageStateException("Version file " + versionFile + " is missing");
        }
        LOG.debug("Reading Version file from disk, {}", (Object)versionFile);
        this.storageID = HddsVolumeUtil.getStorageID(props, versionFile);
        this.clusterID = HddsVolumeUtil.getClusterID(props, versionFile, this.clusterID);
        this.datanodeUuid = HddsVolumeUtil.getDatanodeUUID(props, versionFile, this.datanodeUuid);
        this.cTime = HddsVolumeUtil.getCreationTime(props, versionFile);
        this.layoutVersion = HddsVolumeUtil.getLayOutVersion(props, versionFile);
    }

    private File getVersionFile() {
        return HddsVolumeUtil.getVersionFile(super.getStorageDir());
    }

    public File getHddsRootDir() {
        return super.getStorageDir();
    }

    @Override
    public String getStorageID() {
        return this.storageID;
    }

    public String getClusterID() {
        return this.clusterID;
    }

    public String getDatanodeUuid() {
        return this.datanodeUuid;
    }

    public long getCTime() {
        return this.cTime;
    }

    public int getLayoutVersion() {
        return this.layoutVersion;
    }

    public VolumeState getStorageState() {
        return this.state;
    }

    public void setState(VolumeState state) {
        this.state = state;
    }

    public boolean isFailed() {
        return this.state == VolumeState.FAILED;
    }

    public VolumeIOStats getVolumeIOStats() {
        return this.volumeIOStats;
    }

    @Override
    public void failVolume() {
        this.setState(VolumeState.FAILED);
        super.failVolume();
        if (this.volumeIOStats != null) {
            this.volumeIOStats.unregister();
        }
    }

    @Override
    public void shutdown() {
        this.state = VolumeState.NON_EXISTENT;
        super.shutdown();
        if (this.volumeIOStats != null) {
            this.volumeIOStats.unregister();
        }
    }

    public long incCommittedBytes(long delta) {
        return this.committedBytes.addAndGet(delta);
    }

    public long getCommittedBytes() {
        return this.committedBytes.get();
    }

    public static enum VolumeState {
        NORMAL,
        FAILED,
        NON_EXISTENT,
        INCONSISTENT,
        NOT_FORMATTED,
        NOT_INITIALIZED;

    }

    public static class Builder
    extends StorageVolume.Builder<Builder> {
        private String datanodeUuid;
        private String clusterID;

        public Builder(String volumeRootStr) {
            super(volumeRootStr, HddsVolume.HDDS_VOLUME_DIR);
        }

        @Override
        public Builder getThis() {
            return this;
        }

        public Builder datanodeUuid(String datanodeUUID) {
            this.datanodeUuid = datanodeUUID;
            return this;
        }

        public Builder clusterID(String cid) {
            this.clusterID = cid;
            return this;
        }

        @Override
        public HddsVolume build() throws IOException {
            return new HddsVolume(this);
        }
    }
}

