/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.server.events;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.hdds.server.events.Event;
import org.apache.hadoop.hdds.server.events.EventExecutor;
import org.apache.hadoop.hdds.server.events.EventHandler;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.server.events.SingleThreadExecutor;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventQueue
implements EventPublisher,
AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(EventQueue.class);
    private static final String EXECUTOR_NAME_SEPARATOR = "For";
    private final Map<Event, Map<EventExecutor, List<EventHandler>>> executors = new HashMap<Event, Map<EventExecutor, List<EventHandler>>>();
    private final AtomicLong queuedCount = new AtomicLong(0L);
    private final AtomicLong eventCount = new AtomicLong(0L);
    private boolean isRunning = true;
    private static final Gson TRACING_SERIALIZER = new GsonBuilder().create();
    private boolean isSilent = false;

    public <PAYLOAD, EVENT_TYPE extends Event<PAYLOAD>> void addHandler(EVENT_TYPE event, EventHandler<PAYLOAD> handler) {
        Preconditions.checkNotNull(handler, (Object)"Handler should not be null.");
        this.validateEvent(event);
        String executorName = EventQueue.getExecutorName(event, handler);
        this.addHandler(event, new SingleThreadExecutor(executorName), handler);
    }

    public static <PAYLOAD> String getExecutorName(Event<PAYLOAD> event, EventHandler<PAYLOAD> eventHandler) {
        return StringUtils.camelize((String)event.getName()) + EXECUTOR_NAME_SEPARATOR + EventQueue.generateHandlerName(eventHandler);
    }

    private <EVENT_TYPE extends Event<?>> void validateEvent(EVENT_TYPE event) {
        Preconditions.checkArgument((!event.getName().contains(EXECUTOR_NAME_SEPARATOR) ? 1 : 0) != 0, (Object)"Event name should not contain For string.");
    }

    private static <PAYLOAD> String generateHandlerName(EventHandler<PAYLOAD> handler) {
        if (!handler.getClass().isAnonymousClass()) {
            return handler.getClass().getSimpleName();
        }
        return handler.getClass().getName();
    }

    public <PAYLOAD, EVENT_TYPE extends Event<PAYLOAD>> void addHandler(EVENT_TYPE event, EventExecutor<PAYLOAD> executor, EventHandler<PAYLOAD> handler) {
        if (!this.isRunning) {
            LOG.warn("Not adding handler for {}, EventQueue is not running", event);
            return;
        }
        this.validateEvent(event);
        String executorName = EventQueue.getExecutorName(event, handler);
        Preconditions.checkState((boolean)executorName.equals(executor.getName()), (Object)("Event Executor name is not matching the specified format. It should be " + executorName + " but it is " + executor.getName()));
        this.executors.putIfAbsent(event, new HashMap());
        this.executors.get(event).putIfAbsent(executor, new ArrayList());
        this.executors.get(event).get(executor).add(handler);
    }

    @Override
    public <PAYLOAD, EVENT_TYPE extends Event<PAYLOAD>> void fireEvent(EVENT_TYPE event, PAYLOAD payload) {
        if (!this.isRunning) {
            LOG.warn("Processing of {} is skipped, EventQueue is not running", event);
            return;
        }
        Map<EventExecutor, List<EventHandler>> eventExecutorListMap = this.executors.get(event);
        this.eventCount.incrementAndGet();
        if (eventExecutorListMap != null) {
            for (Map.Entry<EventExecutor, List<EventHandler>> executorAndHandlers : eventExecutorListMap.entrySet()) {
                for (EventHandler handler : executorAndHandlers.getValue()) {
                    this.queuedCount.incrementAndGet();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Delivering [event={}] to executor/handler {}: <json>{}</json>", new Object[]{event.getName(), executorAndHandlers.getKey().getName(), TRACING_SERIALIZER.toJson(payload).replaceAll("\n", "\\\\n")});
                    } else if (LOG.isDebugEnabled()) {
                        LOG.debug("Delivering [event={}] to executor/handler {}: {}", new Object[]{event.getName(), executorAndHandlers.getKey().getName(), payload.getClass().getSimpleName()});
                    }
                    executorAndHandlers.getKey().onMessage(handler, payload, this);
                }
            }
        } else if (!this.isSilent) {
            LOG.warn("No event handler registered for event {}", event);
        }
    }

    @VisibleForTesting
    public void processAll(long timeout) {
        long processed;
        long currentTime = Time.now();
        do {
            if (!this.isRunning) {
                LOG.warn("Processing of event skipped. EventQueue is not running");
                return;
            }
            processed = 0L;
            Stream allExecutor = this.executors.values().stream().flatMap(handlerMap -> handlerMap.keySet().stream());
            boolean allIdle = allExecutor.allMatch(executor -> executor.queuedEvents() == executor.successfulEvents() + executor.failedEvents());
            if (allIdle) {
                return;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted exception while sleeping.", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        } while (Time.now() <= currentTime + timeout);
        throw new AssertionError((Object)("Messages are not processed in the given timeframe. Queued: " + this.queuedCount.get() + " Processed: " + processed));
    }

    @Override
    public void close() {
        this.isRunning = false;
        Set<EventExecutor> allExecutors = this.executors.values().stream().flatMap(handlerMap -> handlerMap.keySet().stream()).collect(Collectors.toSet());
        allExecutors.forEach(executor -> {
            try {
                executor.close();
            }
            catch (Exception ex) {
                LOG.error("Can't close the executor " + executor.getName(), (Throwable)ex);
            }
        });
    }

    public void setSilent(boolean silent) {
        this.isSilent = silent;
    }

    @VisibleForTesting
    public Map<EventExecutor, List<EventHandler>> getExecutorAndHandler(Event event) {
        return this.executors.get(event);
    }
}

