/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.authority;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyPair;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.sql.Date;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.validator.routines.DomainValidator;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.metadata.SCMMetadataStore;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CRLApprover;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateApprover;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateServer;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateStore;
import org.apache.hadoop.hdds.security.x509.certificate.authority.DefaultApprover;
import org.apache.hadoop.hdds.security.x509.certificate.authority.DefaultCRLApprover;
import org.apache.hadoop.hdds.security.x509.certificate.authority.PKIProfiles.PKIProfile;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateCodec;
import org.apache.hadoop.hdds.security.x509.certificates.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.security.x509.certificates.utils.SelfSignedCertificate;
import org.apache.hadoop.hdds.security.x509.crl.CRLInfo;
import org.apache.hadoop.hdds.security.x509.exceptions.CertificateException;
import org.apache.hadoop.hdds.security.x509.keys.HDDSKeyGenerator;
import org.apache.hadoop.hdds.security.x509.keys.KeyCodec;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.bouncycastle.asn1.x509.CRLReason;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCAServer
implements CertificateServer {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultCAServer.class);
    private final String subject;
    private final String clusterID;
    private final String scmID;
    private String componentName;
    private Path caKeysPath;
    private Path caRootX509Path;
    private SecurityConfig config;
    private PKIProfile profile;
    private CertificateApprover approver;
    private CRLApprover crlApprover;
    private CertificateStore store;
    private Lock lock;

    public DefaultCAServer(String subject, String clusterID, String scmID, CertificateStore certificateStore, PKIProfile pkiProfile, String componentName) {
        this.subject = subject;
        this.clusterID = clusterID;
        this.scmID = scmID;
        this.store = certificateStore;
        this.profile = pkiProfile;
        this.componentName = componentName;
        this.lock = new ReentrantLock();
    }

    @Override
    public void init(SecurityConfig securityConfig, CertificateServer.CAType type) throws IOException {
        this.caKeysPath = securityConfig.getKeyLocation(this.componentName);
        this.caRootX509Path = securityConfig.getCertificateLocation(this.componentName);
        this.config = securityConfig;
        this.approver = new DefaultApprover(this.profile, this.config);
        VerificationStatus status = this.verifySelfSignedCA(securityConfig);
        Consumer<SecurityConfig> caInitializer = this.processVerificationStatus(status, type);
        caInitializer.accept(securityConfig);
        this.crlApprover = new DefaultCRLApprover(securityConfig, this.getCAKeys().getPrivate());
    }

    @Override
    public X509CertificateHolder getCACertificate() throws IOException {
        CertificateCodec certificateCodec = new CertificateCodec(this.config, this.componentName);
        try {
            return certificateCodec.readCertificate();
        }
        catch (java.security.cert.CertificateException e) {
            throw new IOException(e);
        }
    }

    @Override
    public X509Certificate getCertificate(String certSerialId) throws IOException {
        return this.store.getCertificateByID(new BigInteger(certSerialId), CertificateStore.CertType.VALID_CERTS);
    }

    private KeyPair getCAKeys() throws IOException {
        KeyCodec keyCodec = new KeyCodec(this.config, this.componentName);
        try {
            return new KeyPair(keyCodec.readPublicKey(), keyCodec.readPrivateKey());
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new IOException(e);
        }
    }

    @Override
    public Future<X509CertificateHolder> requestCertificate(PKCS10CertificationRequest csr, CertificateApprover.ApprovalType approverType, HddsProtos.NodeType role) {
        LocalDate beginDate = LocalDate.now().atStartOfDay().toLocalDate();
        LocalDateTime temp = LocalDateTime.of(beginDate, LocalTime.MIDNIGHT);
        LocalDate endDate = role == HddsProtos.NodeType.SCM ? temp.plus(this.config.getMaxCertificateDuration()).toLocalDate() : temp.plus(this.config.getDefaultCertDuration()).toLocalDate();
        CompletableFuture<X509CertificateHolder> xcertHolder = this.approver.inspectCSR(csr);
        if (xcertHolder.isCompletedExceptionally()) {
            return xcertHolder;
        }
        try {
            switch (approverType) {
                case MANUAL: {
                    xcertHolder.completeExceptionally(new SCMSecurityException("Manual approval is not yet implemented."));
                    break;
                }
                case KERBEROS_TRUSTED: 
                case TESTING_AUTOMATIC: {
                    X509CertificateHolder xcert;
                    try {
                        xcert = this.signAndStoreCertificate(beginDate, endDate, csr, role);
                    }
                    catch (SCMSecurityException e) {
                        LOG.error("Certificate storage failed, retrying one more time.", (Throwable)e);
                        xcert = this.signAndStoreCertificate(beginDate, endDate, csr, role);
                    }
                    xcertHolder.complete(xcert);
                    break;
                }
                default: {
                    return null;
                }
            }
        }
        catch (IOException | java.security.cert.CertificateException | OperatorCreationException e) {
            LOG.error("Unable to issue a certificate.", e);
            xcertHolder.completeExceptionally(new SCMSecurityException((Exception)e, SCMSecurityException.ErrorCode.UNABLE_TO_ISSUE_CERTIFICATE));
        }
        return xcertHolder;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private X509CertificateHolder signAndStoreCertificate(LocalDate beginDate, LocalDate endDate, PKCS10CertificationRequest csr, HddsProtos.NodeType role) throws IOException, OperatorCreationException, java.security.cert.CertificateException {
        X509CertificateHolder xcert;
        this.lock.lock();
        try {
            xcert = this.approver.sign(this.config, this.getCAKeys().getPrivate(), this.getCACertificate(), Date.valueOf(beginDate), Date.valueOf(endDate), csr, this.scmID, this.clusterID);
            if (this.store != null) {
                this.store.checkValidCertID(xcert.getSerialNumber());
                this.store.storeValidCertificate(xcert.getSerialNumber(), CertificateCodec.getX509Certificate((X509CertificateHolder)xcert), role);
            }
        }
        finally {
            this.lock.unlock();
        }
        return xcert;
    }

    @Override
    public Future<X509CertificateHolder> requestCertificate(String csr, CertificateApprover.ApprovalType type, HddsProtos.NodeType nodeType) throws IOException {
        PKCS10CertificationRequest request = CertificateSignRequest.getCertificationRequest(csr);
        return this.requestCertificate(request, type, nodeType);
    }

    @Override
    public Future<Optional<Long>> revokeCertificates(List<BigInteger> certificates, CRLReason reason, java.util.Date revocationTime) {
        CompletableFuture<Optional<Long>> revoked = new CompletableFuture<Optional<Long>>();
        if (CollectionUtils.isEmpty(certificates)) {
            revoked.completeExceptionally(new SCMSecurityException("Certificates cannot be null or empty"));
            return revoked;
        }
        try {
            revoked.complete(this.store.revokeCertificates(certificates, this.getCACertificate(), reason, revocationTime, this.crlApprover));
        }
        catch (IOException ex) {
            LOG.error("Revoking the certificate failed.", ex.getCause());
            revoked.completeExceptionally(new SCMSecurityException((Throwable)ex));
        }
        return revoked;
    }

    @Override
    public List<X509Certificate> listCertificate(HddsProtos.NodeType role, long startSerialId, int count, boolean isRevoked) throws IOException {
        return this.store.listCertificate(role, BigInteger.valueOf(startSerialId), count, isRevoked ? CertificateStore.CertType.REVOKED_CERTS : CertificateStore.CertType.VALID_CERTS);
    }

    @Override
    public void reinitialize(SCMMetadataStore scmMetadataStore) {
        this.store.reinitialize(scmMetadataStore);
    }

    @Override
    public List<CRLInfo> getCrls(List<Long> crlIds) throws IOException {
        return this.store.getCrls(crlIds);
    }

    @Override
    public long getLatestCrlId() {
        return this.store.getLatestCrlId();
    }

    private void generateSelfSignedCA(SecurityConfig securityConfig) throws NoSuchAlgorithmException, NoSuchProviderException, IOException {
        KeyPair keyPair = this.generateKeys(securityConfig);
        this.generateRootCertificate(securityConfig, keyPair);
    }

    private VerificationStatus verifySelfSignedCA(SecurityConfig securityConfig) {
        boolean keyStatus = this.checkIfKeysExist();
        boolean certStatus = this.checkIfCertificatesExist();
        if (certStatus == keyStatus && certStatus) {
            return VerificationStatus.SUCCESS;
        }
        if (certStatus == keyStatus) {
            return VerificationStatus.INITIALIZE;
        }
        if (certStatus) {
            return VerificationStatus.MISSING_KEYS;
        }
        return VerificationStatus.MISSING_CERTIFICATE;
    }

    private boolean checkIfKeysExist() {
        if (!Files.exists(this.caKeysPath, new LinkOption[0])) {
            return false;
        }
        return Files.exists(Paths.get(this.caKeysPath.toString(), this.config.getPrivateKeyFileName()), new LinkOption[0]);
    }

    private boolean checkIfCertificatesExist() {
        if (!Files.exists(this.caRootX509Path, new LinkOption[0])) {
            return false;
        }
        return Files.exists(Paths.get(this.caRootX509Path.toString(), this.config.getCertificateFileName()), new LinkOption[0]);
    }

    @VisibleForTesting
    Consumer<SecurityConfig> processVerificationStatus(VerificationStatus status, CertificateServer.CAType type) {
        Consumer<SecurityConfig> consumer = null;
        switch (status) {
            case SUCCESS: {
                consumer = arg -> LOG.info("CertificateServer validation is successful");
                break;
            }
            case MISSING_KEYS: {
                consumer = arg -> {
                    LOG.error("We have found the Certificate for this CertificateServer, but keys used by this CertificateServer is missing. This is a non-recoverable error. Please restart the system after locating the Keys used by the CertificateServer.");
                    LOG.error("Exiting due to unrecoverable CertificateServer error.");
                    throw new IllegalStateException("Missing Keys, cannot continue.");
                };
                break;
            }
            case MISSING_CERTIFICATE: {
                consumer = arg -> {
                    LOG.error("We found the keys, but the root certificate for this CertificateServer is missing. Please restart SCM after locating the Certificates.");
                    LOG.error("Exiting due to unrecoverable CertificateServer error.");
                    throw new IllegalStateException("Missing Root Certs, cannot continue.");
                };
                break;
            }
            case INITIALIZE: {
                if (type == CertificateServer.CAType.SELF_SIGNED_CA) {
                    consumer = arg -> {
                        try {
                            this.generateSelfSignedCA((SecurityConfig)arg);
                        }
                        catch (IOException | NoSuchAlgorithmException | NoSuchProviderException e) {
                            LOG.error("Unable to initialize CertificateServer.", (Throwable)e);
                        }
                        VerificationStatus newStatus = this.verifySelfSignedCA((SecurityConfig)arg);
                        if (newStatus != VerificationStatus.SUCCESS) {
                            LOG.error("Unable to initialize CertificateServer, failed in verification.");
                        }
                    };
                    break;
                }
                if (type != CertificateServer.CAType.INTERMEDIARY_CA) break;
                consumer = arg -> {
                    LOG.error("Sub SCM CA Server is missing keys/certs. SCM is started with out init/bootstrap");
                    throw new IllegalStateException("INTERMEDIARY_CA Should not be in Initialize State during startup.");
                };
                break;
            }
        }
        return consumer;
    }

    private KeyPair generateKeys(SecurityConfig securityConfig) throws NoSuchProviderException, NoSuchAlgorithmException, IOException {
        HDDSKeyGenerator keyGenerator = new HDDSKeyGenerator(securityConfig);
        KeyPair keys = keyGenerator.generateKey();
        KeyCodec keyPEMWriter = new KeyCodec(securityConfig, this.componentName);
        keyPEMWriter.writeKey(keys);
        return keys;
    }

    private void generateRootCertificate(SecurityConfig securityConfig, KeyPair key) throws IOException, SCMSecurityException {
        Preconditions.checkNotNull((Object)this.config);
        LocalDate beginDate = LocalDate.now().atStartOfDay().toLocalDate();
        LocalDateTime temp = LocalDateTime.of(beginDate, LocalTime.MIDNIGHT);
        LocalDate endDate = temp.plus(securityConfig.getMaxCertificateDuration()).toLocalDate();
        SelfSignedCertificate.Builder builder = SelfSignedCertificate.newBuilder().setSubject(this.subject).setScmID(this.scmID).setClusterID(this.clusterID).setBeginDate(beginDate).setEndDate(endDate).makeCA().setConfiguration(securityConfig.getConfiguration()).setKey(key);
        try {
            DomainValidator validator = DomainValidator.getInstance();
            OzoneSecurityUtil.getValidInetsForCurrentHost().forEach(ip -> {
                builder.addIpAddress(ip.getHostAddress());
                if (validator.isValid(ip.getCanonicalHostName())) {
                    builder.addDnsName(ip.getCanonicalHostName());
                }
            });
        }
        catch (IOException e) {
            throw new CertificateException("Error while adding ip to CA self signed certificate", e, CertificateException.ErrorCode.CSR_ERROR);
        }
        X509CertificateHolder selfSignedCertificate = builder.build();
        CertificateCodec certCodec = new CertificateCodec(this.config, this.componentName);
        certCodec.writeCertificate(selfSignedCertificate);
    }

    @VisibleForTesting
    static enum VerificationStatus {
        SUCCESS,
        MISSING_KEYS,
        MISSING_CERTIFICATE,
        INITIALIZE;

    }
}

