/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.XceiverClientReply;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.storage.BufferPool;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitWatcher {
    private static final Logger LOG = LoggerFactory.getLogger(CommitWatcher.class);
    private BufferPool bufferPool;
    private Map<Long, List<ChunkBuffer>> commitIndex2flushedDataMap;
    private ConcurrentHashMap<Long, CompletableFuture<ContainerProtos.ContainerCommandResponseProto>> futureMap;
    private XceiverClientSpi xceiverClient;
    private long totalAckDataLength;

    public CommitWatcher(BufferPool bufferPool, XceiverClientSpi xceiverClient) {
        this.bufferPool = bufferPool;
        this.xceiverClient = xceiverClient;
        this.commitIndex2flushedDataMap = new ConcurrentSkipListMap<Long, List<ChunkBuffer>>();
        this.totalAckDataLength = 0L;
        this.futureMap = new ConcurrentHashMap();
    }

    private long releaseBuffers(List<Long> indexes) {
        Preconditions.checkArgument((!this.commitIndex2flushedDataMap.isEmpty() ? 1 : 0) != 0);
        for (long index : indexes) {
            Preconditions.checkState((boolean)this.commitIndex2flushedDataMap.containsKey(index));
            List<ChunkBuffer> buffers = this.commitIndex2flushedDataMap.remove(index);
            long length = buffers.stream().mapToLong(ChunkBuffer::position).sum();
            this.totalAckDataLength += length;
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> remove = this.futureMap.remove(this.totalAckDataLength);
            if (remove == null) {
                LOG.error("Couldn't find required future for " + this.totalAckDataLength);
                for (Long key : this.futureMap.keySet()) {
                    LOG.error("Existing acknowledged data: " + key);
                }
            }
            Preconditions.checkNotNull(remove);
            for (ChunkBuffer byteBuffer : buffers) {
                this.bufferPool.releaseBuffer(byteBuffer);
            }
        }
        return this.totalAckDataLength;
    }

    public void updateCommitInfoMap(long index, List<ChunkBuffer> buffers) {
        this.commitIndex2flushedDataMap.computeIfAbsent(index, k -> new LinkedList()).addAll(buffers);
    }

    int getCommitInfoMapSize() {
        return this.commitIndex2flushedDataMap.size();
    }

    public XceiverClientReply watchOnFirstIndex() throws IOException {
        if (!this.commitIndex2flushedDataMap.isEmpty()) {
            long index = this.commitIndex2flushedDataMap.keySet().stream().mapToLong(v -> v).min().getAsLong();
            if (LOG.isDebugEnabled()) {
                LOG.debug("waiting for first index {} to catch up", (Object)index);
            }
            return this.watchForCommit(index);
        }
        return null;
    }

    public XceiverClientReply watchOnLastIndex() throws IOException {
        if (!this.commitIndex2flushedDataMap.isEmpty()) {
            long index = this.commitIndex2flushedDataMap.keySet().stream().mapToLong(v -> v).max().getAsLong();
            if (LOG.isDebugEnabled()) {
                LOG.debug("waiting for last flush Index {} to catch up", (Object)index);
            }
            return this.watchForCommit(index);
        }
        return null;
    }

    private void adjustBuffers(long commitIndex) {
        List<Long> keyList = this.commitIndex2flushedDataMap.keySet().stream().filter(p -> p <= commitIndex).collect(Collectors.toList());
        if (!keyList.isEmpty()) {
            this.releaseBuffers(keyList);
        }
    }

    void releaseBuffersOnException() {
        this.adjustBuffers(this.xceiverClient.getReplicatedMinCommitIndex());
    }

    public XceiverClientReply watchForCommit(long commitIndex) throws IOException {
        try {
            XceiverClientReply reply = this.xceiverClient.watchForCommit(commitIndex);
            long index = reply == null ? 0L : reply.getLogIndex();
            this.adjustBuffers(index);
            return reply;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw this.getIOExceptionForWatchForCommit(commitIndex, e);
        }
        catch (ExecutionException | TimeoutException e) {
            throw this.getIOExceptionForWatchForCommit(commitIndex, e);
        }
    }

    private IOException getIOExceptionForWatchForCommit(long commitIndex, Exception e) {
        LOG.warn("watchForCommit failed for index {}", (Object)commitIndex, (Object)e);
        IOException ioException = new IOException("Unexpected Storage Container Exception: " + e.toString(), e);
        this.releaseBuffersOnException();
        return ioException;
    }

    @VisibleForTesting
    public Map<Long, List<ChunkBuffer>> getCommitIndex2flushedDataMap() {
        return this.commitIndex2flushedDataMap;
    }

    public ConcurrentMap<Long, CompletableFuture<ContainerProtos.ContainerCommandResponseProto>> getFutureMap() {
        return this.futureMap;
    }

    public long getTotalAckDataLength() {
        return this.totalAckDataLength;
    }

    public void cleanup() {
        if (this.commitIndex2flushedDataMap != null) {
            this.commitIndex2flushedDataMap.clear();
        }
        if (this.futureMap != null) {
            this.futureMap.clear();
        }
        this.commitIndex2flushedDataMap = null;
    }
}

