/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.ratis;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.hdds.utils.DBStoreHAManager;
import org.apache.hadoop.hdds.utils.TransactionInfo;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.common.ha.ratis.RatisSnapshotInfo;
import org.apache.hadoop.ozone.om.OzoneManager;
import org.apache.hadoop.ozone.om.OzoneManagerPrepareState;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.OMRatisHelper;
import org.apache.hadoop.ozone.om.ratis.OzoneManagerDoubleBuffer;
import org.apache.hadoop.ozone.om.ratis.OzoneManagerRatisServer;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.ozone.protocolPB.OzoneManagerRequestHandler;
import org.apache.hadoop.ozone.protocolPB.RequestHandler;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.exceptions.RaftException;
import org.apache.ratis.protocol.exceptions.StateMachineException;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.ExitUtils;
import org.apache.ratis.util.LifeCycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OzoneManagerStateMachine
extends BaseStateMachine {
    static final Logger LOG = LoggerFactory.getLogger(OzoneManagerStateMachine.class);
    private final SimpleStateMachineStorage storage = new SimpleStateMachineStorage();
    private final OzoneManagerRatisServer omRatisServer;
    private final OzoneManager ozoneManager;
    private RequestHandler handler;
    private RaftGroupId raftGroupId;
    private OzoneManagerDoubleBuffer ozoneManagerDoubleBuffer;
    private final RatisSnapshotInfo snapshotInfo;
    private final ExecutorService executorService;
    private final ExecutorService installSnapshotExecutor;
    private final boolean isTracingEnabled;
    private ConcurrentMap<Long, Long> applyTransactionMap = new ConcurrentSkipListMap<Long, Long>();
    private ConcurrentMap<Long, Long> ratisTransactionMap = new ConcurrentSkipListMap<Long, Long>();

    public OzoneManagerStateMachine(OzoneManagerRatisServer ratisServer, boolean isTracingEnabled) throws IOException {
        this.omRatisServer = ratisServer;
        this.isTracingEnabled = isTracingEnabled;
        this.ozoneManager = this.omRatisServer.getOzoneManager();
        this.snapshotInfo = this.ozoneManager.getSnapshotInfo();
        this.loadSnapshotInfoFromDB();
        this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
        this.handler = new OzoneManagerRequestHandler(this.ozoneManager, this.ozoneManagerDoubleBuffer);
        ThreadFactory build = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("OM StateMachine ApplyTransaction Thread - %d").build();
        this.executorService = HadoopExecutors.newSingleThreadExecutor((ThreadFactory)build);
        this.installSnapshotExecutor = HadoopExecutors.newSingleThreadExecutor();
    }

    public void initialize(RaftServer server, RaftGroupId id, RaftStorage raftStorage) throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            super.initialize(server, id, raftStorage);
            this.raftGroupId = id;
            this.storage.init(raftStorage);
        }, new Class[0]);
    }

    public void reinitialize() throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            this.loadSnapshotInfoFromDB();
            this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
            this.handler.updateDoubleBuffer(this.ozoneManagerDoubleBuffer);
        }, new Class[0]);
    }

    public SnapshotInfo getLatestSnapshot() {
        LOG.debug("Latest Snapshot Info {}", (Object)this.snapshotInfo);
        return this.snapshotInfo;
    }

    public void notifyTermIndexUpdated(long currentTerm, long index) {
        this.computeAndUpdateLastAppliedIndex(index, currentTerm, null, false);
    }

    public void notifyConfigurationChanged(long term, long index, RaftProtos.RaftConfigurationProto newRaftConfiguration) {
        List newPeers = newRaftConfiguration.getPeersList();
        LOG.info("Received Configuration change notification from Ratis. New Peer list:\n{}", (Object)newPeers);
        ArrayList<String> newPeerIds = new ArrayList<String>();
        for (RaftProtos.RaftPeerProto raftPeerProto : newPeers) {
            newPeerIds.add(RaftPeerId.valueOf((ByteString)raftPeerProto.getId()).toString());
        }
        this.ozoneManager.updatePeerList(newPeerIds);
    }

    public TransactionContext startTransaction(RaftClientRequest raftClientRequest) throws IOException {
        ByteString messageContent = raftClientRequest.getMessage().getContent();
        OzoneManagerProtocolProtos.OMRequest omRequest = OMRatisHelper.convertByteStringToOMRequest((ByteString)messageContent);
        Preconditions.checkArgument((boolean)raftClientRequest.getRaftGroupId().equals((Object)this.raftGroupId));
        try {
            this.handler.validateRequest(omRequest);
        }
        catch (IOException ioe) {
            TransactionContext ctxt = TransactionContext.newBuilder().setClientRequest(raftClientRequest).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER).build();
            ctxt.setException((Exception)ioe);
            return ctxt;
        }
        return this.handleStartTransactionRequests(raftClientRequest, omRequest);
    }

    public TransactionContext preAppendTransaction(TransactionContext trx) throws IOException {
        OzoneManagerProtocolProtos.OMRequest request = OMRatisHelper.convertByteStringToOMRequest((ByteString)trx.getStateMachineLogEntry().getLogData());
        OzoneManagerProtocolProtos.Type cmdType = request.getCmdType();
        OzoneManagerPrepareState prepareState = this.ozoneManager.getPrepareState();
        if (cmdType == OzoneManagerProtocolProtos.Type.Prepare) {
            String username = request.getUserInfo().getUserName();
            if (this.ozoneManager.getAclsEnabled() && !this.ozoneManager.isAdmin(username)) {
                String message = "Access denied for user " + username + ". " + "Superuser privilege is required to prepare ozone managers.";
                OMException cause = new OMException(message, OMException.ResultCodes.ACCESS_DENIED);
                throw new StateMachineException(message, (Throwable)cause, false);
            }
            prepareState.enablePrepareGate();
        }
        if (prepareState.requestAllowed(cmdType)) {
            return trx;
        }
        String message = "Cannot apply write request " + request.getCmdType().name() + " when OM is in prepare mode.";
        OMException cause = new OMException(message, OMException.ResultCodes.NOT_SUPPORTED_OPERATION_WHEN_PREPARED);
        throw new StateMachineException(message, (Throwable)cause, false);
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        try {
            OzoneManagerProtocolProtos.OMRequest request = OMRatisHelper.convertByteStringToOMRequest((ByteString)trx.getStateMachineLogEntry().getLogData());
            long trxLogIndex = trx.getLogEntry().getIndex();
            CompletableFuture<Message> ratisFuture = new CompletableFuture<Message>();
            this.applyTransactionMap.put(trxLogIndex, trx.getLogEntry().getTerm());
            CompletableFuture<OzoneManagerProtocolProtos.OMResponse> future = CompletableFuture.supplyAsync(() -> this.runCommand(request, trxLogIndex), this.executorService);
            future.thenApply(omResponse -> {
                if (!omResponse.getSuccess()) {
                    if (omResponse.getStatus() == OzoneManagerProtocolProtos.Status.INTERNAL_ERROR) {
                        this.terminate((OzoneManagerProtocolProtos.OMResponse)omResponse, OMException.ResultCodes.INTERNAL_ERROR);
                    } else if (omResponse.getStatus() == OzoneManagerProtocolProtos.Status.METADATA_ERROR) {
                        this.terminate((OzoneManagerProtocolProtos.OMResponse)omResponse, OMException.ResultCodes.METADATA_ERROR);
                    }
                }
                ratisFuture.complete(OMRatisHelper.convertResponseToMessage((OzoneManagerProtocolProtos.OMResponse)omResponse));
                return ratisFuture;
            });
            return ratisFuture;
        }
        catch (Exception e) {
            return OzoneManagerStateMachine.completeExceptionally(e);
        }
    }

    private void terminate(OzoneManagerProtocolProtos.OMResponse omResponse, OMException.ResultCodes resultCode) {
        OMException exception = new OMException(omResponse.getMessage(), resultCode);
        String errorMessage = "OM Ratis Server has received unrecoverable error, to avoid further DB corruption, terminating OM. Error Response received is:" + omResponse;
        ExitUtils.terminate((int)1, (String)errorMessage, (Throwable)exception, (Logger)LOG);
    }

    public CompletableFuture<Message> query(Message request) {
        try {
            OzoneManagerProtocolProtos.OMRequest omRequest = OMRatisHelper.convertByteStringToOMRequest((ByteString)request.getContent());
            return CompletableFuture.completedFuture(this.queryCommand(omRequest));
        }
        catch (IOException e) {
            return OzoneManagerStateMachine.completeExceptionally(e);
        }
    }

    public void pause() {
        this.getLifeCycle().transition(LifeCycle.State.PAUSING);
        this.getLifeCycle().transition(LifeCycle.State.PAUSED);
        this.ozoneManagerDoubleBuffer.stop();
    }

    public void unpause(long newLastAppliedSnaphsotIndex, long newLastAppliedSnapShotTermIndex) {
        this.getLifeCycle().startAndTransition(() -> {
            this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
            this.handler.updateDoubleBuffer(this.ozoneManagerDoubleBuffer);
            this.setLastAppliedTermIndex(TermIndex.valueOf((long)newLastAppliedSnapShotTermIndex, (long)newLastAppliedSnaphsotIndex));
        }, new Class[0]);
    }

    public OzoneManagerDoubleBuffer buildDoubleBufferForRatis() {
        return new OzoneManagerDoubleBuffer.Builder().setOmMetadataManager(this.ozoneManager.getMetadataManager()).setOzoneManagerRatisSnapShot(this::updateLastAppliedIndex).setIndexToTerm(this::getTermForIndex).enableRatis(true).enableTracing(this.isTracingEnabled).build();
    }

    public long takeSnapshot() throws IOException {
        LOG.info("Current Snapshot Index {}", (Object)this.getLastAppliedTermIndex());
        TermIndex lastTermIndex = this.getLastAppliedTermIndex();
        long lastAppliedIndex = lastTermIndex.getIndex();
        this.snapshotInfo.updateTermIndex(lastTermIndex.getTerm(), lastAppliedIndex);
        TransactionInfo build = new TransactionInfo.Builder().setTransactionIndex(lastAppliedIndex).setCurrentTerm(lastTermIndex.getTerm()).build();
        Table txnInfoTable = this.ozoneManager.getMetadataManager().getTransactionInfoTable();
        txnInfoTable.put((Object)"#TRANSACTIONINFO", (Object)build);
        this.ozoneManager.getMetadataManager().getStore().flushDB();
        return lastAppliedIndex;
    }

    public CompletableFuture<TermIndex> notifyInstallSnapshotFromLeader(RaftProtos.RoleInfoProto roleInfoProto, TermIndex firstTermIndexInLog) {
        String leaderNodeId = RaftPeerId.valueOf((ByteString)roleInfoProto.getFollowerInfo().getLeaderInfo().getId().getId()).toString();
        LOG.info("Received install snapshot notification from OM leader: {} with term index: {}", (Object)leaderNodeId, (Object)firstTermIndexInLog);
        CompletableFuture<TermIndex> future = CompletableFuture.supplyAsync(() -> this.ozoneManager.installSnapshotFromLeader(leaderNodeId), this.installSnapshotExecutor);
        return future;
    }

    public void notifyNotLeader(Collection<TransactionContext> pendingEntries) throws IOException {
    }

    public String toStateMachineLogEntryString(RaftProtos.StateMachineLogEntryProto proto) {
        return OMRatisHelper.smProtoToString((RaftProtos.StateMachineLogEntryProto)proto);
    }

    public void close() throws IOException {
        LOG.info("StateMachine has shutdown. Shutdown OzoneManager if not already shutdown.");
        this.ozoneManager.shutdown((Exception)new RaftException("RaftServer called shutdown on StateMachine"));
    }

    private TransactionContext handleStartTransactionRequests(RaftClientRequest raftClientRequest, OzoneManagerProtocolProtos.OMRequest omRequest) {
        return TransactionContext.newBuilder().setClientRequest(raftClientRequest).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER).setLogData(raftClientRequest.getMessage().getContent()).build();
    }

    private OzoneManagerProtocolProtos.OMResponse runCommand(OzoneManagerProtocolProtos.OMRequest request, long trxLogIndex) {
        try {
            return this.handler.handleWriteRequest(request, trxLogIndex).getOMResponse();
        }
        catch (Throwable e) {
            String errorMessage = "Request " + request + "failed with exception";
            ExitUtils.terminate((int)1, (String)errorMessage, (Throwable)e, (Logger)LOG);
            return null;
        }
    }

    public void updateLastAppliedIndex(List<Long> flushedEpochs) {
        Preconditions.checkArgument((flushedEpochs.size() > 0 ? 1 : 0) != 0);
        this.computeAndUpdateLastAppliedIndex(flushedEpochs.get(flushedEpochs.size() - 1), -1L, flushedEpochs, true);
    }

    private synchronized void computeAndUpdateLastAppliedIndex(long lastFlushedIndex, long currentTerm, List<Long> flushedEpochs, boolean checkMap) {
        if (checkMap) {
            ArrayList<Long> flushedTrans = new ArrayList<Long>(flushedEpochs);
            Long appliedTerm = null;
            long appliedIndex = -1L;
            long i = this.getLastAppliedTermIndex().getIndex() + 1L;
            while (true) {
                Long removed;
                if (flushedTrans.contains(i)) {
                    appliedIndex = i;
                    appliedTerm = removed = (Long)this.applyTransactionMap.remove(i);
                    flushedTrans.remove(i);
                } else if (this.ratisTransactionMap.containsKey(i)) {
                    appliedTerm = removed = (Long)this.ratisTransactionMap.remove(i);
                    appliedIndex = i;
                } else {
                    Iterator iterator = flushedTrans.iterator();
                    while (iterator.hasNext()) {
                        long epoch = (Long)iterator.next();
                        this.ratisTransactionMap.put(epoch, (Long)this.applyTransactionMap.remove(epoch));
                    }
                    if (!LOG.isDebugEnabled() || flushedTrans.isEmpty()) break;
                    LOG.debug("ComputeAndUpdateLastAppliedIndex due to SM added to map remaining {}", flushedTrans);
                    break;
                }
                ++i;
            }
            if (appliedTerm != null) {
                this.updateLastAppliedTermIndex(appliedTerm, appliedIndex);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("ComputeAndUpdateLastAppliedIndex due to SM is {}", (Object)this.getLastAppliedTermIndex());
                }
            }
        } else if (this.getLastAppliedTermIndex().getIndex() + 1L == lastFlushedIndex) {
            this.updateLastAppliedTermIndex(currentTerm, lastFlushedIndex);
            if (LOG.isDebugEnabled()) {
                LOG.debug("ComputeAndUpdateLastAppliedIndex due to notifyIndex {}", (Object)this.getLastAppliedTermIndex());
            }
        } else {
            this.ratisTransactionMap.put(lastFlushedIndex, currentTerm);
            if (LOG.isDebugEnabled()) {
                LOG.debug("ComputeAndUpdateLastAppliedIndex due to notifyIndex added to map. Passed Term {} index {}, where as lastApplied Index {}", new Object[]{currentTerm, lastFlushedIndex, this.getLastAppliedTermIndex()});
            }
        }
    }

    public void loadSnapshotInfoFromDB() throws IOException {
        TransactionInfo transactionInfo = TransactionInfo.readTransactionInfo((DBStoreHAManager)this.ozoneManager.getMetadataManager());
        if (transactionInfo != null) {
            this.setLastAppliedTermIndex(TermIndex.valueOf((long)transactionInfo.getTerm(), (long)transactionInfo.getTransactionIndex()));
            this.snapshotInfo.updateTermIndex(transactionInfo.getTerm(), transactionInfo.getTransactionIndex());
        }
        LOG.info("LastAppliedIndex is set from TransactionInfo from OM DB as {}", (Object)this.getLastAppliedTermIndex());
    }

    private Message queryCommand(OzoneManagerProtocolProtos.OMRequest request) {
        OzoneManagerProtocolProtos.OMResponse response = this.handler.handleReadRequest(request);
        return OMRatisHelper.convertResponseToMessage((OzoneManagerProtocolProtos.OMResponse)response);
    }

    private static <T> CompletableFuture<T> completeExceptionally(Exception e) {
        CompletableFuture future = new CompletableFuture();
        future.completeExceptionally(e);
        return future;
    }

    @VisibleForTesting
    public void setHandler(OzoneManagerRequestHandler handler) {
        this.handler = handler;
    }

    @VisibleForTesting
    public void setRaftGroupId(RaftGroupId raftGroupId) {
        this.raftGroupId = raftGroupId;
    }

    public void stop() {
        this.ozoneManagerDoubleBuffer.stop();
        HadoopExecutors.shutdown((ExecutorService)this.executorService, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        HadoopExecutors.shutdown((ExecutorService)this.installSnapshotExecutor, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
    }

    @VisibleForTesting
    void addApplyTransactionTermIndex(long term, long index) {
        this.applyTransactionMap.put(index, term);
    }

    public long getTermForIndex(long transactionIndex) {
        return (Long)this.applyTransactionMap.get(transactionIndex);
    }
}

