/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.crypto.key.KeyProvider;
import org.apache.hadoop.crypto.key.KeyProviderCryptoExtension;
import org.apache.hadoop.hdds.protocol.StorageType;
import org.apache.hadoop.ozone.OzoneAcl;
import org.apache.hadoop.ozone.om.BucketManager;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.BucketEncryptionKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmBucketArgs;
import org.apache.hadoop.ozone.om.helpers.OmBucketInfo;
import org.apache.hadoop.ozone.om.helpers.OmVolumeArgs;
import org.apache.hadoop.ozone.om.helpers.OzoneAclUtil;
import org.apache.hadoop.ozone.om.lock.OzoneManagerLock;
import org.apache.hadoop.ozone.security.acl.OzoneObj;
import org.apache.hadoop.ozone.security.acl.RequestContext;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BucketManagerImpl
implements BucketManager {
    private static final Logger LOG = LoggerFactory.getLogger(BucketManagerImpl.class);
    private final OMMetadataManager metadataManager;
    private final KeyProviderCryptoExtension kmsProvider;

    public BucketManagerImpl(OMMetadataManager metadataManager) {
        this(metadataManager, null, false);
    }

    public BucketManagerImpl(OMMetadataManager metadataManager, KeyProviderCryptoExtension kmsProvider) {
        this(metadataManager, kmsProvider, false);
    }

    public BucketManagerImpl(OMMetadataManager metadataManager, KeyProviderCryptoExtension kmsProvider, boolean isRatisEnabled) {
        this.metadataManager = metadataManager;
        this.kmsProvider = kmsProvider;
    }

    KeyProviderCryptoExtension getKMSProvider() {
        return this.kmsProvider;
    }

    @Override
    public void createBucket(OmBucketInfo bucketInfo) throws IOException {
        Preconditions.checkNotNull((Object)bucketInfo);
        String volumeName = bucketInfo.getVolumeName();
        String bucketName = bucketInfo.getBucketName();
        boolean acquiredBucketLock = false;
        this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.VOLUME_LOCK, new String[]{volumeName});
        try {
            try {
                boolean hasSourceBucket;
                acquiredBucketLock = this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
                String volumeKey = this.metadataManager.getVolumeKey(volumeName);
                String bucketKey = this.metadataManager.getBucketKey(volumeName, bucketName);
                OmVolumeArgs volumeArgs = (OmVolumeArgs)this.metadataManager.getVolumeTable().get((Object)volumeKey);
                if (volumeArgs == null) {
                    LOG.debug("volume: {} not found ", (Object)volumeName);
                    throw new OMException("Volume doesn't exist", OMException.ResultCodes.VOLUME_NOT_FOUND);
                }
                if (this.metadataManager.getBucketTable().get((Object)bucketKey) != null) {
                    LOG.debug("bucket: {} already exists ", (Object)bucketName);
                    throw new OMException("Bucket already exist", OMException.ResultCodes.BUCKET_ALREADY_EXISTS);
                }
                BucketEncryptionKeyInfo bek = bucketInfo.getEncryptionKeyInfo();
                boolean hasSourceVolume = bucketInfo.getSourceVolume() != null;
                boolean bl = hasSourceBucket = bucketInfo.getSourceBucket() != null;
                if (hasSourceBucket != hasSourceVolume) {
                    throw new OMException("Both source volume and source bucket are required for bucket links", OMException.ResultCodes.INVALID_REQUEST);
                }
                if (bek != null && hasSourceBucket) {
                    throw new OMException("Encryption cannot be set for bucket links", OMException.ResultCodes.INVALID_REQUEST);
                }
                BucketEncryptionKeyInfo.Builder bekb = this.createBucketEncryptionKeyInfoBuilder(bek);
                OmBucketInfo.Builder omBucketInfoBuilder = bucketInfo.toBuilder().setCreationTime(Time.now());
                OzoneAclUtil.inheritDefaultAcls((List)omBucketInfoBuilder.getAcls(), (List)volumeArgs.getDefaultAcls());
                if (bekb != null) {
                    omBucketInfoBuilder.setBucketEncryptionKey(bekb.build());
                }
                OmBucketInfo omBucketInfo = omBucketInfoBuilder.build();
                this.commitBucketInfoToDB(omBucketInfo);
                if (hasSourceBucket) {
                    LOG.debug("created link {}/{} to bucket: {}/{}", new Object[]{volumeName, bucketName, omBucketInfo.getSourceVolume(), omBucketInfo.getSourceBucket()});
                } else {
                    LOG.debug("created bucket: {} in volume: {}", (Object)bucketName, (Object)volumeName);
                }
            }
            catch (IOException ex) {
                if (!(ex instanceof OMException)) {
                    LOG.error("Bucket creation failed for bucket:{} in volume:{}", new Object[]{bucketName, volumeName, ex});
                }
                throw ex;
            }
        }
        catch (Throwable throwable) {
            if (acquiredBucketLock) {
                this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
            }
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.VOLUME_LOCK, new String[]{volumeName});
            throw throwable;
        }
        if (acquiredBucketLock) {
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
        }
        this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.VOLUME_LOCK, new String[]{volumeName});
    }

    @Nullable
    public BucketEncryptionKeyInfo.Builder createBucketEncryptionKeyInfoBuilder(BucketEncryptionKeyInfo bek) throws IOException {
        BucketEncryptionKeyInfo.Builder bekb = null;
        if (bek != null) {
            if (this.kmsProvider == null) {
                throw new OMException("Invalid KMS provider, check configuration hadoop.security.key.provider.path", OMException.ResultCodes.INVALID_KMS_PROVIDER);
            }
            if (bek.getKeyName() == null) {
                throw new OMException("Bucket encryption key needed.", OMException.ResultCodes.BUCKET_ENCRYPTION_KEY_NOT_FOUND);
            }
            KeyProvider.Metadata metadata = this.getKMSProvider().getMetadata(bek.getKeyName());
            if (metadata == null) {
                throw new OMException("Bucket encryption key " + bek.getKeyName() + " doesn't exist.", OMException.ResultCodes.BUCKET_ENCRYPTION_KEY_NOT_FOUND);
            }
            this.kmsProvider.warmUpEncryptedKeys(new String[]{bek.getKeyName()});
            bekb = new BucketEncryptionKeyInfo.Builder().setKeyName(bek.getKeyName()).setVersion(CryptoProtocolVersion.ENCRYPTION_ZONES).setSuite(CipherSuite.convert((String)metadata.getCipher()));
        }
        return bekb;
    }

    private void commitBucketInfoToDB(OmBucketInfo omBucketInfo) throws IOException {
        String dbBucketKey = this.metadataManager.getBucketKey(omBucketInfo.getVolumeName(), omBucketInfo.getBucketName());
        this.metadataManager.getBucketTable().put((Object)dbBucketKey, (Object)omBucketInfo);
    }

    @Override
    public OmBucketInfo getBucketInfo(String volumeName, String bucketName) throws IOException {
        OmBucketInfo omBucketInfo;
        Preconditions.checkNotNull((Object)volumeName);
        Preconditions.checkNotNull((Object)bucketName);
        this.metadataManager.getLock().acquireReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
        try {
            String bucketKey = this.metadataManager.getBucketKey(volumeName, bucketName);
            OmBucketInfo value = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)bucketKey);
            if (value == null) {
                LOG.debug("bucket: {} not found in volume: {}.", (Object)bucketName, (Object)volumeName);
                throw new OMException("Bucket not found", OMException.ResultCodes.BUCKET_NOT_FOUND);
            }
            omBucketInfo = value;
        }
        catch (IOException ex) {
            try {
                if (!(ex instanceof OMException)) {
                    LOG.error("Exception while getting bucket info for bucket: {}", (Object)bucketName, (Object)ex);
                }
                throw ex;
            }
            catch (Throwable throwable) {
                this.metadataManager.getLock().releaseReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
                throw throwable;
            }
        }
        this.metadataManager.getLock().releaseReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
        return omBucketInfo;
    }

    @Override
    public void setBucketProperty(OmBucketArgs args) throws IOException {
        Preconditions.checkNotNull((Object)args);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
        try {
            try {
                String bucketKey = this.metadataManager.getBucketKey(volumeName, bucketName);
                OmBucketInfo oldBucketInfo = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)bucketKey);
                if (oldBucketInfo == null) {
                    LOG.debug("bucket: {} not found ", (Object)bucketName);
                    throw new OMException("Bucket doesn't exist", OMException.ResultCodes.BUCKET_NOT_FOUND);
                }
                OmBucketInfo.Builder bucketInfoBuilder = OmBucketInfo.newBuilder();
                bucketInfoBuilder.setVolumeName(oldBucketInfo.getVolumeName()).setBucketName(oldBucketInfo.getBucketName());
                bucketInfoBuilder.addAllMetadata(args.getMetadata());
                StorageType storageType = args.getStorageType();
                if (storageType != null) {
                    bucketInfoBuilder.setStorageType(storageType);
                    LOG.debug("Updating bucket storage type for bucket: {} in volume: {}", (Object)bucketName, (Object)volumeName);
                } else {
                    bucketInfoBuilder.setStorageType(oldBucketInfo.getStorageType());
                }
                Boolean versioning = args.getIsVersionEnabled();
                if (versioning != null) {
                    bucketInfoBuilder.setIsVersionEnabled(versioning);
                    LOG.debug("Updating bucket versioning for bucket: {} in volume: {}", (Object)bucketName, (Object)volumeName);
                } else {
                    bucketInfoBuilder.setIsVersionEnabled(Boolean.valueOf(oldBucketInfo.getIsVersionEnabled()));
                }
                bucketInfoBuilder.setCreationTime(oldBucketInfo.getCreationTime());
                if (oldBucketInfo.getAcls() != null) {
                    bucketInfoBuilder.setAcls(oldBucketInfo.getAcls());
                }
                OmBucketInfo omBucketInfo = bucketInfoBuilder.build();
                this.commitBucketInfoToDB(omBucketInfo);
            }
            catch (IOException ex) {
                if (!(ex instanceof OMException)) {
                    LOG.error("Setting bucket property failed for bucket:{} in volume:{}", new Object[]{bucketName, volumeName, ex});
                }
                throw ex;
            }
        }
        catch (Throwable throwable) {
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
            throw throwable;
        }
        this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
    }

    @Override
    public void deleteBucket(String volumeName, String bucketName) throws IOException {
        Preconditions.checkNotNull((Object)volumeName);
        Preconditions.checkNotNull((Object)bucketName);
        this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
        try {
            try {
                String bucketKey = this.metadataManager.getBucketKey(volumeName, bucketName);
                if (this.metadataManager.getBucketTable().get((Object)bucketKey) == null) {
                    LOG.debug("bucket: {} not found ", (Object)bucketName);
                    throw new OMException("Bucket doesn't exist", OMException.ResultCodes.BUCKET_NOT_FOUND);
                }
                if (!this.metadataManager.isBucketEmpty(volumeName, bucketName)) {
                    LOG.debug("bucket: {} is not empty ", (Object)bucketName);
                    throw new OMException("Bucket is not empty", OMException.ResultCodes.BUCKET_NOT_EMPTY);
                }
                this.commitDeleteBucketInfoToOMDB(bucketKey);
            }
            catch (IOException ex) {
                if (!(ex instanceof OMException)) {
                    LOG.error("Delete bucket failed for bucket:{} in volume:{}", new Object[]{bucketName, volumeName, ex});
                }
                throw ex;
            }
        }
        catch (Throwable throwable) {
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
            throw throwable;
        }
        this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volumeName, bucketName});
    }

    private void commitDeleteBucketInfoToOMDB(String dbBucketKey) throws IOException {
        this.metadataManager.getBucketTable().delete((Object)dbBucketKey);
    }

    @Override
    public List<OmBucketInfo> listBuckets(String volumeName, String startBucket, String bucketPrefix, int maxNumOfBuckets) throws IOException {
        Preconditions.checkNotNull((Object)volumeName);
        return this.metadataManager.listBuckets(volumeName, startBucket, bucketPrefix, maxNumOfBuckets);
    }

    @Override
    public boolean addAcl(OzoneObj obj, OzoneAcl acl) throws IOException {
        Objects.requireNonNull(obj);
        Objects.requireNonNull(acl);
        if (!obj.getResourceType().equals((Object)OzoneObj.ResourceType.BUCKET)) {
            throw new IllegalArgumentException("Unexpected argument passed to BucketManager. OzoneObj type:" + obj.getResourceType());
        }
        String volume = obj.getVolumeName();
        String bucket = obj.getBucketName();
        boolean changed = false;
        this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        try {
            try {
                String dbBucketKey = this.metadataManager.getBucketKey(volume, bucket);
                OmBucketInfo bucketInfo = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)dbBucketKey);
                if (bucketInfo == null) {
                    LOG.debug("Bucket:{}/{} does not exist", (Object)volume, (Object)bucket);
                    throw new OMException("Bucket " + bucket + " is not found", OMException.ResultCodes.BUCKET_NOT_FOUND);
                }
                changed = bucketInfo.addAcl(acl);
                if (changed) {
                    this.metadataManager.getBucketTable().put((Object)dbBucketKey, (Object)bucketInfo);
                }
            }
            catch (IOException ex) {
                if (!(ex instanceof OMException)) {
                    LOG.error("Add acl operation failed for bucket:{}/{} acl:{}", new Object[]{volume, bucket, acl, ex});
                }
                throw ex;
            }
        }
        catch (Throwable throwable) {
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
            throw throwable;
        }
        this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        return changed;
    }

    @Override
    public boolean removeAcl(OzoneObj obj, OzoneAcl acl) throws IOException {
        Objects.requireNonNull(obj);
        Objects.requireNonNull(acl);
        if (!obj.getResourceType().equals((Object)OzoneObj.ResourceType.BUCKET)) {
            throw new IllegalArgumentException("Unexpected argument passed to BucketManager. OzoneObj type:" + obj.getResourceType());
        }
        String volume = obj.getVolumeName();
        String bucket = obj.getBucketName();
        boolean removed = false;
        this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        try {
            try {
                String dbBucketKey = this.metadataManager.getBucketKey(volume, bucket);
                OmBucketInfo bucketInfo = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)dbBucketKey);
                if (bucketInfo == null) {
                    LOG.debug("Bucket:{}/{} does not exist", (Object)volume, (Object)bucket);
                    throw new OMException("Bucket " + bucket + " is not found", OMException.ResultCodes.BUCKET_NOT_FOUND);
                }
                removed = bucketInfo.removeAcl(acl);
                if (removed) {
                    this.metadataManager.getBucketTable().put((Object)dbBucketKey, (Object)bucketInfo);
                }
            }
            catch (IOException ex) {
                if (!(ex instanceof OMException)) {
                    LOG.error("Remove acl operation failed for bucket:{}/{} acl:{}", new Object[]{volume, bucket, acl, ex});
                }
                throw ex;
            }
        }
        catch (Throwable throwable) {
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
            throw throwable;
        }
        this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        return removed;
    }

    @Override
    public boolean setAcl(OzoneObj obj, List<OzoneAcl> acls) throws IOException {
        Objects.requireNonNull(obj);
        Objects.requireNonNull(acls);
        if (!obj.getResourceType().equals((Object)OzoneObj.ResourceType.BUCKET)) {
            throw new IllegalArgumentException("Unexpected argument passed to BucketManager. OzoneObj type:" + obj.getResourceType());
        }
        String volume = obj.getVolumeName();
        String bucket = obj.getBucketName();
        this.metadataManager.getLock().acquireWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        try {
            try {
                String dbBucketKey = this.metadataManager.getBucketKey(volume, bucket);
                OmBucketInfo bucketInfo = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)dbBucketKey);
                if (bucketInfo == null) {
                    LOG.debug("Bucket:{}/{} does not exist", (Object)volume, (Object)bucket);
                    throw new OMException("Bucket " + bucket + " is not found", OMException.ResultCodes.BUCKET_NOT_FOUND);
                }
                bucketInfo.setAcls(acls);
                this.metadataManager.getBucketTable().put((Object)dbBucketKey, (Object)bucketInfo);
            }
            catch (IOException ex) {
                if (!(ex instanceof OMException)) {
                    LOG.error("Set acl operation failed for bucket:{}/{} acl:{}", new Object[]{volume, bucket, StringUtils.join((CharSequence)",", acls), ex});
                }
                throw ex;
            }
        }
        catch (Throwable throwable) {
            this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
            throw throwable;
        }
        this.metadataManager.getLock().releaseWriteLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        return true;
    }

    @Override
    public List<OzoneAcl> getAcl(OzoneObj obj) throws IOException {
        List list;
        Objects.requireNonNull(obj);
        if (!obj.getResourceType().equals((Object)OzoneObj.ResourceType.BUCKET)) {
            throw new IllegalArgumentException("Unexpected argument passed to BucketManager. OzoneObj type:" + obj.getResourceType());
        }
        String volume = obj.getVolumeName();
        String bucket = obj.getBucketName();
        this.metadataManager.getLock().acquireReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        try {
            String dbBucketKey = this.metadataManager.getBucketKey(volume, bucket);
            OmBucketInfo bucketInfo = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)dbBucketKey);
            if (bucketInfo == null) {
                LOG.debug("Bucket:{}/{} does not exist", (Object)volume, (Object)bucket);
                throw new OMException("Bucket " + bucket + " is not found", OMException.ResultCodes.BUCKET_NOT_FOUND);
            }
            list = bucketInfo.getAcls();
        }
        catch (IOException ex) {
            try {
                if (!(ex instanceof OMException)) {
                    LOG.error("Get acl operation failed for bucket:{}/{}.", new Object[]{volume, bucket, ex});
                }
                throw ex;
            }
            catch (Throwable throwable) {
                this.metadataManager.getLock().releaseReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
                throw throwable;
            }
        }
        this.metadataManager.getLock().releaseReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        return list;
    }

    @Override
    public boolean checkAccess(OzoneObj ozObject, RequestContext context) throws OMException {
        boolean bl;
        Objects.requireNonNull(ozObject);
        Objects.requireNonNull(context);
        String volume = ozObject.getVolumeName();
        String bucket = ozObject.getBucketName();
        this.metadataManager.getLock().acquireReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        try {
            String dbBucketKey = this.metadataManager.getBucketKey(volume, bucket);
            OmBucketInfo bucketInfo = (OmBucketInfo)this.metadataManager.getBucketTable().get((Object)dbBucketKey);
            if (bucketInfo == null) {
                LOG.debug("Bucket:{}/{} does not exist", (Object)volume, (Object)bucket);
                throw new OMException("Bucket " + bucket + " is not found", OMException.ResultCodes.BUCKET_NOT_FOUND);
            }
            boolean hasAccess = OzoneAclUtil.checkAclRights((List)bucketInfo.getAcls(), (RequestContext)context);
            if (LOG.isDebugEnabled()) {
                LOG.debug("user:{} has access rights for bucket:{} :{} ", new Object[]{context.getClientUgi(), ozObject.getBucketName(), hasAccess});
            }
            bl = hasAccess;
        }
        catch (IOException ex) {
            try {
                if (ex instanceof OMException) {
                    throw (OMException)((Object)ex);
                }
                LOG.error("CheckAccess operation failed for bucket:{}/{}.", new Object[]{volume, bucket, ex});
                throw new OMException("Check access operation failed for bucket:" + bucket, (Throwable)ex, OMException.ResultCodes.INTERNAL_ERROR);
            }
            catch (Throwable throwable) {
                this.metadataManager.getLock().releaseReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
                throw throwable;
            }
        }
        this.metadataManager.getLock().releaseReadLock(OzoneManagerLock.Resource.BUCKET_LOCK, new String[]{volume, bucket});
        return bl;
    }
}

