/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.management.ObjectName;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.utils.HddsServerUtil;
import org.apache.hadoop.hdds.utils.RocksDBStoreMBean;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.CodecRegistry;
import org.apache.hadoop.hdds.utils.db.DBCheckpoint;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.apache.hadoop.hdds.utils.db.DBStoreBuilder;
import org.apache.hadoop.hdds.utils.db.DBUpdatesWrapper;
import org.apache.hadoop.hdds.utils.db.RDBBatchOperation;
import org.apache.hadoop.hdds.utils.db.RDBCheckpointManager;
import org.apache.hadoop.hdds.utils.db.RDBMetrics;
import org.apache.hadoop.hdds.utils.db.RDBTable;
import org.apache.hadoop.hdds.utils.db.SequenceNumberNotFoundException;
import org.apache.hadoop.hdds.utils.db.StringCodec;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableConfig;
import org.apache.hadoop.hdds.utils.db.TypedTable;
import org.apache.hadoop.hdds.utils.db.cache.TableCache;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.DBOptions;
import org.rocksdb.FlushOptions;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.TransactionLogIterator;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBStore
implements DBStore {
    private static final Logger LOG = LoggerFactory.getLogger(RDBStore.class);
    private RocksDB db;
    private File dbLocation;
    private final WriteOptions writeOptions;
    private final DBOptions dbOptions;
    private final CodecRegistry codecRegistry;
    private final Map<String, ColumnFamilyHandle> handleTable;
    private ObjectName statMBeanName;
    private RDBCheckpointManager checkPointManager;
    private String checkpointsParentDir;
    private List<ColumnFamilyHandle> columnFamilyHandles;
    private RDBMetrics rdbMetrics;

    @VisibleForTesting
    public RDBStore(File dbFile, DBOptions options, Set<TableConfig> families) throws IOException {
        this(dbFile, options, new WriteOptions(), families, new CodecRegistry(), false);
    }

    public RDBStore(File dbFile, DBOptions options, WriteOptions writeOptions, Set<TableConfig> families, CodecRegistry registry, boolean readOnly) throws IOException {
        Preconditions.checkNotNull((Object)dbFile, (Object)"DB file location cannot be null");
        Preconditions.checkNotNull(families);
        Preconditions.checkArgument((!families.isEmpty() ? 1 : 0) != 0);
        this.handleTable = new HashMap<String, ColumnFamilyHandle>();
        this.codecRegistry = registry;
        ArrayList<ColumnFamilyDescriptor> columnFamilyDescriptors = new ArrayList<ColumnFamilyDescriptor>();
        this.columnFamilyHandles = new ArrayList<ColumnFamilyHandle>();
        for (TableConfig family : families) {
            columnFamilyDescriptors.add(family.getDescriptor());
        }
        this.dbOptions = options;
        this.dbLocation = dbFile;
        this.writeOptions = writeOptions;
        try {
            boolean success;
            List<TableConfig> columnFamiliesInDb = this.getColumnFamiliesInExistingDb();
            List<TableConfig> extraCf = columnFamiliesInDb.stream().filter(cf -> !families.contains(cf)).collect(Collectors.toList());
            if (!extraCf.isEmpty()) {
                LOG.info("Found the following extra column families in existing DB : {}", extraCf);
                extraCf.forEach(cf -> columnFamilyDescriptors.add(cf.getDescriptor()));
            }
            this.db = readOnly ? RocksDB.openReadOnly((DBOptions)this.dbOptions, (String)this.dbLocation.getAbsolutePath(), columnFamilyDescriptors, this.columnFamilyHandles) : RocksDB.open((DBOptions)this.dbOptions, (String)this.dbLocation.getAbsolutePath(), columnFamilyDescriptors, this.columnFamilyHandles);
            for (int x = 0; x < this.columnFamilyHandles.size(); ++x) {
                this.handleTable.put(StringUtils.bytes2String((byte[])this.columnFamilyHandles.get(x).getName()), this.columnFamilyHandles.get(x));
            }
            if (this.dbOptions.statistics() != null) {
                HashMap<String, String> jmxProperties = new HashMap<String, String>();
                jmxProperties.put("dbName", dbFile.getName());
                this.statMBeanName = HddsUtils.registerWithJmxProperties((String)"Ozone", (String)"RocksDbStore", jmxProperties, (Object)RocksDBStoreMBean.create(this.dbOptions.statistics(), dbFile.getName()));
                if (this.statMBeanName == null) {
                    LOG.warn("jmx registration failed during RocksDB init, db path :{}", (Object)dbFile.getAbsolutePath());
                }
            }
            this.checkpointsParentDir = Paths.get(this.dbLocation.getParent(), "db.checkpoints").toString();
            File checkpointsDir = new File(this.checkpointsParentDir);
            if (!checkpointsDir.exists() && !(success = checkpointsDir.mkdir())) {
                LOG.warn("Unable to create RocksDB checkpoint directory");
            }
            this.checkPointManager = new RDBCheckpointManager(this.db, this.dbLocation.getName());
            this.rdbMetrics = RDBMetrics.create();
        }
        catch (RocksDBException e) {
            String msg = "Failed init RocksDB, db path : " + dbFile.getAbsolutePath() + ", exception :" + (e.getCause() == null ? ((Object)((Object)e)).getClass().getCanonicalName() + " " + e.getMessage() : e.getCause().getClass().getCanonicalName() + " " + e.getCause().getMessage());
            throw HddsServerUtil.toIOException(msg, e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("RocksDB successfully opened.");
            LOG.debug("[Option] dbLocation= {}", (Object)this.dbLocation.getAbsolutePath());
            LOG.debug("[Option] createIfMissing = {}", (Object)options.createIfMissing());
            LOG.debug("[Option] maxOpenFiles= {}", (Object)options.maxOpenFiles());
        }
    }

    private List<TableConfig> getColumnFamiliesInExistingDb() throws RocksDBException {
        List bytes = RocksDB.listColumnFamilies((Options)new Options(), (String)this.dbLocation.getAbsolutePath());
        List<TableConfig> columnFamiliesInDb = bytes.stream().map(cfbytes -> new TableConfig(StringUtils.bytes2String((byte[])cfbytes), DBStoreBuilder.HDDS_DEFAULT_DB_PROFILE.getColumnFamilyOptions())).collect(Collectors.toList());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found column Families in DB : {}", columnFamiliesInDb);
        }
        return columnFamiliesInDb;
    }

    @Override
    public void compactDB() throws IOException {
        if (this.db != null) {
            try {
                this.db.compactRange();
            }
            catch (RocksDBException e) {
                throw HddsServerUtil.toIOException("Failed to compact db", e);
            }
        }
    }

    @Override
    public void close() throws IOException {
        for (ColumnFamilyHandle handle : this.handleTable.values()) {
            handle.close();
        }
        if (this.statMBeanName != null) {
            MBeans.unregister((ObjectName)this.statMBeanName);
            this.statMBeanName = null;
        }
        RDBMetrics.unRegister();
        if (this.db != null) {
            this.db.close();
        }
        if (this.dbOptions != null) {
            this.dbOptions.close();
        }
        if (this.writeOptions != null) {
            this.writeOptions.close();
        }
    }

    public <K, V> void move(K key, Table<K, V> source, Table<K, V> dest) throws IOException {
        try (BatchOperation batchOperation = this.initBatchOperation();){
            V value = source.get(key);
            dest.putWithBatch(batchOperation, key, value);
            source.deleteWithBatch(batchOperation, key);
            this.commitBatchOperation(batchOperation);
        }
    }

    public <K, V> void move(K key, V value, Table<K, V> source, Table<K, V> dest) throws IOException {
        this.move(key, key, value, source, dest);
    }

    public <K, V> void move(K sourceKey, K destKey, V value, Table<K, V> source, Table<K, V> dest) throws IOException {
        try (BatchOperation batchOperation = this.initBatchOperation();){
            dest.putWithBatch(batchOperation, destKey, value);
            source.deleteWithBatch(batchOperation, sourceKey);
            this.commitBatchOperation(batchOperation);
        }
    }

    @Override
    public long getEstimatedKeyCount() throws IOException {
        try {
            return this.db.getLongProperty("rocksdb.estimate-num-keys");
        }
        catch (RocksDBException e) {
            throw HddsServerUtil.toIOException("Unable to get the estimated count.", e);
        }
    }

    @Override
    public BatchOperation initBatchOperation() {
        return new RDBBatchOperation();
    }

    @Override
    public void commitBatchOperation(BatchOperation operation) throws IOException {
        ((RDBBatchOperation)operation).commit(this.db, this.writeOptions);
    }

    @VisibleForTesting
    protected ObjectName getStatMBeanName() {
        return this.statMBeanName;
    }

    @Override
    public Table<byte[], byte[]> getTable(String name) throws IOException {
        ColumnFamilyHandle handle = this.handleTable.get(name);
        if (handle == null) {
            throw new IOException("No such table in this DB. TableName : " + name);
        }
        return new RDBTable(this.db, handle, this.writeOptions, this.rdbMetrics);
    }

    public <K, V> Table<K, V> getTable(String name, Class<K> keyType, Class<V> valueType) throws IOException {
        return new TypedTable<K, V>(this.getTable(name), this.codecRegistry, keyType, valueType);
    }

    public <K, V> Table<K, V> getTable(String name, Class<K> keyType, Class<V> valueType, TableCache.CacheType cacheType) throws IOException {
        return new TypedTable<K, V>(this.getTable(name), this.codecRegistry, keyType, valueType, cacheType);
    }

    @Override
    public ArrayList<Table> listTables() {
        ArrayList<Table> returnList = new ArrayList<Table>();
        for (ColumnFamilyHandle handle : this.handleTable.values()) {
            returnList.add(new RDBTable(this.db, handle, this.writeOptions, this.rdbMetrics));
        }
        return returnList;
    }

    @Override
    public void flushDB() throws IOException {
        try (FlushOptions flushOptions = new FlushOptions();){
            flushOptions.setWaitForFlush(true);
            this.db.flush(flushOptions);
        }
        catch (RocksDBException e) {
            throw HddsServerUtil.toIOException("Unable to Flush RocksDB data", e);
        }
    }

    @Override
    public void flushLog(boolean sync) throws IOException {
        if (this.db != null) {
            try {
                this.db.flushWal(sync);
            }
            catch (RocksDBException e) {
                throw HddsServerUtil.toIOException("Failed to flush db", e);
            }
        }
    }

    @Override
    public DBCheckpoint getCheckpoint(boolean flush) throws IOException {
        if (flush) {
            this.flushDB();
        }
        return this.checkPointManager.createCheckpoint(this.checkpointsParentDir);
    }

    @Override
    public File getDbLocation() {
        return this.dbLocation;
    }

    @Override
    public Map<Integer, String> getTableNames() {
        HashMap<Integer, String> tableNames = new HashMap<Integer, String>();
        StringCodec stringCodec = new StringCodec();
        for (ColumnFamilyHandle columnFamilyHandle : this.columnFamilyHandles) {
            try {
                tableNames.put(columnFamilyHandle.getID(), stringCodec.fromPersistedFormat(columnFamilyHandle.getName()));
            }
            catch (IOException | RocksDBException e) {
                LOG.error("Unexpected exception while reading column family handle name", e);
            }
        }
        return tableNames;
    }

    @Override
    public CodecRegistry getCodecRegistry() {
        return this.codecRegistry;
    }

    @Override
    public DBUpdatesWrapper getUpdatesSince(long sequenceNumber) throws SequenceNumberNotFoundException {
        DBUpdatesWrapper dbUpdatesWrapper = new DBUpdatesWrapper();
        try {
            TransactionLogIterator transactionLogIterator = this.db.getUpdatesSince(sequenceNumber);
            boolean checkValidStartingSeqNumber = true;
            while (transactionLogIterator.isValid()) {
                TransactionLogIterator.BatchResult result = transactionLogIterator.getBatch();
                long currSequenceNumber = result.sequenceNumber();
                if (checkValidStartingSeqNumber && currSequenceNumber > 1L + sequenceNumber) {
                    throw new SequenceNumberNotFoundException("Unable to read data from RocksDB wal to get delta updates. It may have already beenflushed to SSTs.");
                }
                checkValidStartingSeqNumber = false;
                if (currSequenceNumber <= sequenceNumber) {
                    transactionLogIterator.next();
                    continue;
                }
                dbUpdatesWrapper.addWriteBatch(result.writeBatch().data(), result.sequenceNumber());
                transactionLogIterator.next();
            }
        }
        catch (RocksDBException e) {
            LOG.error("Unable to get delta updates since sequenceNumber {} ", (Object)sequenceNumber, (Object)e);
        }
        return dbUpdatesWrapper;
    }

    @VisibleForTesting
    public RocksDB getDb() {
        return this.db;
    }

    public RDBMetrics getMetrics() {
        return this.rdbMetrics;
    }
}

