/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nonnull;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.ozone.container.common.volume.AsyncChecker;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.ThrottledAsyncChecker;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.Timer;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HddsVolumeChecker {
    public static final int MAX_VOLUME_FAILURE_TOLERATED_LIMIT = -1;
    public static final Logger LOG = LoggerFactory.getLogger(HddsVolumeChecker.class);
    private AsyncChecker<Boolean, VolumeCheckResult> delegateChecker;
    private final AtomicLong numVolumeChecks = new AtomicLong(0L);
    private final AtomicLong numAllVolumeChecks = new AtomicLong(0L);
    private final AtomicLong numSkippedChecks = new AtomicLong(0L);
    private final long maxAllowedTimeForCheckMs;
    private final long minDiskCheckGapMs;
    private long lastAllVolumesCheck;
    private final Timer timer;
    private final ExecutorService checkVolumeResultHandlerExecutorService;

    public HddsVolumeChecker(ConfigurationSource conf, Timer timer) throws DiskChecker.DiskErrorException {
        this.maxAllowedTimeForCheckMs = conf.getTimeDuration("dfs.datanode.disk.check.timeout", "10m", TimeUnit.MILLISECONDS);
        if (this.maxAllowedTimeForCheckMs <= 0L) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.disk.check.timeout - " + this.maxAllowedTimeForCheckMs + " (should be > 0)");
        }
        this.timer = timer;
        int maxVolumeFailuresTolerated = conf.getInt("dfs.datanode.failed.volumes.tolerated", 0);
        this.minDiskCheckGapMs = conf.getTimeDuration("dfs.datanode.disk.check.min.gap", "15m", TimeUnit.MILLISECONDS);
        if (this.minDiskCheckGapMs < 0L) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.disk.check.min.gap - " + this.minDiskCheckGapMs + " (should be >= 0)");
        }
        long diskCheckTimeout = conf.getTimeDuration("dfs.datanode.disk.check.timeout", "10m", TimeUnit.MILLISECONDS);
        if (diskCheckTimeout < 0L) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.disk.check.timeout - " + diskCheckTimeout + " (should be >= 0)");
        }
        this.lastAllVolumesCheck = timer.monotonicNow() - this.minDiskCheckGapMs;
        if (maxVolumeFailuresTolerated < -1) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.failed.volumes.tolerated - " + maxVolumeFailuresTolerated + " " + "should be greater than or equal to -1");
        }
        this.delegateChecker = new ThrottledAsyncChecker<Boolean, VolumeCheckResult>(timer, this.minDiskCheckGapMs, diskCheckTimeout, Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("DataNode DiskChecker thread %d").setDaemon(true).build()));
        this.checkVolumeResultHandlerExecutorService = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("VolumeCheck ResultHandler thread %d").setDaemon(true).build());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<HddsVolume> checkAllVolumes(Collection<HddsVolume> volumes) throws InterruptedException {
        long gap = this.timer.monotonicNow() - this.lastAllVolumesCheck;
        if (gap < this.minDiskCheckGapMs) {
            this.numSkippedChecks.incrementAndGet();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Skipped checking all volumes, time since last check {} is less than the minimum gap between checks ({} ms).", (Object)gap, (Object)this.minDiskCheckGapMs);
            }
            return Collections.emptySet();
        }
        this.lastAllVolumesCheck = this.timer.monotonicNow();
        HashSet<HddsVolume> healthyVolumes = new HashSet<HddsVolume>();
        HashSet<HddsVolume> failedVolumes = new HashSet<HddsVolume>();
        HashSet<HddsVolume> allVolumes = new HashSet<HddsVolume>();
        AtomicLong numVolumes = new AtomicLong(volumes.size());
        CountDownLatch latch = new CountDownLatch(1);
        for (HddsVolume v : volumes) {
            Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(v, null);
            LOG.info("Scheduled health check for volume {}", (Object)v);
            if (olf.isPresent()) {
                allVolumes.add(v);
                Futures.addCallback(olf.get(), (FutureCallback)new ResultHandler(v, healthyVolumes, failedVolumes, numVolumes, (ignored1, ignored2) -> latch.countDown()), (Executor)MoreExecutors.directExecutor());
                continue;
            }
            if (numVolumes.decrementAndGet() != 0L) continue;
            latch.countDown();
        }
        if (!latch.await(this.maxAllowedTimeForCheckMs, TimeUnit.MILLISECONDS)) {
            LOG.warn("checkAllVolumes timed out after {} ms", (Object)this.maxAllowedTimeForCheckMs);
        }
        this.numAllVolumeChecks.incrementAndGet();
        HddsVolumeChecker hddsVolumeChecker = this;
        synchronized (hddsVolumeChecker) {
            return new HashSet<HddsVolume>((Collection<HddsVolume>)Sets.difference(allVolumes, healthyVolumes));
        }
    }

    public boolean checkVolume(HddsVolume volume, Callback callback) {
        if (volume == null) {
            LOG.debug("Cannot schedule check on null volume");
            return false;
        }
        Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(volume, null);
        if (olf.isPresent()) {
            this.numVolumeChecks.incrementAndGet();
            Futures.addCallback(olf.get(), (FutureCallback)new ResultHandler(volume, new HashSet<HddsVolume>(), new HashSet<HddsVolume>(), new AtomicLong(1L), callback), (Executor)this.checkVolumeResultHandlerExecutorService);
            return true;
        }
        return false;
    }

    void shutdownAndWait(int gracePeriod, TimeUnit timeUnit) {
        try {
            this.delegateChecker.shutdownAndWait(gracePeriod, timeUnit);
        }
        catch (InterruptedException e) {
            LOG.warn("{} interrupted during shutdown.", (Object)this.getClass().getSimpleName());
            Thread.currentThread().interrupt();
        }
    }

    @VisibleForTesting
    void setDelegateChecker(AsyncChecker<Boolean, VolumeCheckResult> testDelegate) {
        this.delegateChecker = testDelegate;
    }

    public long getNumVolumeChecks() {
        return this.numVolumeChecks.get();
    }

    public long getNumAllVolumeChecks() {
        return this.numAllVolumeChecks.get();
    }

    public long getNumSkippedChecks() {
        return this.numSkippedChecks.get();
    }

    private class ResultHandler
    implements FutureCallback<VolumeCheckResult> {
        private final HddsVolume volume;
        private final Set<HddsVolume> failedVolumes;
        private final Set<HddsVolume> healthyVolumes;
        private final AtomicLong volumeCounter;
        private final @Nullable Callback callback;

        ResultHandler(HddsVolume volume, Set<HddsVolume> healthyVolumes, Set<HddsVolume> failedVolumes, @Nullable AtomicLong volumeCounter, Callback callback) {
            this.volume = volume;
            this.healthyVolumes = healthyVolumes;
            this.failedVolumes = failedVolumes;
            this.volumeCounter = volumeCounter;
            this.callback = callback;
        }

        public void onSuccess(@Nullable VolumeCheckResult result) {
            if (result == null) {
                LOG.error("Unexpected empty health check result for volume {}", (Object)this.volume);
                this.markHealthy();
            } else {
                switch (result) {
                    case HEALTHY: 
                    case DEGRADED: {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Volume {} is {}.", (Object)this.volume, (Object)result);
                        }
                        this.markHealthy();
                        break;
                    }
                    case FAILED: {
                        LOG.warn("Volume {} detected as being unhealthy", (Object)this.volume);
                        this.markFailed();
                        break;
                    }
                    default: {
                        LOG.error("Unexpected health check result {} for volume {}", (Object)result, (Object)this.volume);
                        this.markHealthy();
                    }
                }
            }
            this.cleanup();
        }

        public void onFailure(@Nonnull Throwable t) {
            Throwable exception = t instanceof ExecutionException ? t.getCause() : t;
            LOG.warn("Exception running disk checks against volume {}", (Object)this.volume, (Object)exception);
            this.markFailed();
            this.cleanup();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void markHealthy() {
            HddsVolumeChecker hddsVolumeChecker = HddsVolumeChecker.this;
            synchronized (hddsVolumeChecker) {
                this.healthyVolumes.add(this.volume);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void markFailed() {
            HddsVolumeChecker hddsVolumeChecker = HddsVolumeChecker.this;
            synchronized (hddsVolumeChecker) {
                this.failedVolumes.add(this.volume);
            }
        }

        private void cleanup() {
            this.invokeCallback();
        }

        private void invokeCallback() {
            try {
                long remaining = this.volumeCounter.decrementAndGet();
                if (this.callback != null && remaining == 0L) {
                    this.callback.call(this.healthyVolumes, this.failedVolumes);
                }
            }
            catch (Exception e) {
                LOG.warn("Unexpected exception", (Throwable)e);
            }
        }
    }

    public static interface Callback {
        public void call(Set<HddsVolume> var1, Set<HddsVolume> var2);
    }
}

