/*
 * Decompiled with CFR 0.152.
 */
package org.jodconverter.local.office;

import com.sun.star.beans.PropertyValue;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XServiceInfo;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.jodconverter.core.document.DocumentFamily;
import org.jodconverter.core.office.OfficeException;
import org.jodconverter.core.util.AssertUtils;
import org.jodconverter.core.util.OSUtils;
import org.jodconverter.core.util.StringUtils;
import org.jodconverter.local.office.OfficeUrl;
import org.jodconverter.local.office.utils.Lo;
import org.jodconverter.local.office.utils.Props;
import org.jodconverter.local.process.FreeBSDProcessManager;
import org.jodconverter.local.process.MacProcessManager;
import org.jodconverter.local.process.ProcessManager;
import org.jodconverter.local.process.PureJavaProcessManager;
import org.jodconverter.local.process.UnixProcessManager;
import org.jodconverter.local.process.WindowsProcessManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LocalOfficeUtils {
    public static final int DEFAULT_PORT = 2002;
    private static final String EXECUTABLE_DEFAULT = "program/soffice.bin";
    private static final String EXECUTABLE_MAC = "program/soffice";
    private static final String EXECUTABLE_MAC_41 = "MacOS/soffice";
    private static final String EXECUTABLE_WINDOWS = "program/soffice.exe";
    private static final Logger LOGGER = LoggerFactory.getLogger(LocalOfficeUtils.class);

    public static @NonNull ProcessManager findBestProcessManager() {
        if (OSUtils.IS_OS_MAC) {
            return MacProcessManager.getDefault();
        }
        if (OSUtils.IS_OS_FREE_BSD) {
            return FreeBSDProcessManager.getDefault();
        }
        if (OSUtils.IS_OS_UNIX) {
            return UnixProcessManager.getDefault();
        }
        if (OSUtils.IS_OS_WINDOWS) {
            WindowsProcessManager windowsProcessManager = WindowsProcessManager.getDefault();
            return windowsProcessManager.isUsable() ? windowsProcessManager : PureJavaProcessManager.getDefault();
        }
        return PureJavaProcessManager.getDefault();
    }

    static @NonNull List<@NonNull OfficeUrl> buildOfficeUrls(@Nullable List<@NonNull Integer> portNumbers, @Nullable List<@NonNull String> pipeNames) {
        return LocalOfficeUtils.buildOfficeUrls(null, portNumbers, pipeNames, null);
    }

    static @NonNull List<@NonNull OfficeUrl> buildOfficeUrls(@Nullable String host, @Nullable List<@NonNull Integer> portNumbers, @Nullable List<@NonNull String> pipeNames, @Nullable List<@NonNull String> websocketUrls) {
        if (!(portNumbers != null && !portNumbers.isEmpty() || pipeNames != null && !pipeNames.isEmpty() || websocketUrls != null && !websocketUrls.isEmpty())) {
            return Collections.singletonList(new OfficeUrl(host, 2002));
        }
        ArrayList<OfficeUrl> officeUrls = new ArrayList<OfficeUrl>();
        if (portNumbers != null) {
            portNumbers.stream().map(p -> new OfficeUrl(host, (int)p)).forEach(officeUrls::add);
        }
        if (pipeNames != null) {
            pipeNames.stream().map(OfficeUrl::new).forEach(officeUrls::add);
        }
        if (websocketUrls != null) {
            websocketUrls.stream().map(OfficeUrl::createForWebsocket).forEach(officeUrls::add);
        }
        return officeUrls;
    }

    public static @NonNull File getDefaultOfficeHome() {
        return DefaultOfficeHomeHolder.INSTANCE;
    }

    public static @Nullable DocumentFamily getDocumentFamilySilently(@NonNull XComponent document) {
        AssertUtils.notNull((Object)document, (String)"document must not be null");
        XServiceInfo serviceInfo = Lo.qi(XServiceInfo.class, document);
        if (serviceInfo.supportsService("com.sun.star.text.WebDocument")) {
            return DocumentFamily.WEB;
        }
        if (serviceInfo.supportsService("com.sun.star.text.GenericTextDocument")) {
            return DocumentFamily.TEXT;
        }
        if (serviceInfo.supportsService("com.sun.star.sheet.SpreadsheetDocument")) {
            return DocumentFamily.SPREADSHEET;
        }
        if (serviceInfo.supportsService("com.sun.star.presentation.PresentationDocument")) {
            return DocumentFamily.PRESENTATION;
        }
        if (serviceInfo.supportsService("com.sun.star.drawing.DrawingDocument")) {
            return DocumentFamily.DRAWING;
        }
        return null;
    }

    public static @NonNull DocumentFamily getDocumentFamily(@NonNull XComponent document) throws OfficeException {
        DocumentFamily family = LocalOfficeUtils.getDocumentFamilySilently(document);
        if (family == null) {
            throw new OfficeException("Document of unknown family: " + document.getClass().getName());
        }
        return family;
    }

    public static @NonNull File getOfficeExecutable(@NonNull File officeHome) {
        if (OSUtils.IS_OS_MAC) {
            File executableFile = new File(officeHome, EXECUTABLE_MAC_41);
            if (!executableFile.isFile()) {
                executableFile = new File(officeHome, EXECUTABLE_MAC);
            }
            return executableFile;
        }
        if (OSUtils.IS_OS_WINDOWS) {
            return new File(officeHome, EXECUTABLE_WINDOWS);
        }
        return new File(officeHome, EXECUTABLE_DEFAULT);
    }

    public static @NonNull PropertyValue[] toUnoProperties(@NonNull Map<@NonNull String, @NonNull Object> properties) {
        ArrayList<PropertyValue> propertyValues = new ArrayList<PropertyValue>(properties.size());
        for (Map.Entry<String, Object> entry : properties.entrySet()) {
            PropertyValue[] value = entry.getValue();
            if (value instanceof Map) {
                Map subProperties = (Map)value;
                value = LocalOfficeUtils.toUnoProperties(subProperties);
            }
            propertyValues.add(Props.makeProperty(entry.getKey(), value));
        }
        return propertyValues.toArray(new PropertyValue[0]);
    }

    public static @NonNull String toUrl(@NonNull File file) {
        String path = file.toURI().getRawPath();
        String url = path.startsWith("//") ? "file:" + path : "file://" + path;
        return url.endsWith("/") ? url.substring(0, url.length() - 1) : url;
    }

    public static void validateOfficeHome(@NonNull File officeHome) {
        AssertUtils.notNull((Object)officeHome, (String)"officeHome must not be null");
        if (!officeHome.isDirectory()) {
            throw new IllegalStateException("officeHome doesn't exist or is not a directory: " + officeHome);
        }
        if (!LocalOfficeUtils.getOfficeExecutable(officeHome).isFile()) {
            throw new IllegalStateException("Invalid officeHome: it doesn't contain soffice.bin: " + officeHome);
        }
    }

    public static void validateOfficeTemplateProfileDirectory(@Nullable File templateProfileDir) {
        if (templateProfileDir == null || new File(templateProfileDir, "user").isDirectory()) {
            return;
        }
        throw new IllegalStateException("templateProfileDir doesn't appear to contain a user profile: " + templateProfileDir);
    }

    private LocalOfficeUtils() {
        throw new AssertionError((Object)"Utility class must not be instantiated");
    }

    private static class DefaultOfficeHomeHolder {
        static final File INSTANCE;

        private DefaultOfficeHomeHolder() {
        }

        private static File findOfficeHome(String executablePath, String ... homePaths) {
            return Stream.of(homePaths).filter(homePath -> Files.isRegularFile(Paths.get(homePath, executablePath), new LinkOption[0])).findFirst().map(File::new).orElse(null);
        }

        static {
            if (StringUtils.isNotBlank((String)System.getProperty("office.home"))) {
                INSTANCE = new File(System.getProperty("office.home"));
            } else if (OSUtils.IS_OS_WINDOWS) {
                String programFiles64 = System.getenv("ProgramFiles");
                String programFiles32 = System.getenv("ProgramFiles(x86)");
                INSTANCE = DefaultOfficeHomeHolder.findOfficeHome(LocalOfficeUtils.EXECUTABLE_WINDOWS, programFiles64 + File.separator + "LibreOffice", programFiles32 + File.separator + "LibreOffice", programFiles64 + File.separator + "LibreOffice 5", programFiles32 + File.separator + "LibreOffice 5", programFiles64 + File.separator + "LibreOffice 4", programFiles32 + File.separator + "LibreOffice 4", programFiles32 + File.separator + "OpenOffice 4", programFiles64 + File.separator + "LibreOffice 3", programFiles32 + File.separator + "LibreOffice 3", programFiles32 + File.separator + "OpenOffice.org 3");
            } else if (OSUtils.IS_OS_MAC) {
                File homeDir = DefaultOfficeHomeHolder.findOfficeHome(LocalOfficeUtils.EXECUTABLE_MAC_41, "/Applications/LibreOffice.app/Contents", "/Applications/OpenOffice.app/Contents", "/Applications/OpenOffice.org.app/Contents");
                if (homeDir == null) {
                    homeDir = DefaultOfficeHomeHolder.findOfficeHome(LocalOfficeUtils.EXECUTABLE_MAC, "/Applications/LibreOffice.app/Contents", "/Applications/OpenOffice.app/Contents", "/Applications/OpenOffice.org.app/Contents");
                }
                INSTANCE = homeDir;
            } else {
                INSTANCE = DefaultOfficeHomeHolder.findOfficeHome(LocalOfficeUtils.EXECUTABLE_DEFAULT, "/usr/lib64/libreoffice", "/usr/lib/libreoffice", "/usr/local/lib64/libreoffice", "/usr/local/lib/libreoffice", "/opt/libreoffice", "/usr/lib64/openoffice", "/usr/lib64/openoffice.org3", "/usr/lib64/openoffice.org", "/usr/lib/openoffice", "/usr/lib/openoffice.org3", "/usr/lib/openoffice.org", "/opt/openoffice4", "/opt/openoffice.org3");
            }
            LOGGER.debug("Default office home set to {}", (Object)INSTANCE);
        }
    }
}

