/**
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once

#include <chrono>
#include <fstream>
#include <memory>
#include <string>

#include "utils/file/FileUtils.h"
#include "utils/Environment.h"
#include "utils/Id.h"
#include "utils/TimeUtil.h"

namespace org {
namespace apache {
namespace nifi {
namespace minifi {
namespace utils {

std::string putFileToDir(const std::string& dir_path, const std::string& file_name, const std::string& content) {
  std::string file_path(file::FileUtils::concat_path(dir_path, file_name));
  std::ofstream out_file(file_path, std::ios::binary | std::ios::out);
  if (out_file.is_open()) {
    out_file << content;
  }
  return file_path;
}

std::string getFileContent(const std::string& file_name) {
  std::ifstream file_handle(file_name, std::ios::binary | std::ios::in);
  assert(file_handle.is_open());
  std::string file_content{ (std::istreambuf_iterator<char>(file_handle)), (std::istreambuf_iterator<char>()) };
  return file_content;
}

Identifier generateUUID() {
  // TODO(hunyadi): Will make the Id generator manage lifetime using a unique_ptr and return a raw ptr on access
  static std::shared_ptr<utils::IdGenerator> id_generator = utils::IdGenerator::getIdGenerator();
  return id_generator->generate();
}

class ManualClock : public timeutils::Clock {
 public:
  [[nodiscard]] std::chrono::milliseconds timeSinceEpoch() const override { return time_; }
  void advance(std::chrono::milliseconds elapsed_time) { time_ += elapsed_time; }

 private:
  std::chrono::milliseconds time_{0};
};

}  // namespace utils
}  // namespace minifi
}  // namespace nifi
}  // namespace apache
}  // namespace org
