/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.bootstrap.property;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HexFormat;
import java.util.Objects;
import java.util.Properties;
import javax.security.auth.x500.X500Principal;
import org.apache.nifi.bootstrap.property.ApplicationPropertyHandler;
import org.apache.nifi.security.cert.builder.StandardCertificateBuilder;
import org.slf4j.Logger;

public class SecurityApplicationPropertyHandler
implements ApplicationPropertyHandler {
    protected static final String ENTRY_ALIAS = "generated";
    protected static final X500Principal CERTIFICATE_ISSUER = new X500Principal("CN=localhost");
    private static final String DIGEST_ALGORITHM = "SHA-256";
    private static final String KEY_ALGORITHM = "RSA";
    private static final int KEY_SIZE = 4096;
    private static final String LOCALHOST = "localhost";
    private static final Duration CERTIFICATE_VALIDITY_PERIOD = Duration.ofDays(60L);
    private static final int RANDOM_BYTE_LENGTH = 16;
    private static final String PROPERTY_SEPARATOR = "=";
    private final Logger logger;

    public SecurityApplicationPropertyHandler(Logger logger) {
        this.logger = Objects.requireNonNull(logger, "Logger required");
    }

    @Override
    public void handleProperties(Path applicationPropertiesLocation) {
        Objects.requireNonNull(applicationPropertiesLocation);
        Properties applicationProperties = this.loadProperties(applicationPropertiesLocation);
        if (this.isCertificateGenerationRequired(applicationProperties)) {
            this.processApplicationProperties(applicationProperties);
            this.writePasswordProperties(applicationProperties, applicationPropertiesLocation);
        }
    }

    private void processApplicationProperties(Properties applicationProperties) {
        KeyPair keyPair = this.generateKeyPair();
        Collection<String> subjectAlternativeNames = this.getSubjectAlternativeNames(applicationProperties);
        X509Certificate certificate = new StandardCertificateBuilder(keyPair, CERTIFICATE_ISSUER, CERTIFICATE_VALIDITY_PERIOD).setDnsSubjectAlternativeNames(subjectAlternativeNames).build();
        String certificateDigestEncoded = SecurityApplicationPropertyHandler.getDigest(certificate);
        this.logger.info("Generated Self-Signed Certificate Expiration: {}", (Object)LocalDate.now().plusDays(CERTIFICATE_VALIDITY_PERIOD.toDays()));
        this.logger.info("Generated Self-Signed Certificate SHA-256: {}", (Object)certificateDigestEncoded);
        PrivateKey privateKey = keyPair.getPrivate();
        this.writeKeyStore(applicationProperties, certificate, privateKey);
        this.writeTrustStore(applicationProperties, certificate);
    }

    private void writeTrustStore(Properties applicationProperties, X509Certificate certificate) {
        String storeType = applicationProperties.getProperty(SecurityProperty.TRUSTSTORE_TYPE.getName());
        KeyStore trustStore = this.newKeyStore(storeType);
        try {
            trustStore.load(null, null);
            trustStore.setCertificateEntry(ENTRY_ALIAS, certificate);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException("Trust Store creation failed", e);
        }
        Path outputLocation = Paths.get(applicationProperties.getProperty(SecurityProperty.TRUSTSTORE.getName()), new String[0]);
        try (OutputStream outputStream = Files.newOutputStream(outputLocation, new OpenOption[0]);){
            String truststorePasswd = this.generatePassword();
            trustStore.store(outputStream, truststorePasswd.toCharArray());
            applicationProperties.setProperty(SecurityProperty.TRUSTSTORE_PASSWD.getName(), truststorePasswd);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException("Trust Store storage failed", e);
        }
    }

    private void writeKeyStore(Properties applicationProperties, X509Certificate certificate, PrivateKey privateKey) {
        String keystorePasswd = this.generatePassword();
        char[] password = keystorePasswd.toCharArray();
        String storeType = applicationProperties.getProperty(SecurityProperty.KEYSTORE_TYPE.getName());
        KeyStore keyStore = this.newKeyStore(storeType);
        try {
            keyStore.load(null, null);
            Certificate[] certificates = new X509Certificate[]{certificate};
            keyStore.setKeyEntry(ENTRY_ALIAS, privateKey, password, certificates);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException("Key Store creation failed", e);
        }
        Path outputLocation = Paths.get(applicationProperties.getProperty(SecurityProperty.KEYSTORE.getName()), new String[0]);
        try (OutputStream outputStream = Files.newOutputStream(outputLocation, new OpenOption[0]);){
            keyStore.store(outputStream, password);
            applicationProperties.setProperty(SecurityProperty.KEYSTORE_PASSWD.getName(), keystorePasswd);
            applicationProperties.setProperty(SecurityProperty.KEY_PASSWD.getName(), keystorePasswd);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException("Key Store storage failed", e);
        }
    }

    private void writePasswordProperties(Properties applicationProperties, Path applicationPropertiesLocation) {
        ByteArrayOutputStream propertiesOutputStream = new ByteArrayOutputStream();
        try (BufferedReader reader = Files.newBufferedReader(applicationPropertiesLocation);
             PrintWriter writer = new PrintWriter(propertiesOutputStream);){
            String line = reader.readLine();
            while (line != null) {
                if (line.startsWith(SecurityProperty.KEYSTORE_PASSWD.getName())) {
                    this.writeProperty(writer, SecurityProperty.KEYSTORE_PASSWD, applicationProperties);
                } else if (line.startsWith(SecurityProperty.KEY_PASSWD.getName())) {
                    this.writeProperty(writer, SecurityProperty.KEY_PASSWD, applicationProperties);
                } else if (line.startsWith(SecurityProperty.TRUSTSTORE_PASSWD.getName())) {
                    this.writeProperty(writer, SecurityProperty.TRUSTSTORE_PASSWD, applicationProperties);
                } else {
                    writer.println(line);
                }
                line = reader.readLine();
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Read Application Properties failed", e);
        }
        byte[] updatedProperties = propertiesOutputStream.toByteArray();
        try (OutputStream outputStream = Files.newOutputStream(applicationPropertiesLocation, new OpenOption[0]);){
            outputStream.write(updatedProperties);
        }
        catch (IOException e) {
            throw new IllegalStateException("Write Application Properties failed", e);
        }
    }

    private void writeProperty(PrintWriter writer, SecurityProperty securityProperty, Properties applicationProperties) {
        writer.print(securityProperty.getName());
        writer.print(PROPERTY_SEPARATOR);
        String propertyValue = applicationProperties.getProperty(securityProperty.getName());
        writer.println(propertyValue);
    }

    private KeyStore newKeyStore(String storeType) {
        try {
            return KeyStore.getInstance(storeType);
        }
        catch (KeyStoreException e) {
            throw new IllegalStateException("Key Store Type [%s] instantiation failed".formatted(storeType), e);
        }
    }

    private boolean isCertificateGenerationRequired(Properties applicationProperties) {
        boolean required;
        String keystoreLocation = applicationProperties.getProperty(SecurityProperty.KEYSTORE.getName());
        String truststoreLocation = applicationProperties.getProperty(SecurityProperty.TRUSTSTORE.getName());
        if (this.isBlank(applicationProperties.getProperty(SecurityProperty.HTTPS_PORT.getName()))) {
            required = false;
        } else if (this.isBlank(keystoreLocation)) {
            required = false;
        } else if (this.isBlank(truststoreLocation)) {
            required = false;
        } else if (this.isBlank(applicationProperties.getProperty(SecurityProperty.KEYSTORE_PASSWD.getName())) && this.isBlank(applicationProperties.getProperty(SecurityProperty.TRUSTSTORE_PASSWD.getName()))) {
            Path keystorePath = Paths.get(keystoreLocation, new String[0]);
            Path truststorePath = Paths.get(truststoreLocation, new String[0]);
            required = Files.notExists(keystorePath, new LinkOption[0]) && Files.notExists(truststorePath, new LinkOption[0]);
        } else {
            required = false;
        }
        return required;
    }

    private Collection<String> getSubjectAlternativeNames(Properties applicationProperties) {
        try {
            InetAddress localHost = InetAddress.getLocalHost();
            String localHostName = localHost.getHostName();
            ArrayList<String> subjectAlternativeNames = new ArrayList<String>();
            subjectAlternativeNames.add(LOCALHOST);
            subjectAlternativeNames.add(localHostName);
            String proxyHost = applicationProperties.getProperty(SecurityProperty.WEB_PROXY_HOST.getName());
            if (!this.isBlank(proxyHost)) {
                subjectAlternativeNames.add(proxyHost);
            }
            return subjectAlternativeNames;
        }
        catch (UnknownHostException e) {
            return Collections.emptyList();
        }
    }

    private KeyPair generateKeyPair() {
        KeyPairGenerator keyPairGenerator;
        try {
            keyPairGenerator = KeyPairGenerator.getInstance(KEY_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Key Pair Algorithm not supported [%s]".formatted(KEY_ALGORITHM), e);
        }
        keyPairGenerator.initialize(4096);
        return keyPairGenerator.generateKeyPair();
    }

    protected String generatePassword() {
        SecureRandom secureRandom = new SecureRandom();
        byte[] bytes = new byte[16];
        secureRandom.nextBytes(bytes);
        return HexFormat.of().formatHex(bytes);
    }

    private static String getDigest(X509Certificate certificate) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance(DIGEST_ALGORITHM);
            byte[] certificateEncoded = certificate.getEncoded();
            byte[] digest = messageDigest.digest(certificateEncoded);
            return HexFormat.of().formatHex(digest).toUpperCase();
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Message Digest Algorithm not found", e);
        }
        catch (CertificateEncodingException e) {
            throw new IllegalArgumentException("Certificate encoding processing failed", e);
        }
    }

    private Properties loadProperties(Path applicationPropertiesLocation) {
        Properties properties;
        block8: {
            InputStream inputStream = Files.newInputStream(applicationPropertiesLocation, new OpenOption[0]);
            try {
                Properties properties2 = new Properties();
                properties2.load(inputStream);
                properties = properties2;
                if (inputStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new IllegalStateException("Reading Application Properties failed [%s]".formatted(applicationPropertiesLocation), e);
                }
            }
            inputStream.close();
        }
        return properties;
    }

    private boolean isBlank(String propertyValue) {
        return propertyValue == null || propertyValue.isBlank();
    }

    protected static enum SecurityProperty {
        HTTPS_PORT("nifi.web.https.port"),
        WEB_PROXY_HOST("nifi.web.proxy.host"),
        KEYSTORE("nifi.security.keystore"),
        KEYSTORE_TYPE("nifi.security.keystoreType"),
        KEYSTORE_PASSWD("nifi.security.keystorePasswd"),
        KEY_PASSWD("nifi.security.keyPasswd"),
        TRUSTSTORE("nifi.security.truststore"),
        TRUSTSTORE_TYPE("nifi.security.truststoreType"),
        TRUSTSTORE_PASSWD("nifi.security.truststorePasswd");

        private final String name;

        private SecurityProperty(String name) {
            this.name = name;
        }

        public String getName() {
            return this.name;
        }
    }
}

