/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.bootstrap;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.management.ManagementFactory;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.nifi.bootstrap.NiFiListener;
import org.apache.nifi.bootstrap.ShutdownHook;
import org.apache.nifi.bootstrap.process.RuntimeValidatorExecutor;
import org.apache.nifi.bootstrap.property.SecurityApplicationPropertyHandler;
import org.apache.nifi.bootstrap.util.DumpFileValidator;
import org.apache.nifi.util.file.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RunNiFi {
    public static final String DEFAULT_CONFIG_FILE = "./conf/bootstrap.conf";
    public static final String DEFAULT_JAVA_CMD = "java";
    public static final String DEFAULT_PID_DIR = "bin";
    public static final String DEFAULT_LOG_DIR = "./logs";
    public static final String DEFAULT_STATUS_HISTORY_DAYS = "1";
    public static final String GRACEFUL_SHUTDOWN_PROP = "graceful.shutdown.seconds";
    public static final String DEFAULT_GRACEFUL_SHUTDOWN_VALUE = "20";
    public static final String NIFI_PID_DIR_PROP = "org.apache.nifi.bootstrap.config.pid.dir";
    public static final String NIFI_PID_FILE_NAME = "nifi.pid";
    public static final String NIFI_STATUS_FILE_NAME = "nifi.status";
    public static final String NIFI_LOCK_FILE_NAME = "nifi.lock";
    public static final String NIFI_BOOTSTRAP_LISTEN_PORT_PROP = "nifi.bootstrap.listen.port";
    public static final String PID_KEY = "pid";
    public static final int STARTUP_WAIT_SECONDS = 60;
    public static final long GRACEFUL_SHUTDOWN_RETRY_MILLIS = 2000L;
    public static final String SHUTDOWN_CMD = "SHUTDOWN";
    public static final String DECOMMISSION_CMD = "DECOMMISSION";
    public static final String PING_CMD = "PING";
    public static final String DUMP_CMD = "DUMP";
    public static final String DIAGNOSTICS_CMD = "DIAGNOSTICS";
    public static final String CLUSTER_STATUS_CMD = "CLUSTER_STATUS";
    public static final String IS_LOADED_CMD = "IS_LOADED";
    public static final String STATUS_HISTORY_CMD = "STATUS_HISTORY";
    private static final int UNINITIALIZED_CC_PORT = -1;
    private static final int INVALID_CMD_ARGUMENT = -1;
    private volatile boolean autoRestartNiFi = true;
    private volatile int ccPort = -1;
    private volatile long nifiPid = -1L;
    private volatile String secretKey;
    private volatile ShutdownHook shutdownHook;
    private volatile boolean nifiStarted;
    private final Lock startedLock = new ReentrantLock();
    private final Lock lock = new ReentrantLock();
    private final Condition startupCondition = this.lock.newCondition();
    private final File bootstrapConfigFile;
    private final Logger cmdLogger = LoggerFactory.getLogger((String)"org.apache.nifi.bootstrap.Command");
    private final Logger defaultLogger = LoggerFactory.getLogger(RunNiFi.class);
    private final ExecutorService loggingExecutor;
    private final RuntimeValidatorExecutor runtimeValidatorExecutor;
    private volatile Set<Future<?>> loggingFutures = new HashSet(2);

    public RunNiFi(File bootstrapConfigFile) throws IOException {
        this.bootstrapConfigFile = bootstrapConfigFile;
        this.loggingExecutor = Executors.newFixedThreadPool(2, runnable -> {
            Thread t = Executors.defaultThreadFactory().newThread(runnable);
            t.setDaemon(true);
            t.setName("NiFi logging handler");
            return t;
        });
        this.runtimeValidatorExecutor = new RuntimeValidatorExecutor();
    }

    private static void printUsage() {
        System.out.println("Usage:");
        System.out.println();
        System.out.println("java org.apache.nifi.bootstrap.RunNiFi <command> [options]");
        System.out.println();
        System.out.println("Valid commands include:");
        System.out.println();
        System.out.println("Start : Start a new instance of Apache NiFi");
        System.out.println("Stop : Stop a running instance of Apache NiFi");
        System.out.println("Restart : Stop Apache NiFi, if it is running, and then start a new instance");
        System.out.println("Decommission : Disconnects Apache NiFi from its cluster, offloads its data to other nodes in the cluster, removes itself from the cluster, and shuts down the instance");
        System.out.println("Status : Determine if there is a running instance of Apache NiFi");
        System.out.println("Dump : Write a Thread Dump to the file specified by [options], or to the log if no file is given");
        System.out.println("Diagnostics : Write diagnostic information to the file specified by [options], or to the log if no file is given. The --verbose flag may be provided as an option before the filename, which may result in additional diagnostic information being written.");
        System.out.println("Status-history : Save the status history to the file specified by [options]. The expected command parameters are: status-history <number of days> <dumpFile>. The <number of days> parameter is optional and defaults to 1 day.");
        System.out.println("Run : Start a new instance of Apache NiFi and monitor the Process, restarting if the instance dies");
        System.out.println();
    }

    public static void main(String[] args) throws IOException {
        if (args.length < 1 || args.length > 3) {
            RunNiFi.printUsage();
            return;
        }
        File dumpFile = null;
        boolean verbose = false;
        String statusHistoryDays = null;
        String cmd = args[0];
        if (cmd.equalsIgnoreCase("dump")) {
            if (args.length > 1) {
                dumpFile = new File(args[1]);
            }
        } else if (cmd.equalsIgnoreCase("diagnostics")) {
            if (args.length > 2) {
                verbose = args[1].equalsIgnoreCase("--verbose");
                dumpFile = new File(args[2]);
            } else if (args.length > 1) {
                if (args[1].equalsIgnoreCase("--verbose")) {
                    verbose = true;
                } else {
                    dumpFile = new File(args[1]);
                }
            }
        } else if (cmd.equalsIgnoreCase("cluster-status")) {
            if (args.length > 1) {
                dumpFile = new File(args[1]);
            }
        } else if (cmd.equalsIgnoreCase("status-history")) {
            if (args.length < 2) {
                System.err.printf("Wrong number of arguments: %d instead of 1 or 2, the command parameters are: status-history <number of days> <dumpFile>%n", 0);
                System.exit(-1);
            }
            if (args.length == 3) {
                statusHistoryDays = args[1];
                try {
                    int numberOfDays = Integer.parseInt(statusHistoryDays);
                    if (numberOfDays < 1) {
                        System.err.println("The <number of days> parameter must be positive and greater than zero. The command parameters are: status-history <number of days> <dumpFile>");
                        System.exit(-1);
                    }
                }
                catch (NumberFormatException e) {
                    System.err.println("The <number of days> parameter value is not a number. The command parameters are: status-history <number of days> <dumpFile>");
                    System.exit(-1);
                }
                try {
                    Paths.get(args[2], new String[0]);
                }
                catch (InvalidPathException e) {
                    System.err.println("Invalid filename. The command parameters are: status-history <number of days> <dumpFile>");
                    System.exit(-1);
                }
                dumpFile = new File(args[2]);
            } else {
                boolean isValid = DumpFileValidator.validate(args[1]);
                if (isValid) {
                    statusHistoryDays = DEFAULT_STATUS_HISTORY_DAYS;
                    dumpFile = new File(args[1]);
                } else {
                    System.exit(-1);
                }
            }
        }
        switch (cmd.toLowerCase()) {
            case "start": 
            case "run": 
            case "stop": 
            case "decommission": 
            case "status": 
            case "is_loaded": 
            case "dump": 
            case "diagnostics": 
            case "status-history": 
            case "restart": 
            case "env": 
            case "cluster-status": {
                break;
            }
            default: {
                RunNiFi.printUsage();
                return;
            }
        }
        File configFile = RunNiFi.getDefaultBootstrapConfFile();
        RunNiFi runNiFi = new RunNiFi(configFile);
        Integer exitStatus = null;
        switch (cmd.toLowerCase()) {
            case "start": {
                runNiFi.start(true);
                break;
            }
            case "run": {
                runNiFi.start(true);
                break;
            }
            case "stop": {
                runNiFi.stop();
                break;
            }
            case "decommission": {
                boolean shutdown = args.length < 2 || !"--shutdown=false".equals(args[1]);
                exitStatus = runNiFi.decommission(shutdown);
                break;
            }
            case "status": {
                exitStatus = runNiFi.status();
                break;
            }
            case "is_loaded": {
                try {
                    System.out.println(runNiFi.isNiFiFullyLoaded());
                }
                catch (NiFiNotRunningException e) {
                    System.out.println("not_running");
                }
                break;
            }
            case "restart": {
                runNiFi.stop();
                runNiFi.start(true);
                break;
            }
            case "dump": {
                runNiFi.dump(dumpFile);
                break;
            }
            case "diagnostics": {
                runNiFi.diagnostics(dumpFile, verbose);
                break;
            }
            case "cluster-status": {
                runNiFi.clusterStatus(dumpFile);
                break;
            }
            case "status-history": {
                runNiFi.statusHistory(dumpFile, statusHistoryDays);
                break;
            }
            case "env": {
                runNiFi.env();
            }
        }
        if (exitStatus != null) {
            System.exit(exitStatus);
        }
    }

    private static File getDefaultBootstrapConfFile() {
        String nifiHome;
        String configFilename = System.getProperty("org.apache.nifi.bootstrap.config.file");
        if (configFilename == null && (nifiHome = System.getenv("NIFI_HOME")) != null) {
            File nifiHomeFile = new File(nifiHome.trim());
            File configFile = new File(nifiHomeFile, DEFAULT_CONFIG_FILE);
            configFilename = configFile.getAbsolutePath();
        }
        if (configFilename == null) {
            configFilename = DEFAULT_CONFIG_FILE;
        }
        return new File(configFilename);
    }

    protected File getBootstrapFile(Logger logger, String directory, String defaultDirectory, String fileName) throws IOException {
        File confDir = this.bootstrapConfigFile.getParentFile();
        File nifiHome = confDir.getParentFile();
        String confFileDir = System.getProperty(directory);
        File fileDir = confFileDir != null ? new File(confFileDir.trim()) : new File(nifiHome, defaultDirectory);
        FileUtils.ensureDirectoryExistAndCanAccess((File)fileDir);
        File statusFile = new File(fileDir, fileName);
        logger.debug("Status File: {}", (Object)statusFile);
        return statusFile;
    }

    protected File getPidFile(Logger logger) throws IOException {
        return this.getBootstrapFile(logger, NIFI_PID_DIR_PROP, DEFAULT_PID_DIR, NIFI_PID_FILE_NAME);
    }

    protected File getStatusFile(Logger logger) throws IOException {
        return this.getBootstrapFile(logger, NIFI_PID_DIR_PROP, DEFAULT_PID_DIR, NIFI_STATUS_FILE_NAME);
    }

    protected File getLockFile(Logger logger) throws IOException {
        return this.getBootstrapFile(logger, NIFI_PID_DIR_PROP, DEFAULT_PID_DIR, NIFI_LOCK_FILE_NAME);
    }

    protected File getStatusFile() throws IOException {
        return this.getStatusFile(this.defaultLogger);
    }

    private Properties loadProperties(Logger logger) throws IOException {
        Properties props = new Properties();
        File statusFile = this.getStatusFile(logger);
        if (statusFile == null || !statusFile.exists()) {
            logger.debug("No status file to load properties from");
            return props;
        }
        try (FileInputStream fis = new FileInputStream(this.getStatusFile(logger));){
            props.load(fis);
        }
        HashMap<Object, Object> modified = new HashMap<Object, Object>(props);
        modified.remove("secret.key");
        logger.debug("Properties: {}", modified);
        return props;
    }

    private synchronized void savePidProperties(Properties pidProperties, Logger logger) throws IOException {
        File statusFile;
        String pid = pidProperties.getProperty(PID_KEY);
        if (pid != null && !pid.isBlank()) {
            this.writePidFile(pid, logger);
        }
        if ((statusFile = this.getStatusFile(logger)).exists() && !statusFile.delete()) {
            logger.warn("Failed to delete {}", (Object)statusFile);
        }
        if (!statusFile.createNewFile()) {
            throw new IOException("Failed to create file " + String.valueOf(statusFile));
        }
        try {
            HashSet<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
            perms.add(PosixFilePermission.OWNER_READ);
            perms.add(PosixFilePermission.OWNER_WRITE);
            Files.setPosixFilePermissions(statusFile.toPath(), perms);
        }
        catch (Exception e) {
            logger.warn("Failed to set permissions so that only the owner can read status file {}; this may allows others to have access to the key needed to communicate with NiFi. Permissions should be changed so that only the owner can read this file", (Object)statusFile);
        }
        try (FileOutputStream fos = new FileOutputStream(statusFile);){
            pidProperties.store(fos, null);
            fos.getFD().sync();
        }
        logger.debug("Saved Properties {} to {}", (Object)pidProperties, (Object)statusFile);
    }

    private synchronized void writePidFile(String pid, Logger logger) throws IOException {
        File pidFile = this.getPidFile(logger);
        if (pidFile.exists() && !pidFile.delete()) {
            logger.warn("Failed to delete {}", (Object)pidFile);
        }
        if (!pidFile.createNewFile()) {
            throw new IOException("Failed to create file " + String.valueOf(pidFile));
        }
        try {
            HashSet<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
            perms.add(PosixFilePermission.OWNER_WRITE);
            perms.add(PosixFilePermission.OWNER_READ);
            perms.add(PosixFilePermission.GROUP_READ);
            perms.add(PosixFilePermission.OTHERS_READ);
            Files.setPosixFilePermissions(pidFile.toPath(), perms);
        }
        catch (Exception e) {
            logger.warn("Failed to set permissions so that only the owner can read pid file {}; this may allows others to have access to the key needed to communicate with NiFi. Permissions should be changed so that only the owner can read this file", (Object)pidFile);
        }
        try (FileOutputStream fos = new FileOutputStream(pidFile);){
            fos.write(pid.getBytes(StandardCharsets.UTF_8));
            fos.getFD().sync();
        }
        logger.debug("Saved PID [{}] to [{}]", (Object)pid, (Object)pidFile);
    }

    private boolean isPingSuccessful(int port, String secretKey, Logger logger) {
        boolean bl;
        logger.debug("Pinging {}", (Object)port);
        Socket socket = new Socket("localhost", port);
        try {
            OutputStream out = socket.getOutputStream();
            out.write(("PING " + secretKey + "\n").getBytes(StandardCharsets.UTF_8));
            out.flush();
            logger.debug("Sent PING command");
            socket.setSoTimeout(5000);
            InputStream in = socket.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(in));
            String response = reader.readLine();
            logger.debug("PING response: {}", (Object)response);
            out.close();
            reader.close();
            bl = PING_CMD.equals(response);
        }
        catch (Throwable throwable) {
            try {
                try {
                    socket.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ioe) {
                return false;
            }
        }
        socket.close();
        return bl;
    }

    private Integer getCurrentPort(Logger logger) throws IOException {
        Properties props = this.loadProperties(logger);
        String portVal = props.getProperty("port");
        if (portVal == null) {
            logger.debug("No Port found in status file");
            return null;
        }
        logger.debug("Port defined in status file: {}", (Object)portVal);
        int port = Integer.parseInt(portVal);
        boolean success = this.isPingSuccessful(port, props.getProperty("secret.key"), logger);
        if (success) {
            logger.debug("Successful PING on port {}", (Object)port);
            return port;
        }
        String pid = props.getProperty(PID_KEY);
        logger.debug("PID in status file is {}", (Object)pid);
        if (pid != null) {
            boolean procRunning = this.isProcessRunning(pid, logger);
            if (procRunning) {
                return port;
            }
            return null;
        }
        return null;
    }

    private boolean isProcessRunning(String pid, Logger logger) {
        try {
            ProcessBuilder builder = new ProcessBuilder(new String[0]);
            builder.command("ps", "-p", pid);
            Process proc = builder.start();
            boolean running = false;
            try (InputStream in = proc.getInputStream();
                 InputStreamReader streamReader = new InputStreamReader(in);
                 BufferedReader reader = new BufferedReader(streamReader);){
                String line;
                while ((line = reader.readLine()) != null) {
                    if (!line.trim().startsWith(pid)) continue;
                    running = true;
                }
            }
            if (running) {
                logger.debug("Process with PID {} is running", (Object)pid);
            } else {
                logger.debug("Process with PID {} is not running", (Object)pid);
            }
            return running;
        }
        catch (IOException ioe) {
            System.err.println("Failed to determine if Process " + pid + " is running; assuming that it is not");
            return false;
        }
    }

    private Status getStatus(Logger logger) {
        Properties props;
        try {
            props = this.loadProperties(logger);
        }
        catch (IOException ioe) {
            return new Status(null, null, false, false);
        }
        if (props == null) {
            return new Status(null, null, false, false);
        }
        String portValue = props.getProperty("port");
        String pid = props.getProperty(PID_KEY);
        String secretKey = props.getProperty("secret.key");
        if (portValue == null && pid == null) {
            return new Status(null, null, false, false);
        }
        Integer port = null;
        boolean pingSuccess = false;
        if (portValue != null) {
            try {
                port = Integer.parseInt(portValue);
                pingSuccess = this.isPingSuccessful(port, secretKey, logger);
            }
            catch (NumberFormatException nfe) {
                return new Status(null, null, false, false);
            }
        }
        if (pingSuccess) {
            return new Status(port, pid, true, true);
        }
        boolean alive = pid != null && this.isProcessRunning(pid, logger);
        return new Status(port, pid, pingSuccess, alive);
    }

    public int status() throws IOException {
        Logger logger = this.cmdLogger;
        Status status = this.getStatus(logger);
        if (status.isRespondingToPing()) {
            logger.info("Apache NiFi PID [{}] running with Bootstrap Port [{}]", (Object)status.getPid(), (Object)status.getPort());
            return 0;
        }
        if (status.isProcessRunning()) {
            logger.info("Apache NiFi PID [{}] running but not responding with Bootstrap Port [{}]", (Object)status.getPid(), (Object)status.getPort());
            return 4;
        }
        if (status.getPort() == null) {
            logger.info("Apache NiFi is not running");
            return 3;
        }
        if (status.getPid() == null) {
            logger.info("Apache NiFi is not responding to Ping requests. The process may have died or may be hung");
        } else {
            logger.info("Apache NiFi is not running");
        }
        return 3;
    }

    public void env() {
        Object virtualMachine;
        Method detachMethod;
        Method attachMethod;
        Class<?> virtualMachineClass;
        Logger logger = this.cmdLogger;
        Status status = this.getStatus(logger);
        if (status.getPid() == null) {
            logger.info("Apache NiFi is not running");
            return;
        }
        try {
            virtualMachineClass = Class.forName("com.sun.tools.attach.VirtualMachine");
        }
        catch (ClassNotFoundException cnfe) {
            logger.error("Seems tools.jar (Linux / Windows JDK) or classes.jar (Mac OS) is not available in classpath");
            return;
        }
        try {
            attachMethod = virtualMachineClass.getMethod("attach", String.class);
            detachMethod = virtualMachineClass.getDeclaredMethod("detach", new Class[0]);
        }
        catch (Exception e) {
            logger.error("Methods required for getting environment not available", (Throwable)e);
            return;
        }
        try {
            virtualMachine = attachMethod.invoke(null, status.getPid());
        }
        catch (Throwable t) {
            logger.error("Problem attaching to NiFi", t);
            return;
        }
        try {
            Method getSystemPropertiesMethod = virtualMachine.getClass().getMethod("getSystemProperties", new Class[0]);
            Properties sysProps = (Properties)getSystemPropertiesMethod.invoke(virtualMachine, new Object[0]);
            for (Map.Entry<Object, Object> syspropEntry : sysProps.entrySet()) {
                logger.info("{} = {}", syspropEntry.getKey(), syspropEntry.getValue());
            }
        }
        catch (Throwable t) {
            throw new RuntimeException(t);
        }
        finally {
            try {
                detachMethod.invoke(virtualMachine, new Object[0]);
            }
            catch (Exception e) {
                logger.warn("Caught exception detaching from process", (Throwable)e);
            }
        }
    }

    public void diagnostics(File dumpFile, boolean verbose) throws IOException {
        String args = verbose ? "--verbose=true" : null;
        this.makeRequest(DIAGNOSTICS_CMD, args, dumpFile, null, "diagnostics information");
    }

    public void clusterStatus(File dumpFile) throws IOException {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            this.makeRequest(CLUSTER_STATUS_CMD, null, dumpFile, baos, "cluster status");
            String response = baos.toString(StandardCharsets.UTF_8);
            System.out.println("Cluster Status: " + response);
        }
    }

    public void dump(File dumpFile) throws IOException {
        this.makeRequest(DUMP_CMD, null, dumpFile, null, "thread dump");
    }

    public void statusHistory(File dumpFile, String days) throws IOException {
        this.makeRequest(STATUS_HISTORY_CMD, days, dumpFile, null, "status history information");
    }

    private boolean isNiFiFullyLoaded() throws IOException, NiFiNotRunningException {
        Logger logger = this.defaultLogger;
        Integer port = this.getCurrentPort(logger);
        if (port == null) {
            logger.info("Apache NiFi is not currently running");
            throw new NiFiNotRunningException();
        }
        try (Socket socket = new Socket();){
            boolean bl;
            this.sendRequest(socket, port, IS_LOADED_CMD, null, logger);
            InputStream in = socket.getInputStream();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                String line = reader.readLine();
                bl = Boolean.parseBoolean(line);
            }
            return bl;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void makeRequest(String request, String arguments, File dumpFile, OutputStream outputStream, String contentsDescription) throws IOException {
        Logger logger = this.defaultLogger;
        Integer port = this.getCurrentPort(logger);
        if (port == null) {
            this.cmdLogger.info("Apache NiFi is not currently running");
            logger.info("Apache NiFi is not currently running");
            return;
        }
        FileOutputStream fileOut = dumpFile == null ? null : new FileOutputStream(dumpFile);
        try (Socket socket = new Socket();){
            this.sendRequest(socket, port, request, arguments, logger);
            InputStream in = socket.getInputStream();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                String line;
                while ((line = reader.readLine()) != null) {
                    boolean written = false;
                    if (fileOut != null) {
                        ((OutputStream)fileOut).write(line.getBytes(StandardCharsets.UTF_8));
                        ((OutputStream)fileOut).write(10);
                        written = true;
                    }
                    if (outputStream != null) {
                        outputStream.write(line.getBytes(StandardCharsets.UTF_8));
                        outputStream.write(10);
                        written = true;
                    }
                    if (written) continue;
                    logger.info(line);
                }
            }
        }
        finally {
            if (fileOut != null) {
                ((OutputStream)fileOut).close();
                this.cmdLogger.info("Successfully wrote {} to {}", (Object)contentsDescription, (Object)dumpFile.getAbsolutePath());
            }
        }
    }

    private void sendRequest(Socket socket, Integer port, String request, String arguments, Logger logger) throws IOException {
        logger.debug("Connecting to NiFi instance");
        socket.setSoTimeout(60000);
        socket.connect(new InetSocketAddress("localhost", (int)port));
        logger.debug("Established connection to NiFi instance.");
        socket.setSoTimeout(60000);
        logger.debug("Sending {} Command to port {}", (Object)request, (Object)port);
        OutputStream socketOut = socket.getOutputStream();
        Properties nifiProps = this.loadProperties(logger);
        String secretKey = nifiProps.getProperty("secret.key");
        if (arguments == null) {
            socketOut.write((request + " " + secretKey + "\n").getBytes(StandardCharsets.UTF_8));
        } else {
            socketOut.write((request + " " + secretKey + " " + arguments + "\n").getBytes(StandardCharsets.UTF_8));
        }
        socketOut.flush();
    }

    public Integer decommission(boolean shutdown) throws IOException {
        Logger logger = this.cmdLogger;
        Integer port = this.getCurrentPort(logger);
        if (port == null) {
            logger.info("Apache NiFi is not currently running");
            return 15;
        }
        File lockFile = this.getLockFile(logger);
        if (!lockFile.exists()) {
            lockFile.createNewFile();
        }
        Properties nifiProps = this.loadProperties(logger);
        String secretKey = nifiProps.getProperty("secret.key");
        String pid = nifiProps.getProperty(PID_KEY);
        File statusFile = this.getStatusFile(logger);
        File pidFile = this.getPidFile(logger);
        try {
            String response;
            Socket socket;
            block14: {
                Integer n;
                socket = new Socket();
                try {
                    logger.debug("Connecting to NiFi instance");
                    socket.setSoTimeout(10000);
                    socket.connect(new InetSocketAddress("localhost", (int)port));
                    logger.debug("Established connection to NiFi instance.");
                    socket.setSoTimeout(0);
                    logger.debug("Sending DECOMMISSION Command to port {}", (Object)port);
                    OutputStream out = socket.getOutputStream();
                    String command = "DECOMMISSION " + secretKey + " --shutdown=" + shutdown + "\n";
                    out.write(command.getBytes(StandardCharsets.UTF_8));
                    out.flush();
                    socket.shutdownOutput();
                    response = this.readResponse(socket.getInputStream());
                    if (!DECOMMISSION_CMD.equals(response)) break block14;
                    logger.debug("Received response to DECOMMISSION command: {}", (Object)response);
                    if (pid != null) {
                        this.waitForShutdown(pid, logger, statusFile, pidFile);
                    }
                    n = null;
                }
                catch (Throwable throwable) {
                    try {
                        socket.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                socket.close();
                return n;
            }
            logger.error("When sending DECOMMISSION command to NiFi, got unexpected response {}", (Object)response);
            Integer n = 18;
            socket.close();
            return n;
        }
        finally {
            if (lockFile.exists() && !lockFile.delete()) {
                logger.error("Failed to delete lock file {}; this file should be cleaned up manually", (Object)lockFile);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() throws IOException {
        Logger logger = this.cmdLogger;
        Integer port = this.getCurrentPort(logger);
        if (port == null) {
            logger.info("Apache NiFi is not currently running");
            return;
        }
        File lockFile = this.getLockFile(logger);
        if (!lockFile.exists()) {
            lockFile.createNewFile();
        }
        Properties nifiProps = this.loadProperties(logger);
        String secretKey = nifiProps.getProperty("secret.key");
        String pid = nifiProps.getProperty(PID_KEY);
        File statusFile = this.getStatusFile(logger);
        File pidFile = this.getPidFile(logger);
        try (Socket socket = new Socket();){
            logger.debug("Connecting to NiFi instance");
            socket.setSoTimeout(10000);
            socket.connect(new InetSocketAddress("localhost", (int)port));
            logger.debug("Established connection to NiFi instance.");
            socket.setSoTimeout(10000);
            logger.debug("Sending SHUTDOWN Command to port {}", (Object)port);
            OutputStream out = socket.getOutputStream();
            out.write(("SHUTDOWN " + secretKey + "\n").getBytes(StandardCharsets.UTF_8));
            out.flush();
            socket.shutdownOutput();
            String response = this.readResponse(socket.getInputStream());
            logger.debug("Received response to SHUTDOWN command: {}", (Object)response);
            if (SHUTDOWN_CMD.equals(response)) {
                logger.info("Apache NiFi has accepted the Shutdown Command and is shutting down now");
                if (pid != null) {
                    this.waitForShutdown(pid, logger, statusFile, pidFile);
                }
            } else {
                logger.error("When sending SHUTDOWN command to NiFi, got unexpected response: {}", (Object)response);
            }
        }
        catch (IOException ioe) {
            if (pid == null) {
                logger.error("Failed to send shutdown command to port {} due to {}. No PID found for the NiFi process, so unable to kill process; the process should be killed manually.", (Object)port, (Object)ioe.toString());
            } else {
                logger.error("Failed to send shutdown command to port {} due to {}. Will kill the NiFi Process with PID {}.", new Object[]{port, ioe, pid});
                this.killProcessTree(pid, logger);
                if (statusFile.exists() && !statusFile.delete()) {
                    logger.error("Failed to delete status file {}; this file should be cleaned up manually", (Object)statusFile);
                }
            }
        }
        finally {
            if (lockFile.exists() && !lockFile.delete()) {
                logger.error("Failed to delete lock file {}; this file should be cleaned up manually", (Object)lockFile);
            }
        }
    }

    private String readResponse(InputStream in) throws IOException {
        int lastChar;
        StringBuilder sb = new StringBuilder();
        while ((lastChar = in.read()) > -1) {
            sb.append((char)lastChar);
        }
        return sb.toString().trim();
    }

    private void waitForShutdown(String pid, Logger logger, File statusFile, File pidFile) throws IOException {
        int gracefulShutdownSeconds;
        Properties bootstrapProperties = new Properties();
        try (FileInputStream fis = new FileInputStream(this.bootstrapConfigFile);){
            bootstrapProperties.load(fis);
        }
        String gracefulShutdown = bootstrapProperties.getProperty(GRACEFUL_SHUTDOWN_PROP, DEFAULT_GRACEFUL_SHUTDOWN_VALUE);
        try {
            gracefulShutdownSeconds = Integer.parseInt(gracefulShutdown);
        }
        catch (NumberFormatException nfe) {
            gracefulShutdownSeconds = Integer.parseInt(DEFAULT_GRACEFUL_SHUTDOWN_VALUE);
        }
        long startWait = System.nanoTime();
        while (this.isProcessRunning(pid, logger)) {
            logger.info("NiFi PID [{}] shutdown in progress...", (Object)pid);
            long waitNanos = System.nanoTime() - startWait;
            long waitSeconds = TimeUnit.NANOSECONDS.toSeconds(waitNanos);
            if (waitSeconds >= (long)gracefulShutdownSeconds && gracefulShutdownSeconds > 0) {
                if (!this.isProcessRunning(pid, logger)) break;
                logger.warn("NiFi PID [{}] shutdown not completed after {} seconds: Killing process", (Object)pid, (Object)gracefulShutdownSeconds);
                try {
                    this.killProcessTree(pid, logger);
                }
                catch (IOException ioe) {
                    logger.error("Failed to kill Process with PID {}", (Object)pid);
                }
                break;
            }
            try {
                Thread.sleep(2000L);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (statusFile.exists() && !statusFile.delete()) {
            logger.error("Failed to delete status file {}; this file should be cleaned up manually", (Object)statusFile);
        }
        if (pidFile.exists() && !pidFile.delete()) {
            logger.error("Failed to delete pid file {}; this file should be cleaned up manually", (Object)pidFile);
        }
        logger.info("NiFi PID [{}] shutdown completed", (Object)pid);
    }

    private static List<String> getChildProcesses(String ppid) throws IOException {
        Process proc = Runtime.getRuntime().exec(new String[]{"ps", "-o", PID_KEY, "--no-headers", "--ppid", ppid});
        ArrayList<String> childPids = new ArrayList<String>();
        try (InputStream in = proc.getInputStream();
             BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
            String line;
            while ((line = reader.readLine()) != null) {
                childPids.add(line.trim());
            }
        }
        return childPids;
    }

    private void killProcessTree(String pid, Logger logger) throws IOException {
        logger.debug("Killing Process Tree for PID {}", (Object)pid);
        List<String> children = RunNiFi.getChildProcesses(pid);
        logger.debug("Children of PID {}: {}", (Object)pid, children);
        for (String childPid : children) {
            this.killProcessTree(childPid, logger);
        }
        Runtime.getRuntime().exec(new String[]{"kill", "-9", pid});
    }

    public static boolean isAlive(Process process) {
        try {
            process.exitValue();
            return false;
        }
        catch (IllegalStateException | IllegalThreadStateException itse) {
            return true;
        }
    }

    public void start(boolean monitor) throws IOException {
        int gracefulShutdownSeconds;
        String fileExtension;
        File javaFile2;
        String javaHome;
        Integer port = this.getCurrentPort(this.cmdLogger);
        if (port != null) {
            this.cmdLogger.info("Apache NiFi is already running, listening to Bootstrap on port {}", (Object)port);
            return;
        }
        File prevLockFile = this.getLockFile(this.cmdLogger);
        if (prevLockFile.exists() && !prevLockFile.delete()) {
            this.cmdLogger.warn("Failed to delete previous lock file {}; this file should be cleaned up manually", (Object)prevLockFile);
        }
        this.runtimeValidatorExecutor.execute();
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        if (!this.bootstrapConfigFile.exists()) {
            throw new FileNotFoundException(this.bootstrapConfigFile.getAbsolutePath());
        }
        Properties properties = new Properties();
        try (FileInputStream fis = new FileInputStream(this.bootstrapConfigFile);){
            properties.load(fis);
        }
        HashMap<Object, Object> props = new HashMap<Object, Object>();
        props.putAll(properties);
        String specifiedWorkingDir = (String)props.get("working.dir");
        if (specifiedWorkingDir != null) {
            builder.directory(new File(specifiedWorkingDir));
        }
        File bootstrapConfigAbsoluteFile = this.bootstrapConfigFile.getAbsoluteFile();
        File binDir = bootstrapConfigAbsoluteFile.getParentFile();
        File workingDir = binDir.getParentFile();
        if (specifiedWorkingDir == null) {
            builder.directory(workingDir);
        }
        String nifiLogDir = this.replaceNull(System.getProperty("org.apache.nifi.bootstrap.config.log.dir"), DEFAULT_LOG_DIR).trim();
        String libFilename = this.replaceNull((String)props.get("lib.dir"), "./lib").trim();
        File libDir = this.getFile(libFilename, workingDir);
        String confFilename = this.replaceNull((String)props.get("conf.dir"), "./conf").trim();
        File confDir = this.getFile(confFilename, workingDir);
        String nifiPropsFilename = (String)props.get("props.file");
        if (nifiPropsFilename == null) {
            nifiPropsFilename = confDir.exists() ? new File(confDir, "nifi.properties").getAbsolutePath() : DEFAULT_CONFIG_FILE;
        }
        nifiPropsFilename = nifiPropsFilename.trim();
        String maximumHeapSize = null;
        String minimumHeapSize = null;
        ArrayList<File[]> javaAdditionalArgs = new ArrayList<File[]>();
        for (Map.Entry entry : props.entrySet()) {
            String key = (String)entry.getKey();
            File[] value = (File[])entry.getValue();
            if (!key.startsWith("java.arg")) continue;
            javaAdditionalArgs.add(value);
            if (value.startsWith("-Xms")) {
                minimumHeapSize = value.replace("-Xms", "");
            }
            if (!value.startsWith("-Xmx")) continue;
            maximumHeapSize = value.replace("-Xmx", "");
        }
        File[] libFiles = libDir.listFiles(new FilenameFilter(this){

            @Override
            public boolean accept(File dir, String filename) {
                return filename.toLowerCase().endsWith(".jar");
            }
        });
        if (libFiles == null || libFiles.length == 0) {
            throw new RuntimeException("Could not find lib directory at " + libDir.getAbsolutePath());
        }
        File[] confFiles = confDir.listFiles();
        if (confFiles == null || confFiles.length == 0) {
            throw new RuntimeException("Could not find conf directory at " + confDir.getAbsolutePath());
        }
        ArrayList<String> cpFiles = new ArrayList<String>(confFiles.length + libFiles.length);
        cpFiles.add(confDir.getAbsolutePath());
        for (File file : libFiles) {
            cpFiles.add(file.getAbsolutePath());
        }
        this.defaultLogger.info(this.getPlatformDetails(minimumHeapSize, maximumHeapSize));
        StringBuilder classPathBuilder = new StringBuilder();
        for (int i = 0; i < cpFiles.size(); ++i) {
            String filename = (String)cpFiles.get(i);
            classPathBuilder.append(filename);
            if (i >= cpFiles.size() - 1) continue;
            classPathBuilder.append(File.pathSeparatorChar);
        }
        String classPath = classPathBuilder.toString();
        String javaCmd = (String)props.get(DEFAULT_JAVA_CMD);
        if (javaCmd == null) {
            javaCmd = DEFAULT_JAVA_CMD;
        }
        if (javaCmd.equals(DEFAULT_JAVA_CMD) && (javaHome = System.getenv("JAVA_HOME")) != null && (javaFile2 = new File(javaHome + File.separatorChar + DEFAULT_PID_DIR + File.separatorChar + DEFAULT_JAVA_CMD + (fileExtension = this.isWindows() ? ".exe" : ""))).exists() && javaFile2.canExecute()) {
            javaCmd = javaFile2.getAbsolutePath();
        }
        try {
            SecurityApplicationPropertyHandler propertyHandler = new SecurityApplicationPropertyHandler(this.cmdLogger);
            Path applicationPropertiesLocation = Paths.get(nifiPropsFilename, new String[0]);
            propertyHandler.handleProperties(applicationPropertiesLocation);
        }
        catch (RuntimeException e) {
            this.cmdLogger.error("Self-Signed Certificate Generation Failed", (Throwable)e);
        }
        String listenPortPropString = (String)props.get(NIFI_BOOTSTRAP_LISTEN_PORT_PROP);
        int listenPortPropInt = 0;
        if (listenPortPropString != null) {
            try {
                listenPortPropInt = Integer.parseInt(listenPortPropString.trim());
            }
            catch (Exception javaFile2) {
                // empty catch block
            }
        }
        NiFiListener listener = new NiFiListener();
        int listenPort = listener.start(this, listenPortPropInt);
        ArrayList<String> cmd = new ArrayList<String>();
        cmd.add(javaCmd);
        cmd.add("-classpath");
        cmd.add(classPath);
        cmd.addAll(javaAdditionalArgs);
        cmd.add("-Dnifi.properties.file.path=" + nifiPropsFilename);
        cmd.add("-Dnifi.bootstrap.listen.port=" + listenPort);
        cmd.add("-Dapp=NiFi");
        cmd.add("-Dorg.apache.nifi.bootstrap.config.log.dir=" + nifiLogDir);
        cmd.add("org.apache.nifi.NiFi");
        builder.command(cmd);
        StringBuilder cmdBuilder = new StringBuilder();
        for (String s : cmd) {
            cmdBuilder.append(s).append(" ");
        }
        this.cmdLogger.info("Starting Apache NiFi...");
        this.cmdLogger.info("Working Directory: {}", (Object)workingDir.getAbsolutePath());
        this.cmdLogger.info("Command: {}", (Object)cmdBuilder);
        String gracefulShutdown = (String)props.get(GRACEFUL_SHUTDOWN_PROP);
        if (gracefulShutdown == null) {
            gracefulShutdown = DEFAULT_GRACEFUL_SHUTDOWN_VALUE;
        }
        try {
            gracefulShutdownSeconds = Integer.parseInt(gracefulShutdown);
        }
        catch (NumberFormatException nfe) {
            throw new NumberFormatException("The 'graceful.shutdown.seconds' property in Bootstrap Config File " + bootstrapConfigAbsoluteFile.getAbsolutePath() + " has an invalid value. Must be a non-negative integer");
        }
        if (gracefulShutdownSeconds < 0) {
            throw new NumberFormatException("The 'graceful.shutdown.seconds' property in Bootstrap Config File " + bootstrapConfigAbsoluteFile.getAbsolutePath() + " has an invalid value. Must be a non-negative integer");
        }
        Process process = builder.start();
        this.handleLogging(process);
        this.nifiPid = process.pid();
        Properties pidProperties = new Properties();
        pidProperties.setProperty(PID_KEY, String.valueOf(this.nifiPid));
        this.savePidProperties(pidProperties, this.cmdLogger);
        this.cmdLogger.info("Application Process [{}] launched", (Object)this.nifiPid);
        this.shutdownHook = new ShutdownHook(process, this.nifiPid, this, this.secretKey, gracefulShutdownSeconds, this.loggingExecutor);
        if (monitor) {
            Runtime runtime = Runtime.getRuntime();
            runtime.addShutdownHook(this.shutdownHook);
            while (true) {
                boolean alive;
                if (alive = RunNiFi.isAlive(process)) {
                    try {
                        Thread.sleep(1000L);
                    }
                    catch (InterruptedException interruptedException) {}
                    continue;
                }
                try {
                    runtime.removeShutdownHook(this.shutdownHook);
                }
                catch (IllegalStateException illegalStateException) {
                    // empty catch block
                }
                if (!this.autoRestartNiFi) break;
                File statusFile = this.getStatusFile(this.defaultLogger);
                if (!statusFile.exists()) {
                    this.defaultLogger.info("Status File no longer exists. Will not restart NiFi");
                    return;
                }
                File lockFile = this.getLockFile(this.defaultLogger);
                if (lockFile.exists()) {
                    this.defaultLogger.info("A shutdown was initiated. Will not restart NiFi");
                    return;
                }
                boolean previouslyStarted = this.getNifiStarted();
                if (!previouslyStarted) {
                    this.defaultLogger.info("NiFi never started. Will not restart NiFi");
                    return;
                }
                this.setNiFiStarted(false);
                this.defaultLogger.warn("Apache NiFi appears to have died. Restarting...");
                this.secretKey = null;
                process = builder.start();
                this.handleLogging(process);
                this.nifiPid = process.pid();
                pidProperties.setProperty(PID_KEY, String.valueOf(this.nifiPid));
                this.savePidProperties(pidProperties, this.defaultLogger);
                this.cmdLogger.info("Application Process [{}] launched", (Object)this.nifiPid);
                this.shutdownHook = new ShutdownHook(process, this.nifiPid, this, this.secretKey, gracefulShutdownSeconds, this.loggingExecutor);
                runtime.addShutdownHook(this.shutdownHook);
                boolean started = this.waitForStart();
                if (started) {
                    this.cmdLogger.info("Application Process [{}] started", (Object)this.nifiPid);
                    continue;
                }
                this.defaultLogger.error("Application Process [{}] not started", (Object)this.nifiPid);
            }
            return;
        }
    }

    private String getPlatformDetails(String minimumHeapSize, String maximumHeapSize) {
        LinkedHashMap<String, String> details = new LinkedHashMap<String, String>(6);
        details.put("javaVersion", System.getProperty("java.version"));
        details.put("availableProcessors", Integer.toString(Runtime.getRuntime().availableProcessors()));
        try {
            Object totalPhysicalMemory;
            ObjectName osObjectName = ManagementFactory.getOperatingSystemMXBean().getObjectName();
            MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
            Object maxOpenFileCount = mBeanServer.getAttribute(osObjectName, "MaxFileDescriptorCount");
            if (maxOpenFileCount != null) {
                details.put("maxOpenFileDescriptors", String.valueOf(maxOpenFileCount));
            }
            if ((totalPhysicalMemory = mBeanServer.getAttribute(osObjectName, "TotalPhysicalMemorySize")) != null) {
                details.put("totalPhysicalMemoryMB", String.valueOf((Long)totalPhysicalMemory / 0x100000L));
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        if (minimumHeapSize != null) {
            details.put("minimumHeapSize", minimumHeapSize);
        }
        if (maximumHeapSize != null) {
            details.put("maximumHeapSize", maximumHeapSize);
        }
        return ((Object)details).toString();
    }

    private void handleLogging(final Process process) {
        Set<Future<?>> existingFutures = this.loggingFutures;
        if (existingFutures != null) {
            for (Future<?> future : existingFutures) {
                future.cancel(false);
            }
        }
        Future<?> stdOutFuture = this.loggingExecutor.submit(new Runnable(){

            @Override
            public void run() {
                Logger stdOutLogger = LoggerFactory.getLogger((String)"org.apache.nifi.StdOut");
                InputStream in = process.getInputStream();
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        stdOutLogger.info(line);
                    }
                }
                catch (IOException e) {
                    RunNiFi.this.defaultLogger.error("Failed to read from NiFi's Standard Out stream", (Throwable)e);
                }
            }
        });
        Future<?> stdErrFuture = this.loggingExecutor.submit(new Runnable(){

            @Override
            public void run() {
                Logger stdErrLogger = LoggerFactory.getLogger((String)"org.apache.nifi.StdErr");
                InputStream in = process.getErrorStream();
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        stdErrLogger.error(line);
                    }
                }
                catch (IOException e) {
                    RunNiFi.this.defaultLogger.error("Failed to read from NiFi's Standard Error stream", (Throwable)e);
                }
            }
        });
        HashSet futures = new HashSet();
        futures.add(stdOutFuture);
        futures.add(stdErrFuture);
        this.loggingFutures = futures;
    }

    private boolean isWindows() {
        String osName = System.getProperty("os.name");
        return osName != null && osName.toLowerCase().contains("win");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean waitForStart() {
        this.lock.lock();
        try {
            long startTime = System.nanoTime();
            while (this.ccPort < 1) {
                try {
                    this.startupCondition.await(1L, TimeUnit.SECONDS);
                }
                catch (InterruptedException ie) {
                    boolean bl = false;
                    this.lock.unlock();
                    return bl;
                }
                long waitNanos = System.nanoTime() - startTime;
                long waitSeconds = TimeUnit.NANOSECONDS.toSeconds(waitNanos);
                if (waitSeconds <= 60L) continue;
                boolean bl = false;
                return bl;
            }
        }
        finally {
            this.lock.unlock();
        }
        return true;
    }

    private File getFile(String filename, File workingDir) {
        File file = new File(filename);
        if (!file.isAbsolute()) {
            file = new File(workingDir, filename);
        }
        return file;
    }

    private String replaceNull(String value, String replacement) {
        return value == null ? replacement : value;
    }

    void setAutoRestartNiFi(boolean restart) {
        this.autoRestartNiFi = restart;
    }

    void setNiFiCommandControlPort(int port, String secretKey) throws IOException {
        if (this.secretKey != null && this.ccPort != -1) {
            this.defaultLogger.warn("Blocking attempt to change NiFi command port and secret after they have already been initialized. requestedPort={}", (Object)port);
            return;
        }
        this.ccPort = port;
        this.secretKey = secretKey;
        if (this.shutdownHook != null) {
            this.shutdownHook.setSecretKey(secretKey);
        }
        File statusFile = this.getStatusFile(this.defaultLogger);
        Properties nifiProps = new Properties();
        if (this.nifiPid != -1L) {
            nifiProps.setProperty(PID_KEY, String.valueOf(this.nifiPid));
        }
        nifiProps.setProperty("port", String.valueOf(this.ccPort));
        nifiProps.setProperty("secret.key", secretKey);
        try {
            this.savePidProperties(nifiProps, this.defaultLogger);
        }
        catch (IOException ioe) {
            this.defaultLogger.warn("Apache NiFi has started but failed to persist NiFi Port information to {} due to {}", (Object)statusFile.getAbsolutePath(), (Object)ioe);
        }
        this.defaultLogger.info("Apache NiFi now running and listening for Bootstrap requests on port {}", (Object)port);
    }

    int getNiFiCommandControlPort() {
        return this.ccPort;
    }

    void setNiFiStarted(boolean nifiStarted) {
        this.startedLock.lock();
        try {
            this.nifiStarted = nifiStarted;
        }
        finally {
            this.startedLock.unlock();
        }
    }

    boolean getNifiStarted() {
        this.startedLock.lock();
        try {
            boolean bl = this.nifiStarted;
            return bl;
        }
        finally {
            this.startedLock.unlock();
        }
    }

    private static class NiFiNotRunningException
    extends Exception {
        private NiFiNotRunningException() {
        }

        @Override
        public synchronized Throwable fillInStackTrace() {
            return this;
        }
    }

    private static class Status {
        private final Integer port;
        private final String pid;
        private final Boolean respondingToPing;
        private final Boolean processRunning;

        public Status(Integer port, String pid, Boolean respondingToPing, Boolean processRunning) {
            this.port = port;
            this.pid = pid;
            this.respondingToPing = respondingToPing;
            this.processRunning = processRunning;
        }

        public String getPid() {
            return this.pid;
        }

        public Integer getPort() {
            return this.port;
        }

        public boolean isRespondingToPing() {
            return Boolean.TRUE.equals(this.respondingToPing);
        }

        public boolean isProcessRunning() {
            return Boolean.TRUE.equals(this.processRunning);
        }
    }
}

