/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.apache.nifi.BootstrapListener;
import org.apache.nifi.NiFiEntryPoint;
import org.apache.nifi.NiFiServer;
import org.apache.nifi.bundle.Bundle;
import org.apache.nifi.diagnostics.DiagnosticsDump;
import org.apache.nifi.nar.ExtensionMapping;
import org.apache.nifi.nar.NarClassLoaders;
import org.apache.nifi.nar.NarClassLoadersHolder;
import org.apache.nifi.nar.NarUnpackMode;
import org.apache.nifi.nar.NarUnpacker;
import org.apache.nifi.nar.SystemBundle;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.util.DiagnosticUtils;
import org.apache.nifi.util.FileUtils;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.bridge.SLF4JBridgeHandler;

public class NiFi
implements NiFiEntryPoint {
    public static final String BOOTSTRAP_PORT_PROPERTY = "nifi.bootstrap.listen.port";
    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss");
    private static final Logger LOGGER = LoggerFactory.getLogger(NiFi.class);
    private static final String KEY_FILE_FLAG = "-K";
    private final NiFiServer nifiServer;
    private final BootstrapListener bootstrapListener;
    private final NiFiProperties properties;
    private volatile boolean shutdown = false;

    public NiFi(NiFiProperties properties) throws ClassNotFoundException, IOException, IllegalArgumentException {
        this(properties, ClassLoader.getSystemClassLoader());
    }

    public NiFi(NiFiProperties properties, ClassLoader rootClassLoader) throws ClassNotFoundException, IOException, IllegalArgumentException {
        this.properties = properties;
        File kerberosConfigFile = properties.getKerberosConfigurationFile();
        if (kerberosConfigFile != null) {
            String kerberosConfigFilePath = kerberosConfigFile.getAbsolutePath();
            LOGGER.debug("Setting java.security.krb5.conf to {}", (Object)kerberosConfigFilePath);
            System.setProperty("java.security.krb5.conf", kerberosConfigFilePath);
        }
        this.setDefaultUncaughtExceptionHandler();
        this.addShutdownHook();
        String bootstrapPort = System.getProperty(BOOTSTRAP_PORT_PROPERTY);
        if (bootstrapPort != null) {
            try {
                int port = Integer.parseInt(bootstrapPort);
                if (port < 1 || port > 65535) {
                    throw new RuntimeException("Failed to start NiFi because system property 'nifi.bootstrap.listen.port' is not a valid integer in the range 1 - 65535");
                }
                this.bootstrapListener = new BootstrapListener(this, port);
                this.bootstrapListener.start(properties.getDefaultListenerBootstrapPort());
            }
            catch (NumberFormatException nfe) {
                throw new RuntimeException("Failed to start NiFi because system property 'nifi.bootstrap.listen.port' is not a valid integer in the range 1 - 65535");
            }
        } else {
            LOGGER.info("NiFi started without Bootstrap Port information provided; will not listen for requests from Bootstrap");
            this.bootstrapListener = null;
        }
        File webWorkingDir = properties.getWebWorkingDirectory();
        FileUtils.deleteFilesInDirectory((File)webWorkingDir, null, (Logger)LOGGER, (boolean)true, (boolean)true);
        FileUtils.deleteFile((File)webWorkingDir, (Logger)LOGGER, (int)3);
        this.detectTimingIssues();
        this.initLogging();
        Bundle systemBundle = SystemBundle.create((NiFiProperties)properties, (ClassLoader)rootClassLoader);
        NarUnpackMode unpackMode = properties.isUnpackNarsToUberJar() ? NarUnpackMode.UNPACK_TO_UBER_JAR : NarUnpackMode.UNPACK_INDIVIDUAL_JARS;
        ExtensionMapping extensionMapping = NarUnpacker.unpackNars((NiFiProperties)properties, (Bundle)systemBundle, (NarUnpackMode)unpackMode);
        NarClassLoaders narClassLoaders = NarClassLoadersHolder.getInstance();
        narClassLoaders.init(rootClassLoader, properties.getFrameworkWorkingDirectory(), properties.getExtensionsWorkingDirectory(), true);
        ClassLoader frameworkClassLoader = narClassLoaders.getFrameworkBundle().getClassLoader();
        if (frameworkClassLoader == null) {
            throw new IllegalStateException("Unable to find the framework NAR ClassLoader.");
        }
        Set narBundles = narClassLoaders.getBundles();
        long startTime = System.nanoTime();
        this.nifiServer = narClassLoaders.getServer();
        if (this.nifiServer == null) {
            throw new IllegalStateException("Unable to find a NiFiServer implementation.");
        }
        Thread.currentThread().setContextClassLoader(this.nifiServer.getClass().getClassLoader());
        this.nifiServer.initialize(properties, systemBundle, narBundles, extensionMapping);
        if (this.shutdown) {
            LOGGER.info("NiFi has been shutdown via NiFi Bootstrap. Will not start Controller");
        } else {
            this.nifiServer.start();
            if (this.bootstrapListener != null) {
                this.bootstrapListener.setNiFiLoaded(true);
                this.bootstrapListener.sendStartedStatus(true);
            }
            long duration = System.nanoTime() - startTime;
            double durationSeconds = (double)TimeUnit.NANOSECONDS.toMillis(duration) / 1000.0;
            LOGGER.info("Started Application Controller in {} seconds ({} ns)", (Object)durationSeconds, (Object)duration);
        }
    }

    @Override
    public NiFiServer getServer() {
        return this.nifiServer;
    }

    protected void setDefaultUncaughtExceptionHandler() {
        Thread.setDefaultUncaughtExceptionHandler((thread, exception) -> LOGGER.error("An Unknown Error Occurred in Thread {}: {}", new Object[]{thread, exception.toString(), exception}));
    }

    protected void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> this.shutdownHook(false)));
    }

    protected void initLogging() {
        SLF4JBridgeHandler.removeHandlersForRootLogger();
        SLF4JBridgeHandler.install();
    }

    private static ClassLoader createBootstrapClassLoader() {
        ArrayList urls = new ArrayList();
        try (Stream<Path> files = Files.list(Paths.get("lib/bootstrap", new String[0]));){
            files.forEach(p -> {
                try {
                    urls.add(p.toUri().toURL());
                }
                catch (MalformedURLException mef) {
                    LOGGER.warn("Unable to load bootstrap library [{}]", (Object)p.getFileName(), (Object)mef);
                }
            });
        }
        catch (IOException ioe) {
            LOGGER.warn("Unable to access lib/bootstrap to create bootstrap classloader", (Throwable)ioe);
        }
        return new URLClassLoader(urls.toArray(new URL[0]), Thread.currentThread().getContextClassLoader());
    }

    @Override
    public void shutdownHook(boolean isReload) {
        try {
            this.runDiagnosticsOnShutdown();
            this.shutdown();
        }
        catch (Throwable t) {
            LOGGER.warn("Problem occurred ensuring Jetty web server was properly terminated", t);
        }
    }

    private void runDiagnosticsOnShutdown() throws IOException {
        if (this.properties.isDiagnosticsOnShutdownEnabled()) {
            String diagnosticDirectoryPath = this.properties.getDiagnosticsOnShutdownDirectory();
            boolean isCreated = DiagnosticUtils.createDiagnosticDirectory(diagnosticDirectoryPath);
            if (isCreated) {
                LOGGER.debug("Diagnostic directory has successfully been created.");
            }
            while (DiagnosticUtils.isFileCountExceeded(diagnosticDirectoryPath, this.properties.getDiagnosticsOnShutdownMaxFileCount()) || DiagnosticUtils.isSizeExceeded(diagnosticDirectoryPath, DataUnit.parseDataSize((String)this.properties.getDiagnosticsOnShutdownDirectoryMaxSize(), (DataUnit)DataUnit.B).longValue())) {
                Path oldestFile = DiagnosticUtils.getOldestFile(diagnosticDirectoryPath);
                Files.delete(oldestFile);
            }
            String fileName = String.format("%s/diagnostic-%s.log", diagnosticDirectoryPath, DATE_TIME_FORMATTER.format(LocalDateTime.now()));
            this.diagnose(new File(fileName), this.properties.isDiagnosticsOnShutdownVerbose());
        }
    }

    private void diagnose(File file, boolean verbose) throws IOException {
        DiagnosticsDump diagnosticsDump = this.getServer().getDiagnosticsFactory().create(verbose);
        try (FileOutputStream fileOutputStream = new FileOutputStream(file);){
            diagnosticsDump.writeTo((OutputStream)fileOutputStream);
        }
    }

    protected void shutdown() {
        this.shutdown = true;
        LOGGER.info("Application Server shutdown started");
        if (this.nifiServer != null) {
            this.nifiServer.stop();
        }
        if (this.bootstrapListener != null) {
            this.bootstrapListener.stop();
        }
        LOGGER.info("Application Server shutdown completed");
    }

    private void detectTimingIssues() {
        int minRequiredOccurrences = 25;
        int maxOccurrencesOutOfRange = 15;
        AtomicLong lastTriggerMillis = new AtomicLong(System.currentTimeMillis());
        final ScheduledExecutorService service = Executors.newScheduledThreadPool(1, new ThreadFactory(this){
            private final ThreadFactory defaultFactory = Executors.defaultThreadFactory();

            @Override
            public Thread newThread(Runnable runnable) {
                Thread t = this.defaultFactory.newThread(runnable);
                t.setDaemon(true);
                t.setName("Detect Timing Issues");
                return t;
            }
        });
        final AtomicInteger occurrencesOutOfRange = new AtomicInteger(0);
        final AtomicInteger occurrences = new AtomicInteger(0);
        Runnable command = () -> {
            long curMillis = System.currentTimeMillis();
            long difference = curMillis - lastTriggerMillis.get();
            long millisOff = Math.abs(difference - 2000L);
            occurrences.incrementAndGet();
            if (millisOff > 500L) {
                occurrencesOutOfRange.incrementAndGet();
            }
            lastTriggerMillis.set(curMillis);
        };
        final ScheduledFuture<?> future = service.scheduleWithFixedDelay(command, 2000L, 2000L, TimeUnit.MILLISECONDS);
        TimerTask timerTask = new TimerTask(this){

            @Override
            public void run() {
                future.cancel(true);
                service.shutdownNow();
                if (occurrences.get() < 25 || occurrencesOutOfRange.get() > 15) {
                    LOGGER.warn("NiFi has detected that this box is not responding within the expected timing interval, which may cause Processors to be scheduled erratically. Please see the NiFi documentation for more information.");
                }
            }
        };
        Timer timer = new Timer(true);
        timer.schedule(timerTask, 60000L);
    }

    public static void main(String[] args) {
        LOGGER.info("Launching NiFi...");
        try {
            NiFiProperties properties = NiFi.convertArgumentsToValidatedNiFiProperties(args);
            new NiFi(properties);
        }
        catch (Throwable t) {
            LOGGER.error("Failure to launch NiFi", t);
        }
    }

    protected static NiFiProperties convertArgumentsToValidatedNiFiProperties(String[] args) {
        return NiFi.convertArgumentsToValidatedNiFiProperties(args, NiFi.createBootstrapClassLoader());
    }

    protected static NiFiProperties convertArgumentsToValidatedNiFiProperties(String[] args, ClassLoader bootstrapClassLoader) {
        NiFiProperties properties = NiFi.initializeProperties(args, bootstrapClassLoader);
        properties.validate();
        return properties;
    }

    private static NiFiProperties initializeProperties(String[] args, ClassLoader boostrapLoader) {
        String key;
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            key = NiFi.loadFormattedKey(args);
        }
        catch (IllegalArgumentException e) {
            String msg = "The bootstrap process did not provide a valid key";
            throw new IllegalArgumentException("The bootstrap process did not provide a valid key", e);
        }
        Thread.currentThread().setContextClassLoader(boostrapLoader);
        try {
            Class<?> propsLoaderClass = Class.forName("org.apache.nifi.properties.NiFiPropertiesLoader", true, boostrapLoader);
            Method withKeyMethod = propsLoaderClass.getMethod("withKey", String.class);
            Object loaderInstance = withKeyMethod.invoke(null, key);
            Method getMethod = propsLoaderClass.getMethod("get", new Class[0]);
            NiFiProperties properties = (NiFiProperties)getMethod.invoke(loaderInstance, new Object[0]);
            LOGGER.info("Application Properties loaded [{}]", (Object)properties.size());
            NiFiProperties niFiProperties = properties;
            return niFiProperties;
        }
        catch (InvocationTargetException wrappedException) {
            String msg = "There was an issue decrypting protected properties";
            throw new IllegalArgumentException("There was an issue decrypting protected properties", wrappedException.getCause() == null ? wrappedException : wrappedException.getCause());
        }
        catch (ClassNotFoundException | IllegalAccessException | NoSuchMethodException reex) {
            String msg = "Unable to access properties loader in the expected manner - apparent classpath or build issue";
            throw new IllegalArgumentException("Unable to access properties loader in the expected manner - apparent classpath or build issue", reex);
        }
        catch (RuntimeException e) {
            String msg = "There was an issue decrypting protected properties";
            throw new IllegalArgumentException("There was an issue decrypting protected properties", e);
        }
        finally {
            Thread.currentThread().setContextClassLoader(contextClassLoader);
        }
    }

    private static String loadFormattedKey(String[] args) {
        String key = null;
        List<String> parsedArgs = NiFi.parseArgs(args);
        if (parsedArgs.contains(KEY_FILE_FLAG)) {
            key = NiFi.getKeyFromKeyFileAndPrune(parsedArgs);
            key = NiFi.formatHexKey(key);
        }
        if (null == key) {
            return "";
        }
        if (!NiFi.isHexKeyValid(key)) {
            throw new IllegalArgumentException("The key was not provided in valid hex format and of the correct length");
        }
        return key;
    }

    private static String getKeyFromKeyFileAndPrune(List<String> parsedArgs) {
        String key = null;
        LOGGER.debug("The bootstrap process provided the {} flag", (Object)KEY_FILE_FLAG);
        int i = parsedArgs.indexOf(KEY_FILE_FLAG);
        if (parsedArgs.size() <= i + 1) {
            LOGGER.error("The bootstrap process passed the {} flag without a filename", (Object)KEY_FILE_FLAG);
            throw new IllegalArgumentException("The bootstrap process provided the -K flag but no key");
        }
        try {
            String passwordfilePath = parsedArgs.get(i + 1);
            byte[] encoded = Files.readAllBytes(Paths.get(passwordfilePath, new String[0]));
            key = new String(encoded, StandardCharsets.UTF_8);
            if (0 == key.length()) {
                throw new IllegalArgumentException("Key in keyfile " + passwordfilePath + " yielded an empty key");
            }
            LOGGER.debug("Overwriting temporary bootstrap key file [{}]", (Object)passwordfilePath);
            File passwordFile = new File(passwordfilePath);
            FileWriter overwriter = new FileWriter(passwordFile, false);
            Random random = new Random();
            StringBuffer sb = new StringBuffer();
            while (sb.length() < encoded.length) {
                sb.append(Integer.toHexString(random.nextInt()));
            }
            String pad = sb.toString();
            overwriter.write(pad);
            overwriter.close();
            LOGGER.debug("Removing temporary bootstrap key file [{}]", (Object)passwordfilePath);
            passwordFile.delete();
        }
        catch (IOException e) {
            LOGGER.error("Caught IOException while retrieving the {} -passed keyfile; aborting: {}", (Object)KEY_FILE_FLAG, (Object)e.toString());
            System.exit(1);
        }
        return key;
    }

    private static List<String> parseArgs(String[] args) {
        ArrayList<String> parsedArgs = new ArrayList<String>(Arrays.asList(args));
        for (int i = 0; i < parsedArgs.size(); ++i) {
            if (!((String)parsedArgs.get(i)).startsWith("-K ")) continue;
            String[] split = ((String)parsedArgs.get(i)).split(" ", 2);
            parsedArgs.set(i, split[0]);
            parsedArgs.add(i + 1, split[1]);
            break;
        }
        return parsedArgs;
    }

    private static String formatHexKey(String input) {
        if (input == null || input.trim().isEmpty()) {
            return "";
        }
        return input.replaceAll("[^0-9a-fA-F]", "").toLowerCase();
    }

    private static boolean isHexKeyValid(String key) {
        if (key == null || key.trim().isEmpty()) {
            return false;
        }
        return Arrays.asList(128, 196, 256).contains(key.length() * 4) && key.matches("^[0-9a-fA-F]*$");
    }
}

