/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.properties;

import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.nifi.properties.ProtectedPropertyContext;
import org.apache.nifi.properties.SensitivePropertyProtectionException;
import org.apache.nifi.properties.SensitivePropertyProvider;

public class AesGcmSensitivePropertyProvider
implements SensitivePropertyProvider {
    private static final String ALGORITHM = "AES/GCM/NoPadding";
    private static final String SECRET_KEY_ALGORITHM = "AES";
    private static final String DELIMITER = "||";
    private static final int IV_LENGTH = 12;
    private static final int TAG_LENGTH = 128;
    private static final int MIN_CIPHER_TEXT_LENGTH = 16 + "||".length() + 1;
    private static final List<Integer> VALID_KEY_LENGTHS = Arrays.asList(128, 192, 256);
    private static final String IDENTIFIER_KEY_FORMAT = "aes/gcm/%d";
    private static final int BITS_PER_BYTE = 8;
    private static final Base64.Encoder BASE_64_ENCODER = Base64.getEncoder();
    private static final Base64.Decoder BASE_64_DECODER = Base64.getDecoder();
    private final SecureRandom secureRandom;
    private final Cipher cipher;
    private final SecretKey key;
    private final String identifierKey;

    public AesGcmSensitivePropertyProvider(String keyHex) {
        byte[] keyBytes = this.validateKey(keyHex);
        this.secureRandom = new SecureRandom();
        try {
            this.cipher = Cipher.getInstance(ALGORITHM);
            this.key = new SecretKeySpec(keyBytes, SECRET_KEY_ALGORITHM);
            int keySize = keyBytes.length * 8;
            this.identifierKey = String.format(IDENTIFIER_KEY_FORMAT, keySize);
        }
        catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new SensitivePropertyProtectionException(String.format("Cipher [%s] initialization failed", ALGORITHM), (Throwable)e);
        }
    }

    public boolean isSupported() {
        return true;
    }

    public String getIdentifierKey() {
        return this.identifierKey;
    }

    public String protect(String unprotectedValue, ProtectedPropertyContext context) throws SensitivePropertyProtectionException {
        Objects.requireNonNull(unprotectedValue, "Value required");
        byte[] iv = this.generateIV();
        try {
            this.cipher.init(1, (Key)this.key, new GCMParameterSpec(128, iv));
            byte[] plainBytes = unprotectedValue.getBytes(StandardCharsets.UTF_8);
            byte[] cipherBytes = this.cipher.doFinal(plainBytes);
            return BASE_64_ENCODER.encodeToString(iv) + DELIMITER + BASE_64_ENCODER.encodeToString(cipherBytes);
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            throw new SensitivePropertyProtectionException("AES-GCM encryption failed", (Throwable)e);
        }
    }

    public String unprotect(String protectedValue, ProtectedPropertyContext context) throws SensitivePropertyProtectionException {
        if (protectedValue == null || protectedValue.trim().length() < MIN_CIPHER_TEXT_LENGTH) {
            throw new IllegalArgumentException("Cannot decrypt a cipher text shorter than " + MIN_CIPHER_TEXT_LENGTH + " chars");
        }
        if (!protectedValue.contains(DELIMITER)) {
            throw new IllegalArgumentException("The cipher text does not contain the delimiter || -- it should be of the form Base64(IV) || Base64(cipherText)");
        }
        String trimmedProtectedValue = protectedValue.trim();
        String armoredIV = trimmedProtectedValue.substring(0, trimmedProtectedValue.indexOf(DELIMITER));
        byte[] iv = BASE_64_DECODER.decode(armoredIV);
        if (iv.length < 12) {
            throw new IllegalArgumentException(String.format("The IV (%s bytes) must be at least %s bytes", iv.length, 12));
        }
        String encodedCipherBytes = trimmedProtectedValue.substring(trimmedProtectedValue.indexOf(DELIMITER) + 2);
        try {
            byte[] cipherBytes = BASE_64_DECODER.decode(encodedCipherBytes);
            this.cipher.init(2, (Key)this.key, new GCMParameterSpec(128, iv));
            byte[] plainBytes = this.cipher.doFinal(cipherBytes);
            return new String(plainBytes, StandardCharsets.UTF_8);
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            throw new SensitivePropertyProtectionException("AES-GCM decryption failed", (Throwable)e);
        }
    }

    public void cleanUp() {
    }

    private byte[] generateIV() {
        byte[] iv = new byte[12];
        this.secureRandom.nextBytes(iv);
        return iv;
    }

    private byte[] validateKey(String keyHex) {
        byte[] key;
        if (keyHex == null || keyHex.isEmpty()) {
            throw new SensitivePropertyProtectionException("AES Key not provided");
        }
        if (!AesGcmSensitivePropertyProvider.isHexKeyValid(keyHex = AesGcmSensitivePropertyProvider.formatHexKey(keyHex))) {
            throw new SensitivePropertyProtectionException("AES Key not hexadecimal");
        }
        try {
            key = Hex.decodeHex((String)keyHex);
        }
        catch (DecoderException e) {
            throw new SensitivePropertyProtectionException("AES Key Hexadecimal decoding failed", (Throwable)e);
        }
        int keyLengthBits = key.length * 8;
        if (!VALID_KEY_LENGTHS.contains(keyLengthBits)) {
            throw new SensitivePropertyProtectionException(String.format("AES Key length not valid [%d]", keyLengthBits));
        }
        return key;
    }

    private static String formatHexKey(String input) {
        if (input == null || input.isEmpty()) {
            return "";
        }
        return input.replaceAll("[^0-9a-fA-F]", "").toLowerCase();
    }

    private static boolean isHexKeyValid(String key) {
        if (key == null || key.isEmpty()) {
            return false;
        }
        return key.matches("^[0-9a-fA-F]*$");
    }
}

