/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.properties;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.nifi.properties.ProtectedNiFiProperties;
import org.apache.nifi.properties.SensitivePropertyProtectionException;
import org.apache.nifi.properties.SensitivePropertyProvider;
import org.apache.nifi.properties.SensitivePropertyProviderFactory;
import org.apache.nifi.property.protection.loader.PropertyProtectionURLClassLoader;
import org.apache.nifi.property.protection.loader.PropertyProviderFactoryLoader;
import org.apache.nifi.util.NiFiBootstrapUtils;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NiFiPropertiesLoader {
    private static final Logger logger = LoggerFactory.getLogger(NiFiPropertiesLoader.class);
    private static final Base64.Encoder KEY_ENCODER = Base64.getEncoder().withoutPadding();
    private static final int SENSITIVE_PROPERTIES_KEY_LENGTH = 24;
    private static final String EMPTY_SENSITIVE_PROPERTIES_KEY = String.format("%s=", "nifi.sensitive.props.key");
    private static final String MIGRATION_INSTRUCTIONS = "See Admin Guide section [Updating the Sensitive Properties Key]";
    private static final String PROPERTIES_KEY_MESSAGE = String.format("Sensitive Properties Key [%s] not found: %s", "nifi.sensitive.props.key", "See Admin Guide section [Updating the Sensitive Properties Key]");
    private static final String SET_KEY_METHOD = "setKeyHex";
    private final String defaultPropertiesFilePath = NiFiBootstrapUtils.getDefaultApplicationPropertiesFilePath();
    private NiFiProperties instance;
    private String keyHex;

    public static NiFiPropertiesLoader withKey(String keyHex) {
        NiFiPropertiesLoader loader = new NiFiPropertiesLoader();
        loader.setKeyHex(keyHex);
        return loader;
    }

    public void setKeyHex(String keyHex) {
        if (this.keyHex != null && !this.keyHex.trim().isEmpty()) {
            throw new RuntimeException("Cannot overwrite an existing key");
        }
        this.keyHex = keyHex;
    }

    public static NiFiProperties loadDefaultWithKeyFromBootstrap() {
        try {
            return new NiFiPropertiesLoader().loadDefault();
        }
        catch (Exception e) {
            logger.warn("Encountered an error naively loading the nifi.properties file because one or more properties are protected: {}", (Object)e.getLocalizedMessage());
            throw e;
        }
    }

    ProtectedNiFiProperties loadProtectedProperties(File file) {
        if (file == null || !file.exists() || !file.canRead()) {
            throw new IllegalArgumentException(String.format("Application Properties [%s] not found", file));
        }
        logger.info("Loading Application Properties [{}]", (Object)file);
        DuplicateDetectingProperties rawProperties = new DuplicateDetectingProperties();
        try (BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(file));){
            rawProperties.load(inputStream);
        }
        catch (Exception e) {
            throw new RuntimeException(String.format("Loading Application Properties [%s] failed", file), e);
        }
        if (!rawProperties.redundantKeySet().isEmpty()) {
            logger.warn("Duplicate property keys with the same value were detected in the properties file: {}", (Object)String.join((CharSequence)", ", rawProperties.redundantKeySet()));
        }
        if (!rawProperties.duplicateKeySet().isEmpty()) {
            throw new IllegalArgumentException("Duplicate property keys with different values were detected in the properties file: " + String.join((CharSequence)", ", rawProperties.duplicateKeySet()));
        }
        Properties properties = new Properties();
        Set<String> keys = rawProperties.stringPropertyNames();
        for (String key : keys) {
            String property = rawProperties.getProperty(key);
            properties.setProperty(key, property.trim());
        }
        return new ProtectedNiFiProperties(properties);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NiFiProperties load(File file) {
        NiFiProperties properties;
        ProtectedNiFiProperties protectedProperties = this.loadProtectedProperties(file);
        if (protectedProperties.hasProtectedKeys()) {
            ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
            try {
                PropertyProtectionURLClassLoader protectionClassLoader = new PropertyProtectionURLClassLoader(contextClassLoader);
                Thread.currentThread().setContextClassLoader((ClassLoader)protectionClassLoader);
                PropertyProviderFactoryLoader factoryLoader = new PropertyProviderFactoryLoader();
                SensitivePropertyProviderFactory providerFactory = factoryLoader.getPropertyProviderFactory();
                this.setBootstrapKey(providerFactory);
                providerFactory.getSupportedProviders().forEach(protectedProperties::addSensitivePropertyProvider);
                properties = protectedProperties.getUnprotectedProperties();
                providerFactory.getSupportedProviders().forEach(SensitivePropertyProvider::cleanUp);
            }
            finally {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        } else {
            properties = protectedProperties.getUnprotectedProperties();
        }
        return properties;
    }

    public NiFiProperties load(String path) {
        if (path != null && !path.trim().isEmpty()) {
            return this.load(new File(path));
        }
        return this.loadDefault();
    }

    public NiFiProperties get() {
        if (this.instance == null) {
            this.instance = this.getDefaultProperties();
        }
        return this.instance;
    }

    private NiFiProperties loadDefault() {
        return this.load(this.defaultPropertiesFilePath);
    }

    private NiFiProperties getDefaultProperties() {
        NiFiProperties defaultProperties = this.loadDefault();
        if (NiFiPropertiesLoader.isKeyGenerationRequired(defaultProperties)) {
            if (defaultProperties.isClustered()) {
                logger.error("Clustered Configuration Found: Shared Sensitive Properties Key [{}] required for cluster nodes", (Object)"nifi.sensitive.props.key");
                throw new SensitivePropertyProtectionException(PROPERTIES_KEY_MESSAGE);
            }
            File flowConfigurationFile = defaultProperties.getFlowConfigurationFile();
            if (flowConfigurationFile.exists()) {
                logger.error("Flow Configuration [{}] Found: Migration Required for blank Sensitive Properties Key [{}]", (Object)flowConfigurationFile, (Object)"nifi.sensitive.props.key");
                throw new SensitivePropertyProtectionException(PROPERTIES_KEY_MESSAGE);
            }
            this.setSensitivePropertiesKey();
            defaultProperties = this.loadDefault();
        }
        return defaultProperties;
    }

    private void setSensitivePropertiesKey() {
        logger.warn("Generating Random Sensitive Properties Key [{}]", (Object)"nifi.sensitive.props.key");
        SecureRandom secureRandom = new SecureRandom();
        byte[] sensitivePropertiesKeyBinary = new byte[24];
        secureRandom.nextBytes(sensitivePropertiesKeyBinary);
        String sensitivePropertiesKey = KEY_ENCODER.encodeToString(sensitivePropertiesKeyBinary);
        try {
            File niFiPropertiesFile = new File(this.defaultPropertiesFilePath);
            Path niFiPropertiesPath = Paths.get(niFiPropertiesFile.toURI());
            List<String> lines = Files.readAllLines(niFiPropertiesPath);
            List updatedLines = lines.stream().map(line -> {
                if (line.equals(EMPTY_SENSITIVE_PROPERTIES_KEY)) {
                    return line + sensitivePropertiesKey;
                }
                return line;
            }).collect(Collectors.toList());
            Files.write(niFiPropertiesPath, updatedLines, new OpenOption[0]);
            logger.info("NiFi Properties [{}] updated with Sensitive Properties Key", (Object)niFiPropertiesPath);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to set Sensitive Properties Key", e);
        }
    }

    private static boolean isKeyGenerationRequired(NiFiProperties properties) {
        String configuredSensitivePropertiesKey = properties.getProperty("nifi.sensitive.props.key");
        return configuredSensitivePropertiesKey == null || configuredSensitivePropertiesKey.length() == 0;
    }

    private void setBootstrapKey(SensitivePropertyProviderFactory providerFactory) {
        if (this.keyHex == null) {
            logger.debug("Bootstrap Sensitive Key not configured");
        } else {
            Class factoryClass = providerFactory.getClass();
            try {
                Method setMethod = factoryClass.getMethod(SET_KEY_METHOD, String.class);
                setMethod.invoke((Object)providerFactory, this.keyHex);
            }
            catch (NoSuchMethodException e) {
                logger.warn("Method [{}] on Class [{}] not found", (Object)SET_KEY_METHOD, (Object)factoryClass.getName());
            }
            catch (IllegalAccessException e) {
                logger.warn("Method [{}] on Class [{}] access not allowed", (Object)SET_KEY_METHOD, (Object)factoryClass.getName());
            }
            catch (InvocationTargetException e) {
                throw new SensitivePropertyProtectionException("Set Bootstrap Key on Provider Factory failed", (Throwable)e);
            }
        }
    }

    private static class DuplicateDetectingProperties
    extends Properties {
        private final Set<String> duplicateKeys = new HashSet<String>();
        private final Set<String> redundantKeys = new HashSet<String>();

        public Set<String> duplicateKeySet() {
            return this.duplicateKeys;
        }

        public Set<String> redundantKeySet() {
            return this.redundantKeys;
        }

        @Override
        public Object put(Object key, Object value) {
            Object existingValue = super.put(key, value);
            if (existingValue != null) {
                if (existingValue.toString().equals(value.toString())) {
                    this.redundantKeys.add(key.toString());
                    return existingValue;
                }
                this.duplicateKeys.add(key.toString());
            }
            return value;
        }
    }
}

