/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.registry.provider.flow.git;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.apache.nifi.registry.provider.flow.git.Bucket;
import org.apache.nifi.registry.provider.flow.git.Flow;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.LsRemoteCommand;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.api.errors.NoHeadException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.GpgConfig;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectStream;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryCache;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.jgit.util.FS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;

class GitFlowMetaData {
    static final int CURRENT_LAYOUT_VERSION = 1;
    static final String LAYOUT_VERSION = "layoutVer";
    static final String BUCKET_ID = "bucketId";
    static final String FLOWS = "flows";
    static final String VER = "ver";
    static final String FILE = "file";
    static final String FLOW_NAME = "flowName";
    static final String FLOW_DESC = "flowDesc";
    static final String AUTHOR = "author";
    static final String COMMENTS = "comments";
    static final String CREATED = "created";
    static final String BUCKET_FILENAME = "bucket.yml";
    private static final Logger logger = LoggerFactory.getLogger(GitFlowMetaData.class);
    private Repository gitRepo;
    private String remoteToPush;
    private CredentialsProvider credentialsProvider;
    private final BlockingQueue<Long> pushQueue = new ArrayBlockingQueue<Long>(1);
    private Map<String, Bucket> buckets = new HashMap<String, Bucket>();

    GitFlowMetaData() {
    }

    public void setRemoteToPush(String remoteToPush) {
        this.remoteToPush = remoteToPush;
    }

    public void setRemoteCredential(String userName, String password) {
        this.credentialsProvider = new UsernamePasswordCredentialsProvider(userName, password);
    }

    private Repository openRepository(File gitProjectRootDir) throws IOException {
        if (!gitProjectRootDir.isDirectory()) {
            throw new IOException(String.format("'%s' is not a directory or does not exist.", gitProjectRootDir));
        }
        if (!gitProjectRootDir.canRead() || !gitProjectRootDir.canWrite()) {
            throw new IOException(String.format("Directory '%s' does not have read/write privilege.", gitProjectRootDir));
        }
        FileRepositoryBuilder builder = (FileRepositoryBuilder)((FileRepositoryBuilder)((FileRepositoryBuilder)((FileRepositoryBuilder)new FileRepositoryBuilder().readEnvironment()).setMustExist(true)).addCeilingDirectory(gitProjectRootDir)).findGitDir(gitProjectRootDir);
        if (builder.getGitDir() == null) {
            throw new IOException(String.format("Directory '%s' does not contain a .git directory. Please init and configure the directory with 'git init' command before using it from NiFi Registry.", gitProjectRootDir));
        }
        return builder.build();
    }

    private static boolean hasAtLeastOneReference(Repository repo) throws IOException {
        logger.info("Checking references for repository {}", (Object)repo.toString());
        for (Ref ref : repo.getRefDatabase().getRefs()) {
            if (ref.getObjectId() == null) continue;
            return true;
        }
        return false;
    }

    public boolean localRepoExists(File localRepo) throws IOException {
        if (!localRepo.isDirectory()) {
            logger.info("{} is not a directory or does not exist.", (Object)localRepo.getPath());
            return false;
        }
        if (RepositoryCache.FileKey.isGitRepository((File)new File(localRepo.getPath() + "/.git"), (FS)FS.DETECTED)) {
            Git git = Git.open((File)new File(localRepo.getPath() + "/.git"));
            Repository repository = git.getRepository();
            logger.info("Checking for git references in {}", (Object)localRepo.getPath());
            boolean referenceExists = GitFlowMetaData.hasAtLeastOneReference(repository);
            if (referenceExists) {
                logger.info("{} local repository exists with references so no need to clone remote", (Object)localRepo.getPath());
            }
            return true;
        }
        return false;
    }

    public void remoteRepoExists(String remoteRepository) throws GitAPIException, IOException {
        Git git = new Git(FileRepositoryBuilder.create((File)new File(remoteRepository)));
        LsRemoteCommand lsCmd = git.lsRemote();
        lsCmd.setRemote(remoteRepository);
        lsCmd.setCredentialsProvider(this.credentialsProvider);
        lsCmd.call();
    }

    public void cloneRepository(File localRepo, String remoteRepository) throws GitAPIException {
        logger.info("Cloning the repository {} in {}", (Object)remoteRepository, (Object)localRepo.getPath());
        ((CloneCommand)Git.cloneRepository().setURI(remoteRepository).setCredentialsProvider(this.credentialsProvider)).setDirectory(localRepo).call();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void loadGitRepository(File gitProjectRootDir) throws IOException, GitAPIException {
        this.gitRepo = this.openRepository(gitProjectRootDir);
        try (Git git = new Git(this.gitRepo);){
            List remotes;
            boolean isRemoteExist2;
            if (!StringUtils.isEmpty((CharSequence)this.remoteToPush) && !(isRemoteExist2 = (remotes = git.remoteList().call()).stream().anyMatch(remote -> remote.getName().equals(this.remoteToPush)))) {
                List remoteNames = remotes.stream().map(RemoteConfig::getName).collect(Collectors.toList());
                throw new IllegalArgumentException(String.format("The configured remote '%s' to push does not exist. Available remotes are %s", this.remoteToPush, remoteNames));
            }
            boolean isLatestCommit = true;
            try {
                Iterator isRemoteExist2 = git.log().call().iterator();
                while (isRemoteExist2.hasNext()) {
                    HashMap<String, ObjectId> flowSnapshotObjectIds;
                    HashMap<String, ObjectId> bucketObjectIds;
                    TreeWalk treeWalk;
                    RevCommit commit;
                    block20: {
                        commit = (RevCommit)isRemoteExist2.next();
                        String shortCommitId = commit.getId().abbreviate(7).name();
                        logger.debug("Processing a commit: {}", (Object)shortCommitId);
                        RevTree tree = commit.getTree();
                        treeWalk = new TreeWalk(this.gitRepo);
                        treeWalk.addTree((AnyObjectId)tree);
                        bucketObjectIds = new HashMap<String, ObjectId>();
                        flowSnapshotObjectIds = new HashMap<String, ObjectId>();
                        while (treeWalk.next()) {
                            if (treeWalk.isSubtree()) {
                                treeWalk.enterSubtree();
                                continue;
                            }
                            String pathString = treeWalk.getPathString();
                            if (pathString.endsWith("/bucket.yml")) {
                                bucketObjectIds.put(pathString, treeWalk.getObjectId(0));
                                continue;
                            }
                            if (!pathString.endsWith(".snapshot")) continue;
                            flowSnapshotObjectIds.put(pathString, treeWalk.getObjectId(0));
                        }
                        if (!bucketObjectIds.isEmpty()) break block20;
                        logger.debug("Tree at commit {} does not contain any bucket.yml. Stop loading commits here.", (Object)shortCommitId);
                        treeWalk.close();
                        return;
                    }
                    try {
                        this.loadBuckets(this.gitRepo, commit, isLatestCommit, bucketObjectIds, flowSnapshotObjectIds);
                        isLatestCommit = false;
                    }
                    finally {
                        treeWalk.close();
                    }
                }
                return;
            }
            catch (NoHeadException e) {
                logger.debug("'{}' does not have any commit yet. Starting with empty buckets.", (Object)gitProjectRootDir);
                return;
            }
        }
    }

    void startPushThread() {
        if (StringUtils.isEmpty((CharSequence)this.remoteToPush)) {
            return;
        }
        BasicThreadFactory threadFactory = new BasicThreadFactory.Builder().daemon(true).namingPattern(this.getClass().getSimpleName() + " Push thread").build();
        ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor((ThreadFactory)threadFactory);
        executorService.scheduleWithFixedDelay(() -> {
            Long offeredTimestamp;
            try {
                offeredTimestamp = this.pushQueue.take();
            }
            catch (InterruptedException e) {
                logger.warn("Waiting for push request has been interrupted due to {}", (Object)e.getMessage(), (Object)e);
                return;
            }
            logger.debug("Took a push request sent at {} to {}...", (Object)offeredTimestamp, (Object)this.remoteToPush);
            PushCommand pushCommand = new Git(this.gitRepo).push().setRemote(this.remoteToPush);
            if (this.credentialsProvider != null) {
                pushCommand.setCredentialsProvider(this.credentialsProvider);
            }
            try {
                Iterable pushResults = pushCommand.call();
                for (PushResult pushResult : pushResults) {
                    logger.debug(pushResult.getMessages());
                }
            }
            catch (GitAPIException e) {
                logger.error(String.format("Failed to push commits to %s due to %s", new Object[]{this.remoteToPush, e}), (Throwable)e);
            }
        }, 10L, 10L, TimeUnit.SECONDS);
    }

    private void loadBuckets(Repository gitRepo, RevCommit commit, boolean isLatestCommit, Map<String, ObjectId> bucketObjectIds, Map<String, ObjectId> flowSnapshotObjectIds) throws IOException {
        Yaml yaml = new Yaml();
        for (String bucketFilePath : bucketObjectIds.keySet()) {
            Bucket bucket;
            Map bucketMeta;
            ObjectId bucketObjectId = bucketObjectIds.get(bucketFilePath);
            try (ObjectStream bucketIn = gitRepo.newObjectReader().open((AnyObjectId)bucketObjectId).openStream();){
                bucketMeta = (Map)yaml.load((InputStream)bucketIn);
            }
            if (!this.validateRequiredValue(bucketMeta, bucketFilePath, LAYOUT_VERSION, BUCKET_ID, FLOWS)) continue;
            int layoutVersion = (Integer)bucketMeta.get(LAYOUT_VERSION);
            if (layoutVersion > 1) {
                logger.warn("{} has unsupported {} {}. This Registry can only support {} or lower. Skipping it.", new Object[]{bucketFilePath, LAYOUT_VERSION, layoutVersion, 1});
                continue;
            }
            String bucketId = (String)bucketMeta.get(BUCKET_ID);
            if (isLatestCommit) {
                bucket = this.getBucketOrCreate(bucketId);
            } else {
                Optional<Bucket> bucketOpt = this.getBucket(bucketId);
                if (bucketOpt.isPresent()) {
                    bucket = bucketOpt.get();
                } else {
                    logger.debug("Bucket {} does not exist any longer. It may have been deleted.", (Object)bucketId);
                    continue;
                }
            }
            String bucketDirName = bucketFilePath.substring(0, bucketFilePath.lastIndexOf("/"));
            if (StringUtils.isEmpty((CharSequence)bucket.getBucketDirName())) {
                bucket.setBucketDirName(bucketDirName);
            }
            Map flows = (Map)bucketMeta.get(FLOWS);
            this.loadFlows(commit, isLatestCommit, bucket, bucketFilePath, flows, flowSnapshotObjectIds);
        }
    }

    private void loadFlows(RevCommit commit, boolean isLatestCommit, Bucket bucket, String backetFilePath, Map<String, Object> flows, Map<String, ObjectId> flowSnapshotObjectIds) {
        for (String flowId : flows.keySet()) {
            Flow flow;
            Map flowMeta = (Map)flows.get(flowId);
            if (!this.validateRequiredValue(flowMeta, backetFilePath + ":" + flowId, VER, FILE)) continue;
            if (isLatestCommit) {
                flow = bucket.getFlowOrCreate(flowId);
            } else {
                Optional<Flow> flowOpt = bucket.getFlow(flowId);
                if (flowOpt.isPresent()) {
                    flow = flowOpt.get();
                } else {
                    logger.debug("Flow {} does not exist in bucket {}:{} any longer. It may have been deleted.", new Object[]{flowId, bucket.getBucketDirName(), bucket.getBucketId()});
                    continue;
                }
            }
            int version = (Integer)flowMeta.get(VER);
            String flowSnapshotFilename = (String)flowMeta.get(FILE);
            if (flow.hasVersion(version)) continue;
            Flow.FlowPointer pointer = new Flow.FlowPointer(flowSnapshotFilename);
            File flowSnapshotFile = new File(new File(backetFilePath).getParent(), flowSnapshotFilename);
            ObjectId objectId = flowSnapshotObjectIds.get(flowSnapshotFile.getPath());
            if (objectId == null) {
                logger.warn("Git object id for Flow {} version {} with path {} in bucket {}:{} was not found. Ignoring this entry.", new Object[]{flowId, version, flowSnapshotFile.getPath(), bucket.getBucketDirName(), bucket.getBucketId()});
                continue;
            }
            pointer.setGitRev(commit.getName());
            pointer.setObjectId(objectId.getName());
            if (flowMeta.containsKey(FLOW_NAME)) {
                pointer.setFlowName((String)flowMeta.get(FLOW_NAME));
            }
            if (flowMeta.containsKey(FLOW_DESC)) {
                pointer.setFlowDescription((String)flowMeta.get(FLOW_DESC));
            }
            if (flowMeta.containsKey(AUTHOR)) {
                pointer.setAuthor((String)flowMeta.get(AUTHOR));
            }
            if (flowMeta.containsKey(COMMENTS)) {
                pointer.setComment((String)flowMeta.get(COMMENTS));
            }
            if (flowMeta.containsKey(CREATED)) {
                pointer.setCreated((long)((Long)flowMeta.get(CREATED)));
            }
            flow.putVersion(version, pointer);
        }
    }

    private boolean validateRequiredValue(Map<String, Object> map, String nameOfMap, Object ... keys) {
        for (Object key : keys) {
            if (map.containsKey(key)) continue;
            logger.warn("{} does not have {}. Skipping it.", (Object)nameOfMap, key);
            return false;
        }
        return true;
    }

    public Bucket getBucketOrCreate(String bucketId) {
        return this.buckets.computeIfAbsent(bucketId, k -> new Bucket(bucketId));
    }

    public Optional<Bucket> getBucket(String bucketId) {
        return Optional.ofNullable(this.buckets.get(bucketId));
    }

    Map<String, Bucket> getBuckets() {
        return this.buckets;
    }

    void saveBucket(Bucket bucket, File bucketDir) throws IOException {
        Yaml yaml = new Yaml();
        Map<String, Object> serializedBucket = bucket.serialize();
        File bucketFile = new File(bucketDir, BUCKET_FILENAME);
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(bucketFile), StandardCharsets.UTF_8);){
            yaml.dump(serializedBucket, (Writer)writer);
        }
    }

    boolean isGitDirectoryClean() throws GitAPIException {
        Status status = new Git(this.gitRepo).status().call();
        return status.isClean() && !status.hasUncommittedChanges();
    }

    void commit(String author, String message, Bucket bucket, Flow.FlowPointer flowPointer) throws GitAPIException, IOException {
        try (Git git = new Git(this.gitRepo);){
            long offeredTimestamp;
            git.add().addFilepattern(".").call();
            git.add().addFilepattern(".").setUpdate(true).call();
            String commitMessage = StringUtils.isEmpty((CharSequence)author) ? message : String.format("%s\n\nBy NiFi Registry user: %s", message, author);
            Config unusedConfig = new Config();
            unusedConfig.setEnum("gpg", null, "format", (Enum)GpgConfig.GpgFormat.OPENPGP);
            RevCommit commit = git.commit().setMessage(commitMessage).setSign(Boolean.valueOf(false)).setGpgConfig(new GpgConfig(unusedConfig)).call();
            if (flowPointer != null) {
                RevTree tree = commit.getTree();
                String bucketDirName = bucket.getBucketDirName();
                String flowSnapshotPath = new File(bucketDirName, flowPointer.getFileName()).getPath();
                try (TreeWalk treeWalk = new TreeWalk(this.gitRepo);){
                    treeWalk.addTree((AnyObjectId)tree);
                    while (treeWalk.next()) {
                        if (treeWalk.isSubtree()) {
                            treeWalk.enterSubtree();
                            continue;
                        }
                        String pathString = treeWalk.getPathString();
                        if (!pathString.equals(flowSnapshotPath)) continue;
                        String flowSnapshotObjectId = treeWalk.getObjectId(0).getName();
                        flowPointer.setObjectId(flowSnapshotObjectId);
                        break;
                    }
                }
                flowPointer.setGitRev(commit.getName());
            }
            if (!StringUtils.isEmpty((CharSequence)this.remoteToPush) && this.pushQueue.offer(offeredTimestamp = System.currentTimeMillis())) {
                logger.debug("New push request is offered at {}.", (Object)offeredTimestamp);
            }
        }
    }

    byte[] getContent(String objectId) throws IOException {
        ObjectId flowSnapshotObjectId = this.gitRepo.resolve(objectId);
        ObjectStream objStream = this.gitRepo.newObjectReader().open((AnyObjectId)flowSnapshotObjectId).openStream();
        return IOUtils.toByteArray((InputStream)objStream);
    }
}

