/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.registry;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.management.LockInfo;
import java.lang.management.ManagementFactory;
import java.lang.management.MonitorInfo;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.nifi.registry.NiFiRegistry;
import org.apache.nifi.registry.util.LimitingInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BootstrapListener {
    private static final Logger logger = LoggerFactory.getLogger(BootstrapListener.class);
    private final NiFiRegistry nifi;
    private final int bootstrapPort;
    private final String secretKey;
    private volatile Listener listener;
    private volatile ServerSocket serverSocket;

    public BootstrapListener(NiFiRegistry nifi, int bootstrapPort) {
        this.nifi = nifi;
        this.bootstrapPort = bootstrapPort;
        this.secretKey = UUID.randomUUID().toString();
    }

    public void start() throws IOException {
        logger.debug("Starting Bootstrap Listener to communicate with Bootstrap Port {}", (Object)this.bootstrapPort);
        this.serverSocket = new ServerSocket();
        this.serverSocket.bind(new InetSocketAddress("localhost", 0));
        this.serverSocket.setSoTimeout(2000);
        int localPort = this.serverSocket.getLocalPort();
        logger.info("Started Bootstrap Listener, Listening for incoming requests on port {}", (Object)localPort);
        this.listener = new Listener(this.serverSocket);
        Thread listenThread = new Thread(this.listener);
        listenThread.setDaemon(true);
        listenThread.setName("Listen to Bootstrap");
        listenThread.start();
        logger.debug("Notifying Bootstrap that local port is {}", (Object)localPort);
        this.sendCommand("PORT", new String[]{String.valueOf(localPort), this.secretKey});
    }

    public void stop() {
        if (this.listener != null) {
            this.listener.stop();
        }
    }

    public void sendStartedStatus(boolean status) throws IOException {
        logger.debug("Notifying Bootstrap that the status of starting NiFi Registry is {}", (Object)status);
        this.sendCommand("STARTED", new String[]{String.valueOf(status)});
    }

    private void sendCommand(String command, String[] args) throws IOException {
        try (Socket socket = new Socket();){
            socket.setSoTimeout(60000);
            socket.connect(new InetSocketAddress("localhost", this.bootstrapPort));
            socket.setSoTimeout(60000);
            StringBuilder commandBuilder = new StringBuilder(command);
            for (String arg : args) {
                commandBuilder.append(" ").append(arg);
            }
            commandBuilder.append("\n");
            String commandWithArgs = commandBuilder.toString();
            logger.debug("Sending command to Bootstrap: " + commandWithArgs);
            OutputStream out = socket.getOutputStream();
            out.write(commandWithArgs.getBytes(StandardCharsets.UTF_8));
            out.flush();
            logger.debug("Awaiting response from Bootstrap...");
            BufferedReader reader = new BufferedReader(new InputStreamReader(socket.getInputStream()));
            String response = reader.readLine();
            if ("OK".equals(response)) {
                logger.info("Successfully initiated communication with Bootstrap");
            } else {
                logger.error("Failed to communicate with Bootstrap. Bootstrap may be unable to issue or receive commands from NiFi Registry ");
            }
        }
    }

    private static void writeDump(OutputStream out) throws IOException {
        ThreadMXBean mbean = ManagementFactory.getThreadMXBean();
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out));
        ThreadInfo[] infos = mbean.dumpAllThreads(true, true);
        long[] deadlockedThreadIds = mbean.findDeadlockedThreads();
        long[] monitorDeadlockThreadIds = mbean.findMonitorDeadlockedThreads();
        ArrayList<ThreadInfo> sortedInfos = new ArrayList<ThreadInfo>(infos.length);
        for (ThreadInfo info : infos) {
            sortedInfos.add(info);
        }
        Collections.sort(sortedInfos, new Comparator<ThreadInfo>(){

            @Override
            public int compare(ThreadInfo o1, ThreadInfo o2) {
                return o1.getThreadName().toLowerCase().compareTo(o2.getThreadName().toLowerCase());
            }
        });
        StringBuilder sb = new StringBuilder();
        for (ThreadInfo info : sortedInfos) {
            StackTraceElement[] stackTraces;
            sb.append("\n");
            sb.append("\"").append(info.getThreadName()).append("\" Id=");
            sb.append(info.getThreadId()).append(" ");
            sb.append(info.getThreadState().toString()).append(" ");
            switch (info.getThreadState()) {
                case BLOCKED: 
                case TIMED_WAITING: 
                case WAITING: {
                    sb.append(" on ");
                    sb.append(info.getLockInfo());
                    break;
                }
            }
            if (info.isSuspended()) {
                sb.append(" (suspended)");
            }
            if (info.isInNative()) {
                sb.append(" (in native code)");
            }
            if (deadlockedThreadIds != null && deadlockedThreadIds.length > 0) {
                for (Object id : (ThreadInfo)deadlockedThreadIds) {
                    if (id != info.getThreadId()) continue;
                    sb.append(" ** DEADLOCKED THREAD **");
                }
            }
            if (monitorDeadlockThreadIds != null && monitorDeadlockThreadIds.length > 0) {
                for (Object id : (ThreadInfo)monitorDeadlockThreadIds) {
                    if (id != info.getThreadId()) continue;
                    sb.append(" ** MONITOR-DEADLOCKED THREAD **");
                }
            }
            for (StackTraceElement element : stackTraces = info.getStackTrace()) {
                MonitorInfo[] monitors;
                sb.append("\n\tat ").append(element);
                for (MonitorInfo monitor : monitors = info.getLockedMonitors()) {
                    if (!monitor.getLockedStackFrame().equals(element)) continue;
                    sb.append("\n\t- waiting on ").append(monitor);
                }
            }
            LockInfo[] lockInfos = info.getLockedSynchronizers();
            if (lockInfos.length > 0) {
                sb.append("\n\t");
                sb.append("Number of Locked Synchronizers: ").append(lockInfos.length);
                for (LockInfo lockInfo : lockInfos) {
                    sb.append("\n\t- ").append(lockInfo.toString());
                }
            }
            sb.append("\n");
        }
        if (deadlockedThreadIds != null && deadlockedThreadIds.length > 0) {
            sb.append("\n\nDEADLOCK DETECTED!");
            sb.append("\nThe following thread IDs are deadlocked:");
            for (Object id : (Object)deadlockedThreadIds) {
                sb.append("\n").append((long)id);
            }
        }
        if (monitorDeadlockThreadIds != null && monitorDeadlockThreadIds.length > 0) {
            sb.append("\n\nMONITOR DEADLOCK DETECTED!");
            sb.append("\nThe following thread IDs are deadlocked:");
            for (Object id : (Object)monitorDeadlockThreadIds) {
                sb.append("\n").append((long)id);
            }
        }
        writer.write(sb.toString());
        writer.flush();
    }

    private void echoPing(OutputStream out) throws IOException {
        out.write("PING\n".getBytes(StandardCharsets.UTF_8));
        out.flush();
    }

    private void echoShutdown(OutputStream out) throws IOException {
        out.write("SHUTDOWN\n".getBytes(StandardCharsets.UTF_8));
        out.flush();
    }

    private BootstrapRequest readRequest(InputStream in) throws IOException {
        LimitingInputStream limitingIn = new LimitingInputStream(in, 4096L);
        BufferedReader reader = new BufferedReader(new InputStreamReader(limitingIn));
        String line = reader.readLine();
        String[] splits = line.split(" ");
        if (splits.length < 1) {
            throw new IOException("Received invalid request from Bootstrap: " + line);
        }
        String requestType = splits[0];
        if (splits.length == 1) {
            throw new IOException("Received invalid request from Bootstrap; request did not have a secret key; request type = " + requestType);
        }
        String[] args = splits.length == 2 ? new String[]{} : Arrays.copyOfRange(splits, 2, splits.length);
        String requestKey = splits[1];
        if (!this.secretKey.equals(requestKey)) {
            throw new IOException("Received invalid Secret Key for request type " + requestType);
        }
        try {
            return new BootstrapRequest(requestType, args);
        }
        catch (Exception e) {
            throw new IOException("Received invalid request from Bootstrap; request type = " + requestType);
        }
    }

    private class Listener
    implements Runnable {
        private final ServerSocket serverSocket;
        private final ExecutorService executor;
        private volatile boolean stopped = false;

        public Listener(ServerSocket serverSocket) {
            this.serverSocket = serverSocket;
            this.executor = Executors.newFixedThreadPool(2);
        }

        public void stop() {
            this.stopped = true;
            this.executor.shutdownNow();
            try {
                this.serverSocket.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }

        @Override
        public void run() {
            while (!this.stopped) {
                try {
                    Socket socket;
                    try {
                        logger.debug("Listening for Bootstrap Requests");
                        socket = this.serverSocket.accept();
                    }
                    catch (SocketTimeoutException ste) {
                        if (!this.stopped) continue;
                        return;
                    }
                    catch (IOException ioe) {
                        if (this.stopped) {
                            return;
                        }
                        throw ioe;
                    }
                    logger.debug("Received connection from Bootstrap");
                    socket.setSoTimeout(5000);
                    this.executor.submit(new Runnable(){

                        /*
                         * WARNING - Removed try catching itself - possible behaviour change.
                         * Enabled aggressive block sorting
                         * Enabled unnecessary exception pruning
                         * Enabled aggressive exception aggregation
                         */
                        @Override
                        public void run() {
                            try {
                                BootstrapRequest request = BootstrapListener.this.readRequest(socket.getInputStream());
                                BootstrapRequest.RequestType requestType = request.getRequestType();
                                switch (requestType.ordinal()) {
                                    case 2: {
                                        logger.debug("Received PING request from Bootstrap; responding");
                                        BootstrapListener.this.echoPing(socket.getOutputStream());
                                        logger.debug("Responded to PING request from Bootstrap");
                                        return;
                                    }
                                    case 0: {
                                        logger.info("Received SHUTDOWN request from Bootstrap");
                                        BootstrapListener.this.echoShutdown(socket.getOutputStream());
                                        BootstrapListener.this.nifi.shutdownHook();
                                        return;
                                    }
                                    case 1: {
                                        logger.info("Received DUMP request from Bootstrap");
                                        BootstrapListener.writeDump(socket.getOutputStream());
                                        return;
                                    }
                                }
                                return;
                            }
                            catch (Throwable t) {
                                logger.error("Failed to process request from Bootstrap due to " + t.toString(), t);
                                return;
                            }
                            finally {
                                try {
                                    socket.close();
                                }
                                catch (IOException ioe) {
                                    logger.warn("Failed to close socket to Bootstrap due to {}", (Object)ioe.toString());
                                }
                            }
                        }
                    });
                }
                catch (Throwable t) {
                    logger.error("Failed to process request from Bootstrap due to " + t.toString(), t);
                }
            }
        }
    }

    private static class BootstrapRequest {
        private final RequestType requestType;
        private final String[] args;

        public BootstrapRequest(String request, String[] args) {
            this.requestType = RequestType.valueOf(request);
            this.args = args;
        }

        public RequestType getRequestType() {
            return this.requestType;
        }

        public String[] getArgs() {
            return this.args;
        }

        public static enum RequestType {
            SHUTDOWN,
            DUMP,
            PING;

        }
    }
}

