/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.nifi.registry.flow.diff;

public enum DifferenceType {
    /**
     * The component does not exist in Flow A but exists in Flow B
     */
    COMPONENT_ADDED("Component Added"),

    /**
     * The component exists in Flow A but does not exist in Flow B
     */
    COMPONENT_REMOVED("Component Removed"),

    /**
     * The component has a different name in each of the flows
     */
    NAME_CHANGED("Component Name Changed"),

    /**
     * The component has a different type in each of the flows
     */
    TYPE_CHANGED("Component Type Changed"),

    /**
     * The component has a different bundle in each of the flows
     */
    BUNDLE_CHANGED("Component Bundle Changed"),

    /**
     * The component has a different penalty duration in each of the flows
     */
    PENALTY_DURATION_CHANGED("Penalty Duration Changed"),

    /**
     * The component has a different yield duration in each of the flows
     */
    YIELD_DURATION_CHANGED("Yield Duration Changed"),

    /**
     * The component has a different retry count in each of the flows
     */
    RETRY_COUNT_CHANGED("Retry Count Changed"),

    /**
     * The component has a different retried relationship in each of the flows
     */
    RETRIED_RELATIONSHIPS_CHANGED("Retried Relationships Changed"),

    /**
     * The component has a different backoff mechanism in each of the flows
     */
    BACKOFF_MECHANISM_CHANGED("Backoff Mechanism Changed"),

    /**
     * The component has a different max backoff period in each of the flows
     */
    MAX_BACKOFF_PERIOD_CHANGED("Max Backoff Period Changed"),

    /**
     * The component has a different bulletin level in each of the flows
     */
    BULLETIN_LEVEL_CHANGED("Bulletin Level Changed"),

    /**
     * The component has a different set of Auto-Terminated Relationships in each of the flows
     */
    AUTO_TERMINATED_RELATIONSHIPS_CHANGED("Auto-Terminated Relationships Changed"),

    /**
     * The component has a different scheduling strategy in each of the flows
     */
    SCHEDULING_STRATEGY_CHANGED("Scheduling Strategy Changed"),

    /**
     * The component has a different maximum number of concurrent tasks in each of the flows
     */
    CONCURRENT_TASKS_CHANGED("Concurrent Tasks Changed"),

    /**
     * The component has a different enforcement policy in each of the flows
     */
    ENFORCEMENT_POLICY_CHANGED("Enforcement Policy Changed"),

    /**
     * The component has a different value for the timeout in each of the flows
     */
    TIMEOUT_CHANGED("Timeout Changed"),

    /**
     * The component has a different run schedule in each of the flows
     */
    RUN_SCHEDULE_CHANGED("Run Schedule Changed"),

    /**
     * The component has a different scheduled state (enabled/disabled) in each of the flows
     */
    SCHEDULED_STATE_CHANGED("Scheduled State Changed"),

    /**
     * The component has a different execution mode in each of the flows
     */
    EXECUTION_MODE_CHANGED("Execution Mode Changed"),

    /**
     * The component has a different run duration in each of the flows
     */
    RUN_DURATION_CHANGED("Run Duration Changed"),

    /**
     * The component has a different Port Function in each of the flows
     */
    PORT_FUNCTION_CHANGED("Port Function Changed"),

    /**
     * The component has a different value in each of the flows for a specific property
     */
    PROPERTY_CHANGED("Property Value Changed"),

    /**
     * Property does not exist in Flow A but does exist in Flow B
     */
    PROPERTY_ADDED("Property Added"),

    /**
     * Property exists in Flow A but does not exist in Flow B
     */
    PROPERTY_REMOVED("Property Removed"),

    /**
     * The sensitive status of the property changed
     */
    PROPERTY_SENSITIVITY_CHANGED("Property Sensitivity Changed"),

    /**
     * Property is unset or set to an explicit value in Flow A but set to (exactly) a parameter reference in Flow B. Note that if Flow A
     * has a property set to "#{param1} abc" and it is changed to "#{param1} abc #{param2}" this would indicate a Difference Type of @{link #PROPERTY_CHANGED}, not
     * PROPERTY_PARAMETERIZED
     */
    PROPERTY_PARAMETERIZED("Property Parameterized"),

    /**
     * Property is set to (exactly) a parameter reference in Flow A but either unset or set to an explicit value in Flow B.
     */
    PROPERTY_PARAMETERIZATION_REMOVED("Property Parameterization Removed"),

    /**
     * Parameter exists for a given Parameter Context in Flow B but does not exist in Flow A
     */
    PARAMETER_ADDED("Parameter Added"),

    /**
     * Parameter exists for a given Parameter Context in Flow A but does not exist in Flow B
     */
    PARAMETER_REMOVED("Parameter Removed"),

    /**
     * The value of the Parameter is different in each of the flows
     */
    PARAMETER_VALUE_CHANGED("Parameter Value Changed"),

    /**
     * The description of the Parameter is different in each of the flows
     */
    PARAMETER_DESCRIPTION_CHANGED("Parameter Description Changed"),

    /**
     * The Parameter Context bound to a Process Group is different in each of the flows
     */
    PARAMETER_CONTEXT_CHANGED("Parameter Context Changed"),

    /**
     * The Parameter Context inherits from a different collection of Parameter Contexts in each of the flows
     */
    INHERITED_CONTEXTS_CHANGED("Inherited Parameter Contexts Changed"),

    /**
     * The Parameter Context references a different Parameter Group Name from a Parameter Provider in each of the flows
     */
    PARAMETER_GROUP_NAME_CHANGED("Mapped Parameter Group Name Changed"),

    /**
     * The Parameter Context synchronization to fetched Parameter Groups is different in each of the flows
     */
    PARAMETER_PROVIDER_SYNCHRONIZED_CHANGED("Parameter Context Synchronization Changed"),

    /**
     * The component has a different value for the Annotation Data in each of the flows
     */
    ANNOTATION_DATA_CHANGED("Annotation Data (Advanced UI Configuration) Changed"),

    /**
     * The component has a different comment in each of the flows
     */
    COMMENTS_CHANGED("Comments Changed"),

    /**
     * The component has a different description in each of the flows
     */
    DESCRIPTION_CHANGED("Description Changed"),

    /**
     * The position of the component on the graph is different in each of the flows
     */
    POSITION_CHANGED("Position Changed"),

    /**
     * The height and/or width size of the component is different in each of the flows
     */
    SIZE_CHANGED("Size Changed"),

    /**
     * The stylistic configuration of the component is different in each of the flows
     */
    STYLE_CHANGED("Style Changed"),

    /**
     * The Relationships included in a connection is different in each of the flows
     */
    SELECTED_RELATIONSHIPS_CHANGED("Selected Relationships Changed"),

    /**
     * The Connection has a different set of Prioritizers in each of the flows
     */
    PRIORITIZERS_CHANGED("Prioritizers Changed"),

    /**
     * The Connection has a different value for the FlowFile Expiration in each of the flows
     */
    FLOWFILE_EXPIRATION_CHANGED("FlowFile Expiration Changed"),

    /**
     * The Connection has a different value for the Object Backpressure Threshold in each of the flows
     */
    BACKPRESSURE_OBJECT_THRESHOLD_CHANGED("Backpressure Object Threshold Changed"),

    /**
     * The Connection has a different value for the Data Size Backpressure Threshold in each of the flows
     */
    BACKPRESSURE_DATA_SIZE_THRESHOLD_CHANGED("Backpressure Data Size Threshold Changed"),

    /**
     * The Connection has a different value for the Load Balance Strategy in each of the flows
     */
    LOAD_BALANCE_STRATEGY_CHANGED("Load-Balance Strategy Changed"),

    /**
     * The Connection has a different value for the Partitioning Attribute in each of the flows
     */
    PARTITIONING_ATTRIBUTE_CHANGED("Partitioning Attribute Changed"),

    /**
     * The Connection has a different value for the Load Balancing Compression in each of the flows
     */
    LOAD_BALANCE_COMPRESSION_CHANGED("Load-Balance Compression Changed"),

    /**
     * The Connection has a different set of Bend Points in each of the flows
     */
    BENDPOINTS_CHANGED("Connection Bend Points Changed"),

    /**
     * The Connection has a difference Source in each of the flows
     */
    SOURCE_CHANGED("Connection Source Changed"),

    /**
     * The Connection has a difference Destination in each of the flows
     */
    DESTINATION_CHANGED("Connection Destination Changed"),

    /**
     * The value in the Label is different in each of the flows
     */
    LABEL_VALUE_CHANGED("Label Text Changed"),

    /**
     * The zindex value is different in each of the components
     */
    ZINDEX_CHANGED("z-index Changed"),

    /**
     * The API of the Controller Service is different in each of the flows
     */
    SERVICE_API_CHANGED("Controller Service API Changed"),

    /**
     * The Remote Process Group has a different Transport Protocol in each of the flows
     */
    RPG_TRANSPORT_PROTOCOL_CHANGED("Remote Process Group Transport Protocol Changed"),

    /**
     * The Remote Process Group has a different Proxy Host in each of the flows
     */
    RPG_PROXY_HOST_CHANGED("Remote Process Group Proxy Host Changed"),

    /**
     * The Remote Process Group has a different Proxy Port in each of the flows
     */
    RPG_PROXY_PORT_CHANGED("Remote Process Group Proxy Port Changed"),

    /**
     * The Remote Process Group has a different Proxy User in each of the flows
     */
    RPG_PROXY_USER_CHANGED("Remote Process Group Proxy User Changed"),

    /**
     * The Remote Process Group has a different Network Interface chosen in each of the flows
     */
    RPG_NETWORK_INTERFACE_CHANGED("Remote Process Group Network Interface Changed"),

    /**
     * The Remote Process Group has a different Communications Timeout in each of the flows
     */
    RPG_COMMS_TIMEOUT_CHANGED("Remote Process Group Communications Timeout Changed"),

    /**
     * The Remote Process Group has a different URL in each of the flows
     */
    RPG_URL_CHANGED("Remote Process Group URL Changed"),

    /**
     * The Remote Input Port or Remote Output Port has a different Batch Size in each of the flows
     */
    REMOTE_PORT_BATCH_SIZE_CHANGED("Remote Process Group Port's Batch Size Changed"),

    /**
     * The Remote Input Port or Remote Output Port has a different value for the Compression flag in each of the flows
     */
    REMOTE_PORT_COMPRESSION_CHANGED("Remote Process Group Port's Compression Flag Changed"),

    /**
     * The Process Group points to a different Versioned Flow in each of the flows
     */
    VERSIONED_FLOW_COORDINATES_CHANGED("Versioned Flow Coordinates Changed"),

    /**
     * The Process Group's configured FlowFile Concurrency is different in each of the flows
     */
    FLOWFILE_CONCURRENCY_CHANGED("FlowFile Concurrency Changed"),

    /**
     * The Process Group's configured FlowFile Outbound Policy is different in each of the flows
     */
    FLOWFILE_OUTBOUND_POLICY_CHANGED("FlowFile Outbound Policy Changed"),

    /**
     * The Process Group's Default value for connections' Backpressure Data Size Threshold is different in each of the flows
     */
    DEFAULT_BACKPRESSURE_DATA_SIZE_CHANGED("Default Back Pressure Data Size Changed"),

    /**
     * The Process Group's Default value for connections' Backpressure Object Threshold is different in each of the flows
     */
    DEFAULT_BACKPRESSURE_OBJECT_COUNT_CHANGED("Default Back Pressure Object Count Changed"),

    /**
     * The Process Group's Default value for connections' FlowFile Expiration is different in each of the flows
     */
    DEFAULT_FLOWFILE_EXPIRATION_CHANGED("Default FlowFile Expiration Changed"),

    /**
     * The Process Group's Log File Suffix value is different in each of the flows
     */
    LOG_FILE_SUFFIX_CHANGED("Log File Suffix Changed"),

    /**
     * The Process Group's Execution Engine is different in each of the flows
     */
    EXECUTION_ENGINE_CHANGED("Execution Engine Changed");


    private final String description;

    DifferenceType(final String description) {
        this.description = description;
    }

    public String getDescription() {
        return description;
    }
}
