/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.extensions;

import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.stream.Collectors;
import org.apache.nifi.bundle.Bundle;
import org.apache.nifi.bundle.BundleCoordinate;
import org.apache.nifi.extensions.ExtensionClient;
import org.apache.nifi.extensions.exception.BundleNotFoundException;
import org.apache.nifi.nar.ExtensionManager;
import org.apache.nifi.nar.NarManifestEntry;
import org.apache.nifi.stream.io.StreamUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DownloadQueue {
    private static final Logger logger = LoggerFactory.getLogger(DownloadQueue.class);
    private static final Lock fileRenameLock = new ReentrantLock();
    private final ExtensionManager extensionManager;
    private final ExecutorService executorService;
    private final int concurrentDownloads;
    private final File narLibDirectory;
    private final List<ExtensionClient> clients;
    private final BlockingQueue<BundleCoordinate> toDownload = new LinkedBlockingQueue<BundleCoordinate>();
    private final Set<BundleCoordinate> allDownloads = Collections.synchronizedSet(new HashSet());

    public DownloadQueue(ExtensionManager extensionManager, ExecutorService executorService, int concurrentDownloads, Collection<BundleCoordinate> bundles, File narLibDirectory, List<ExtensionClient> clients) {
        this.extensionManager = extensionManager;
        this.executorService = executorService;
        this.concurrentDownloads = concurrentDownloads;
        this.narLibDirectory = narLibDirectory;
        this.clients = clients;
        if (!narLibDirectory.exists()) {
            boolean created;
            boolean bl = created = narLibDirectory.mkdirs() || narLibDirectory.exists();
            if (!created) {
                logger.error("Extensions directory {} did not exist and could not be created.", (Object)narLibDirectory.getAbsolutePath());
            }
        }
        this.toDownload.addAll(bundles);
        this.allDownloads.addAll(bundles);
    }

    public CompletableFuture<Void> download() {
        CompletableFuture[] futures = new CompletableFuture[this.concurrentDownloads];
        for (int i = 0; i < this.concurrentDownloads; ++i) {
            CompletableFuture<Void> completableFuture = new CompletableFuture<Void>();
            this.executorService.submit(new DownloadTask(this.toDownload, completableFuture, this.allDownloads));
            futures[i] = completableFuture;
        }
        return CompletableFuture.allOf(futures);
    }

    public Set<BundleCoordinate> getDownloadedCoordinates() {
        return this.allDownloads.stream().filter(this::isDownloadable).collect(Collectors.toSet());
    }

    public Set<File> getDownloadedFiles() {
        return this.allDownloads.stream().filter(this::isDownloadable).map(this::getBundleFile).collect(Collectors.toSet());
    }

    private boolean isDownloadable(BundleCoordinate coordinate) {
        return !"nifi-jetty-bundle".equals(coordinate.getId()) && !"nifi-framework-nar".equals(coordinate.getId());
    }

    private File getBundleFile(BundleCoordinate coordinate) {
        String filename = coordinate.getId() + "-" + coordinate.getVersion() + ".nar";
        return new File(this.narLibDirectory, filename);
    }

    private class DownloadTask
    implements Runnable {
        private final BlockingQueue<BundleCoordinate> downloadQueue;
        private final CompletableFuture<Void> completableFuture;
        private final Set<BundleCoordinate> downloads;

        public DownloadTask(BlockingQueue<BundleCoordinate> downloadQueue2, CompletableFuture<Void> completableFuture, Set<BundleCoordinate> filesDownloaded) {
            this.downloadQueue = downloadQueue2;
            this.completableFuture = completableFuture;
            this.downloads = filesDownloaded;
        }

        @Override
        public void run() {
            BundleCoordinate coordinate;
            while ((coordinate = (BundleCoordinate)this.downloadQueue.poll()) != null) {
                try {
                    this.downloadBundleAndParents(coordinate);
                }
                catch (Exception e) {
                    logger.error("Failed to download {}", (Object)coordinate, (Object)e);
                    this.completableFuture.completeExceptionally(e);
                }
            }
            this.completableFuture.complete(null);
        }

        private void downloadBundleAndParents(BundleCoordinate coordinate) throws IOException {
            Bundle existingBundle;
            if (coordinate == null) {
                return;
            }
            this.downloads.add(coordinate);
            File downloaded = this.download(coordinate);
            if (downloaded != null) {
                BundleCoordinate parentCoordinate = this.getParentCoordinate(downloaded);
                this.downloadBundleAndParents(parentCoordinate);
            }
            if ((existingBundle = DownloadQueue.this.extensionManager.getBundle(coordinate)) != null) {
                BundleCoordinate parentCoordinate = existingBundle.getBundleDetails().getDependencyCoordinate();
                this.downloadBundleAndParents(parentCoordinate);
            }
        }

        private BundleCoordinate getParentCoordinate(File narFile) throws IOException {
            try (JarFile nar = new JarFile(narFile);){
                Manifest manifest = nar.getManifest();
                Attributes attributes = manifest.getMainAttributes();
                String groupId = attributes.getValue(NarManifestEntry.NAR_DEPENDENCY_GROUP.getManifestName());
                String narId = attributes.getValue(NarManifestEntry.NAR_DEPENDENCY_ID.getManifestName());
                String version = attributes.getValue(NarManifestEntry.NAR_DEPENDENCY_VERSION.getManifestName());
                if (groupId == null || narId == null || version == null) {
                    BundleCoordinate bundleCoordinate = null;
                    return bundleCoordinate;
                }
                BundleCoordinate bundleCoordinate = new BundleCoordinate(groupId, narId, version);
                return bundleCoordinate;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private File download(BundleCoordinate coordinate) throws BundleNotFoundException {
            ArrayList<Exception> suppressed = new ArrayList<Exception>();
            File destinationFile = DownloadQueue.this.getBundleFile(coordinate);
            if (!DownloadQueue.this.isDownloadable(coordinate)) {
                logger.debug("Requested to download {} but only a single NAR of this type is allowed to exist so will not download.", (Object)coordinate);
                return null;
            }
            if (destinationFile.exists()) {
                logger.debug("Requested to download {} but destination file {} already exists. Will not download.", (Object)coordinate, (Object)destinationFile);
                return destinationFile;
            }
            for (ExtensionClient extensionClient : DownloadQueue.this.clients) {
                InputStream extensionStream = null;
                try {
                    extensionStream = extensionClient.getExtension(coordinate);
                    if (extensionStream == null) continue;
                    long start = System.currentTimeMillis();
                    File tmpFile = new File(destinationFile.getParentFile(), destinationFile.getName() + ".download." + String.valueOf(UUID.randomUUID()));
                    try (FileOutputStream out = new FileOutputStream(tmpFile);){
                        StreamUtils.copy((InputStream)extensionStream, (OutputStream)out);
                    }
                    fileRenameLock.lock();
                    try {
                        if (destinationFile.exists()) {
                            logger.debug("Finished downloading {} but the destination file {} already exists. Assuming that another thread has already downloaded the file.", (Object)tmpFile, (Object)destinationFile.getAbsolutePath());
                            if (!tmpFile.delete()) {
                                logger.warn("Failed to remove temporary file {}. This file should be removed manually.", (Object)tmpFile.getAbsolutePath());
                            }
                        } else {
                            Files.move(tmpFile.toPath(), destinationFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                        }
                    }
                    finally {
                        fileRenameLock.unlock();
                    }
                    long millis = System.currentTimeMillis() - start;
                    logger.info("Successfully downloaded {} to {} in {} millis", new Object[]{coordinate, destinationFile.getAbsolutePath(), millis});
                    File file = destinationFile;
                    return file;
                }
                catch (Exception e) {
                    logger.error("Failed to fetch extension {} from {}", new Object[]{coordinate, extensionClient, e});
                    suppressed.add(e);
                }
                finally {
                    this.closeQuietly(extensionStream);
                }
            }
            BundleNotFoundException bnfe = new BundleNotFoundException(coordinate, "Could not fetch bundle " + String.valueOf(coordinate) + " from any client");
            suppressed.forEach(bnfe::addSuppressed);
            throw bnfe;
        }

        private void closeQuietly(Closeable closeable) {
            if (closeable == null) {
                return;
            }
            try {
                closeable.close();
            }
            catch (IOException ioe) {
                logger.warn("Failed to close {}", (Object)closeable);
            }
        }
    }
}

