/*
 * Decompiled with CFR 0.152.
 */
package io.questdb.cairo.pool;

import io.questdb.cairo.CairoConfiguration;
import io.questdb.cairo.CairoException;
import io.questdb.cairo.EntryUnavailableException;
import io.questdb.cairo.TableToken;
import io.questdb.cairo.pool.AbstractPool;
import io.questdb.cairo.pool.PoolConstants;
import io.questdb.cairo.pool.PoolListener;
import io.questdb.cairo.pool.PoolTenant;
import io.questdb.cairo.pool.ResourcePool;
import io.questdb.cairo.pool.ex.EntryLockedException;
import io.questdb.cairo.pool.ex.PoolClosedException;
import io.questdb.log.Log;
import io.questdb.log.LogFactory;
import io.questdb.std.ConcurrentHashMap;
import io.questdb.std.Os;
import io.questdb.std.Unsafe;
import java.util.Arrays;
import java.util.Map;

public abstract class AbstractMultiTenantPool<T extends PoolTenant>
extends AbstractPool
implements ResourcePool<T> {
    public static final int ENTRY_SIZE = 32;
    public static final String NO_LOCK_REASON = "unknown";
    private static final long LOCK_OWNER = Unsafe.getFieldOffset(Entry.class, "lockOwner");
    private static final int NEXT_ALLOCATED = 1;
    private static final int NEXT_LOCKED = 2;
    private static final int NEXT_OPEN = 0;
    private static final long NEXT_STATUS = Unsafe.getFieldOffset(Entry.class, "nextStatus");
    private static final long UNLOCKED = -1L;
    private final Log LOG = LogFactory.getLog(this.getClass());
    private final ConcurrentHashMap<Entry<T>> entries = new ConcurrentHashMap();
    private final int maxEntries;
    private final int maxSegments;

    public AbstractMultiTenantPool(CairoConfiguration configuration) {
        super(configuration, configuration.getInactiveReaderTTL());
        this.maxSegments = configuration.getReaderPoolMaxSegments();
        this.maxEntries = this.maxSegments * 32;
    }

    public Map<CharSequence, Entry<T>> entries() {
        return this.entries;
    }

    @Override
    public T get(TableToken tableToken) {
        Entry<PoolTenant> e = this.getEntry(tableToken);
        long lockOwner = e.lockOwner;
        long thread = Thread.currentThread().getId();
        if (lockOwner != -1L) {
            this.LOG.info().$('\'').utf8(tableToken.getDirName()).$("' is locked [owner=").$(lockOwner).$(']').$();
            throw EntryLockedException.instance(NO_LOCK_REASON);
        }
        do {
            for (int i = 0; i < 32; ++i) {
                if (!Unsafe.cas(e.allocations, i, -1L, thread)) continue;
                Unsafe.arrayPutOrdered(e.releaseOrAcquireTimes, i, this.clock.getTicks());
                PoolTenant tenant = (PoolTenant)e.getTenant(i);
                if (tenant == null) {
                    try {
                        this.LOG.info().$("open '").utf8(tableToken.getDirName()).$("' [at=").$(e.index).$(':').$(i).I$();
                        tenant = this.newTenant(tableToken, e, i);
                    }
                    catch (CairoException ex) {
                        Unsafe.arrayPutOrdered(e.allocations, i, -1L);
                        throw ex;
                    }
                    e.assignTenant(i, tenant);
                    this.notifyListener(thread, tableToken, (short)10, e.index, i);
                } else {
                    try {
                        tenant.refresh();
                    }
                    catch (Throwable th) {
                        tenant.goodbye();
                        tenant.close();
                        e.assignTenant(i, null);
                        Unsafe.arrayPutOrdered(e.allocations, i, -1L);
                        throw th;
                    }
                    this.notifyListener(thread, tableToken, (short)11, e.index, i);
                }
                if (this.isClosed()) {
                    e.assignTenant(i, null);
                    tenant.goodbye();
                    this.LOG.info().$('\'').utf8(tableToken.getDirName()).$("' born free").$();
                    tenant.updateTableToken(tableToken);
                    return (T)tenant;
                }
                this.LOG.debug().$('\'').utf8(tableToken.getDirName()).$("' is assigned [at=").$(e.index).$(':').$(i).$(", thread=").$(thread).$(']').$();
                tenant.updateTableToken(tableToken);
                return (T)tenant;
            }
            this.LOG.debug().$("Thread ").$(thread).$(" is moving to entry ").$(e.index + 1).$();
            if (Unsafe.getUnsafe().compareAndSwapInt(e, NEXT_STATUS, 0, 1)) {
                this.LOG.debug().$("Thread ").$(thread).$(" allocated entry ").$(e.index + 1).$();
                e.next = new Entry(e.index + 1, this.clock.getTicks());
                continue;
            }
            while (e.next == null && e.nextStatus == 1) {
                Os.pause();
            }
        } while ((e = e.next) != null && e.index < this.maxSegments);
        this.notifyListener(thread, tableToken, (short)25, -1, -1);
        this.LOG.info().$("could not get, busy [table=`").utf8(tableToken.getDirName()).$("`, thread=").$(thread).$(", retries=").$(this.maxSegments).$(']').$();
        throw EntryUnavailableException.instance(NO_LOCK_REASON);
    }

    public int getBusyCount() {
        int count = 0;
        for (Map.Entry<CharSequence, Entry<T>> me : this.entries.entrySet()) {
            Entry<T> e = me.getValue();
            do {
                for (int i = 0; i < 32; ++i) {
                    if (Unsafe.arrayGetVolatile(e.allocations, i) == -1L || e.getTenant(i) == null) continue;
                    ++count;
                }
            } while ((e = e.next) != null);
        }
        return count;
    }

    public int getMaxEntries() {
        return this.maxEntries;
    }

    public boolean lock(TableToken tableToken) {
        long thread;
        Entry<T> e = this.getEntry(tableToken);
        if (Unsafe.cas(e, LOCK_OWNER, -1L, thread = Thread.currentThread().getId()) || e.lockOwner == thread) {
            do {
                for (int i = 0; i < 32; ++i) {
                    if (Unsafe.cas(e.allocations, i, -1L, thread)) {
                        this.closeTenant(thread, e, i, (short)19, 2);
                        continue;
                    }
                    if (Unsafe.arrayGetVolatile(e.allocations, i) == thread) {
                        if (e.getTenant(i) == null) continue;
                        e.lockOwner = -1L;
                        return false;
                    }
                    this.LOG.info().$("could not lock, busy [table=`").utf8(tableToken.getDirName()).$("`, at=").$(e.index).$(':').$(i).$(", owner=").$(e.allocations[i]).$(", thread=").$(thread).$(']').$();
                    e.lockOwner = -1L;
                    return false;
                }
                if (e.next != null) continue;
                if (!Unsafe.getUnsafe().compareAndSwapInt(e, NEXT_STATUS, 0, 2)) {
                    if (e.nextStatus != 1) continue;
                    while (e.next == null) {
                        Os.pause();
                    }
                }
                break;
            } while ((e = e.next) != null);
        } else {
            this.LOG.error().$("already locked [table=`").utf8(tableToken.getDirName()).$("`, owner=").$(e.lockOwner).$(']').$();
            this.notifyListener(thread, tableToken, (short)7, -1, -1);
            return false;
        }
        this.notifyListener(thread, tableToken, (short)6, -1, -1);
        this.LOG.debug().$("locked [table=`").utf8(tableToken.getDirName()).$("`, thread=").$(thread).$(']').$();
        return true;
    }

    public void unlock(TableToken tableToken) {
        Entry<T> e = this.entries.get(tableToken.getDirName());
        long thread = Thread.currentThread().getId();
        if (e == null) {
            this.LOG.info().$("not found, cannot unlock [table=`").$(tableToken).$("`]").$();
            this.notifyListener(thread, tableToken, (short)9, -1, -1);
            return;
        }
        if (e.lockOwner == thread) {
            this.entries.remove(tableToken.getDirName());
            while (e != null) {
                e = e.next;
            }
        } else {
            this.notifyListener(thread, tableToken, (short)12);
            throw CairoException.nonCritical().put("Not the lock owner of ").put(tableToken.getDirName());
        }
        this.notifyListener(thread, tableToken, (short)8, -1, -1);
        this.LOG.debug().$("unlocked [table=`").utf8(tableToken.getDirName()).$("`]").$();
    }

    private void checkClosed() {
        if (this.isClosed()) {
            this.LOG.info().$("is closed").$();
            throw PoolClosedException.INSTANCE;
        }
    }

    private void closeTenant(long thread, Entry<T> entry, int index, short ev, int reason) {
        PoolTenant tenant = (PoolTenant)entry.getTenant(index);
        if (tenant != null) {
            tenant.goodbye();
            tenant.close();
            this.LOG.info().$("closed '").utf8(tenant.getTableToken().getDirName()).$("' [at=").$(entry.index).$(':').$(index).$(", reason=").$(PoolConstants.closeReasonText(reason)).I$();
            this.notifyListener(thread, tenant.getTableToken(), ev, entry.index, index);
            entry.assignTenant(index, null);
        }
    }

    private Entry<T> getEntry(TableToken name) {
        this.checkClosed();
        Entry<T> e = this.entries.get(name.getDirName());
        if (e == null) {
            e = new Entry(0, this.clock.getTicks());
            Entry<T> other = this.entries.putIfAbsent(name.getDirName(), e);
            if (other != null) {
                e = other;
            }
        }
        return e;
    }

    private void notifyListener(long thread, TableToken name, short event, int segment, int position) {
        PoolListener listener = this.getPoolListener();
        if (listener != null) {
            listener.onEvent(this.getListenerSrc(), thread, name, event, (short)segment, (short)position);
        }
    }

    @Override
    protected void closePool() {
        super.closePool();
        this.LOG.info().$("closed").$();
    }

    protected void expelFromPool(T tenant) {
        Entry<Object> e = tenant.getEntry();
        if (e == null) {
            return;
        }
        TableToken tableToken = tenant.getTableToken();
        long thread = Thread.currentThread().getId();
        int index = tenant.getIndex();
        long owner = Unsafe.arrayGetVolatile(e.allocations, index);
        if (owner != -1L) {
            this.LOG.debug().$('\'').utf8(tableToken.getDirName()).$("' is expelled [at=").$(e.index).$(':').$(index).$(", thread=").$(thread).$(']').$();
            this.notifyListener(thread, tableToken, (short)2, e.index, index);
            e.assignTenant(index, null);
            Unsafe.cas(e.allocations, index, owner, -1L);
        }
    }

    protected abstract byte getListenerSrc();

    protected abstract T newTenant(TableToken var1, Entry<T> var2, int var3);

    @Override
    protected boolean releaseAll(long deadline) {
        long thread = Thread.currentThread().getId();
        boolean removed = false;
        int casFailures = 0;
        int closeReason = deadline < Long.MAX_VALUE ? 3 : 1;
        TableToken leftBehind = null;
        for (Entry<T> e : this.entries.values()) {
            do {
                for (int i = 0; i < 32; ++i) {
                    PoolTenant r;
                    if (deadline <= Unsafe.arrayGetVolatile(e.releaseOrAcquireTimes, i) || (r = (PoolTenant)e.getTenant(i)) == null) continue;
                    if (Unsafe.cas(e.allocations, i, -1L, thread)) {
                        if (deadline > e.releaseOrAcquireTimes[i]) {
                            removed = true;
                            this.closeTenant(thread, e, i, (short)17, closeReason);
                        }
                        Unsafe.arrayPutOrdered(e.allocations, i, -1L);
                        continue;
                    }
                    ++casFailures;
                    if (deadline != Long.MAX_VALUE) continue;
                    r.goodbye();
                    Unsafe.arrayPutOrdered(e.allocations, i, -1L);
                    this.LOG.info().$("shutting down. '").utf8(r.getTableToken().getDirName()).$("' is left behind").$();
                    leftBehind = r.getTableToken();
                }
            } while ((e = e.next) != null);
        }
        if (leftBehind != null) {
            throw CairoException.nonCritical().put('\'').put(leftBehind.getDirName()).put("' is left behind on pool shutdown");
        }
        if (closeReason == 3) {
            return removed;
        }
        return casFailures == 0;
    }

    protected boolean returnToPool(T tenant) {
        Entry e = tenant.getEntry();
        if (e == null) {
            return false;
        }
        TableToken tableToken = tenant.getTableToken();
        long thread = Thread.currentThread().getId();
        int index = tenant.getIndex();
        long owner = Unsafe.arrayGetVolatile(e.allocations, index);
        if (owner != -1L) {
            this.LOG.debug().$('\'').utf8(tableToken.getDirName()).$("' is back [at=").$(e.index).$(':').$(index).$(", thread=").$(thread).$(']').$();
            this.notifyListener(thread, tableToken, (short)1, e.index, index);
            e.releaseOrAcquireTimes[index] = this.clock.getTicks();
            Unsafe.arrayPutOrdered(e.allocations, index, -1L);
            boolean closed = this.isClosed();
            return !closed || !Unsafe.cas(e.allocations, index, -1L, owner);
        }
        throw CairoException.critical(0).put("double close [table=").put(tableToken.getDirName()).put(", index=").put(index).put(']');
    }

    public static final class Entry<T> {
        private final long[] allocations = new long[32];
        private final int index;
        private final long[] releaseOrAcquireTimes = new long[32];
        private final T[] tenants = new Object[32];
        int nextStatus = 0;
        private volatile long lockOwner = -1L;
        private volatile Entry<T> next;

        public Entry(int index, long currentMicros) {
            this.index = index;
            Arrays.fill(this.allocations, -1L);
            Arrays.fill(this.releaseOrAcquireTimes, currentMicros);
        }

        public void assignTenant(int pos, T tenant) {
            this.tenants[pos] = tenant;
        }

        public Entry<T> getNext() {
            return this.next;
        }

        public long getOwnerVolatile(int pos) {
            return Unsafe.arrayGetVolatile(this.allocations, pos);
        }

        public long getReleaseOrAcquireTime(int pos) {
            return this.releaseOrAcquireTimes[pos];
        }

        public T getTenant(int pos) {
            return this.tenants[pos];
        }
    }
}

