/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.persistence;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Calendar;
import java.util.Comparator;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.util.FormatUtils;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlowConfigurationArchiveManager {
    private static final Logger logger = LoggerFactory.getLogger(FlowConfigurationArchiveManager.class);
    private final Pattern archiveFilenamePattern = Pattern.compile("^([\\d]{8}T[\\d]{6}([\\+\\-][\\d]{4}|Z))_.+$");
    private final Path archiveDir;
    private final Integer maxCount;
    private final Long maxTimeMillis;
    private final Long maxStorageBytes;

    public FlowConfigurationArchiveManager(NiFiProperties properties) {
        Path archiveDirectory;
        this.maxCount = properties.getFlowConfigurationArchiveMaxCount();
        String maxTime = properties.getFlowConfigurationArchiveMaxTime();
        String maxStorage = properties.getFlowConfigurationArchiveMaxStorage();
        if (this.maxCount == null && StringUtils.isBlank((String)maxTime) && StringUtils.isBlank((String)maxStorage)) {
            maxTime = "30 days";
            maxStorage = "500 MB";
            logger.info("None of archive max limitation is specified, fall back to the default configuration, maxTime={}, maxStorage={}", (Object)maxTime, (Object)maxStorage);
        }
        this.maxTimeMillis = StringUtils.isBlank((String)maxTime) ? null : Long.valueOf(FormatUtils.getTimeDuration((String)maxTime, (TimeUnit)TimeUnit.MILLISECONDS));
        this.maxStorageBytes = StringUtils.isBlank((String)maxStorage) ? null : Long.valueOf(DataUnit.parseDataSize((String)maxStorage, (DataUnit)DataUnit.B).longValue());
        String archiveDirVal = properties.getFlowConfigurationArchiveDir();
        if (archiveDirVal == null || archiveDirVal.trim().isEmpty()) {
            File persistenceFile = properties.getFlowConfigurationFile();
            archiveDirectory = archiveDirVal == null || archiveDirVal.equals("") ? persistenceFile.toPath().getParent().resolve("archive") : new File(archiveDirVal).toPath();
        } else {
            archiveDirectory = Paths.get(archiveDirVal, new String[0]);
        }
        this.archiveDir = archiveDirectory;
    }

    private String createArchiveFileName(String originalFlowConfigFileName) {
        TimeZone tz = TimeZone.getDefault();
        Calendar cal = GregorianCalendar.getInstance(tz);
        int offsetInMillis = tz.getOffset(cal.getTimeInMillis());
        int year = cal.get(1);
        int month = cal.get(2) + 1;
        int day = cal.get(5);
        int hour = cal.get(11);
        int min = cal.get(12);
        int sec = cal.get(13);
        String offset = String.format("%s%02d%02d", offsetInMillis >= 0 ? "+" : "-", Math.abs(offsetInMillis / 3600000), Math.abs(offsetInMillis / 60000 % 60));
        return String.format("%d%02d%02dT%02d%02d%02d%s_%s", year, month, day, hour, min, sec, offset, originalFlowConfigFileName);
    }

    private File setupArchiveFile(String originalFilename) throws IOException {
        Files.createDirectories(this.archiveDir, new FileAttribute[0]);
        if (!Files.isDirectory(this.archiveDir, new LinkOption[0])) {
            throw new IOException("Archive directory doesn't appear to be a directory " + String.valueOf(this.archiveDir));
        }
        Path archiveFile = this.archiveDir.resolve(this.createArchiveFileName(originalFilename));
        return archiveFile.toFile();
    }

    public File archive(File file) throws IOException {
        String originalFlowConfigFileName = file.getName();
        File archiveFile = this.setupArchiveFile(file.getName());
        long now = System.currentTimeMillis();
        AtomicLong totalArchiveSize = new AtomicLong(0L);
        List archives = Files.walk(this.archiveDir, 1, new FileVisitOption[0]).filter(p -> {
            Matcher matcher;
            String filename = p.getFileName().toString();
            if (Files.isRegularFile(p, new LinkOption[0]) && filename.endsWith("_" + originalFlowConfigFileName) && (matcher = this.archiveFilenamePattern.matcher(filename)).matches() && filename.equals(matcher.group(1) + "_" + originalFlowConfigFileName)) {
                try {
                    totalArchiveSize.getAndAdd(Files.size(p));
                }
                catch (IOException e) {
                    logger.warn("Failed to get file size of {} due to {}", p, (Object)e);
                }
                return true;
            }
            return false;
        }).collect(Collectors.toList());
        archives.sort(Comparator.comparingLong(path -> path.toFile().lastModified()));
        logger.debug("archives={}", archives);
        int archiveCount = archives.size();
        long flowConfigFileSize = file.length();
        IntStream.range(0, archiveCount).filter(i -> {
            boolean old = this.maxCount != null && this.maxCount > 0 && archiveCount - i > this.maxCount - 1;
            File archive = ((Path)archives.get(i)).toFile();
            old = old || this.maxTimeMillis != null && this.maxTimeMillis > 0L && now - archive.lastModified() > this.maxTimeMillis;
            boolean bl = old = old || this.maxStorageBytes != null && this.maxStorageBytes > 0L && totalArchiveSize.get() + flowConfigFileSize > this.maxStorageBytes;
            if (old) {
                totalArchiveSize.getAndAdd(archive.length() * -1L);
                logger.info("Removing old archive file {} to reduce storage usage. currentSize={}", (Object)archive, (Object)totalArchiveSize);
            }
            return old;
        }).forEach(i -> {
            try {
                Files.delete((Path)archives.get(i));
            }
            catch (IOException e) {
                logger.warn("Failed to delete {} to reduce storage usage, due to {}", archives.get(i), (Object)e);
            }
        });
        Files.copy(file.toPath(), archiveFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        if (this.maxStorageBytes != null && this.maxStorageBytes > 0L && flowConfigFileSize > this.maxStorageBytes) {
            logger.warn("Size of {} ({}) exceeds configured maxStorage size ({}). Archive won't be able to keep old files.", new Object[]{file, flowConfigFileSize, this.maxStorageBytes});
        }
        return archiveFile;
    }
}

