/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.remote.protocol;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.zip.CRC32;
import java.util.zip.CheckedInputStream;
import java.util.zip.CheckedOutputStream;
import org.apache.nifi.connectable.Connection;
import org.apache.nifi.connectable.Port;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.flowfile.attributes.SiteToSiteAttributes;
import org.apache.nifi.groups.ProcessGroup;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.io.InputStreamCallback;
import org.apache.nifi.remote.Peer;
import org.apache.nifi.remote.PortAuthorizationResult;
import org.apache.nifi.remote.PublicPort;
import org.apache.nifi.remote.cluster.NodeInformant;
import org.apache.nifi.remote.codec.FlowFileCodec;
import org.apache.nifi.remote.exception.HandshakeException;
import org.apache.nifi.remote.exception.ProtocolException;
import org.apache.nifi.remote.io.CompressionInputStream;
import org.apache.nifi.remote.io.CompressionOutputStream;
import org.apache.nifi.remote.protocol.CommunicationsSession;
import org.apache.nifi.remote.protocol.DataPacket;
import org.apache.nifi.remote.protocol.FlowFileTransaction;
import org.apache.nifi.remote.protocol.HandshakeProperties;
import org.apache.nifi.remote.protocol.HandshakeProperty;
import org.apache.nifi.remote.protocol.Response;
import org.apache.nifi.remote.protocol.ResponseCode;
import org.apache.nifi.remote.protocol.ServerProtocol;
import org.apache.nifi.remote.util.StandardDataPacket;
import org.apache.nifi.util.FormatUtils;
import org.apache.nifi.util.StopWatch;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractFlowFileServerProtocol
implements ServerProtocol {
    protected ProcessGroup rootGroup;
    protected PublicPort port;
    protected boolean handshakeCompleted;
    protected boolean shutdown = false;
    protected FlowFileCodec negotiatedFlowFileCodec = null;
    protected HandshakeProperties handshakeProperties;
    protected static final long DEFAULT_BATCH_NANOS = TimeUnit.SECONDS.toNanos(5L);
    protected final Logger logger = LoggerFactory.getLogger(this.getClass());

    public void setRootProcessGroup(ProcessGroup group) {
        if (!group.isRootGroup()) {
            throw new IllegalArgumentException("Specified group was not a root group.");
        }
        this.rootGroup = group;
    }

    public boolean isHandshakeSuccessful() {
        return this.handshakeCompleted;
    }

    protected void validateHandshakeRequest(HandshakeProperties confirmed, Peer peer, Map<String, String> properties) throws HandshakeException {
        Boolean useGzip = null;
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            HandshakeProperty property;
            String propertyName = entry.getKey();
            String value = entry.getValue();
            try {
                property = HandshakeProperty.valueOf((String)propertyName);
            }
            catch (Exception e) {
                throw new HandshakeException(ResponseCode.UNKNOWN_PROPERTY_NAME, "Received unknown property: " + propertyName);
            }
            try {
                switch (property) {
                    case GZIP: {
                        useGzip = Boolean.parseBoolean(value);
                        confirmed.setUseGzip(useGzip);
                        break;
                    }
                    case REQUEST_EXPIRATION_MILLIS: {
                        confirmed.setExpirationMillis(Long.parseLong(value));
                        break;
                    }
                    case BATCH_COUNT: {
                        confirmed.setBatchCount(Integer.parseInt(value));
                        break;
                    }
                    case BATCH_SIZE: {
                        confirmed.setBatchBytes(Long.parseLong(value));
                        break;
                    }
                    case BATCH_DURATION: {
                        confirmed.setBatchDurationNanos(TimeUnit.MILLISECONDS.toNanos(Long.parseLong(value)));
                        break;
                    }
                    case PORT_IDENTIFIER: {
                        this.checkPortStatus(peer, value);
                    }
                }
            }
            catch (NumberFormatException nfe) {
                throw new HandshakeException(ResponseCode.ILLEGAL_PROPERTY_VALUE, "Received invalid value for property '" + String.valueOf(property) + "'; invalid value: " + value);
            }
        }
        if (useGzip == null) {
            this.logger.debug("Responding with ResponseCode MISSING_PROPERTY because GZIP Property missing");
            throw new HandshakeException(ResponseCode.MISSING_PROPERTY, "Missing Property " + HandshakeProperty.GZIP.name());
        }
    }

    protected void checkPortStatus(Peer peer, String portId) throws HandshakeException {
        Port receivedPort = this.rootGroup.findInputPort(portId);
        if (receivedPort == null) {
            receivedPort = this.rootGroup.findOutputPort(portId);
        }
        if (receivedPort == null) {
            this.logger.debug("Responding with ResponseCode UNKNOWN_PORT for identifier {}", (Object)portId);
            throw new HandshakeException(ResponseCode.UNKNOWN_PORT, "Received unknown port identifier: " + portId);
        }
        if (!(receivedPort instanceof PublicPort)) {
            this.logger.debug("Responding with ResponseCode UNKNOWN_PORT for identifier {}", (Object)portId);
            throw new HandshakeException(ResponseCode.UNKNOWN_PORT, "Received port identifier " + portId + ", but this Port is not remotely accessible");
        }
        this.port = (PublicPort)receivedPort;
        PortAuthorizationResult portAuthResult = this.port.checkUserAuthorization(peer.getCommunicationsSession().getUserDn());
        if (!portAuthResult.isAuthorized()) {
            this.logger.debug("Responding with ResponseCode UNAUTHORIZED: {}", (Object)portAuthResult.getExplanation());
            throw new HandshakeException(ResponseCode.UNAUTHORIZED, portAuthResult.getExplanation());
        }
        if (!receivedPort.isValid()) {
            this.logger.debug("Responding with ResponseCode PORT_NOT_IN_VALID_STATE for {}", (Object)receivedPort);
            throw new HandshakeException(ResponseCode.PORT_NOT_IN_VALID_STATE, "Port is not valid");
        }
        if (!receivedPort.isRunning()) {
            this.logger.debug("Responding with ResponseCode PORT_NOT_IN_VALID_STATE for {}", (Object)receivedPort);
            throw new HandshakeException(ResponseCode.PORT_NOT_IN_VALID_STATE, "Port not running");
        }
        if (this.getVersionNegotiator().getVersion() > 1) {
            for (Connection connection : this.port.getConnections()) {
                if (!connection.getFlowFileQueue().isFull()) continue;
                this.logger.debug("Responding with ResponseCode PORTS_DESTINATION_FULL for {}", (Object)this.port);
                throw new HandshakeException(ResponseCode.PORTS_DESTINATION_FULL, "Received port identifier " + portId + ", but its destination is full");
            }
        }
    }

    public PublicPort getPort() {
        return this.port;
    }

    public FlowFileCodec getPreNegotiatedCodec() {
        return this.negotiatedFlowFileCodec;
    }

    public final void handshake(Peer peer) throws IOException, HandshakeException {
        if (this.handshakeCompleted) {
            throw new IllegalStateException("Handshake has already been completed");
        }
        if (this.shutdown) {
            throw new IllegalStateException("Protocol is shutdown");
        }
        this.logger.debug("{} Handshaking with {}", (Object)this, (Object)peer);
        this.handshakeProperties = this.doHandshake(peer);
        this.logger.debug("{} Finished handshake with {}", (Object)this, (Object)peer);
        this.handshakeCompleted = true;
    }

    protected abstract HandshakeProperties doHandshake(Peer var1) throws IOException, HandshakeException;

    public int transferFlowFiles(Peer peer, ProcessContext context, ProcessSession session, final FlowFileCodec codec) throws IOException, ProtocolException {
        Object flowFile;
        if (!this.handshakeCompleted) {
            throw new IllegalStateException("Handshake has not been completed");
        }
        if (this.shutdown) {
            throw new IllegalStateException("Protocol is shutdown");
        }
        this.logger.debug("{} Sending FlowFiles to {}", (Object)this, (Object)peer);
        CommunicationsSession commsSession = peer.getCommunicationsSession();
        String remoteDn = commsSession.getUserDn();
        if (remoteDn == null) {
            remoteDn = "none";
        }
        if ((flowFile = session.get()) == null) {
            this.logger.debug("{} No data to send to {}", (Object)this, (Object)peer);
            this.writeTransactionResponse(true, ResponseCode.NO_MORE_DATA, commsSession);
            return 0;
        }
        this.logger.debug("{} Data is available to send to {}", (Object)this, (Object)peer);
        this.writeTransactionResponse(true, ResponseCode.MORE_DATA, commsSession);
        StopWatch stopWatch = new StopWatch(true);
        long bytesSent = 0L;
        HashSet<FlowFile> flowFilesSent = new HashSet<FlowFile>();
        CRC32 crc = new CRC32();
        boolean continueTransaction = true;
        long startNanos = System.nanoTime();
        String calculatedCRC = "";
        DataOutputStream os = new DataOutputStream(commsSession.getOutput().getOutputStream());
        while (continueTransaction) {
            double batchBytes;
            boolean useGzip = this.handshakeProperties.isUseGzip();
            DataOutputStream flowFileOutputStream = useGzip ? new CompressionOutputStream((OutputStream)os) : os;
            this.logger.debug("{} Sending {} to {}", new Object[]{this, flowFile, peer});
            final CheckedOutputStream checkedOutputStream = new CheckedOutputStream(flowFileOutputStream, crc);
            StopWatch transferWatch = new StopWatch(true);
            final FlowFile toSend = flowFile;
            session.read(flowFile, new InputStreamCallback(){

                public void process(InputStream in) throws IOException {
                    StandardDataPacket dataPacket = new StandardDataPacket(toSend.getAttributes(), in, toSend.getSize());
                    codec.encode((DataPacket)dataPacket, (OutputStream)checkedOutputStream);
                }
            });
            long transmissionMillis = transferWatch.getElapsed(TimeUnit.MILLISECONDS);
            if (useGzip) {
                checkedOutputStream.close();
            }
            flowFilesSent.add((FlowFile)flowFile);
            bytesSent += flowFile.getSize();
            String transitUri = this.createTransitUri(peer, flowFile.getAttribute(CoreAttributes.UUID.key()));
            session.getProvenanceReporter().send(flowFile, transitUri, "Remote Host=" + peer.getHost() + ", Remote DN=" + remoteDn, transmissionMillis, false);
            session.remove(flowFile);
            long sendingNanos = System.nanoTime() - startNanos;
            boolean poll = true;
            double batchDurationNanos = this.handshakeProperties.getBatchDurationNanos();
            if ((double)sendingNanos >= batchDurationNanos && batchDurationNanos > 0.0) {
                poll = false;
            }
            if ((double)bytesSent >= (batchBytes = (double)this.handshakeProperties.getBatchBytes()) && batchBytes > 0.0) {
                poll = false;
            }
            double batchCount = this.handshakeProperties.getBatchCount();
            if ((double)flowFilesSent.size() >= batchCount && batchCount > 0.0) {
                poll = false;
            }
            if (batchDurationNanos == 0.0 && batchBytes == 0.0 && batchCount == 0.0) {
                poll = sendingNanos < DEFAULT_BATCH_NANOS;
            }
            flowFile = poll ? session.get() : null;
            boolean bl = continueTransaction = flowFile != null;
            if (continueTransaction) {
                this.logger.debug("{} Sending ContinueTransaction indicator to {}", (Object)this, (Object)peer);
                this.writeTransactionResponse(true, ResponseCode.CONTINUE_TRANSACTION, commsSession);
                continue;
            }
            this.logger.debug("{} Sending FinishTransaction indicator to {}", (Object)this, (Object)peer);
            this.writeTransactionResponse(true, ResponseCode.FINISH_TRANSACTION, commsSession);
            calculatedCRC = String.valueOf(checkedOutputStream.getChecksum().getValue());
        }
        FlowFileTransaction transaction = new FlowFileTransaction(session, context, stopWatch, bytesSent, flowFilesSent, calculatedCRC);
        return this.commitTransferTransaction(peer, transaction);
    }

    protected String createTransitUri(Peer peer, String sourceFlowFileIdentifier) {
        return peer.createTransitUri(sourceFlowFileIdentifier);
    }

    protected int commitTransferTransaction(Peer peer, FlowFileTransaction transaction) throws IOException {
        Response transactionResponse;
        ProcessSession session = transaction.getSession();
        Set<FlowFile> flowFilesSent = transaction.getFlowFilesSent();
        CommunicationsSession commsSession = peer.getCommunicationsSession();
        Response transactionConfirmationResponse = this.readTransactionResponse(true, commsSession);
        if (transactionConfirmationResponse.getCode() == ResponseCode.CONFIRM_TRANSACTION) {
            String calculatedCRC;
            this.logger.debug("{} Received {}  from {}", new Object[]{this, transactionConfirmationResponse, peer});
            String receivedCRC = transactionConfirmationResponse.getMessage();
            if (this.getVersionNegotiator().getVersion() > 3 && !receivedCRC.equals(calculatedCRC = transaction.getCalculatedCRC())) {
                this.writeTransactionResponse(true, ResponseCode.BAD_CHECKSUM, commsSession);
                session.rollback();
                throw new IOException(String.valueOf(this) + " Sent data to peer " + String.valueOf(peer) + " but calculated CRC32 Checksum as " + calculatedCRC + " while peer calculated CRC32 Checksum as " + receivedCRC + "; canceling transaction and rolling back session");
            }
        } else {
            throw new ProtocolException("Expected to receive 'Confirm Transaction' response from peer " + String.valueOf(peer) + " but received " + String.valueOf(transactionConfirmationResponse));
        }
        this.writeTransactionResponse(true, ResponseCode.CONFIRM_TRANSACTION, commsSession, "");
        Object flowFileDescription = flowFilesSent.size() < 20 ? flowFilesSent.toString() : flowFilesSent.size() + " FlowFiles";
        try {
            transactionResponse = this.readTransactionResponse(true, commsSession);
        }
        catch (IOException e) {
            this.logger.error("{} Failed to receive a response from {} when expecting a TransactionFinished Indicator. It is unknown whether or not the peer successfully received/processed the data. Therefore, {} will be rolled back, possibly resulting in data duplication of {}", new Object[]{this, peer, session, flowFileDescription});
            session.rollback();
            throw e;
        }
        this.logger.debug("{} received {} from {}", new Object[]{this, transactionResponse, peer});
        if (transactionResponse.getCode() == ResponseCode.TRANSACTION_FINISHED_BUT_DESTINATION_FULL) {
            peer.penalize(this.port.getIdentifier(), this.port.getYieldPeriod(TimeUnit.MILLISECONDS));
        } else if (transactionResponse.getCode() != ResponseCode.TRANSACTION_FINISHED) {
            throw new ProtocolException("After sending data, expected TRANSACTION_FINISHED response but got " + String.valueOf(transactionResponse));
        }
        session.commitAsync();
        StopWatch stopWatch = transaction.getStopWatch();
        long bytesSent = transaction.getBytesSent();
        stopWatch.stop();
        String uploadDataRate = stopWatch.calculateDataRate(bytesSent);
        long uploadMillis = stopWatch.getDuration(TimeUnit.MILLISECONDS);
        String dataSize = FormatUtils.formatDataSize((double)bytesSent);
        this.logger.info("{} Successfully sent {} ({}) to {} in {} milliseconds at a rate of {}", new Object[]{this, flowFileDescription, dataSize, peer, uploadMillis, uploadDataRate});
        return flowFilesSent.size();
    }

    protected Response readTransactionResponse(boolean isTransfer, CommunicationsSession commsSession) throws IOException {
        DataInputStream dis = new DataInputStream(commsSession.getInput().getInputStream());
        return Response.read((DataInputStream)dis);
    }

    protected final void writeTransactionResponse(boolean isTransfer, ResponseCode response, CommunicationsSession commsSession) throws IOException {
        this.writeTransactionResponse(isTransfer, response, commsSession, null);
    }

    protected void writeTransactionResponse(boolean isTransfer, ResponseCode response, CommunicationsSession commsSession, String explanation) throws IOException {
        DataOutputStream dos = new DataOutputStream(commsSession.getOutput().getOutputStream());
        if (explanation == null) {
            response.writeResponse(dos);
        } else {
            response.writeResponse(dos, explanation);
        }
    }

    public int receiveFlowFiles(Peer peer, ProcessContext context, ProcessSession session, FlowFileCodec codec) throws IOException, ProtocolException {
        if (!this.handshakeCompleted) {
            throw new IllegalStateException("Handshake has not been completed");
        }
        if (this.shutdown) {
            throw new IllegalStateException("Protocol is shutdown");
        }
        this.logger.debug("{} receiving FlowFiles from {}", (Object)this, (Object)peer);
        CommunicationsSession commsSession = peer.getCommunicationsSession();
        DataInputStream dis = new DataInputStream(commsSession.getInput().getInputStream());
        String remoteDn = commsSession.getUserDn();
        if (remoteDn == null) {
            remoteDn = "none";
        }
        StopWatch stopWatch = new StopWatch(true);
        CRC32 crc = new CRC32();
        HashSet<FlowFile> flowFilesReceived = new HashSet<FlowFile>();
        long bytesReceived = 0L;
        boolean continueTransaction = true;
        block5: while (continueTransaction) {
            long startNanos = System.nanoTime();
            DataInputStream flowFileInputStream = this.handshakeProperties.isUseGzip() ? new CompressionInputStream((InputStream)dis) : dis;
            CheckedInputStream checkedInputStream = new CheckedInputStream(flowFileInputStream, crc);
            DataPacket dataPacket = codec.decode((InputStream)checkedInputStream);
            if (dataPacket == null) {
                this.logger.debug("{} Received null dataPacket indicating the end of transaction from {}", (Object)this, (Object)peer);
                break;
            }
            FlowFile flowFile = session.create();
            if (dataPacket.getSize() > 0L) {
                flowFile = session.importFrom(dataPacket.getData(), flowFile);
            }
            flowFile = session.putAllAttributes(flowFile, dataPacket.getAttributes());
            if (this.handshakeProperties.isUseGzip()) {
                checkedInputStream.close();
            }
            long transferNanos = System.nanoTime() - startNanos;
            long transferMillis = TimeUnit.MILLISECONDS.convert(transferNanos, TimeUnit.NANOSECONDS);
            String sourceSystemFlowFileUuid = (String)dataPacket.getAttributes().get(CoreAttributes.UUID.key());
            String host = StringUtils.isEmpty((String)peer.getHost()) ? "unknown" : peer.getHost();
            String port = peer.getPort() <= 0 ? "unknown" : String.valueOf(peer.getPort());
            HashMap<String, Object> attributes = new HashMap<String, Object>(4);
            attributes.put(CoreAttributes.UUID.key(), UUID.randomUUID().toString());
            attributes.put(SiteToSiteAttributes.S2S_HOST.key(), host);
            attributes.put(SiteToSiteAttributes.S2S_ADDRESS.key(), host + ":" + port);
            flowFile = session.putAllAttributes(flowFile, attributes);
            String transitUri = this.createTransitUri(peer, sourceSystemFlowFileUuid);
            session.getProvenanceReporter().receive(flowFile, transitUri, sourceSystemFlowFileUuid == null ? null : "urn:nifi:" + sourceSystemFlowFileUuid, "Remote Host=" + peer.getHost() + ", Remote DN=" + remoteDn, transferMillis);
            session.transfer(flowFile, Relationship.ANONYMOUS);
            flowFilesReceived.add(flowFile);
            bytesReceived += flowFile.getSize();
            Response transactionResponse = this.readTransactionResponse(false, commsSession);
            switch (transactionResponse.getCode()) {
                case CONTINUE_TRANSACTION: {
                    this.logger.debug("{} Received ContinueTransaction indicator from {}", (Object)this, (Object)peer);
                    continue block5;
                }
                case FINISH_TRANSACTION: {
                    this.logger.debug("{} Received FinishTransaction indicator from {}", (Object)this, (Object)peer);
                    continueTransaction = false;
                    continue block5;
                }
                case CANCEL_TRANSACTION: {
                    this.logger.info("{} Received CancelTransaction indicator from {} with explanation {}", new Object[]{this, peer, transactionResponse.getMessage()});
                    session.rollback();
                    return 0;
                }
            }
            throw new ProtocolException("Received unexpected response from peer: when expecting Continue Transaction or Finish Transaction, received" + String.valueOf(transactionResponse));
        }
        this.logger.debug("{} Sending CONFIRM_TRANSACTION Response Code to {}", (Object)this, (Object)peer);
        String calculatedCRC = String.valueOf(crc.getValue());
        this.writeTransactionResponse(false, ResponseCode.CONFIRM_TRANSACTION, commsSession, calculatedCRC);
        FlowFileTransaction transaction = new FlowFileTransaction(session, context, stopWatch, bytesReceived, flowFilesReceived, calculatedCRC);
        return this.commitReceiveTransaction(peer, transaction);
    }

    protected int commitReceiveTransaction(Peer peer, FlowFileTransaction transaction) throws IOException {
        CommunicationsSession commsSession = peer.getCommunicationsSession();
        ProcessSession session = transaction.getSession();
        Response confirmTransactionResponse = this.readTransactionResponse(false, commsSession);
        this.logger.debug("{} Received {} from {}", new Object[]{this, confirmTransactionResponse, peer});
        switch (confirmTransactionResponse.getCode()) {
            case CONFIRM_TRANSACTION: {
                break;
            }
            case BAD_CHECKSUM: {
                session.rollback();
                throw new IOException(String.valueOf(this) + " Received a BadChecksum response from peer " + String.valueOf(peer));
            }
            default: {
                throw new ProtocolException(String.valueOf(this) + " Received unexpected Response Code from peer " + String.valueOf(peer) + " : " + String.valueOf(confirmTransactionResponse) + "; expected 'Confirm Transaction' Response Code");
            }
        }
        session.commit();
        if (transaction.getContext().getAvailableRelationships().isEmpty()) {
            this.logger.debug("{} Sending TRANSACTION_FINISHED_BUT_DESTINATION_FULL to {}", (Object)this, (Object)peer);
            this.writeTransactionResponse(false, ResponseCode.TRANSACTION_FINISHED_BUT_DESTINATION_FULL, commsSession);
        } else {
            this.logger.debug("{} Sending TRANSACTION_FINISHED to {}", (Object)this, (Object)peer);
            this.writeTransactionResponse(false, ResponseCode.TRANSACTION_FINISHED, commsSession);
        }
        Set<FlowFile> flowFilesReceived = transaction.getFlowFilesSent();
        long bytesReceived = transaction.getBytesSent();
        StopWatch stopWatch = transaction.getStopWatch();
        stopWatch.stop();
        Object flowFileDescription = flowFilesReceived.size() < 20 ? flowFilesReceived.toString() : flowFilesReceived.size() + " FlowFiles";
        String uploadDataRate = stopWatch.calculateDataRate(bytesReceived);
        long uploadMillis = stopWatch.getDuration(TimeUnit.MILLISECONDS);
        String dataSize = FormatUtils.formatDataSize((double)bytesReceived);
        this.logger.info("{} Successfully received {} ({}) from {} in {} milliseconds at a rate of {}", new Object[]{this, flowFileDescription, dataSize, peer, uploadMillis, uploadDataRate});
        return flowFilesReceived.size();
    }

    public void shutdown(Peer peer) {
        this.logger.debug("{} Shutting down with {}", (Object)this, (Object)peer);
        this.shutdown = true;
    }

    public boolean isShutdown() {
        return this.shutdown;
    }

    public void setNodeInformant(NodeInformant nodeInformant) {
    }

    public long getRequestExpiration() {
        return this.handshakeProperties.getExpirationMillis();
    }

    public String toString() {
        String commid = this.handshakeProperties != null ? this.handshakeProperties.getCommsIdentifier() : null;
        return this.getClass().getSimpleName() + "[CommsID=" + commid + "]";
    }
}

