/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.stateless.repository;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.nifi.controller.repository.ContentRepository;
import org.apache.nifi.controller.repository.ContentRepositoryContext;
import org.apache.nifi.controller.repository.claim.ContentClaim;
import org.apache.nifi.controller.repository.claim.ResourceClaim;
import org.apache.nifi.controller.repository.claim.ResourceClaimManager;
import org.apache.nifi.controller.repository.claim.StandardContentClaim;
import org.apache.nifi.controller.repository.claim.StandardResourceClaim;
import org.apache.nifi.controller.repository.io.LimitedInputStream;
import org.apache.nifi.stream.io.StreamUtils;
import org.apache.nifi.stream.io.SynchronizedByteCountingOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StatelessFileSystemContentRepository
implements ContentRepository {
    private static final Logger logger = LoggerFactory.getLogger(StatelessFileSystemContentRepository.class);
    private static final String CONTENT_FILE_REGEX = "\\d+\\.nifi\\.bin";
    private static final String CONTAINER = "stateless";
    private static final String SECTION = "stateless";
    private final File directory;
    private final ConcurrentMap<ResourceClaim, SynchronizedByteCountingOutputStream> writableStreamMap = new ConcurrentHashMap<ResourceClaim, SynchronizedByteCountingOutputStream>();
    private final AtomicLong resourceClaimIndex = new AtomicLong(0L);
    private final BlockingQueue<ResourceClaim> writableClaimQueue = new LinkedBlockingQueue<ResourceClaim>();
    private ResourceClaimManager resourceClaimManager;

    public StatelessFileSystemContentRepository(File directory) {
        this.directory = directory;
    }

    public void initialize(ContentRepositoryContext context) throws IOException {
        this.resourceClaimManager = context.getResourceClaimManager();
        if (!this.directory.exists() && !this.directory.mkdirs()) {
            throw new IOException("Cannot initialize Content Repository because " + this.directory.getAbsolutePath() + " does not exist and cannot be created");
        }
        File[] existingFiles = this.directory.listFiles(file -> file.getName().matches(CONTENT_FILE_REGEX));
        if (existingFiles == null) {
            throw new IOException("Cannot initialize Content Repository because failed to list contents of directory " + this.directory.getAbsolutePath());
        }
        for (File existingFile : existingFiles) {
            logger.info("Found existing file from previous run {}. Removing file.", (Object)existingFile.getName());
            boolean deleted = existingFile.delete();
            if (deleted) continue;
            logger.warn("Failed to remove existing file from previous run {}", (Object)existingFile);
        }
    }

    public void shutdown() {
        this.purge();
    }

    public Set<String> getContainerNames() {
        return Collections.singleton("stateless");
    }

    public long getContainerCapacity(String containerName) {
        return 0L;
    }

    public long getContainerUsableSpace(String containerName) {
        return 0L;
    }

    public String getContainerFileStoreName(String containerName) {
        return "container";
    }

    public ContentClaim create(boolean lossTolerant) throws IOException {
        long offset;
        ResourceClaim resourceClaim = (ResourceClaim)this.writableClaimQueue.poll();
        if (resourceClaim == null) {
            resourceClaim = new StandardResourceClaim(this.resourceClaimManager, "stateless", "stateless", String.valueOf(this.resourceClaimIndex.getAndIncrement()), false);
            offset = 0L;
            File resourceClaimFile = this.getFile(resourceClaim);
            FileOutputStream fos = new FileOutputStream(resourceClaimFile);
            SynchronizedByteCountingOutputStream contentOutputStream = new SynchronizedByteCountingOutputStream((OutputStream)fos);
            this.writableStreamMap.put(resourceClaim, contentOutputStream);
        } else {
            SynchronizedByteCountingOutputStream contentOutputStream = (SynchronizedByteCountingOutputStream)this.writableStreamMap.get(resourceClaim);
            offset = contentOutputStream.getBytesWritten();
        }
        StandardContentClaim contentClaim = new StandardContentClaim(resourceClaim, offset);
        this.resourceClaimManager.incrementClaimantCount(contentClaim.getResourceClaim());
        return contentClaim;
    }

    public int incrementClaimaintCount(ContentClaim claim) {
        if (claim == null) {
            return 0;
        }
        return this.resourceClaimManager.incrementClaimantCount(claim.getResourceClaim());
    }

    public int getClaimantCount(ContentClaim claim) {
        if (claim == null) {
            return 0;
        }
        return this.resourceClaimManager.getClaimantCount(claim.getResourceClaim());
    }

    public int decrementClaimantCount(ContentClaim claim) {
        if (claim == null) {
            return 0;
        }
        return this.resourceClaimManager.decrementClaimantCount(claim.getResourceClaim());
    }

    public boolean remove(ContentClaim claim) {
        return true;
    }

    public ContentClaim clone(ContentClaim original, boolean lossTolerant) throws IOException {
        ContentClaim clone = this.create(lossTolerant);
        try (InputStream in = this.read(original);
             OutputStream out = this.write(clone);){
            StreamUtils.copy((InputStream)in, (OutputStream)out);
        }
        return clone;
    }

    public long importFrom(Path content, ContentClaim claim) throws IOException {
        try (InputStream in = Files.newInputStream(content, StandardOpenOption.READ);){
            long l = this.importFrom(in, claim);
            return l;
        }
    }

    public long importFrom(InputStream content, ContentClaim claim) throws IOException {
        try (OutputStream out = this.write(claim);){
            long l = StreamUtils.copy((InputStream)content, (OutputStream)out);
            return l;
        }
    }

    public long exportTo(ContentClaim claim, Path destination, boolean append) throws IOException {
        OpenOption[] openOptionArray;
        if (append) {
            StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[2];
            standardOpenOptionArray[0] = StandardOpenOption.CREATE;
            openOptionArray = standardOpenOptionArray;
            standardOpenOptionArray[1] = StandardOpenOption.APPEND;
        } else {
            OpenOption[] openOptionArray2 = new StandardOpenOption[1];
            openOptionArray = openOptionArray2;
            openOptionArray2[0] = StandardOpenOption.CREATE;
        }
        OpenOption[] openOptions = openOptionArray;
        try (OutputStream out = Files.newOutputStream(destination, openOptions);){
            long l = this.exportTo(claim, out);
            return l;
        }
    }

    public long exportTo(ContentClaim claim, Path destination, boolean append, long offset, long length) throws IOException {
        OpenOption[] openOptionArray;
        if (append) {
            StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[2];
            standardOpenOptionArray[0] = StandardOpenOption.CREATE;
            openOptionArray = standardOpenOptionArray;
            standardOpenOptionArray[1] = StandardOpenOption.APPEND;
        } else {
            OpenOption[] openOptionArray2 = new StandardOpenOption[1];
            openOptionArray = openOptionArray2;
            openOptionArray2[0] = StandardOpenOption.CREATE;
        }
        OpenOption[] openOptions = openOptionArray;
        try (OutputStream out = Files.newOutputStream(destination, openOptions);){
            long l = this.exportTo(claim, out, offset, length);
            return l;
        }
    }

    public long exportTo(ContentClaim claim, OutputStream destination) throws IOException {
        try (InputStream in = this.read(claim);){
            long l = StreamUtils.copy((InputStream)in, (OutputStream)destination);
            return l;
        }
    }

    public long exportTo(ContentClaim claim, OutputStream destination, long offset, long length) throws IOException {
        try (InputStream in = this.read(claim);){
            StreamUtils.skip((InputStream)in, (long)offset);
            StreamUtils.copy((InputStream)in, (OutputStream)destination, (long)length);
        }
        return length;
    }

    public long size(ContentClaim claim) {
        return claim.getLength();
    }

    public long size(ResourceClaim claim) throws IOException {
        return 0L;
    }

    public InputStream read(ContentClaim claim) throws IOException {
        if (claim == null) {
            return new ByteArrayInputStream(new byte[0]);
        }
        InputStream resourceClaimIn = this.read(claim.getResourceClaim());
        StreamUtils.skip((InputStream)resourceClaimIn, (long)claim.getOffset());
        LimitedInputStream limitedIn = new LimitedInputStream(resourceClaimIn, claim.getLength());
        return limitedIn;
    }

    public InputStream read(ResourceClaim claim) throws IOException {
        this.validateResourceClaim(claim);
        File file = this.getFile(claim);
        return new FileInputStream(file);
    }

    private File getFile(ResourceClaim claim) {
        return new File(this.directory, claim.getId() + ".nifi.bin");
    }

    private void validateResourceClaim(ResourceClaim resourceClaim) {
        if (!"stateless".equals(resourceClaim.getContainer())) {
            this.throwInvalidResourceClaim();
        }
        if (!"stateless".equals(resourceClaim.getSection())) {
            this.throwInvalidResourceClaim();
        }
    }

    public OutputStream write(ContentClaim claim) throws IOException {
        this.validateContentClaimForWriting(claim);
        SynchronizedByteCountingOutputStream out = (SynchronizedByteCountingOutputStream)this.writableStreamMap.get(claim.getResourceClaim());
        if (out == null) {
            this.throwInvalidContentClaim();
        }
        StandardContentClaim scc = (StandardContentClaim)claim;
        scc.setLength(0L);
        return new ContentOutputStream(out, scc);
    }

    private void validateContentClaimForWriting(ContentClaim claim) throws IOException {
        Objects.requireNonNull(claim, "ContentClaim cannot be null");
        if (!(claim instanceof StandardContentClaim)) {
            this.throwInvalidContentClaim();
        }
        this.validateResourceClaim(claim.getResourceClaim());
        if (claim.getLength() >= 0L) {
            throw new IOException("Cannot write to " + String.valueOf(claim) + " because it has already been written to.");
        }
    }

    private void throwInvalidContentClaim() {
        throw new IllegalArgumentException("The given ContentClaim does not belong to this Content Repository");
    }

    private void throwInvalidResourceClaim() {
        throw new IllegalArgumentException("The given ResourceClaim does not belong to this Content Repository");
    }

    public void purge() {
        this.writableClaimQueue.clear();
        for (OutputStream out : this.writableStreamMap.values()) {
            try {
                out.close();
            }
            catch (IOException ioe) {
                logger.warn("Failed to close Content Repository Output Stream", (Throwable)ioe);
            }
        }
        for (ResourceClaim resourceClaim : this.writableStreamMap.keySet()) {
            File file = this.getFile(resourceClaim);
            if (file.delete() || !file.exists()) continue;
            logger.warn("Failed to remove file from Content Repository: " + file.getAbsolutePath());
        }
        this.writableStreamMap.clear();
        this.resourceClaimManager.purge();
    }

    public void cleanup() {
        this.purge();
    }

    public boolean isAccessible(ContentClaim contentClaim) {
        return false;
    }

    private class ContentOutputStream
    extends FilterOutputStream {
        private final StandardContentClaim scc;
        private final SynchronizedByteCountingOutputStream out;
        private final long initialOffset;
        private boolean closed;

        public ContentOutputStream(SynchronizedByteCountingOutputStream out, StandardContentClaim scc) {
            super((OutputStream)out);
            this.closed = false;
            this.scc = scc;
            this.out = out;
            this.initialOffset = out.getBytesWritten();
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.out.write(b, off, len);
        }

        @Override
        public void write(byte[] b) throws IOException {
            this.out.write(b);
        }

        @Override
        public void write(int b) throws IOException {
            this.out.write(b);
        }

        @Override
        public synchronized void close() throws IOException {
            if (this.closed) {
                return;
            }
            this.closed = true;
            super.flush();
            this.scc.setLength(this.out.getBytesWritten() - this.initialOffset);
            StatelessFileSystemContentRepository.this.writableClaimQueue.offer(this.scc.getResourceClaim());
        }
    }
}

