/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.minifi.bootstrap;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiConsumer;
import org.apache.nifi.bootstrap.BootstrapCommunicator;
import org.apache.nifi.minifi.MiNiFiServer;
import org.apache.nifi.minifi.bootstrap.BootstrapRequest;
import org.apache.nifi.minifi.bootstrap.BootstrapRequestReader;
import org.apache.nifi.minifi.bootstrap.DumpUtil;
import org.apache.nifi.minifi.commons.status.FlowStatusReport;
import org.apache.nifi.minifi.status.StatusRequestException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BootstrapListener
implements BootstrapCommunicator {
    private static final Logger logger = LoggerFactory.getLogger(BootstrapListener.class);
    private static final String RELOAD = "RELOAD";
    private static final String SHUTDOWN = "SHUTDOWN";
    private static final String STARTED = "STARTED";
    private static final int LISTENER_EXECUTOR_THREAD_COUNT = 2;
    private final MiNiFiServer minifiServer;
    private final BootstrapRequestReader bootstrapRequestReader;
    private final int bootstrapPort;
    private final String secretKey;
    private final ObjectMapper objectMapper;
    private Listener listener;
    private final Map<String, BiConsumer<String[], OutputStream>> messageHandlers = new HashMap<String, BiConsumer<String[], OutputStream>>();

    public BootstrapListener(MiNiFiServer minifiServer, int bootstrapPort) {
        this.minifiServer = minifiServer;
        this.bootstrapPort = bootstrapPort;
        this.secretKey = UUID.randomUUID().toString();
        this.bootstrapRequestReader = new BootstrapRequestReader(this.secretKey);
        this.objectMapper = new ObjectMapper();
        this.objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        this.registerHandlers();
    }

    public void start() throws IOException {
        logger.debug("Starting Bootstrap Listener to communicate with Bootstrap Port {}", (Object)this.bootstrapPort);
        ServerSocket serverSocket = new ServerSocket();
        serverSocket.bind(new InetSocketAddress("localhost", 0));
        serverSocket.setSoTimeout(2000);
        int localPort = serverSocket.getLocalPort();
        logger.info("Started Bootstrap Listener, Listening for incoming requests on port {}", (Object)localPort);
        this.listener = new Listener(serverSocket);
        Thread listenThread = new Thread(this.listener);
        listenThread.setDaemon(true);
        listenThread.setName("Listen to Bootstrap");
        listenThread.start();
        logger.debug("Notifying Bootstrap that local port is {}", (Object)localPort);
        this.sendCommand("PORT", new String[]{String.valueOf(localPort), this.secretKey});
    }

    public void reload() throws IOException {
        if (this.listener != null) {
            this.listener.stop();
        }
        this.sendCommand(RELOAD, new String[0]);
    }

    public void stop() throws IOException {
        if (this.listener != null) {
            this.listener.stop();
        }
        this.sendCommand(SHUTDOWN, new String[0]);
    }

    public void sendStartedStatus(boolean status) throws IOException {
        logger.debug("Notifying Bootstrap that the status of starting MiNiFi is {}", (Object)status);
        this.sendCommand(STARTED, new String[]{String.valueOf(status)});
    }

    public void sendCommand(String command, String[] args) throws IOException {
        try (Socket socket = new Socket();){
            socket.setSoTimeout(60000);
            socket.connect(new InetSocketAddress("localhost", this.bootstrapPort));
            StringBuilder commandBuilder = new StringBuilder(command);
            Arrays.stream(args).forEach(arg -> commandBuilder.append(" ").append((String)arg));
            commandBuilder.append("\n");
            String commandWithArgs = commandBuilder.toString();
            logger.debug("Sending command to Bootstrap: {}", (Object)commandWithArgs);
            OutputStream out = socket.getOutputStream();
            out.write(commandWithArgs.getBytes(StandardCharsets.UTF_8));
            out.flush();
            logger.debug("Awaiting response from Bootstrap...");
            BufferedReader reader = new BufferedReader(new InputStreamReader(socket.getInputStream()));
            String response = reader.readLine();
            if ("OK".equals(response)) {
                logger.info("Successfully initiated communication with Bootstrap");
            } else {
                logger.error("Failed to communicate with Bootstrap. Bootstrap may be unable to issue or receive commands from MiNiFi");
            }
        }
    }

    public void registerMessageHandler(String command, BiConsumer<String[], OutputStream> handler) {
        this.messageHandlers.putIfAbsent(command, handler);
    }

    private void registerHandlers() {
        this.messageHandlers.putIfAbsent("PING", (args, outputStream) -> {
            logger.debug("Received PING request from Bootstrap; responding");
            this.echoRequestCmd("PING", (OutputStream)outputStream);
            logger.debug("Responded to PING request from Bootstrap");
        });
        this.messageHandlers.putIfAbsent(RELOAD, (args, outputStream) -> {
            logger.info("Received RELOAD request from Bootstrap");
            this.echoRequestCmd(RELOAD, (OutputStream)outputStream);
            logger.info("Responded to RELOAD request from Bootstrap, stopping MiNiFi Server");
            this.minifiServer.stop(true);
        });
        this.messageHandlers.putIfAbsent(SHUTDOWN, (args, outputStream) -> {
            logger.info("Received SHUTDOWN request from Bootstrap");
            this.echoRequestCmd(SHUTDOWN, (OutputStream)outputStream);
            logger.info("Responded to SHUTDOWN request from Bootstrap, stopping MiNiFi Server");
            this.minifiServer.stop(false);
        });
        this.messageHandlers.putIfAbsent("DUMP", (args, outputStream) -> {
            logger.info("Received DUMP request from Bootstrap");
            this.writeDump((OutputStream)outputStream);
        });
        this.messageHandlers.putIfAbsent("FLOW_STATUS_REPORT", (args, outputStream) -> {
            logger.info("Received FLOW_STATUS_REPORT request from Bootstrap");
            String flowStatusRequestString = args[0];
            this.writeStatusReport(flowStatusRequestString, (OutputStream)outputStream);
        });
        this.messageHandlers.putIfAbsent("ENV", (args, outputStream) -> {
            logger.info("Received ENV request from Bootstrap");
            BootstrapListener.writeEnv(outputStream);
        });
    }

    private void writeStatusReport(String flowStatusRequestString, OutputStream out) throws StatusRequestException {
        try {
            FlowStatusReport flowStatusReport = this.minifiServer.getStatusReport(flowStatusRequestString);
            this.objectMapper.writeValue(out, (Object)flowStatusReport);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static void writeEnv(OutputStream out) {
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out));){
            StringBuilder sb = new StringBuilder();
            System.getProperties().forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(key, value) -> sb.append(key).append("=").append(value).append("\n")));
            writer.write(sb.toString());
            writer.flush();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void writeDump(OutputStream out) {
        try {
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out));
            writer.write(DumpUtil.getDump());
            writer.flush();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void echoRequestCmd(String cmd, OutputStream out) {
        try {
            out.write((cmd + "\n").getBytes(StandardCharsets.UTF_8));
            out.flush();
            out.close();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private class Listener
    implements Runnable {
        private final ServerSocket serverSocket;
        private final ExecutorService executor;
        private volatile boolean stopped = false;

        public Listener(ServerSocket serverSocket) {
            this.serverSocket = serverSocket;
            this.executor = Executors.newFixedThreadPool(2);
        }

        public void stop() {
            this.stopped = true;
            this.executor.shutdownNow();
            try {
                this.serverSocket.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }

        @Override
        public void run() {
            while (!this.stopped) {
                try {
                    Socket socket;
                    try {
                        logger.debug("Listening for Bootstrap Requests");
                        socket = this.serverSocket.accept();
                    }
                    catch (SocketTimeoutException ste) {
                        if (!this.stopped) continue;
                        return;
                    }
                    catch (IOException ioe) {
                        if (this.stopped) {
                            return;
                        }
                        throw ioe;
                    }
                    logger.debug("Received connection from Bootstrap");
                    socket.setSoTimeout(5000);
                    this.executor.submit(() -> this.handleBootstrapRequest(socket));
                }
                catch (Throwable t) {
                    logger.error("Failed to process request from Bootstrap due to " + String.valueOf(t), t);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void handleBootstrapRequest(Socket socket) {
            try {
                BootstrapRequest request = BootstrapListener.this.bootstrapRequestReader.readRequest(socket.getInputStream());
                String requestType = request.getRequestType();
                BiConsumer<String[], OutputStream> handler = BootstrapListener.this.messageHandlers.get(requestType);
                if (handler == null) {
                    logger.warn("There is no handler defined for the {}", (Object)requestType);
                } else {
                    handler.accept(request.getArgs(), socket.getOutputStream());
                }
            }
            catch (Throwable t) {
                logger.error("Failed to process request from Bootstrap due to " + String.valueOf(t), t);
            }
            finally {
                try {
                    socket.close();
                }
                catch (IOException ioe) {
                    logger.warn("Failed to close socket to Bootstrap due to {}", (Object)ioe.toString());
                }
            }
        }
    }
}

