/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.nar;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarInputStream;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import org.apache.nifi.bundle.Bundle;
import org.apache.nifi.bundle.BundleCoordinate;
import org.apache.nifi.nar.ExtensionMapping;
import org.apache.nifi.nar.FileDigestUtils;
import org.apache.nifi.nar.NarManifestEntry;
import org.apache.nifi.nar.NarUnpackMode;
import org.apache.nifi.util.FileUtils;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NarUnpacker {
    public static final String BUNDLED_DEPENDENCIES_DIRECTORY = "NAR-INF/bundled-dependencies";
    private static final String BUNDLED_DEPENDENCIES_PREFIX = "META-INF/bundled-dependencies";
    private static final String JAR_DOCUMENTATION_ROOT_PATH = "docs";
    private static final Logger logger = LoggerFactory.getLogger(NarUnpacker.class);
    private static final String HASH_FILENAME = "nar-digest";
    private static final FileFilter NAR_FILTER = pathname -> {
        String nameToTest = pathname.getName().toLowerCase();
        return nameToTest.endsWith(".nar") && pathname.isFile();
    };

    public static ExtensionMapping unpackNars(NiFiProperties props, Bundle systemBundle, NarUnpackMode unpackMode) {
        return NarUnpacker.unpackNars(props, "nifi-framework-nar", systemBundle, unpackMode);
    }

    public static ExtensionMapping unpackNars(NiFiProperties props, String frameworkNarId, Bundle systemBundle, NarUnpackMode unpackMode) {
        List narLibraryDirs = props.getNarLibraryDirectories();
        File frameworkWorkingDir = props.getFrameworkWorkingDirectory();
        File extensionsWorkingDir = props.getExtensionsWorkingDirectory();
        File docsWorkingDir = props.getComponentDocumentationWorkingDirectory();
        return NarUnpacker.unpackNars(systemBundle, frameworkWorkingDir, frameworkNarId, extensionsWorkingDir, docsWorkingDir, narLibraryDirs, unpackMode);
    }

    public static ExtensionMapping unpackNars(Bundle systemBundle, File frameworkWorkingDir, String frameworkNarId, File extensionsWorkingDir, File docsWorkingDir, List<Path> narLibraryDirs, NarUnpackMode unpackMode) {
        return NarUnpacker.unpackNars(systemBundle, frameworkWorkingDir, extensionsWorkingDir, docsWorkingDir, narLibraryDirs, true, frameworkNarId, true, true, unpackMode, coordinate -> true);
    }

    public static ExtensionMapping unpackNars(Bundle systemBundle, File frameworkWorkingDir, File extensionsWorkingDir, File docsWorkingDir, List<Path> narLibraryDirs, boolean requireFrameworkNar, String frameworkNarId, boolean requireJettyNar, boolean verifyHash, NarUnpackMode unpackMode, Predicate<BundleCoordinate> narFilter) {
        try {
            File unpackedJetty = null;
            File unpackedFramework = null;
            HashSet<File> unpackedExtensions = new HashSet<File>();
            ArrayList<File> narFiles = new ArrayList<File>();
            if (requireFrameworkNar) {
                FileUtils.ensureDirectoryExistAndCanReadAndWrite(frameworkWorkingDir);
            }
            FileUtils.ensureDirectoryExistAndCanReadAndWrite(extensionsWorkingDir);
            if (docsWorkingDir != null) {
                FileUtils.ensureDirectoryExistAndCanReadAndWrite(docsWorkingDir);
            }
            for (Path narLibraryDir : narLibraryDirs) {
                File narDir = narLibraryDir.toFile();
                FileUtils.ensureDirectoryExistAndCanRead(narDir);
                File[] fileArray = narDir.listFiles(NAR_FILTER);
                if (fileArray == null) continue;
                List<File> fileList = Arrays.asList(fileArray);
                narFiles.addAll(fileList);
            }
            if (!narFiles.isEmpty()) {
                File[] extensionsWorkingDirContents;
                File[] frameworkWorkingDirContents;
                long startTime = System.nanoTime();
                logger.info("Expanding {} NAR files started", (Object)narFiles.size());
                for (File file : narFiles) {
                    if (!file.canRead()) {
                        throw new IllegalStateException("Unable to read NAR file: " + file.getAbsolutePath());
                    }
                    logger.debug("Expanding NAR file: {}", (Object)file.getAbsolutePath());
                    try (JarFile nar = new JarFile(file);){
                        BundleCoordinate bundleCoordinate = NarUnpacker.createBundleCoordinate(nar.getManifest());
                        if (!narFilter.test(bundleCoordinate)) {
                            logger.debug("Will not expand NAR {} because it does not match the provided filter", (Object)bundleCoordinate);
                            continue;
                        }
                        if (frameworkNarId != null && frameworkNarId.equals(bundleCoordinate.getId())) {
                            if (unpackedFramework != null) {
                                throw new IllegalStateException("Multiple framework NARs discovered. Only one framework is permitted.");
                            }
                            unpackedFramework = NarUnpacker.unpackNar(file, frameworkWorkingDir, verifyHash, unpackMode);
                            continue;
                        }
                        if ("nifi-jetty-bundle".equals(bundleCoordinate.getId())) {
                            if (unpackedJetty != null) {
                                throw new IllegalStateException("Multiple Jetty NARs discovered. Only one Jetty NAR is permitted.");
                            }
                            unpackedJetty = NarUnpacker.unpackNar(file, extensionsWorkingDir, verifyHash, unpackMode);
                            unpackedExtensions.add(unpackedJetty);
                            continue;
                        }
                        File unpackedExtension = NarUnpacker.unpackNar(file, extensionsWorkingDir, verifyHash, unpackMode);
                        unpackedExtensions.add(unpackedExtension);
                    }
                }
                if (requireFrameworkNar) {
                    if (unpackedFramework == null) {
                        throw new IllegalStateException("No framework NAR found.");
                    }
                    if (!unpackedFramework.canRead()) {
                        throw new IllegalStateException("Framework NAR cannot be read.");
                    }
                }
                if (requireJettyNar) {
                    if (unpackedJetty == null) {
                        throw new IllegalStateException("No Jetty NAR found.");
                    }
                    if (!unpackedJetty.canRead()) {
                        throw new IllegalStateException("Jetty NAR cannot be read.");
                    }
                }
                if (unpackedFramework != null && frameworkWorkingDir != null && (frameworkWorkingDirContents = frameworkWorkingDir.listFiles()) != null) {
                    for (File unpackedNar : frameworkWorkingDirContents) {
                        if (unpackedFramework.equals(unpackedNar)) continue;
                        FileUtils.deleteFile(unpackedNar, true);
                    }
                }
                if ((extensionsWorkingDirContents = extensionsWorkingDir.listFiles()) != null) {
                    for (File unpackedNar : extensionsWorkingDirContents) {
                        if (unpackedExtensions.contains(unpackedNar)) continue;
                        FileUtils.deleteFile(unpackedNar, true);
                    }
                }
                long l = System.nanoTime() - startTime;
                double durationSeconds = (double)TimeUnit.NANOSECONDS.toMillis(l) / 1000.0;
                logger.info("Expanded {} NAR files in {} seconds ({} ns)", new Object[]{narFiles.size(), durationSeconds, l});
            }
            HashMap<File, BundleCoordinate> unpackedNars = new HashMap<File, BundleCoordinate>(NarUnpacker.createUnpackedNarBundleCoordinateMap(extensionsWorkingDir));
            ExtensionMapping extensionMapping = new ExtensionMapping();
            NarUnpacker.mapExtensions(unpackedNars, docsWorkingDir, extensionMapping);
            NarUnpacker.unpackBundleDocs(docsWorkingDir, extensionMapping, systemBundle.getBundleDetails().getCoordinate(), systemBundle.getBundleDetails().getWorkingDirectory());
            return extensionMapping;
        }
        catch (IOException e) {
            logger.warn("Unable to load NAR bundles. Proceeding without loading any further NAR bundles", (Throwable)e);
            return null;
        }
    }

    private static Map<File, BundleCoordinate> createUnpackedNarBundleCoordinateMap(File extensionsWorkingDir) {
        File[] unpackedDirs = extensionsWorkingDir.listFiles(file -> file.isDirectory() && file.getName().endsWith("nar-unpacked"));
        if (unpackedDirs == null) {
            return Collections.emptyMap();
        }
        HashMap<File, BundleCoordinate> result = new HashMap<File, BundleCoordinate>();
        for (File unpackedDir : unpackedDirs) {
            Path mf = Paths.get(unpackedDir.getAbsolutePath(), "META-INF", "MANIFEST.MF");
            try (InputStream is = Files.newInputStream(mf, new OpenOption[0]);){
                Manifest manifest = new Manifest(is);
                BundleCoordinate bundleCoordinate = NarUnpacker.createBundleCoordinate(manifest);
                result.put(unpackedDir, bundleCoordinate);
            }
            catch (IOException e) {
                logger.error("Unable to parse NAR information from unpacked directory [{}]", (Object)unpackedDir.getAbsoluteFile(), (Object)e);
            }
        }
        return result;
    }

    private static BundleCoordinate createBundleCoordinate(Manifest manifest) {
        Attributes mainAttributes = manifest.getMainAttributes();
        String groupId = mainAttributes.getValue(NarManifestEntry.NAR_GROUP.getManifestName());
        String narId = mainAttributes.getValue(NarManifestEntry.NAR_ID.getManifestName());
        String version = mainAttributes.getValue(NarManifestEntry.NAR_VERSION.getManifestName());
        return new BundleCoordinate(groupId, narId, version);
    }

    private static void mapExtensions(Map<File, BundleCoordinate> unpackedNars, File docsDirectory, ExtensionMapping mapping) throws IOException {
        for (Map.Entry<File, BundleCoordinate> entry : unpackedNars.entrySet()) {
            File unpackedNar = entry.getKey();
            BundleCoordinate bundleCoordinate = entry.getValue();
            File bundledDependencies = new File(unpackedNar, BUNDLED_DEPENDENCIES_DIRECTORY);
            if (docsDirectory == null) continue;
            NarUnpacker.unpackBundleDocs(docsDirectory, mapping, bundleCoordinate, bundledDependencies);
        }
    }

    public static void mapExtension(File unpackedNar, BundleCoordinate bundleCoordinate, File docsDirectory, ExtensionMapping mapping) throws IOException {
        File bundledDependencies = new File(unpackedNar, BUNDLED_DEPENDENCIES_DIRECTORY);
        NarUnpacker.unpackBundleDocs(docsDirectory, mapping, bundleCoordinate, bundledDependencies);
    }

    private static void unpackBundleDocs(File docsDirectory, ExtensionMapping mapping, BundleCoordinate bundleCoordinate, File bundledDirectory) throws IOException {
        File[] directoryContents = bundledDirectory.listFiles();
        if (directoryContents != null) {
            for (File file : directoryContents) {
                if (!file.getName().toLowerCase().endsWith(".jar")) continue;
                NarUnpacker.unpackDocumentation(bundleCoordinate, file, docsDirectory, mapping);
            }
        }
    }

    public static File unpackNar(File nar, File baseWorkingDirectory, boolean verifyHash, NarUnpackMode unpackMode) throws IOException {
        File narWorkingDirectory = new File(baseWorkingDirectory, nar.getName() + "-unpacked");
        if (!narWorkingDirectory.exists()) {
            NarUnpacker.unpackIndividualJars(nar, narWorkingDirectory, FileDigestUtils.getDigest(nar), unpackMode);
        } else if (verifyHash) {
            byte[] narDigest = FileDigestUtils.getDigest(nar);
            File workingHashFile = new File(narWorkingDirectory, HASH_FILENAME);
            if (!workingHashFile.exists()) {
                FileUtils.deleteFile(narWorkingDirectory, true);
                NarUnpacker.unpackIndividualJars(nar, narWorkingDirectory, narDigest, unpackMode);
            } else {
                byte[] hashFileContents = Files.readAllBytes(workingHashFile.toPath());
                if (!Arrays.equals(hashFileContents, narDigest)) {
                    logger.info("Reloading changed NAR [{}]", (Object)nar.getAbsolutePath());
                    FileUtils.deleteFile(narWorkingDirectory, true);
                    NarUnpacker.unpackIndividualJars(nar, narWorkingDirectory, narDigest, unpackMode);
                }
            }
        } else {
            logger.debug("Directory {} already exists. Will not verify hash. Assuming nothing has changed.", (Object)narWorkingDirectory);
        }
        return narWorkingDirectory;
    }

    private static void unpackIndividualJars(File nar, File workingDirectory, byte[] hash, NarUnpackMode unpackMode) throws IOException {
        switch (unpackMode) {
            case UNPACK_INDIVIDUAL_JARS: {
                NarUnpacker.unpackIndividualJars(nar, workingDirectory, hash);
                return;
            }
            case UNPACK_TO_UBER_JAR: {
                NarUnpacker.unpackToUberJar(nar, workingDirectory, hash);
            }
        }
    }

    private static void unpackIndividualJars(File nar, File workingDirectory, byte[] hash) throws IOException {
        try (JarFile jarFile = new JarFile(nar);){
            Enumeration<JarEntry> jarEntries = jarFile.entries();
            while (jarEntries.hasMoreElements()) {
                JarEntry jarEntry = jarEntries.nextElement();
                File jarEntryFile = NarUnpacker.getMappedJarEntryFile(workingDirectory, jarEntry);
                if (jarEntry.isDirectory()) {
                    FileUtils.ensureDirectoryExistAndCanReadAndWrite(jarEntryFile);
                    continue;
                }
                NarUnpacker.makeFile(jarFile.getInputStream(jarEntry), jarEntryFile);
            }
        }
        File hashFile = new File(workingDirectory, HASH_FILENAME);
        try (FileOutputStream fos = new FileOutputStream(hashFile);){
            fos.write(hash);
        }
    }

    private static void unpackToUberJar(File nar, File workingDirectory, byte[] hash) throws IOException {
        logger.debug("====================================");
        logger.debug("Unpacking NAR {}", (Object)nar.getAbsolutePath());
        File unpackedUberJarFile = new File(workingDirectory, "NAR-INF/bundled-dependencies/" + nar.getName() + ".unpacked.uber.jar");
        Files.createDirectories(workingDirectory.toPath(), new FileAttribute[0]);
        Files.createDirectories(unpackedUberJarFile.getParentFile().toPath(), new FileAttribute[0]);
        HashSet<String> entriesCreated = new HashSet<String>();
        try (JarFile jarFile = new JarFile(nar);
             FileOutputStream out = new FileOutputStream(unpackedUberJarFile);
             BufferedOutputStream bufferedOut = new BufferedOutputStream(out);
             JarOutputStream uberJarOut = new JarOutputStream(bufferedOut);){
            Enumeration<JarEntry> jarEntries = jarFile.entries();
            while (jarEntries.hasMoreElements()) {
                InputStream entryIn;
                JarEntry jarEntry = jarEntries.nextElement();
                File jarEntryFile = NarUnpacker.getMappedJarEntryFile(workingDirectory, jarEntry);
                logger.debug("Unpacking NAR entry {}", (Object)jarEntryFile);
                if (!entriesCreated.add(jarEntry.getName())) continue;
                String jarEntryFilePath = jarEntryFile.getAbsolutePath();
                if (jarEntryFilePath.contains("META-INF") || jarEntryFilePath.contains("NAR-INF") && jarEntryFilePath.endsWith(".war")) {
                    if (jarEntry.isDirectory()) continue;
                    Files.createDirectories(jarEntryFile.getParentFile().toPath(), new FileAttribute[0]);
                    entryIn = jarFile.getInputStream(jarEntry);
                    try {
                        FileOutputStream manifestOut = new FileOutputStream(jarEntryFile);
                        try {
                            NarUnpacker.copy(entryIn, manifestOut);
                            continue;
                        }
                        finally {
                            ((OutputStream)manifestOut).close();
                            continue;
                        }
                    }
                    finally {
                        if (entryIn != null) {
                            entryIn.close();
                        }
                        continue;
                    }
                }
                if (jarEntry.isDirectory()) {
                    uberJarOut.putNextEntry(new JarEntry(jarEntry.getName()));
                    continue;
                }
                if (jarEntryFilePath.endsWith(".jar")) {
                    logger.debug("Unpacking JAR {}", (Object)jarEntryFile);
                    entryIn = jarFile.getInputStream(jarEntry);
                    try {
                        BufferedInputStream in = new BufferedInputStream(entryIn);
                        try {
                            NarUnpacker.copyJarContents(in, uberJarOut, entriesCreated, workingDirectory);
                            continue;
                        }
                        finally {
                            ((InputStream)in).close();
                            continue;
                        }
                    }
                    finally {
                        if (entryIn != null) {
                            entryIn.close();
                        }
                        continue;
                    }
                }
                JarEntry fileEntry = new JarEntry(jarEntry.getName());
                uberJarOut.putNextEntry(fileEntry);
                try (InputStream entryIn2 = jarFile.getInputStream(jarEntry);
                     BufferedInputStream in = new BufferedInputStream(entryIn2);){
                    NarUnpacker.copy(in, uberJarOut);
                }
                uberJarOut.closeEntry();
            }
        }
        File hashFile = new File(workingDirectory, HASH_FILENAME);
        try (FileOutputStream fos = new FileOutputStream(hashFile);){
            fos.write(hash);
        }
    }

    private static void copyJarContents(InputStream in, JarOutputStream out, Set<String> entriesCreated, File workingDirectory) throws IOException {
        try (JarInputStream jarInputStream = new JarInputStream(in);){
            JarEntry jarEntry;
            while ((jarEntry = jarInputStream.getNextJarEntry()) != null) {
                String entryName = jarEntry.getName();
                File outFile = NarUnpacker.getJarEntryFile(workingDirectory, entryName);
                if (entryName.contains("META-INF/") && !entryName.contains("META-INF/MANIFEST.MF") && !jarEntry.isDirectory()) {
                    logger.debug("Found META-INF/services file {}", (Object)entryName);
                    File outDir = outFile.getParentFile();
                    if (!outDir.exists() && !outDir.mkdirs()) {
                        logger.debug("Skipping unpacking {} because parent file does not exist and could not be created", (Object)outFile);
                        continue;
                    }
                    if (!outDir.isDirectory()) {
                        logger.debug("Skipping unpacking {} because parent file is not a directory", (Object)outFile);
                        continue;
                    }
                    FileOutputStream metaInfFileOut = new FileOutputStream(outFile, true);
                    try {
                        BufferedOutputStream bufferedOut = new BufferedOutputStream(metaInfFileOut);
                        try {
                            NarUnpacker.copy(jarInputStream, bufferedOut);
                            ((OutputStream)bufferedOut).write("\n".getBytes(StandardCharsets.UTF_8));
                            continue;
                        }
                        finally {
                            ((OutputStream)bufferedOut).close();
                            continue;
                        }
                    }
                    finally {
                        ((OutputStream)metaInfFileOut).close();
                        continue;
                    }
                }
                if (!entriesCreated.add(entryName)) {
                    logger.debug("Skipping entry {} in {} because an entry with that name already exists", (Object)entryName, (Object)workingDirectory);
                    continue;
                }
                JarEntry outEntry = new JarEntry(jarEntry.getName());
                out.putNextEntry(outEntry);
                if (!jarEntry.isDirectory()) {
                    NarUnpacker.copy(jarInputStream, out);
                }
                out.closeEntry();
            }
        }
    }

    private static void copy(InputStream in, OutputStream out) throws IOException {
        int len;
        byte[] buffer = new byte[4096];
        while ((len = in.read(buffer)) > 0) {
            out.write(buffer, 0, len);
        }
    }

    private static void unpackDocumentation(BundleCoordinate coordinate, File jar, File docsDirectory, ExtensionMapping extensionMapping) throws IOException {
        ExtensionMapping jarExtensionMapping = NarUnpacker.determineDocumentedNiFiComponents(coordinate, jar);
        if (jarExtensionMapping.isEmpty()) {
            return;
        }
        extensionMapping.merge(jarExtensionMapping);
        if (docsDirectory == null) {
            return;
        }
        try (JarFile jarFile = new JarFile(jar);){
            block5: for (String componentName : jarExtensionMapping.getAllExtensionNames().keySet()) {
                String componentDocumentationDirectory = Paths.get(JAR_DOCUMENTATION_ROOT_PATH, componentName).toString();
                Enumeration<JarEntry> jarEnumeration = jarFile.entries();
                while (jarEnumeration.hasMoreElements()) {
                    JarEntry jarEntry = jarEnumeration.nextElement();
                    File jarEntryFile = NarUnpacker.getJarEntryFile(docsDirectory, jarEntry.getName());
                    String jarEntryFileAbsolutePath = jarEntryFile.getAbsolutePath();
                    if (!jarEntryFileAbsolutePath.contains(componentDocumentationDirectory)) continue;
                    String relativePath = StringUtils.substringAfter((String)jarEntryFileAbsolutePath, (String)componentDocumentationDirectory);
                    String outputPath = Paths.get(coordinate.getGroup(), coordinate.getId(), coordinate.getVersion(), componentName, relativePath).toString();
                    if (jarEntry.isDirectory()) {
                        File componentDocsDirectory = new File(docsDirectory, outputPath);
                        if (componentDocsDirectory.exists() || componentDocsDirectory.mkdirs()) continue;
                        logger.warn("Unable to create docs directory {}", (Object)componentDocsDirectory.getAbsolutePath());
                        continue block5;
                    }
                    File componentDoc = new File(docsDirectory, outputPath);
                    NarUnpacker.makeFile(jarFile.getInputStream(jarEntry), componentDoc);
                }
            }
        }
    }

    private static ExtensionMapping determineDocumentedNiFiComponents(BundleCoordinate coordinate, File jar) throws IOException {
        ExtensionMapping mapping = new ExtensionMapping();
        try (JarFile jarFile = new JarFile(jar);){
            JarEntry processorEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.processor.Processor");
            JarEntry reportingTaskEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.reporting.ReportingTask");
            JarEntry flowAnalysisRuleEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.flowanalysis.FlowAnalysisRule");
            JarEntry controllerServiceEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.controller.ControllerService");
            JarEntry parameterProviderEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.parameter.ParameterProvider");
            JarEntry flowRegistryClientEntry = jarFile.getJarEntry("META-INF/services/org.apache.nifi.registry.flow.FlowRegistryClient");
            if (processorEntry == null && reportingTaskEntry == null && flowAnalysisRuleEntry == null && controllerServiceEntry == null && parameterProviderEntry == null) {
                ExtensionMapping extensionMapping = mapping;
                return extensionMapping;
            }
            mapping.addAllProcessors(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, processorEntry));
            mapping.addAllReportingTasks(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, reportingTaskEntry));
            mapping.addAllFlowAnalysisRules(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, flowAnalysisRuleEntry));
            mapping.addAllControllerServices(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, controllerServiceEntry));
            mapping.addAllParameterProviders(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, parameterProviderEntry));
            mapping.addAllFlowRegistryClients(coordinate, NarUnpacker.determineDocumentedNiFiComponents(jarFile, flowRegistryClientEntry));
            ExtensionMapping extensionMapping = mapping;
            return extensionMapping;
        }
    }

    private static List<String> determineDocumentedNiFiComponents(JarFile jarFile, JarEntry jarEntry) throws IOException {
        ArrayList<String> componentNames = new ArrayList<String>();
        if (jarEntry == null) {
            return componentNames;
        }
        try (InputStream entryInputStream = jarFile.getInputStream(jarEntry);
             BufferedReader reader = new BufferedReader(new InputStreamReader(entryInputStream));){
            String line;
            while ((line = reader.readLine()) != null) {
                String trimmedLine = line.trim();
                if (trimmedLine.isEmpty() || trimmedLine.startsWith("#")) continue;
                int indexOfPound = trimmedLine.indexOf("#");
                String effectiveLine = indexOfPound > 0 ? trimmedLine.substring(0, indexOfPound) : trimmedLine;
                componentNames.add(effectiveLine);
            }
        }
        return componentNames;
    }

    private static void makeFile(InputStream inputStream, File file) throws IOException {
        try (InputStream in = inputStream;
             FileOutputStream fos = new FileOutputStream(file);){
            int numRead;
            byte[] bytes = new byte[65536];
            while ((numRead = in.read(bytes)) != -1) {
                fos.write(bytes, 0, numRead);
            }
        }
    }

    private static File getMappedJarEntryFile(File workingDirectory, JarEntry jarEntry) {
        String jarEntryName = jarEntry.getName().replace(BUNDLED_DEPENDENCIES_PREFIX, BUNDLED_DEPENDENCIES_DIRECTORY);
        return NarUnpacker.getJarEntryFile(workingDirectory, jarEntryName);
    }

    private static File getJarEntryFile(File workingDirectory, String jarEntryName) {
        Path workingDirectoryPath = workingDirectory.toPath().normalize();
        Path jarEntryPath = workingDirectoryPath.resolve(jarEntryName).normalize();
        if (jarEntryPath.startsWith(workingDirectoryPath)) {
            return jarEntryPath.toFile();
        }
        throw new IllegalArgumentException(String.format("NAR Entry path not valid [%s]", jarEntryName));
    }

    private NarUnpacker() {
    }
}

