/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.registry.bootstrap;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.attribute.PosixFilePermission;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.bootstrap.util.OSUtils;
import org.apache.nifi.bootstrap.util.RuntimeVersionProvider;
import org.apache.nifi.deprecation.log.DeprecationLogger;
import org.apache.nifi.deprecation.log.DeprecationLoggerFactory;
import org.apache.nifi.registry.bootstrap.NiFiRegistryListener;
import org.apache.nifi.registry.bootstrap.ShutdownHook;
import org.apache.nifi.registry.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RunNiFiRegistry {
    public static final String DEFAULT_CONFIG_FILE = "./conf/bootstrap.conf";
    public static final String DEFAULT_JAVA_CMD = "java";
    public static final String DEFAULT_PID_DIR = "bin";
    public static final String DEFAULT_LOG_DIR = "./logs";
    public static final String DEFAULT_DOCS_DIR = "./docs";
    public static final String GRACEFUL_SHUTDOWN_PROP = "graceful.shutdown.seconds";
    public static final String DEFAULT_GRACEFUL_SHUTDOWN_VALUE = "20";
    public static final String NIFI_REGISTRY_PID_DIR_PROP = "org.apache.nifi.registry.bootstrap.config.pid.dir";
    public static final String NIFI_REGISTRY_PID_FILE_NAME = "nifi-registry.pid";
    public static final String NIFI_REGISTRY_STATUS_FILE_NAME = "nifi-registry.status";
    public static final String NIFI_REGISTRY_LOCK_FILE_NAME = "nifi-registry.lock";
    public static final String NIFI_REGISTRY_BOOTSTRAP_SENSITIVE_KEY = "nifi.registry.bootstrap.sensitive.key";
    public static final String PID_KEY = "pid";
    public static final int STARTUP_WAIT_SECONDS = 60;
    public static final String SHUTDOWN_CMD = "SHUTDOWN";
    public static final String PING_CMD = "PING";
    public static final String DUMP_CMD = "DUMP";
    private static final int UNINITIALIZED_CC_PORT = -1;
    private volatile boolean autoRestartNiFiRegistry = true;
    private volatile int ccPort = -1;
    private volatile long nifiRegistryPid = -1L;
    private volatile String secretKey;
    private volatile ShutdownHook shutdownHook;
    private volatile boolean nifiRegistryStarted;
    private final Lock startedLock = new ReentrantLock();
    private final Lock lock = new ReentrantLock();
    private final Condition startupCondition = this.lock.newCondition();
    private final File bootstrapConfigFile;
    private final Logger cmdLogger = LoggerFactory.getLogger((String)"org.apache.nifi.registry.bootstrap.Command");
    private final Logger defaultLogger = LoggerFactory.getLogger(RunNiFiRegistry.class);
    private final DeprecationLogger deprecationLogger = DeprecationLoggerFactory.getLogger(RunNiFiRegistry.class);
    private final ExecutorService loggingExecutor;
    private volatile Set<Future<?>> loggingFutures = new HashSet(2);

    public RunNiFiRegistry(File bootstrapConfigFile, boolean verbose) throws IOException {
        this.bootstrapConfigFile = bootstrapConfigFile;
        this.loggingExecutor = Executors.newFixedThreadPool(2, new ThreadFactory(){

            @Override
            public Thread newThread(Runnable runnable) {
                Thread t = Executors.defaultThreadFactory().newThread(runnable);
                t.setDaemon(true);
                t.setName("NiFi logging handler");
                return t;
            }
        });
    }

    private static void printUsage() {
        System.out.println("Usage:");
        System.out.println();
        System.out.println("java org.apache.nifi.bootstrap.RunNiFiRegistry [<-verbose>] <command> [options]");
        System.out.println();
        System.out.println("Valid commands include:");
        System.out.println("");
        System.out.println("Start : Start a new instance of Apache NiFi Registry");
        System.out.println("Stop : Stop a running instance of Apache NiFi Registry");
        System.out.println("Restart : Stop Apache NiFi Registry, if it is running, and then start a new instance");
        System.out.println("Status : Determine if there is a running instance of Apache NiFi Registry");
        System.out.println("Dump : Write a Thread Dump to the file specified by [options], or to the log if no file is given");
        System.out.println("Run : Start a new instance of Apache NiFi Registry and monitor the Process, restarting if the instance dies");
        System.out.println();
    }

    private static String[] shift(String[] orig) {
        return Arrays.copyOfRange(orig, 1, orig.length);
    }

    public static void main(String[] args) throws IOException, InterruptedException {
        String cmd;
        if (args.length < 1 || args.length > 3) {
            RunNiFiRegistry.printUsage();
            return;
        }
        File dumpFile = null;
        boolean verbose = false;
        if (args[0].equals("-verbose")) {
            verbose = true;
            args = RunNiFiRegistry.shift(args);
        }
        if ((cmd = args[0]).equals("dump")) {
            dumpFile = args.length > 1 ? new File(args[1]) : null;
        }
        switch (cmd.toLowerCase()) {
            case "start": 
            case "run": 
            case "stop": 
            case "status": 
            case "dump": 
            case "restart": 
            case "env": {
                break;
            }
            default: {
                RunNiFiRegistry.printUsage();
                return;
            }
        }
        File configFile = RunNiFiRegistry.getDefaultBootstrapConfFile();
        RunNiFiRegistry runNiFiRegistry = new RunNiFiRegistry(configFile, verbose);
        Integer exitStatus = null;
        switch (cmd.toLowerCase()) {
            case "start": {
                runNiFiRegistry.start();
                break;
            }
            case "run": {
                runNiFiRegistry.start();
                break;
            }
            case "stop": {
                runNiFiRegistry.stop();
                break;
            }
            case "status": {
                exitStatus = runNiFiRegistry.status();
                break;
            }
            case "restart": {
                runNiFiRegistry.stop();
                runNiFiRegistry.start();
                break;
            }
            case "dump": {
                runNiFiRegistry.dump(dumpFile);
                break;
            }
            case "env": {
                runNiFiRegistry.env();
            }
        }
        if (exitStatus != null) {
            System.exit(exitStatus);
        }
    }

    private static File getDefaultBootstrapConfFile() {
        String nifiRegistryHome;
        String configFilename = System.getProperty("org.apache.nifi.registry.bootstrap.config.file");
        if (configFilename == null && (nifiRegistryHome = System.getenv("NIFI_REGISTRY_HOME")) != null) {
            File nifiRegistryHomeFile = new File(nifiRegistryHome.trim());
            File configFile = new File(nifiRegistryHomeFile, DEFAULT_CONFIG_FILE);
            configFilename = configFile.getAbsolutePath();
        }
        if (configFilename == null) {
            configFilename = DEFAULT_CONFIG_FILE;
        }
        File configFile = new File(configFilename);
        return configFile;
    }

    protected File getBootstrapFile(Logger logger, String directory, String defaultDirectory, String fileName) throws IOException {
        File confDir = this.bootstrapConfigFile.getParentFile();
        File nifiHome = confDir.getParentFile();
        String confFileDir = System.getProperty(directory);
        File fileDir = confFileDir != null ? new File(confFileDir.trim()) : new File(nifiHome, defaultDirectory);
        FileUtils.ensureDirectoryExistAndCanAccess((File)fileDir);
        File statusFile = new File(fileDir, fileName);
        logger.debug("Status File: {}", (Object)statusFile);
        return statusFile;
    }

    protected File getPidFile(Logger logger) throws IOException {
        return this.getBootstrapFile(logger, NIFI_REGISTRY_PID_DIR_PROP, DEFAULT_PID_DIR, NIFI_REGISTRY_PID_FILE_NAME);
    }

    protected File getStatusFile(Logger logger) throws IOException {
        return this.getBootstrapFile(logger, NIFI_REGISTRY_PID_DIR_PROP, DEFAULT_PID_DIR, NIFI_REGISTRY_STATUS_FILE_NAME);
    }

    protected File getLockFile(Logger logger) throws IOException {
        return this.getBootstrapFile(logger, NIFI_REGISTRY_PID_DIR_PROP, DEFAULT_PID_DIR, NIFI_REGISTRY_LOCK_FILE_NAME);
    }

    protected File getStatusFile() throws IOException {
        return this.getStatusFile(this.defaultLogger);
    }

    private Properties loadProperties(Logger logger) throws IOException {
        Properties props = new Properties();
        File statusFile = this.getStatusFile(logger);
        if (statusFile == null || !statusFile.exists()) {
            logger.debug("No status file to load properties from");
            return props;
        }
        try (FileInputStream fis = new FileInputStream(this.getStatusFile(logger));){
            props.load(fis);
        }
        HashMap<Object, Object> modified = new HashMap<Object, Object>(props);
        modified.remove("secret.key");
        logger.debug("Properties: {}", modified);
        return props;
    }

    private synchronized void savePidProperties(Properties pidProperties, Logger logger) throws IOException {
        File statusFile;
        String pid = pidProperties.getProperty(PID_KEY);
        if (!StringUtils.isBlank((CharSequence)pid)) {
            this.writePidFile(pid, logger);
        }
        if ((statusFile = this.getStatusFile(logger)).exists() && !statusFile.delete()) {
            logger.warn("Failed to delete {}", (Object)statusFile);
        }
        if (!statusFile.createNewFile()) {
            throw new IOException("Failed to create file " + statusFile);
        }
        try {
            HashSet<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
            perms.add(PosixFilePermission.OWNER_READ);
            perms.add(PosixFilePermission.OWNER_WRITE);
            Files.setPosixFilePermissions(statusFile.toPath(), perms);
        }
        catch (Exception e) {
            logger.warn("Failed to set permissions so that only the owner can read status file {}; this may allows others to have access to the key needed to communicate with NiFi Registry. Permissions should be changed so that only the owner can read this file", (Object)statusFile);
        }
        try (FileOutputStream fos = new FileOutputStream(statusFile);){
            pidProperties.store(fos, null);
            fos.getFD().sync();
        }
        logger.debug("Saved Properties {} to {}", new Object[]{pidProperties, statusFile});
    }

    private synchronized void writePidFile(String pid, Logger logger) throws IOException {
        File pidFile = this.getPidFile(logger);
        if (pidFile.exists() && !pidFile.delete()) {
            logger.warn("Failed to delete {}", (Object)pidFile);
        }
        if (!pidFile.createNewFile()) {
            throw new IOException("Failed to create file " + pidFile);
        }
        try {
            HashSet<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
            perms.add(PosixFilePermission.OWNER_WRITE);
            perms.add(PosixFilePermission.OWNER_READ);
            perms.add(PosixFilePermission.GROUP_READ);
            perms.add(PosixFilePermission.OTHERS_READ);
            Files.setPosixFilePermissions(pidFile.toPath(), perms);
        }
        catch (Exception e) {
            logger.warn("Failed to set permissions so that only the owner can read pid file {}; this may allows others to have access to the key needed to communicate with NiFi Registry. Permissions should be changed so that only the owner can read this file", (Object)pidFile);
        }
        try (FileOutputStream fos = new FileOutputStream(pidFile);){
            fos.write(pid.getBytes(StandardCharsets.UTF_8));
            fos.getFD().sync();
        }
        logger.debug("Saved Pid {} to {}", new Object[]{pid, pidFile});
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isPingSuccessful(int port, String secretKey, Logger logger) {
        logger.debug("Pinging {}", (Object)port);
        try (Socket socket = new Socket("localhost", port);){
            OutputStream out = socket.getOutputStream();
            out.write(("PING " + secretKey + "\n").getBytes(StandardCharsets.UTF_8));
            out.flush();
            logger.debug("Sent PING command");
            socket.setSoTimeout(5000);
            InputStream in = socket.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(in));
            String response = reader.readLine();
            logger.debug("PING response: {}", (Object)response);
            out.close();
            reader.close();
            boolean bl = PING_CMD.equals(response);
            return bl;
        }
        catch (IOException ioe) {
            return false;
        }
    }

    private Integer getCurrentPort(Logger logger) throws IOException {
        Properties props = this.loadProperties(logger);
        String portVal = props.getProperty("port");
        if (portVal == null) {
            logger.debug("No Port found in status file");
            return null;
        }
        logger.debug("Port defined in status file: {}", (Object)portVal);
        int port = Integer.parseInt(portVal);
        boolean success = this.isPingSuccessful(port, props.getProperty("secret.key"), logger);
        if (success) {
            logger.debug("Successful PING on port {}", (Object)port);
            return port;
        }
        String pid = props.getProperty(PID_KEY);
        logger.debug("PID in status file is {}", (Object)pid);
        if (pid != null) {
            boolean procRunning = this.isProcessRunning(pid, logger);
            if (procRunning) {
                return port;
            }
            return null;
        }
        return null;
    }

    private boolean isProcessRunning(String pid, Logger logger) {
        try {
            ProcessBuilder builder = new ProcessBuilder(new String[0]);
            builder.command("ps", "-p", pid);
            Process proc = builder.start();
            boolean running = false;
            try (InputStream in = proc.getInputStream();
                 InputStreamReader streamReader = new InputStreamReader(in);
                 BufferedReader reader = new BufferedReader(streamReader);){
                String line;
                while ((line = reader.readLine()) != null) {
                    if (!line.trim().startsWith(pid)) continue;
                    running = true;
                }
            }
            if (running) {
                logger.debug("Process with PID {} is running", (Object)pid);
            } else {
                logger.debug("Process with PID {} is not running", (Object)pid);
            }
            return running;
        }
        catch (IOException ioe) {
            System.err.println("Failed to determine if Process " + pid + " is running; assuming that it is not");
            return false;
        }
    }

    private Status getStatus(Logger logger) {
        Properties props;
        try {
            props = this.loadProperties(logger);
        }
        catch (IOException ioe) {
            return new Status(null, null, false, false);
        }
        if (props == null) {
            return new Status(null, null, false, false);
        }
        String portValue = props.getProperty("port");
        String pid = props.getProperty(PID_KEY);
        String secretKey = props.getProperty("secret.key");
        if (portValue == null && pid == null) {
            return new Status(null, null, false, false);
        }
        Integer port = null;
        boolean pingSuccess = false;
        if (portValue != null) {
            try {
                port = Integer.parseInt(portValue);
                pingSuccess = this.isPingSuccessful(port, secretKey, logger);
            }
            catch (NumberFormatException nfe) {
                return new Status(null, null, false, false);
            }
        }
        if (pingSuccess) {
            return new Status(port, pid, true, true);
        }
        boolean alive = pid != null && this.isProcessRunning(pid, logger);
        return new Status(port, pid, pingSuccess, alive);
    }

    public int status() throws IOException {
        Logger logger = this.cmdLogger;
        Status status = this.getStatus(logger);
        if (status.isRespondingToPing()) {
            logger.info("Apache NiFi Registry is currently running, listening to Bootstrap on port {}, PID={}", new Object[]{status.getPort(), status.getPid() == null ? "unknown" : status.getPid()});
            return 0;
        }
        if (status.isProcessRunning()) {
            logger.info("Apache NiFi Registry is running at PID {} but is not responding to ping requests", (Object)status.getPid());
            return 4;
        }
        if (status.getPort() == null) {
            logger.info("Apache NiFi Registry is not running");
            return 3;
        }
        if (status.getPid() == null) {
            logger.info("Apache NiFi Registry is not responding to Ping requests. The process may have died or may be hung");
        } else {
            logger.info("Apache NiFi Registry is not running");
        }
        return 3;
    }

    public void env() {
        Object virtualMachine;
        Method detachMethod;
        Method attachMethod;
        Class<?> virtualMachineClass;
        Logger logger = this.cmdLogger;
        Status status = this.getStatus(logger);
        if (status.getPid() == null) {
            logger.info("Apache NiFi Registry is not running");
            return;
        }
        try {
            virtualMachineClass = Class.forName("com.sun.tools.attach.VirtualMachine");
        }
        catch (ClassNotFoundException cnfe) {
            logger.error("Seems tools.jar (Linux / Windows JDK) or classes.jar (Mac OS) is not available in classpath");
            return;
        }
        try {
            attachMethod = virtualMachineClass.getMethod("attach", String.class);
            detachMethod = virtualMachineClass.getDeclaredMethod("detach", new Class[0]);
        }
        catch (Exception e) {
            logger.error("Methods required for getting environment not available", (Throwable)e);
            return;
        }
        try {
            virtualMachine = attachMethod.invoke(null, status.getPid());
        }
        catch (Throwable t) {
            logger.error("Problem attaching to NiFi", t);
            return;
        }
        try {
            Method getSystemPropertiesMethod = virtualMachine.getClass().getMethod("getSystemProperties", new Class[0]);
            Properties sysProps = (Properties)getSystemPropertiesMethod.invoke(virtualMachine, new Object[0]);
            for (Map.Entry<Object, Object> syspropEntry : sysProps.entrySet()) {
                logger.info(syspropEntry.getKey().toString() + " = " + syspropEntry.getValue().toString());
            }
        }
        catch (Throwable t) {
            throw new RuntimeException(t);
        }
        finally {
            try {
                detachMethod.invoke(virtualMachine, new Object[0]);
            }
            catch (Exception e) {
                logger.warn("Caught exception detaching from process", (Throwable)e);
            }
        }
    }

    public void dump(File dumpFile) throws IOException {
        Logger logger = this.defaultLogger;
        Integer port = this.getCurrentPort(logger);
        if (port == null) {
            logger.info("Apache NiFi Registry is not currently running");
            return;
        }
        Properties nifiRegistryProps = this.loadProperties(logger);
        String secretKey = nifiRegistryProps.getProperty("secret.key");
        StringBuilder sb = new StringBuilder();
        try (Socket socket = new Socket();){
            logger.debug("Connecting to NiFi Registry instance");
            socket.setSoTimeout(60000);
            socket.connect(new InetSocketAddress("localhost", (int)port));
            logger.debug("Established connection to NiFi Registry instance.");
            socket.setSoTimeout(60000);
            logger.debug("Sending DUMP Command to port {}", (Object)port);
            OutputStream out = socket.getOutputStream();
            out.write(("DUMP " + secretKey + "\n").getBytes(StandardCharsets.UTF_8));
            out.flush();
            InputStream in = socket.getInputStream();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                String line;
                while ((line = reader.readLine()) != null) {
                    sb.append(line).append("\n");
                }
            }
        }
        String dump = sb.toString();
        if (dumpFile == null) {
            logger.info(dump);
        } else {
            try (FileOutputStream fos = new FileOutputStream(dumpFile);){
                fos.write(dump.getBytes(StandardCharsets.UTF_8));
            }
            this.cmdLogger.info("Successfully wrote thread dump to {}", (Object)dumpFile.getAbsolutePath());
        }
    }

    public void notifyStop() {
        String hostname = this.getHostname();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");
        String now = sdf.format(System.currentTimeMillis());
        String user = System.getProperty("user.name");
        if (user == null || user.trim().isEmpty()) {
            user = "Unknown User";
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() throws IOException {
        block47: {
            Logger logger = this.cmdLogger;
            Integer port = this.getCurrentPort(logger);
            if (port == null) {
                logger.info("Apache NiFi Registry is not currently running");
                return;
            }
            File lockFile = this.getLockFile(logger);
            if (!lockFile.exists()) {
                lockFile.createNewFile();
            }
            Properties nifiRegistryProps = this.loadProperties(logger);
            String secretKey = nifiRegistryProps.getProperty("secret.key");
            String pid = nifiRegistryProps.getProperty(PID_KEY);
            File statusFile = this.getStatusFile(logger);
            File pidFile = this.getPidFile(logger);
            try (Socket socket = new Socket();){
                int lastChar;
                logger.debug("Connecting to NiFi Registry instance");
                socket.setSoTimeout(10000);
                socket.connect(new InetSocketAddress("localhost", (int)port));
                logger.debug("Established connection to NiFi Registry instance.");
                socket.setSoTimeout(10000);
                logger.debug("Sending SHUTDOWN Command to port {}", (Object)port);
                OutputStream out = socket.getOutputStream();
                out.write(("SHUTDOWN " + secretKey + "\n").getBytes(StandardCharsets.UTF_8));
                out.flush();
                socket.shutdownOutput();
                InputStream in = socket.getInputStream();
                StringBuilder sb = new StringBuilder();
                while ((lastChar = in.read()) > -1) {
                    sb.append((char)lastChar);
                }
                String response = sb.toString().trim();
                logger.debug("Received response to SHUTDOWN command: {}", (Object)response);
                if (SHUTDOWN_CMD.equals(response)) {
                    int gracefulShutdownSeconds;
                    logger.info("Apache NiFi Registry has accepted the Shutdown Command and is shutting down now");
                    if (pid == null) break block47;
                    Properties bootstrapProperties = new Properties();
                    try (FileInputStream fis = new FileInputStream(this.bootstrapConfigFile);){
                        bootstrapProperties.load(fis);
                    }
                    String gracefulShutdown = bootstrapProperties.getProperty(GRACEFUL_SHUTDOWN_PROP, DEFAULT_GRACEFUL_SHUTDOWN_VALUE);
                    try {
                        gracefulShutdownSeconds = Integer.parseInt(gracefulShutdown);
                    }
                    catch (NumberFormatException nfe) {
                        gracefulShutdownSeconds = Integer.parseInt(DEFAULT_GRACEFUL_SHUTDOWN_VALUE);
                    }
                    this.notifyStop();
                    long startWait = System.nanoTime();
                    while (this.isProcessRunning(pid, logger)) {
                        logger.info("Waiting for Apache NiFi Registry to finish shutting down...");
                        long waitNanos = System.nanoTime() - startWait;
                        long waitSeconds = TimeUnit.NANOSECONDS.toSeconds(waitNanos);
                        if (waitSeconds >= (long)gracefulShutdownSeconds && gracefulShutdownSeconds > 0) {
                            if (!this.isProcessRunning(pid, logger)) break;
                            logger.warn("NiFi Registry has not finished shutting down after {} seconds. Killing process.", (Object)gracefulShutdownSeconds);
                            try {
                                this.killProcessTree(pid, logger);
                            }
                            catch (IOException ioe) {
                                logger.error("Failed to kill Process with PID {}", (Object)pid);
                            }
                            break;
                        }
                        try {
                            Thread.sleep(2000L);
                        }
                        catch (InterruptedException interruptedException) {}
                    }
                    if (statusFile.exists() && !statusFile.delete()) {
                        logger.error("Failed to delete status file {}; this file should be cleaned up manually", (Object)statusFile);
                    }
                    if (pidFile.exists() && !pidFile.delete()) {
                        logger.error("Failed to delete pid file {}; this file should be cleaned up manually", (Object)pidFile);
                    }
                    logger.info("NiFi Registry has finished shutting down.");
                    break block47;
                }
                logger.error("When sending SHUTDOWN command to NiFi Registry , got unexpected response {}", (Object)response);
            }
            catch (IOException ioe) {
                if (pid == null) {
                    logger.error("Failed to send shutdown command to port {} due to {}. No PID found for the NiFi Registry process, so unable to kill process; the process should be killed manually.", new Object[]{port, ioe.toString()});
                } else {
                    logger.error("Failed to send shutdown command to port {} due to {}. Will kill the NiFi Registry Process with PID {}.", new Object[]{port, ioe.toString(), pid});
                    this.notifyStop();
                    this.killProcessTree(pid, logger);
                    if (statusFile.exists() && !statusFile.delete()) {
                        logger.error("Failed to delete status file {}; this file should be cleaned up manually", (Object)statusFile);
                    }
                }
            }
            finally {
                if (lockFile.exists() && !lockFile.delete()) {
                    logger.error("Failed to delete lock file {}; this file should be cleaned up manually", (Object)lockFile);
                }
            }
        }
    }

    private static List<String> getChildProcesses(String ppid) throws IOException {
        Process proc = Runtime.getRuntime().exec(new String[]{"ps", "-o", PID_KEY, "--no-headers", "--ppid", ppid});
        ArrayList<String> childPids = new ArrayList<String>();
        try (InputStream in = proc.getInputStream();
             BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
            String line;
            while ((line = reader.readLine()) != null) {
                childPids.add(line.trim());
            }
        }
        return childPids;
    }

    private void killProcessTree(String pid, Logger logger) throws IOException {
        logger.debug("Killing Process Tree for PID {}", (Object)pid);
        List<String> children = RunNiFiRegistry.getChildProcesses(pid);
        logger.debug("Children of PID {}: {}", new Object[]{pid, children});
        for (String childPid : children) {
            this.killProcessTree(childPid, logger);
        }
        Runtime.getRuntime().exec(new String[]{"kill", "-9", pid});
    }

    public static boolean isAlive(Process process) {
        try {
            process.exitValue();
            return false;
        }
        catch (IllegalStateException | IllegalThreadStateException itse) {
            return true;
        }
    }

    private String getHostname() {
        String hostname = "Unknown Host";
        String ip = "Unknown IP Address";
        try {
            InetAddress localhost = InetAddress.getLocalHost();
            hostname = localhost.getHostName();
            ip = localhost.getHostAddress();
        }
        catch (Exception e) {
            this.defaultLogger.warn("Failed to obtain hostname for notification due to:", (Throwable)e);
        }
        return hostname + " (" + ip + ")";
    }

    public void start() throws IOException, InterruptedException {
        int gracefulShutdownSeconds;
        String javaHome;
        String libJava11Filename;
        File libJava11Dir;
        File workingDir;
        Integer port = this.getCurrentPort(this.cmdLogger);
        if (port != null) {
            this.cmdLogger.info("Apache NiFi Registry is already running, listening to Bootstrap on port " + port);
            return;
        }
        File prevLockFile = this.getLockFile(this.cmdLogger);
        if (prevLockFile.exists() && !prevLockFile.delete()) {
            this.cmdLogger.warn("Failed to delete previous lock file {}; this file should be cleaned up manually", (Object)prevLockFile);
        }
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        if (!this.bootstrapConfigFile.exists()) {
            throw new FileNotFoundException(this.bootstrapConfigFile.getAbsolutePath());
        }
        Properties properties = new Properties();
        try (FileInputStream fis = new FileInputStream(this.bootstrapConfigFile);){
            properties.load(fis);
        }
        HashMap<Object, Object> props = new HashMap<Object, Object>();
        props.putAll(properties);
        String specifiedWorkingDir = (String)props.get("working.dir");
        String nifiRegistryHome = System.getenv("NIFI_REGISTRY_HOME");
        File bootstrapConfigAbsoluteFile = this.bootstrapConfigFile.getAbsoluteFile();
        if (!StringUtils.isBlank((CharSequence)specifiedWorkingDir)) {
            workingDir = new File(specifiedWorkingDir.trim());
        } else if (!StringUtils.isBlank((CharSequence)nifiRegistryHome)) {
            workingDir = new File(nifiRegistryHome.trim());
        } else {
            File binDir = bootstrapConfigAbsoluteFile.getParentFile();
            workingDir = binDir.getParentFile();
        }
        builder.directory(workingDir);
        String nifiRegistryLogDir = this.replaceNull(System.getProperty("org.apache.nifi.registry.bootstrap.config.log.dir"), DEFAULT_LOG_DIR).trim();
        String nifiRegistryDocsDir = this.replaceNull((String)props.get("docs.dir"), DEFAULT_DOCS_DIR).trim();
        String libFilename = this.replaceNull((String)props.get("lib.dir"), "./lib").trim();
        File libDir = this.getFile(libFilename, workingDir);
        File libSharedDir = this.getFile(libFilename + "/shared", workingDir);
        String confFilename = this.replaceNull((String)props.get("conf.dir"), "./conf").trim();
        File confDir = this.getFile(confFilename, workingDir);
        String nifiRegistryPropsFilename = (String)props.get("props.file");
        if (nifiRegistryPropsFilename == null) {
            nifiRegistryPropsFilename = confDir.exists() ? new File(confDir, "nifi-registry.properties").getAbsolutePath() : DEFAULT_CONFIG_FILE;
        }
        nifiRegistryPropsFilename = nifiRegistryPropsFilename.trim();
        ArrayList<String> javaAdditionalArgs = new ArrayList<String>();
        for (Map.Entry entry : props.entrySet()) {
            String key = (String)entry.getKey();
            String value = (String)entry.getValue();
            if (!key.startsWith("java.arg")) continue;
            javaAdditionalArgs.add(value);
        }
        File[] libSharedFiles = libSharedDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String filename) {
                return filename.toLowerCase().endsWith(".jar");
            }
        });
        if (libSharedFiles == null || libSharedFiles.length == 0) {
            throw new RuntimeException("Could not find lib shared directory at " + libSharedDir.getAbsolutePath());
        }
        File[] libFiles = libDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String filename) {
                return filename.toLowerCase().endsWith(".jar");
            }
        });
        if (libFiles == null || libFiles.length == 0) {
            throw new RuntimeException("Could not find lib directory at " + libDir.getAbsolutePath());
        }
        File[] confFiles = confDir.listFiles();
        if (confFiles == null || confFiles.length == 0) {
            throw new RuntimeException("Could not find conf directory at " + confDir.getAbsolutePath());
        }
        ArrayList<String> cpFiles = new ArrayList<String>(confFiles.length + libFiles.length + libSharedFiles.length);
        cpFiles.add(confDir.getAbsolutePath());
        for (File file : libSharedFiles) {
            cpFiles.add(file.getAbsolutePath());
        }
        for (File file : libFiles) {
            cpFiles.add(file.getAbsolutePath());
        }
        String runtimeJavaVersion = System.getProperty("java.version");
        this.defaultLogger.info("Runtime Java version: {}", (Object)runtimeJavaVersion);
        int javaMajorVersion = RuntimeVersionProvider.getMajorVersion();
        if (javaMajorVersion >= 11 && (libJava11Dir = this.getFile(libJava11Filename = this.replaceNull((String)props.get("lib.dir"), "./lib").trim() + "/java11", workingDir)).exists()) {
            for (File file : Objects.requireNonNull(libJava11Dir.listFiles((dir, filename) -> filename.toLowerCase().endsWith(".jar")))) {
                cpFiles.add(file.getAbsolutePath());
            }
        }
        if (RuntimeVersionProvider.isMajorVersionDeprecated((int)javaMajorVersion)) {
            this.deprecationLogger.warn("Support for Java {} is deprecated. Java {} is the minimum recommended version", new Object[]{javaMajorVersion, RuntimeVersionProvider.getMinimumMajorVersion()});
        }
        StringBuilder classPathBuilder = new StringBuilder();
        for (int i = 0; i < cpFiles.size(); ++i) {
            String filename2 = (String)cpFiles.get(i);
            classPathBuilder.append(filename2);
            if (i >= cpFiles.size() - 1) continue;
            classPathBuilder.append(File.pathSeparatorChar);
        }
        String classPath = classPathBuilder.toString();
        String javaCmd = (String)props.get(DEFAULT_JAVA_CMD);
        if (javaCmd == null) {
            javaCmd = DEFAULT_JAVA_CMD;
        }
        if (javaCmd.equals(DEFAULT_JAVA_CMD) && (javaHome = System.getenv("JAVA_HOME")) != null) {
            String fileExtension = this.isWindows() ? ".exe" : "";
            File javaFile = new File(javaHome + File.separatorChar + DEFAULT_PID_DIR + File.separatorChar + DEFAULT_JAVA_CMD + fileExtension);
            if (javaFile.exists() && javaFile.canExecute()) {
                javaCmd = javaFile.getAbsolutePath();
            }
        }
        NiFiRegistryListener listener = new NiFiRegistryListener();
        int listenPort = listener.start(this);
        ArrayList<String> cmd = new ArrayList<String>();
        cmd.add(javaCmd);
        cmd.add("-classpath");
        cmd.add(classPath);
        cmd.addAll(javaAdditionalArgs);
        cmd.add("-Dnifi.registry.properties.file.path=" + nifiRegistryPropsFilename);
        cmd.add("-Dnifi.registry.bootstrap.config.file.path=" + this.bootstrapConfigFile.getAbsolutePath());
        cmd.add("-Dnifi.registry.bootstrap.listen.port=" + listenPort);
        cmd.add("-Dnifi.registry.bootstrap.config.docs.dir=" + nifiRegistryDocsDir);
        cmd.add("-Dapp=NiFiRegistry");
        cmd.add("-Dorg.apache.nifi.registry.bootstrap.config.log.dir=" + nifiRegistryLogDir);
        if (runtimeJavaVersion.startsWith("9") || runtimeJavaVersion.startsWith("10")) {
            cmd.add("--add-modules=java.xml.bind");
        }
        cmd.add("org.apache.nifi.registry.NiFiRegistry");
        builder.command(cmd);
        StringBuilder cmdBuilder = new StringBuilder();
        for (String s : cmd) {
            cmdBuilder.append(s).append(" ");
        }
        this.cmdLogger.info("Starting Apache NiFi Registry...");
        this.cmdLogger.info("Working Directory: {}", (Object)workingDir.getAbsolutePath());
        this.cmdLogger.info("Command: {}", (Object)cmdBuilder.toString());
        String gracefulShutdown = (String)props.get(GRACEFUL_SHUTDOWN_PROP);
        if (gracefulShutdown == null) {
            gracefulShutdown = DEFAULT_GRACEFUL_SHUTDOWN_VALUE;
        }
        try {
            gracefulShutdownSeconds = Integer.parseInt(gracefulShutdown);
        }
        catch (NumberFormatException nfe) {
            throw new NumberFormatException("The 'graceful.shutdown.seconds' property in Bootstrap Config File " + bootstrapConfigAbsoluteFile.getAbsolutePath() + " has an invalid value. Must be a non-negative integer");
        }
        if (gracefulShutdownSeconds < 0) {
            throw new NumberFormatException("The 'graceful.shutdown.seconds' property in Bootstrap Config File " + bootstrapConfigAbsoluteFile.getAbsolutePath() + " has an invalid value. Must be a non-negative integer");
        }
        Process process = builder.start();
        this.handleLogging(process);
        Long pid = OSUtils.getProcessId((Process)process, (Logger)this.cmdLogger);
        if (pid == null) {
            this.cmdLogger.warn("Launched Apache NiFi Registry but could not determined the Process ID");
        } else {
            this.nifiRegistryPid = pid;
            Properties pidProperties = new Properties();
            pidProperties.setProperty(PID_KEY, String.valueOf(this.nifiRegistryPid));
            this.savePidProperties(pidProperties, this.cmdLogger);
            this.cmdLogger.info("Launched Apache NiFi Registry with Process ID {}", (Object)pid);
        }
        this.shutdownHook = new ShutdownHook(process, this, this.secretKey, gracefulShutdownSeconds, this.loggingExecutor);
        Runtime runtime = Runtime.getRuntime();
        runtime.addShutdownHook(this.shutdownHook);
        String hostname = this.getHostname();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");
        String now = sdf.format(System.currentTimeMillis());
        String user = System.getProperty("user.name");
        if (user == null || user.trim().isEmpty()) {
            user = "Unknown User";
        }
        while (true) {
            boolean alive;
            if (alive = RunNiFiRegistry.isAlive(process)) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
                continue;
            }
            try {
                runtime.removeShutdownHook(this.shutdownHook);
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
            now = sdf.format(System.currentTimeMillis());
            if (!this.autoRestartNiFiRegistry) break;
            File statusFile = this.getStatusFile(this.defaultLogger);
            if (!statusFile.exists()) {
                this.defaultLogger.info("Status File no longer exists. Will not restart NiFi Registry ");
                return;
            }
            File lockFile = this.getLockFile(this.defaultLogger);
            if (lockFile.exists()) {
                this.defaultLogger.info("A shutdown was initiated. Will not restart NiFi Registry ");
                return;
            }
            boolean previouslyStarted = this.getNifiRegistryStarted();
            if (!previouslyStarted) {
                this.defaultLogger.info("NiFi Registry never started. Will not restart NiFi Registry ");
                return;
            }
            this.setNiFiRegistryStarted(false);
            this.defaultLogger.warn("Apache NiFi Registry appears to have died. Restarting...");
            this.secretKey = null;
            process = builder.start();
            this.handleLogging(process);
            pid = OSUtils.getProcessId((Process)process, (Logger)this.defaultLogger);
            if (pid == null) {
                this.cmdLogger.warn("Launched Apache NiFi Registry but could not obtain the Process ID");
            } else {
                this.nifiRegistryPid = pid;
                Properties pidProperties = new Properties();
                pidProperties.setProperty(PID_KEY, String.valueOf(this.nifiRegistryPid));
                this.savePidProperties(pidProperties, this.defaultLogger);
                this.cmdLogger.info("Launched Apache NiFi Registry with Process ID " + pid);
            }
            this.shutdownHook = new ShutdownHook(process, this, this.secretKey, gracefulShutdownSeconds, this.loggingExecutor);
            runtime.addShutdownHook(this.shutdownHook);
            boolean started = this.waitForStart();
            if (started) {
                this.defaultLogger.info("Successfully started Apache NiFi Registry {}", (Object)(pid == null ? "" : " with PID " + pid));
                continue;
            }
            this.defaultLogger.error("Apache NiFi Registry does not appear to have started");
        }
    }

    private void handleLogging(final Process process) {
        Set<Future<?>> existingFutures = this.loggingFutures;
        if (existingFutures != null) {
            for (Future<?> future : existingFutures) {
                future.cancel(false);
            }
        }
        Future<?> stdOutFuture = this.loggingExecutor.submit(new Runnable(){

            @Override
            public void run() {
                Logger stdOutLogger = LoggerFactory.getLogger((String)"org.apache.nifi.registry.StdOut");
                InputStream in = process.getInputStream();
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        stdOutLogger.info(line);
                    }
                }
                catch (IOException e) {
                    RunNiFiRegistry.this.defaultLogger.error("Failed to read from NiFi Registry's Standard Out stream", (Throwable)e);
                }
            }
        });
        Future<?> stdErrFuture = this.loggingExecutor.submit(new Runnable(){

            @Override
            public void run() {
                Logger stdErrLogger = LoggerFactory.getLogger((String)"org.apache.nifi.registry.StdErr");
                InputStream in = process.getErrorStream();
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(in));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        stdErrLogger.error(line);
                    }
                }
                catch (IOException e) {
                    RunNiFiRegistry.this.defaultLogger.error("Failed to read from NiFi Registry's Standard Error stream", (Throwable)e);
                }
            }
        });
        HashSet futures = new HashSet();
        futures.add(stdOutFuture);
        futures.add(stdErrFuture);
        this.loggingFutures = futures;
    }

    private boolean isWindows() {
        String osName = System.getProperty("os.name");
        return osName != null && osName.toLowerCase().contains("win");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean waitForStart() {
        this.lock.lock();
        try {
            long startTime = System.nanoTime();
            while (this.ccPort < 1) {
                try {
                    this.startupCondition.await(1L, TimeUnit.SECONDS);
                }
                catch (InterruptedException ie) {
                    boolean bl = false;
                    this.lock.unlock();
                    return bl;
                }
                long waitNanos = System.nanoTime() - startTime;
                long waitSeconds = TimeUnit.NANOSECONDS.toSeconds(waitNanos);
                if (waitSeconds <= 60L) continue;
                boolean bl = false;
                return bl;
            }
        }
        finally {
            this.lock.unlock();
        }
        return true;
    }

    private File getFile(String filename, File workingDir) {
        File file = new File(filename);
        if (!file.isAbsolute()) {
            file = new File(workingDir, filename);
        }
        return file;
    }

    private String replaceNull(String value, String replacement) {
        return value == null ? replacement : value;
    }

    void setAutoRestartNiFiRegistry(boolean restart) {
        this.autoRestartNiFiRegistry = restart;
    }

    void setNiFiRegistryCommandControlPort(int port, String secretKey) throws IOException {
        if (this.secretKey != null && this.ccPort != -1) {
            this.defaultLogger.warn("Blocking attempt to change NiFi Registry command port and secret after they have already been initialized. requestedPort={}", (Object)port);
            return;
        }
        this.ccPort = port;
        this.secretKey = secretKey;
        if (this.shutdownHook != null) {
            this.shutdownHook.setSecretKey(secretKey);
        }
        File statusFile = this.getStatusFile(this.defaultLogger);
        Properties nifiProps = new Properties();
        if (this.nifiRegistryPid != -1L) {
            nifiProps.setProperty(PID_KEY, String.valueOf(this.nifiRegistryPid));
        }
        nifiProps.setProperty("port", String.valueOf(this.ccPort));
        nifiProps.setProperty("secret.key", secretKey);
        try {
            this.savePidProperties(nifiProps, this.defaultLogger);
        }
        catch (IOException ioe) {
            this.defaultLogger.warn("Apache NiFi Registry has started but failed to persist NiFi Registry Port information to {} due to {}", new Object[]{statusFile.getAbsolutePath(), ioe});
        }
        this.defaultLogger.info("Apache NiFi Registry now running and listening for Bootstrap requests on port {}", (Object)port);
    }

    int getNiFiRegistryCommandControlPort() {
        return this.ccPort;
    }

    void setNiFiRegistryStarted(boolean nifiStarted) {
        this.startedLock.lock();
        try {
            this.nifiRegistryStarted = nifiStarted;
        }
        finally {
            this.startedLock.unlock();
        }
    }

    boolean getNifiRegistryStarted() {
        this.startedLock.lock();
        try {
            boolean bl = this.nifiRegistryStarted;
            return bl;
        }
        finally {
            this.startedLock.unlock();
        }
    }

    private static class Status {
        private final Integer port;
        private final String pid;
        private final Boolean respondingToPing;
        private final Boolean processRunning;

        public Status(Integer port, String pid, Boolean respondingToPing, Boolean processRunning) {
            this.port = port;
            this.pid = pid;
            this.respondingToPing = respondingToPing;
            this.processRunning = processRunning;
        }

        public String getPid() {
            return this.pid;
        }

        public Integer getPort() {
            return this.port;
        }

        public boolean isRespondingToPing() {
            return Boolean.TRUE.equals(this.respondingToPing);
        }

        public boolean isProcessRunning() {
            return Boolean.TRUE.equals(this.processRunning);
        }
    }
}

