/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.kafka.connect;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.errors.RetriableException;
import org.apache.kafka.connect.header.ConnectHeaders;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.apache.nifi.components.state.Scope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.kafka.connect.StatelessKafkaConnectorUtil;
import org.apache.nifi.kafka.connect.StatelessNiFiSourceConfig;
import org.apache.nifi.stateless.flow.DataflowTrigger;
import org.apache.nifi.stateless.flow.StatelessDataflow;
import org.apache.nifi.stateless.flow.TriggerResult;
import org.apache.nifi.util.FormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StatelessNiFiSourceTask
extends SourceTask {
    private static final Logger logger = LoggerFactory.getLogger(StatelessNiFiSourceTask.class);
    private static final long FAILURE_YIELD_MILLIS = 1000L;
    private StatelessDataflow dataflow;
    private String outputPortName;
    private String topicName;
    private String topicNameAttribute;
    private TriggerResult triggerResult;
    private String keyAttributeName;
    private Pattern headerAttributeNamePattern;
    private long timeoutMillis;
    private String dataflowName;
    private long failureYieldExpiration = 0L;
    private final Map<String, String> clusterStatePartitionMap = Collections.singletonMap("task.index", "CLUSTER");
    private Map<String, String> localStatePartitionMap = new HashMap<String, String>();
    private final AtomicLong unacknowledgedRecords = new AtomicLong(0L);

    public String version() {
        return StatelessKafkaConnectorUtil.getVersion();
    }

    public void start(Map<String, String> properties) {
        logger.info("Starting Source Task");
        StatelessNiFiSourceConfig config = this.createConfig(properties);
        String timeout = config.getDataflowTimeout();
        this.timeoutMillis = (long)FormatUtils.getPreciseTimeDuration((String)timeout, (TimeUnit)TimeUnit.MILLISECONDS);
        this.topicName = config.getTopicName();
        this.topicNameAttribute = config.getTopicNameAttribute();
        this.keyAttributeName = config.getKeyAttribute();
        if (this.topicName == null && this.topicNameAttribute == null) {
            throw new ConfigException("Either the topic.name or topic.name.attribute configuration must be specified");
        }
        String headerRegex = config.getHeaderRegex();
        this.headerAttributeNamePattern = headerRegex == null ? null : Pattern.compile(headerRegex);
        this.dataflow = StatelessKafkaConnectorUtil.createDataflow(config);
        this.dataflow.initialize();
        this.dataflowName = config.getDataflowName();
        this.outputPortName = config.getOutputPortName();
        if (this.outputPortName == null) {
            Set outputPorts = this.dataflow.getOutputPortNames();
            if (outputPorts.isEmpty()) {
                throw new ConfigException("The dataflow specified for <" + this.dataflowName + "> does not have an Output Port at the root level. Dataflows used for a Kafka Connect Source Task must have at least one Output Port at the root level.");
            }
            if (outputPorts.size() > 1) {
                throw new ConfigException("The dataflow specified for <" + this.dataflowName + "> has multiple Output Ports at the root level (" + outputPorts + "). The " + "output.port" + " property must be set to indicate which of these Ports Kafka records should be retrieved from.");
            }
            this.outputPortName = (String)outputPorts.iterator().next();
        }
        String taskIndex = config.getStateMapKey();
        this.localStatePartitionMap.put("task.index", taskIndex);
        Map localStateMap = this.context.offsetStorageReader().offset(this.localStatePartitionMap);
        Map clusterStateMap = this.context.offsetStorageReader().offset(this.clusterStatePartitionMap);
        this.dataflow.setComponentStates(localStateMap, Scope.LOCAL);
        this.dataflow.setComponentStates(clusterStateMap, Scope.CLUSTER);
    }

    public List<SourceRecord> poll() throws InterruptedException {
        Map<String, String> partitionMap;
        long now;
        long yieldExpiration = Math.max(this.failureYieldExpiration, this.dataflow.getSourceYieldExpiration());
        long yieldMillis = yieldExpiration - (now = System.currentTimeMillis());
        if (yieldMillis > 0L) {
            logger.debug("Source of NiFi flow has opted to yield for {} milliseconds. Will pause dataflow until that time period has elapsed.", (Object)yieldMillis);
            Thread.sleep(yieldMillis);
            return null;
        }
        if (this.unacknowledgedRecords.get() > 0L) {
            return null;
        }
        logger.debug("Triggering dataflow");
        long start = System.nanoTime();
        DataflowTrigger trigger = this.dataflow.trigger();
        Optional resultOptional = trigger.getResult(this.timeoutMillis, TimeUnit.MILLISECONDS);
        if (!resultOptional.isPresent()) {
            logger.warn("Dataflow timed out after waiting {} milliseconds. Will cancel the execution.", (Object)this.timeoutMillis);
            trigger.cancel();
            return null;
        }
        this.triggerResult = (TriggerResult)resultOptional.get();
        if (!this.triggerResult.isSuccessful()) {
            logger.error("Dataflow {} failed to execute properly", (Object)this.dataflowName, this.triggerResult.getFailureCause().orElse(null));
            trigger.cancel();
            this.failureYieldExpiration = System.currentTimeMillis() + 1000L;
            return null;
        }
        this.verifyFlowFilesTransferredToProperPort(this.triggerResult, this.outputPortName, trigger);
        long nanos = System.nanoTime() - start;
        List outputFlowFiles = this.triggerResult.getOutputFlowFiles(this.outputPortName);
        ArrayList<SourceRecord> sourceRecords = new ArrayList<SourceRecord>(outputFlowFiles.size());
        Map componentState = this.dataflow.getComponentStates(Scope.CLUSTER);
        if (componentState == null || componentState.isEmpty()) {
            componentState = this.dataflow.getComponentStates(Scope.LOCAL);
            partitionMap = this.localStatePartitionMap;
        } else {
            partitionMap = this.clusterStatePartitionMap;
        }
        try {
            for (FlowFile flowFile : outputFlowFiles) {
                byte[] contents = this.triggerResult.readContentAsByteArray(flowFile);
                SourceRecord sourceRecord = this.createSourceRecord(flowFile, contents, componentState, partitionMap);
                sourceRecords.add(sourceRecord);
            }
        }
        catch (Exception e) {
            logger.error("Failed to obtain contents of Output FlowFiles in order to form Kafka Record", (Throwable)e);
            this.triggerResult.abort((Throwable)e);
            this.failureYieldExpiration = System.currentTimeMillis() + 1000L;
            return null;
        }
        logger.debug("Returning {} records from poll() method (took {} nanos to run dataflow)", (Object)sourceRecords.size(), (Object)nanos);
        if (sourceRecords.size() > 0) {
            this.unacknowledgedRecords.addAndGet(sourceRecords.size());
        } else {
            this.triggerResult.acknowledge();
        }
        return sourceRecords;
    }

    protected StatelessNiFiSourceConfig createConfig(Map<String, String> properties) {
        return new StatelessNiFiSourceConfig(properties);
    }

    private void verifyFlowFilesTransferredToProperPort(TriggerResult triggerResult, String expectedPortName, DataflowTrigger trigger) {
        Map flowFileOutputMap = triggerResult.getOutputFlowFiles();
        for (Map.Entry entry : flowFileOutputMap.entrySet()) {
            String portName = (String)entry.getKey();
            List flowFiles = (List)entry.getValue();
            if (flowFiles.isEmpty() || expectedPortName.equals(portName)) continue;
            logger.error("Dataflow transferred FlowFiles to Port {} but was expecting data to be transferred to {}. Rolling back session.", (Object)portName, (Object)expectedPortName);
            trigger.cancel();
            throw new RetriableException("Data was transferred to unexpected port. Expected: " + expectedPortName + ". Actual: " + portName);
        }
    }

    private SourceRecord createSourceRecord(FlowFile flowFile, byte[] contents, Map<String, ?> componentState, Map<String, ?> partitionMap) {
        String attributeValue;
        Schema valueSchema = contents == null || contents.length == 0 ? null : Schema.BYTES_SCHEMA;
        Integer topicPartition = null;
        String topic = this.topicNameAttribute == null ? this.topicName : ((attributeValue = flowFile.getAttribute(this.topicNameAttribute)) == null ? this.topicName : attributeValue);
        ConnectHeaders headers = new ConnectHeaders();
        if (this.headerAttributeNamePattern != null) {
            for (Map.Entry entry : flowFile.getAttributes().entrySet()) {
                if (!this.headerAttributeNamePattern.matcher((CharSequence)entry.getKey()).matches()) continue;
                String headerName = (String)entry.getKey();
                String headerValue = (String)entry.getValue();
                headers.add(headerName, (Object)headerValue, Schema.STRING_SCHEMA);
            }
        }
        String key = this.keyAttributeName == null ? null : flowFile.getAttribute(this.keyAttributeName);
        Schema keySchema = key == null ? null : Schema.STRING_SCHEMA;
        Long timestamp = System.currentTimeMillis();
        return new SourceRecord(partitionMap, componentState, topic, topicPartition, keySchema, (Object)key, valueSchema, (Object)contents, timestamp, (Iterable)headers);
    }

    public void commitRecord(SourceRecord record, RecordMetadata metadata) throws InterruptedException {
        super.commitRecord(record, metadata);
        long unacked = this.unacknowledgedRecords.decrementAndGet();
        logger.debug("SourceRecord {} committed; number of unacknowledged FlowFiles is now {}", (Object)record, (Object)unacked);
        if (unacked < 1L) {
            logger.debug("Acknowledging trigger result");
            this.triggerResult.acknowledge();
        }
    }

    public void stop() {
        logger.info("Shutting down Source Task for " + this.dataflowName);
        if (this.dataflow != null) {
            this.dataflow.shutdown();
        }
    }

    protected StatelessDataflow getDataflow() {
        return this.dataflow;
    }
}

