/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.nifi.security.util.crypto

import org.apache.commons.codec.binary.Base64
import org.apache.commons.codec.binary.Hex
import org.apache.nifi.security.util.EncryptionMethod
import org.bouncycastle.jce.provider.BouncyCastleProvider
import org.junit.jupiter.api.BeforeAll
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.slf4j.Logger
import org.slf4j.LoggerFactory

import javax.crypto.Cipher
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec
import java.nio.charset.StandardCharsets
import java.security.Security

import static org.junit.jupiter.api.Assertions.assertArrayEquals
import static org.junit.jupiter.api.Assertions.assertEquals
import static org.junit.jupiter.api.Assertions.assertFalse
import static org.junit.jupiter.api.Assertions.assertNotNull
import static org.junit.jupiter.api.Assertions.assertThrows
import static org.junit.jupiter.api.Assertions.assertTrue

class Argon2CipherProviderGroovyTest {
    private static final Logger logger = LoggerFactory.getLogger(Argon2CipherProviderGroovyTest.class)

    private static final String PLAINTEXT = "ExactBlockSizeRequiredForProcess"

    private static List<EncryptionMethod> strongKDFEncryptionMethods

    private static final int DEFAULT_KEY_LENGTH = 128
    private final String SALT_HEX = "0123456789ABCDEFFEDCBA9876543210"
    private static ArrayList<Integer> AES_KEY_LENGTHS

    RandomIVPBECipherProvider cipherProvider
    private final IntRange FULL_SALT_LENGTH_RANGE= (49..53)

    @BeforeAll
    static void setUpOnce() throws Exception {
        Security.addProvider(new BouncyCastleProvider())

        strongKDFEncryptionMethods = EncryptionMethod.values().findAll { it.isCompatibleWithStrongKDFs() }

        logger.metaClass.methodMissing = { String name, args ->
            logger.info("[${name?.toUpperCase()}] ${(args as List).join(" ")}")
        }

        AES_KEY_LENGTHS = [128, 192, 256]
    }

    @BeforeEach
    void setUp() throws Exception {
        // Very fast parameters to test for correctness rather than production values
        cipherProvider = new Argon2CipherProvider(1024, 1, 3)
    }

    @Test
    void testGetCipherShouldBeInternallyConsistent() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = cipherProvider.generateSalt()

        // Act
        for (EncryptionMethod em : strongKDFEncryptionMethods) {
            logger.info("Using algorithm: ${em.getAlgorithm()}")

            // Initialize a cipher for encryption
            Cipher cipher = cipherProvider.getCipher(em, PASSWORD, SALT, DEFAULT_KEY_LENGTH, true)
            byte[] iv = cipher.getIV()
            logger.info("IV: ${Hex.encodeHexString(iv)}")

            byte[] cipherBytes = cipher.doFinal(PLAINTEXT.getBytes("UTF-8"))
            logger.info("Cipher text: ${Hex.encodeHexString(cipherBytes)} ${cipherBytes.length}")

            cipher = cipherProvider.getCipher(em, PASSWORD, SALT, iv, DEFAULT_KEY_LENGTH, false)
            byte[] recoveredBytes = cipher.doFinal(cipherBytes)
            String recovered = new String(recoveredBytes, "UTF-8")
            logger.info("Recovered: ${recovered}")

            // Assert
            assertEquals(PLAINTEXT, recovered)
        }
    }

    @Test
    void testArgon2ShouldSupportExternalCompatibility() throws Exception {
        // Arrange

        // Default values are hashLength = 32, memory = 1024, parallelism = 1, iterations = 3, but the provided salt will contain the parameters used
        cipherProvider = new Argon2CipherProvider()

        final String PLAINTEXT = "This is a plaintext message."
        final String PASSWORD = "thisIsABadPassword"
        final int hashLength = 256

        // These values can be generated by running `$ ./openssl_argon2.rb` in the terminal
        final byte[] SALT = Hex.decodeHex("68d29a1d8021f45954333767358a2492" as char[])
        logger.info("Expected salt: ${Hex.encodeHexString(SALT)}")
        final byte[] IV = Hex.decodeHex("808590f35f9fba14dbda9c2bb2b76a79" as char[])

        final String CIPHER_TEXT = "d672412857916880c79d573aa4f9d4971b85f07438d6f62f38a0e31314caa2e5"
        logger.sanity("Ruby cipher text: ${CIPHER_TEXT}")
        byte[] cipherBytes = Hex.decodeHex(CIPHER_TEXT as char[])
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC

        // Sanity check
        String rubyKeyHex = "8caf581795886d38f0c605e3d674f4961c658ee3625a8e8868be36c902d234ef"
        logger.sanity("Using key: ${rubyKeyHex}")
        logger.sanity("Using IV:  ${Hex.encodeHexString(IV)}")
        Cipher rubyCipher = Cipher.getInstance(encryptionMethod.algorithm, "BC")
        def rubyKey = new SecretKeySpec(Hex.decodeHex(rubyKeyHex as char[]), "AES")
        def ivSpec = new IvParameterSpec(IV)
        rubyCipher.init(Cipher.ENCRYPT_MODE, rubyKey, ivSpec)
        byte[] rubyCipherBytes = rubyCipher.doFinal(PLAINTEXT.bytes)
        logger.sanity("Created cipher text: ${Hex.encodeHexString(rubyCipherBytes)}")
        rubyCipher.init(Cipher.DECRYPT_MODE, rubyKey, ivSpec)
        assertArrayEquals(PLAINTEXT.bytes, rubyCipher.doFinal(rubyCipherBytes))
        logger.sanity("Decrypted generated cipher text successfully")
        assertArrayEquals(PLAINTEXT.bytes, rubyCipher.doFinal(cipherBytes))
        logger.sanity("Decrypted external cipher text successfully")

        // $argon2id$v=19$m=memory,t=iterations,p=parallelism$saltB64$hashB64
        final String FULL_HASH = "\$argon2id\$v=19\$m=256,t=3,p=1\$aNKaHYAh9FlUMzdnNYokkg\$jK9YF5WIbTjwxgXj1nT0lhxljuNiWo6IaL42yQLSNO8"
        logger.info("Full Hash: ${FULL_HASH}")

        final String FULL_SALT = FULL_HASH[0..<FULL_HASH.lastIndexOf("\$")]
        logger.info("Full salt: ${FULL_SALT}")

        final String[] hashComponents = FULL_HASH.split("\\\$")
        logger.info("hashComponents: ${Arrays.toString(hashComponents)}")
        Map<String, String> saltParams = hashComponents[3].split(",").collectEntries { String pair ->
            pair.split("=")
        }
        logger.info("saltParams: ${saltParams}")
        def saltB64 = hashComponents[4]
        byte[] salt = Base64.decodeBase64(saltB64)
        logger.info("Salt: ${Hex.encodeHexString(salt)}")
        assertArrayEquals(SALT, salt)

        logger.info("Using algorithm: ${encryptionMethod.getAlgorithm()}")
        logger.info("External cipher text: ${CIPHER_TEXT} ${cipherBytes.length}")

        // Act
        Cipher cipher = cipherProvider.getCipher(encryptionMethod, PASSWORD, FULL_SALT.bytes, IV, hashLength, false)
        byte[] recoveredBytes = cipher.doFinal(cipherBytes)
        String recovered = new String(recoveredBytes, "UTF-8")
        logger.info("Recovered: ${recovered}")

        // Assert
        assertEquals(PLAINTEXT, recovered)
    }

    @Test
    void testGetCipherShouldRejectInvalidIV() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = Hex.decodeHex(SALT_HEX as char[])
        final def INVALID_IVS = (0..15).collect { int length -> new byte[length] }

        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC

        // Act
        INVALID_IVS.each { byte[] badIV ->
            logger.info("IV: ${Hex.encodeHexString(badIV)} ${badIV.length}")

            // Encrypt should print a warning about the bad IV but overwrite it
            Cipher cipher = cipherProvider.getCipher(encryptionMethod, PASSWORD, SALT, badIV, DEFAULT_KEY_LENGTH, true)

            // Decrypt should fail
            IllegalArgumentException iae = assertThrows(IllegalArgumentException.class,
                    () -> cipherProvider.getCipher(encryptionMethod, PASSWORD, SALT, badIV, DEFAULT_KEY_LENGTH, false))

            // Assert
            assertTrue(iae.getMessage().contains("Cannot decrypt without a valid IV"))
        }
    }

    @Test
    void testGetCipherWithExternalIVShouldBeInternallyConsistent() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = cipherProvider.generateSalt()
        final byte[] IV = Hex.decodeHex("01" * 16 as char[])

        // Act
        for (EncryptionMethod em : strongKDFEncryptionMethods) {
            logger.info("Using algorithm: ${em.getAlgorithm()}")

            // Initialize a cipher for encryption
            Cipher cipher = cipherProvider.getCipher(em, PASSWORD, SALT, IV, DEFAULT_KEY_LENGTH, true)
            logger.info("IV: ${Hex.encodeHexString(IV)}")

            byte[] cipherBytes = cipher.doFinal(PLAINTEXT.getBytes("UTF-8"))
            logger.info("Cipher text: ${Hex.encodeHexString(cipherBytes)} ${cipherBytes.length}")

            cipher = cipherProvider.getCipher(em, PASSWORD, SALT, IV, DEFAULT_KEY_LENGTH, false)
            byte[] recoveredBytes = cipher.doFinal(cipherBytes)
            String recovered = new String(recoveredBytes, "UTF-8")
            logger.info("Recovered: ${recovered}")

            // Assert
            assertEquals(PLAINTEXT, recovered)
        }
    }

    @Test
    void testGetCipherWithUnlimitedStrengthShouldBeInternallyConsistent() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = cipherProvider.generateSalt()

        final int LONG_KEY_LENGTH = 256

        // Act
        for (EncryptionMethod em : strongKDFEncryptionMethods) {
            logger.info("Using algorithm: ${em.getAlgorithm()}")

            // Initialize a cipher for encryption
            Cipher cipher = cipherProvider.getCipher(em, PASSWORD, SALT, LONG_KEY_LENGTH, true)
            byte[] iv = cipher.getIV()
            logger.info("IV: ${Hex.encodeHexString(iv)}")

            byte[] cipherBytes = cipher.doFinal(PLAINTEXT.getBytes("UTF-8"))
            logger.info("Cipher text: ${Hex.encodeHexString(cipherBytes)} ${cipherBytes.length}")

            cipher = cipherProvider.getCipher(em, PASSWORD, SALT, iv, LONG_KEY_LENGTH, false)
            byte[] recoveredBytes = cipher.doFinal(cipherBytes)
            String recovered = new String(recoveredBytes, "UTF-8")
            logger.info("Recovered: ${recovered}")

            // Assert
            assertEquals(PLAINTEXT, recovered)
        }
    }

    @Test
    void testGetCipherShouldNotAcceptInvalidSalts() throws Exception {
        // Arrange
        final String PASSWORD = "thisIsABadPassword"

        final def INVALID_SALTS = ['argon2', '$3a$11$', 'x', '$2a$10$']
        final LENGTH_MESSAGE = "The raw salt must be greater than or equal to 8 bytes"

        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC
        logger.info("Using algorithm: ${encryptionMethod.getAlgorithm()}")

        // Act
        INVALID_SALTS.each { String salt ->
            logger.info("Checking salt ${salt}")

            IllegalArgumentException iae = assertThrows(IllegalArgumentException.class,
                    () -> cipherProvider.getCipher(encryptionMethod, PASSWORD, salt.bytes, DEFAULT_KEY_LENGTH, true))
            logger.expected(iae.getMessage())

            // Assert
            assertTrue(iae.getMessage().contains(LENGTH_MESSAGE))
        }
    }

    @Test
    void testGetCipherShouldHandleUnformattedSalts() throws Exception {
        // Arrange
        final String PASSWORD = "thisIsABadPassword"

        final def RECOVERABLE_SALTS = ['$ab$00$acbdefghijklmnopqrstuv', '$4$1$1$0123456789abcdef', '$400$1$1$abcdefghijklmnopqrstuv']

        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC
        logger.info("Using algorithm: ${encryptionMethod.getAlgorithm()}")

        // Act
        RECOVERABLE_SALTS.each { String salt ->
            logger.info("Checking salt ${salt}")

            Cipher cipher = cipherProvider.getCipher(encryptionMethod, PASSWORD, salt.bytes, DEFAULT_KEY_LENGTH, true)

            // Assert
            assertNotNull(cipher)
        }
    }

    @Test
    void testGetCipherShouldRejectEmptySalt() throws Exception {
        // Arrange
        final String PASSWORD = "thisIsABadPassword"

        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC
        logger.info("Using algorithm: ${encryptionMethod.getAlgorithm()}")

        // Act
        IllegalArgumentException iae = assertThrows(IllegalArgumentException.class,
                () -> cipherProvider.getCipher(encryptionMethod, PASSWORD, new byte[0], DEFAULT_KEY_LENGTH, true))
        logger.expected(iae.getMessage())

        // Assert
        assertTrue(iae.getMessage().contains("The salt cannot be empty. To generate a salt, use Argon2CipherProvider#generateSalt()"))
    }

    @Test
    void testGenerateSaltShouldProvideValidSalt() throws Exception {
        // Arrange
        RandomIVPBECipherProvider cipherProvider = new Argon2CipherProvider()

        // Act
        byte[] saltBytes = cipherProvider.generateSalt()
        logger.info("Generated salt ${Hex.encodeHexString(saltBytes)}")

        String fullSalt = new String(saltBytes, StandardCharsets.UTF_8)
        logger.info("Generated salt (${saltBytes.length}): ${fullSalt}".toString())

        def rawSaltB64 = (fullSalt =~ /\$([\w\+\/]+)\$?$/)[0][1]
        logger.info("Extracted B64 raw salt (${rawSaltB64.size()}): ${rawSaltB64}".toString())

        byte[] rawSaltBytes = Base64.decodeBase64(rawSaltB64)

        // Assert
        boolean isValidFormattedSalt = cipherProvider.isArgon2FormattedSalt(fullSalt)
        logger.info("Salt is Argon2 format: ${isValidFormattedSalt}")
        assertTrue(isValidFormattedSalt)

        boolean fullSaltIsValidLength = FULL_SALT_LENGTH_RANGE.contains(saltBytes.length)
        logger.info("Salt length (${fullSalt.length()}) in valid range (${FULL_SALT_LENGTH_RANGE})")
        assertTrue(fullSaltIsValidLength)

        byte [] notExpected = new byte[16]
        Arrays.fill(notExpected, 0x00 as byte)
        assertFalse(Arrays.equals(notExpected, rawSaltBytes))
    }

    @Test
    void testGetCipherForDecryptShouldRequireIV() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = cipherProvider.generateSalt()
        final byte[] IV = Hex.decodeHex("00" * 16 as char[])

        // Act
        for (EncryptionMethod em : strongKDFEncryptionMethods) {
            logger.info("Using algorithm: ${em.getAlgorithm()}")

            // Initialize a cipher for encryption
            Cipher cipher = cipherProvider.getCipher(em, PASSWORD, SALT, IV, DEFAULT_KEY_LENGTH, true)
            logger.info("IV: ${Hex.encodeHexString(IV)}")

            byte[] cipherBytes = cipher.doFinal(PLAINTEXT.getBytes("UTF-8"))
            logger.info("Cipher text: ${Hex.encodeHexString(cipherBytes)} ${cipherBytes.length}")

            IllegalArgumentException iae = assertThrows(IllegalArgumentException.class,
                    () -> cipherProvider.getCipher(em, PASSWORD, SALT, DEFAULT_KEY_LENGTH, false))
            logger.expected(iae.getMessage())

            // Assert
            assertTrue(iae.getMessage().contains("Cannot decrypt without a valid IV"))
        }
    }

    @Test
    void testGetCipherShouldAcceptValidKeyLengths() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = cipherProvider.generateSalt()
        final byte[] IV = Hex.decodeHex("01" * 16 as char[])

        final def VALID_KEY_LENGTHS = AES_KEY_LENGTHS
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC

        // Act
        VALID_KEY_LENGTHS.each { int keyLength ->
            logger.info("Using algorithm: ${encryptionMethod.getAlgorithm()} with key length ${keyLength}")

            // Initialize a cipher for encryption
            Cipher cipher = cipherProvider.getCipher(encryptionMethod, PASSWORD, SALT, IV, keyLength, true)
            logger.info("IV: ${Hex.encodeHexString(IV)}")

            byte[] cipherBytes = cipher.doFinal(PLAINTEXT.getBytes("UTF-8"))
            logger.info("Cipher text: ${Hex.encodeHexString(cipherBytes)} ${cipherBytes.length}")

            cipher = cipherProvider.getCipher(encryptionMethod, PASSWORD, SALT, IV, keyLength, false)
            byte[] recoveredBytes = cipher.doFinal(cipherBytes)
            String recovered = new String(recoveredBytes, "UTF-8")
            logger.info("Recovered: ${recovered}")

            // Assert
            assertEquals(PLAINTEXT, recovered)
        }
    }

    @Test
    void testGetCipherShouldNotAcceptInvalidKeyLengths() throws Exception {
        // Arrange
        final String PASSWORD = "shortPassword"
        final byte[] SALT = cipherProvider.generateSalt()
        final byte[] IV = Hex.decodeHex("00" * 16 as char[])

        final def INVALID_KEY_LENGTHS = [-1, 40, 64, 112, 512]
        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC

        // Act
        INVALID_KEY_LENGTHS.each { int keyLength ->
            logger.info("Using algorithm: ${encryptionMethod.getAlgorithm()} with key length ${keyLength}")

            // Initialize a cipher for
            IllegalArgumentException iae = assertThrows(IllegalArgumentException.class,
                    () -> cipherProvider.getCipher(encryptionMethod, PASSWORD, SALT, IV, keyLength, true))
            logger.expected(iae.getMessage())

            // Assert
            assertTrue(iae.getMessage().contains(keyLength + " is not a valid key length for AES"))
        }
    }

    @Test
    void testArgon2ShouldNotAcceptInvalidPassword() {
        // Arrange
        String badPassword = ""
        byte[] salt = [0x01 as byte] * 16

        EncryptionMethod encryptionMethod = EncryptionMethod.AES_CBC

        // Act
        IllegalArgumentException iae = assertThrows(IllegalArgumentException.class,
                () -> cipherProvider.getCipher(encryptionMethod, badPassword, salt, DEFAULT_KEY_LENGTH, true))

        // Assert
        assertTrue(iae.getMessage().contains("Encryption with an empty password is not supported"))
    }

    @Test
    void testShouldParseSalt() throws Exception {
        // Arrange
        cipherProvider = cipherProvider as Argon2CipherProvider

        final byte[] EXPECTED_RAW_SALT = Hex.decodeHex("8622b26906d9c900660a60f5cc673233" as char[])
        final int EXPECTED_MEMORY = 1024
        final int EXPECTED_PARALLELISM = 4
        final int EXPECTED_ITERATIONS = 1

        final String FORMATTED_SALT = "\$argon2id\$v=19\$m=1024,t=4,p=1\$hiKyaQbZyQBmCmD1zGcyMw"
        logger.info("Using salt: ${FORMATTED_SALT}")

        byte[] rawSalt = new byte[16]
        def params = []

        // Act
        cipherProvider.parseSalt(FORMATTED_SALT, rawSalt, params)

        // Assert
        assertArrayEquals(EXPECTED_RAW_SALT, rawSalt)
        assertEquals(EXPECTED_MEMORY, params[0])
        assertEquals(EXPECTED_PARALLELISM, params[1])
        assertEquals(EXPECTED_ITERATIONS, params[2])
    }

    @Test
    void testShouldRejectInvalidSalt() throws Exception {
        // Arrange
        cipherProvider = cipherProvider as Argon2CipherProvider

        final String FULL_HASH = "\$argon2id\$v=19\$m=1024,t=4,p=1\$hiKyaQbZyQBmCmD1zGcyMw\$rc+ec+/hQeBcwzjH+OEmUtaTUqhZYKN4ZKJtWzFZYjQ"
        logger.info("Using salt: ${FULL_HASH}")

        byte[] rawSalt = new byte[16]
        List<Integer> params = []

        // Act
        boolean isValid = cipherProvider.isArgon2FormattedSalt(FULL_HASH)
        logger.info("Argon2 formatted salt: ${isValid}")

        // Assert
        assertFalse(isValid)
    }

    @Test
    void testShouldExtractSalt() throws Exception {
        // Arrange
        cipherProvider = cipherProvider as Argon2CipherProvider

        final byte[] EXPECTED_RAW_SALT = Hex.decodeHex("8622b26906d9c900660a60f5cc673233" as char[])

        final String FORMATTED_SALT = "\$argon2id\$v=19\$m=1024,t=4,p=1\$hiKyaQbZyQBmCmD1zGcyMw"
        logger.info("Using salt: ${FORMATTED_SALT}")

        byte[] rawSalt

        // Act
        rawSalt = cipherProvider.extractRawSaltFromArgon2Salt(FORMATTED_SALT)
        logger.info("rawSalt: ${Hex.encodeHexString(rawSalt)}")

        // Assert
        assertArrayEquals(EXPECTED_RAW_SALT, rawSalt)
    }
}