/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix;

import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.Complex;
import smile.math.Math;
import smile.math.matrix.IMatrix;

public class EigenValueDecomposition {
    private static final Logger logger = LoggerFactory.getLogger(EigenValueDecomposition.class);
    private double[] d;
    private double[] e;
    private double[][] V;

    private EigenValueDecomposition(double[][] V, double[] d) {
        this.V = V;
        this.d = d;
    }

    private EigenValueDecomposition(double[][] V, double[] d, double[] e) {
        this.V = V;
        this.d = d;
        this.e = e;
    }

    public double[][] getEigenVectors() {
        return this.V;
    }

    public double[] getEigenValues() {
        return this.d;
    }

    public double[] getRealEigenValues() {
        return this.d;
    }

    public double[] getImagEigenValues() {
        return this.e;
    }

    public double[][] getD() {
        int n = this.V.length;
        double[][] D = new double[n][n];
        for (int i = 0; i < n; ++i) {
            D[i][i] = this.d[i];
            if (this.e == null) continue;
            if (this.e[i] > 0.0) {
                D[i][i + 1] = this.e[i];
                continue;
            }
            if (!(this.e[i] < 0.0)) continue;
            D[i][i - 1] = this.e[i];
        }
        return D;
    }

    public static double eigen(IMatrix A, double[] v) {
        return EigenValueDecomposition.eigen(A, v, Math.max(1.0E-10, (double)A.nrows() * Math.EPSILON));
    }

    public static double eigen(IMatrix A, double[] v, double tol) {
        return EigenValueDecomposition.eigen(A, v, 0.0, tol);
    }

    public static double eigen(IMatrix A, double[] v, double tol, int maxIter) {
        return EigenValueDecomposition.eigen(A, v, 0.0, tol, maxIter);
    }

    public static double eigen(IMatrix A, double[] v, double p, double tol) {
        return EigenValueDecomposition.eigen(A, v, p, tol, Math.max(20, 2 * A.nrows()));
    }

    public static double eigen(IMatrix A, double[] v, double p, double tol, int maxIter) {
        if (A.nrows() != A.ncols()) {
            throw new IllegalArgumentException("Matrix is not square.");
        }
        if (tol <= 0.0) {
            throw new IllegalArgumentException("Invalid tolerance: " + tol);
        }
        if (maxIter <= 0) {
            throw new IllegalArgumentException("Invalid maximum number of iterations: " + maxIter);
        }
        int n = A.nrows();
        tol = Math.max(tol, Math.EPSILON * (double)n);
        double[] z = new double[n];
        double lambda = EigenValueDecomposition.ax(A, v, z, p);
        for (int iter = 1; iter <= maxIter; ++iter) {
            double l = lambda;
            lambda = EigenValueDecomposition.ax(A, v, z, p);
            double eps = Math.abs(lambda - l);
            if (iter % 10 == 0) {
                logger.trace(String.format("Largest eigenvalue after %3d power iterations: %.5f\n", iter, lambda + p));
            }
            if (!(eps < tol)) continue;
            logger.info(String.format("Largest eigenvalue after %3d power iterations: %.5f\n", iter, lambda + p));
            return lambda + p;
        }
        logger.info(String.format("Largest eigenvalue after %3d power iterations: %.5f\n", maxIter, lambda + p));
        logger.error("Power iteration exceeded the maximum number of iterations.");
        return lambda + p;
    }

    public static double[] pagerank(IMatrix A) {
        int n = A.nrows();
        double[] v = new double[n];
        Arrays.fill(v, 1.0 / (double)n);
        return EigenValueDecomposition.pagerank(A, v);
    }

    public static double[] pagerank(IMatrix A, double[] v) {
        return EigenValueDecomposition.pagerank(A, v, 0.85, 1.0E-7, 57);
    }

    public static double[] pagerank(IMatrix A, double[] v, double damping, double tol, int maxIter) {
        if (A.nrows() != A.ncols()) {
            throw new IllegalArgumentException("Matrix is not square.");
        }
        if (tol <= 0.0) {
            throw new IllegalArgumentException("Invalid tolerance: " + tol);
        }
        if (maxIter <= 0) {
            throw new IllegalArgumentException("Invalid maximum number of iterations: " + maxIter);
        }
        int n = A.nrows();
        tol = Math.max(tol, Math.EPSILON * (double)n);
        double[] z = new double[n];
        double[] p = Arrays.copyOf(v, n);
        for (int iter = 1; iter <= maxIter; ++iter) {
            A.ax(p, z);
            double beta = 1.0 - damping * Math.norm1(z);
            double delta = 0.0;
            for (int i = 0; i < n; ++i) {
                double q = damping * z[i] + beta * v[i];
                delta += Math.abs(q - p[i]);
                p[i] = q;
            }
            if (iter % 10 == 0) {
                logger.info(String.format("PageRank residual after %3d power iterations: %.7f\n", iter, delta));
            }
            if (!(delta < tol)) continue;
            logger.info(String.format("PageRank residual after %3d power iterations: %.7f\n", iter, delta));
            return p;
        }
        logger.error("PageRank iteration exceeded the maximum number of iterations.");
        return p;
    }

    private static double ax(IMatrix A, double[] x, double[] y, double p) {
        int i;
        A.ax(x, y);
        if (p != 0.0) {
            for (int i2 = 0; i2 < y.length; ++i2) {
                int n = i2;
                y[n] = y[n] - p * x[i2];
            }
        }
        double lambda = y[0];
        for (i = 1; i < y.length; ++i) {
            if (!(Math.abs(y[i]) > Math.abs(lambda))) continue;
            lambda = y[i];
        }
        for (i = 0; i < y.length; ++i) {
            x[i] = y[i] / lambda;
        }
        return lambda;
    }

    public static EigenValueDecomposition decompose(IMatrix A, int k) {
        return EigenValueDecomposition.decompose(A, k, 1.0E-6);
    }

    public static EigenValueDecomposition decompose(IMatrix A, int k, double kappa) {
        if (A.nrows() != A.ncols()) {
            throw new IllegalArgumentException("Matrix is not square.");
        }
        if (k < 1 || k > A.nrows()) {
            throw new IllegalArgumentException("k is larger than the size of A: " + k + " > " + A.nrows());
        }
        int n = A.nrows();
        int intro = 0;
        double eps = Math.EPSILON * Math.sqrt(n);
        double reps = Math.sqrt(Math.EPSILON);
        double eps34 = reps * Math.sqrt(reps);
        kappa = Math.max(kappa, eps34);
        double[][] wptr = new double[6][n];
        double[] eta = new double[n];
        double[] oldeta = new double[n];
        double[] bnd = new double[n];
        double[] alf = new double[n];
        double[] bet = new double[n + 1];
        double[][] q = new double[n][];
        double[][] p = new double[2][];
        double[] ritz = new double[n + 1];
        double[][] z = null;
        double rnm = EigenValueDecomposition.startv(A, q, wptr, 0);
        double t = 1.0 / rnm;
        Math.scale(t, wptr[0], wptr[1]);
        Math.scale(t, wptr[3]);
        A.ax(wptr[3], wptr[0]);
        alf[0] = Math.dot(wptr[0], wptr[3]);
        Math.axpy(-alf[0], wptr[1], wptr[0]);
        t = Math.dot(wptr[0], wptr[3]);
        Math.axpy(-t, wptr[1], wptr[0]);
        alf[0] = alf[0] + t;
        Math.copy(wptr[0], wptr[4]);
        rnm = Math.norm(wptr[0]);
        double anorm = rnm + Math.abs(alf[0]);
        double tol = reps * anorm;
        if (0.0 == rnm) {
            throw new IllegalStateException("Lanczos method was unable to find a starting vector within range.");
        }
        eta[0] = eps;
        oldeta[0] = eps;
        int neig = 0;
        int ll = 0;
        int first = 1;
        int last = Math.min(k + Math.max(8, k), n);
        int j = 0;
        boolean enough = false;
        while (!enough) {
            int i;
            if (rnm <= tol) {
                rnm = 0.0;
            }
            for (j = first; j < last; ++j) {
                Math.swap((Object[])wptr, 1, 2);
                Math.swap((Object[])wptr, 3, 4);
                EigenValueDecomposition.store(q, j - 1, wptr[2]);
                if (j - 1 < 2) {
                    p[j - 1] = (double[])wptr[4].clone();
                }
                bet[j] = rnm;
                if (0.0 == bet[j]) {
                    rnm = EigenValueDecomposition.startv(A, q, wptr, j);
                    if (rnm < 0.0) {
                        rnm = 0.0;
                        break;
                    }
                    if (rnm == 0.0) {
                        enough = true;
                    }
                }
                if (enough) {
                    Math.swap((Object[])wptr, 1, 2);
                    break;
                }
                t = 1.0 / rnm;
                Math.scale(t, wptr[0], wptr[1]);
                Math.scale(t, wptr[3]);
                A.ax(wptr[3], wptr[0]);
                Math.axpy(-rnm, wptr[2], wptr[0]);
                alf[j] = Math.dot(wptr[0], wptr[3]);
                Math.axpy(-alf[j], wptr[1], wptr[0]);
                if (j <= 2 && Math.abs(alf[j - 1]) > 4.0 * Math.abs(alf[j])) {
                    ll = j;
                }
                for (i = 0; i < Math.min(ll, j - 1); ++i) {
                    t = Math.dot(p[i], wptr[0]);
                    Math.axpy(-t, q[i], wptr[0]);
                    eta[i] = eps;
                    oldeta[i] = eps;
                }
                t = Math.dot(wptr[0], wptr[4]);
                Math.axpy(-t, wptr[2], wptr[0]);
                if (bet[j] > 0.0) {
                    bet[j] = bet[j] + t;
                }
                t = Math.dot(wptr[0], wptr[3]);
                Math.axpy(-t, wptr[1], wptr[0]);
                alf[j] = alf[j] + t;
                Math.copy(wptr[0], wptr[4]);
                rnm = Math.norm(wptr[0]);
                anorm = bet[j] + Math.abs(alf[j]) + rnm;
                tol = reps * anorm;
                EigenValueDecomposition.ortbnd(alf, bet, eta, oldeta, j, rnm, eps);
                rnm = EigenValueDecomposition.purge(ll, q, wptr[0], wptr[1], wptr[4], wptr[3], eta, oldeta, j, rnm, tol, eps, reps);
                if (!(rnm <= tol)) continue;
                rnm = 0.0;
            }
            j = enough ? --j : last - 1;
            first = j + 1;
            bet[j + 1] = rnm;
            System.arraycopy(alf, 0, ritz, 0, j + 1);
            System.arraycopy(bet, 0, wptr[5], 0, j + 1);
            z = new double[j + 1][j + 1];
            for (i = 0; i <= j; ++i) {
                z[i][i] = 1.0;
            }
            EigenValueDecomposition.tql2(z, ritz, wptr[5], j + 1);
            for (i = 0; i <= j; ++i) {
                bnd[i] = rnm * Math.abs(z[j][i]);
            }
            boolean[] ref_enough = new boolean[]{enough};
            neig = EigenValueDecomposition.error_bound(ref_enough, ritz, bnd, j, tol, eps34);
            enough = ref_enough[0];
            if (neig < k) {
                if (0 == neig) {
                    last = first + 9;
                    intro = first;
                } else {
                    last = first + Math.max(3, 1 + (j - intro) * (k - neig) / Math.max(3, neig));
                }
                last = Math.min(last, n);
            } else {
                enough = true;
            }
            enough = enough || first >= n;
        }
        EigenValueDecomposition.store(q, j, wptr[1]);
        k = Math.min(k, neig);
        double[] eigenvalues = new double[k];
        double[][] eigenvectors = new double[n][k];
        int index = 0;
        for (int i = 0; i <= j && index < k; ++i) {
            if (!(bnd[i] <= kappa * Math.abs(ritz[i]))) continue;
            for (int row = 0; row < n; ++row) {
                for (int l = 0; l <= j; ++l) {
                    double[] dArray = eigenvectors[row];
                    int n2 = index;
                    dArray[n2] = dArray[n2] + q[l][row] * z[l][i];
                }
            }
            eigenvalues[index++] = ritz[i];
        }
        return new EigenValueDecomposition(eigenvectors, eigenvalues);
    }

    private static double startv(IMatrix A, double[][] q, double[][] wptr, int step) {
        double rnm = Math.dot(wptr[0], wptr[0]);
        double[] r = wptr[0];
        for (int id = 0; id < 3; ++id) {
            if (id > 0 || step > 0 || rnm == 0.0) {
                for (int i = 0; i < r.length; ++i) {
                    r[i] = Math.random();
                }
            }
            Math.copy(wptr[0], wptr[3]);
            A.ax(wptr[3], wptr[0]);
            Math.copy(wptr[0], wptr[3]);
            rnm = Math.dot(wptr[0], wptr[3]);
            if (rnm > 0.0) break;
        }
        if (rnm <= 0.0) {
            logger.error("Lanczos method was unable to find a starting vector within range.");
            return -1.0;
        }
        if (step > 0) {
            for (int i = 0; i < step; ++i) {
                double t = Math.dot(wptr[3], q[i]);
                Math.axpy(-t, q[i], wptr[0]);
            }
            double t = Math.dot(wptr[4], wptr[0]);
            Math.axpy(-t, wptr[2], wptr[0]);
            Math.copy(wptr[0], wptr[3]);
            t = Math.dot(wptr[3], wptr[0]);
            if (t <= Math.EPSILON * rnm) {
                t = 0.0;
            }
            rnm = t;
        }
        return Math.sqrt(rnm);
    }

    private static void ortbnd(double[] alf, double[] bet, double[] eta, double[] oldeta, int step, double rnm, double eps) {
        int i;
        if (step < 1) {
            return;
        }
        if (0.0 != rnm) {
            if (step > 1) {
                oldeta[0] = (bet[1] * eta[1] + (alf[0] - alf[step]) * eta[0] - bet[step] * oldeta[0]) / rnm + eps;
            }
            for (i = 1; i <= step - 2; ++i) {
                oldeta[i] = (bet[i + 1] * eta[i + 1] + (alf[i] - alf[step]) * eta[i] + bet[i] * eta[i - 1] - bet[step] * oldeta[i]) / rnm + eps;
            }
        }
        oldeta[step - 1] = eps;
        for (i = 0; i < step; ++i) {
            double swap = eta[i];
            eta[i] = oldeta[i];
            oldeta[i] = swap;
        }
        eta[step] = eps;
    }

    private static double purge(int ll, double[][] Q, double[] r, double[] q, double[] ra, double[] qa, double[] eta, double[] oldeta, int step, double rnm, double tol, double eps, double reps) {
        if (step < ll + 2) {
            return rnm;
        }
        int k = EigenValueDecomposition.idamax(step - (ll + 1), eta, ll, 1) + ll;
        if (Math.abs(eta[k]) > reps) {
            int i;
            double reps1 = eps / reps;
            boolean flag = true;
            for (int iteration = 0; iteration < 2 && flag; ++iteration) {
                double t;
                if (!(rnm > tol)) continue;
                double tq = 0.0;
                double tr = 0.0;
                for (i = ll; i < step; ++i) {
                    t = -Math.dot(qa, Q[i]);
                    tq += Math.abs(t);
                    Math.axpy(t, Q[i], q);
                    t = -Math.dot(ra, Q[i]);
                    tr += Math.abs(t);
                    Math.axpy(t, Q[i], r);
                }
                Math.copy(q, qa);
                t = -Math.dot(r, qa);
                tr += Math.abs(t);
                Math.axpy(t, q, r);
                Math.copy(r, ra);
                rnm = Math.sqrt(Math.dot(ra, r));
                if (!(tq <= reps1) || !(tr <= reps1 * rnm)) continue;
                flag = false;
            }
            for (i = ll; i <= step; ++i) {
                eta[i] = eps;
                oldeta[i] = eps;
            }
        }
        return rnm;
    }

    private static int idamax(int n, double[] dx, int ix0, int incx) {
        int ix;
        if (n < 1) {
            return -1;
        }
        if (n == 1) {
            return 0;
        }
        if (incx == 0) {
            return -1;
        }
        int imax = ix = incx < 0 ? ix0 + (-n + 1) * incx : ix0;
        double dmax = Math.abs(dx[ix]);
        for (int i = 1; i < n; ++i) {
            double dtemp = Math.abs(dx[ix += incx]);
            if (!(dtemp > dmax)) continue;
            dmax = dtemp;
            imax = ix;
        }
        return imax;
    }

    private static int error_bound(boolean[] enough, double[] ritz, double[] bnd, int step, double tol, double eps34) {
        int i;
        int i2;
        int mid = EigenValueDecomposition.idamax(step + 1, bnd, 0, 1);
        for (i2 = (step + 1 + (step - 1)) / 2; i2 >= mid + 1; --i2) {
            if (!(Math.abs(ritz[i2 - 1] - ritz[i2]) < eps34 * Math.abs(ritz[i2])) || !(bnd[i2] > tol) || !(bnd[i2 - 1] > tol)) continue;
            bnd[i2 - 1] = Math.sqrt(bnd[i2] * bnd[i2] + bnd[i2 - 1] * bnd[i2 - 1]);
            bnd[i2] = 0.0;
        }
        for (i2 = (step + 1 - (step - 1)) / 2; i2 <= mid - 1; ++i2) {
            if (!(Math.abs(ritz[i2 + 1] - ritz[i2]) < eps34 * Math.abs(ritz[i2])) || !(bnd[i2] > tol) || !(bnd[i2 + 1] > tol)) continue;
            bnd[i2 + 1] = Math.sqrt(bnd[i2] * bnd[i2] + bnd[i2 + 1] * bnd[i2 + 1]);
            bnd[i2] = 0.0;
        }
        int neig = 0;
        double gapl = ritz[step] - ritz[0];
        for (i = 0; i <= step; ++i) {
            double gap = gapl;
            if (i < step) {
                gapl = ritz[i + 1] - ritz[i];
            }
            if ((gap = Math.min(gap, gapl)) > bnd[i]) {
                bnd[i] = bnd[i] * (bnd[i] / gap);
            }
            if (!(bnd[i] <= 16.0 * Math.EPSILON * Math.abs(ritz[i]))) continue;
            ++neig;
            if (enough[0]) continue;
            enough[0] = -Math.EPSILON < ritz[i] && ritz[i] < Math.EPSILON;
        }
        logger.info("Lancozs method found {} converged eigenvalues of the {}-by-{} matrix", new Object[]{neig, step + 1, step + 1});
        if (neig != 0) {
            boolean n = false;
            for (i = 0; i <= step; ++i) {
                if (!(bnd[i] <= 16.0 * Math.EPSILON * Math.abs(ritz[i]))) continue;
                logger.info("ritz[{}] = {}", (Object)i, (Object)ritz[i]);
            }
        }
        return neig;
    }

    private static void store(double[][] q, int j, double[] s) {
        if (null == q[j]) {
            q[j] = (double[])s.clone();
        } else {
            Math.copy(s, q[j]);
        }
    }

    public static EigenValueDecomposition decompose(double[][] A) {
        if (A.length != A[0].length) {
            throw new IllegalArgumentException(String.format("Matrix is not square: %d x %d", A.length, A[0].length));
        }
        int n = A.length;
        double tol = 100.0 * Math.EPSILON;
        boolean symmetric = true;
        for (int i = 0; i < n && symmetric; ++i) {
            for (int j = 0; j < n && symmetric; ++j) {
                symmetric = Math.abs(A[i][j] - A[j][i]) < tol;
            }
        }
        return EigenValueDecomposition.decompose(A, symmetric);
    }

    public static EigenValueDecomposition decompose(double[][] A, boolean symmetric) {
        return EigenValueDecomposition.decompose(A, symmetric, false);
    }

    public static EigenValueDecomposition decompose(double[][] A, boolean symmetric, boolean onlyValues) {
        if (A.length != A[0].length) {
            throw new IllegalArgumentException(String.format("Matrix is not square: %d x %d", A.length, A[0].length));
        }
        int n = A.length;
        double[] d = new double[n];
        double[] e = new double[n];
        if (symmetric) {
            double[][] V = A;
            if (onlyValues) {
                EigenValueDecomposition.tred(V, d, e);
                EigenValueDecomposition.tql(d, e, n);
                return new EigenValueDecomposition(null, d);
            }
            EigenValueDecomposition.tred2(V, d, e);
            EigenValueDecomposition.tql2(V, d, e, n);
            return new EigenValueDecomposition(V, d);
        }
        double[] scale = EigenValueDecomposition.balance(A);
        int[] perm = EigenValueDecomposition.elmhes(A);
        if (onlyValues) {
            EigenValueDecomposition.hqr(A, d, e);
            EigenValueDecomposition.sort(d, e);
            return new EigenValueDecomposition(null, d, e);
        }
        double[][] V = new double[n][n];
        for (int i = 0; i < n; ++i) {
            V[i][i] = 1.0;
        }
        EigenValueDecomposition.eltran(A, V, perm);
        EigenValueDecomposition.hqr2(A, V, d, e);
        EigenValueDecomposition.balbak(V, scale);
        EigenValueDecomposition.sort(d, e, V);
        return new EigenValueDecomposition(V, d, e);
    }

    private static void tred(double[][] V, double[] d, double[] e) {
        int n = V.length;
        System.arraycopy(V[n - 1], 0, d, 0, n);
        for (int i = n - 1; i > 0; --i) {
            int k;
            double scale = 0.0;
            double h = 0.0;
            for (k = 0; k < i; ++k) {
                scale += Math.abs(d[k]);
            }
            if (scale == 0.0) {
                e[i] = d[i - 1];
                for (int j = 0; j < i; ++j) {
                    d[j] = V[i - 1][j];
                    V[i][j] = 0.0;
                    V[j][i] = 0.0;
                }
            } else {
                int j;
                int j2;
                for (k = 0; k < i; ++k) {
                    int n2 = k;
                    d[n2] = d[n2] / scale;
                    h += d[k] * d[k];
                }
                double f = d[i - 1];
                double g = Math.sqrt(h);
                if (f > 0.0) {
                    g = -g;
                }
                e[i] = scale * g;
                h -= f * g;
                d[i - 1] = f - g;
                for (j2 = 0; j2 < i; ++j2) {
                    e[j2] = 0.0;
                }
                for (j2 = 0; j2 < i; ++j2) {
                    V[j2][i] = f = d[j2];
                    g = e[j2] + V[j2][j2] * f;
                    for (int k2 = j2 + 1; k2 <= i - 1; ++k2) {
                        g += V[k2][j2] * d[k2];
                        int n3 = k2;
                        e[n3] = e[n3] + V[k2][j2] * f;
                    }
                    e[j2] = g;
                }
                f = 0.0;
                for (j2 = 0; j2 < i; ++j2) {
                    int n4 = j2;
                    e[n4] = e[n4] / h;
                    f += e[j2] * d[j2];
                }
                double hh = f / (h + h);
                for (j = 0; j < i; ++j) {
                    int n5 = j;
                    e[n5] = e[n5] - hh * d[j];
                }
                for (j = 0; j < i; ++j) {
                    f = d[j];
                    g = e[j];
                    for (int k3 = j; k3 <= i - 1; ++k3) {
                        double[] dArray = V[k3];
                        int n6 = j;
                        dArray[n6] = dArray[n6] - (f * e[k3] + g * d[k3]);
                    }
                    d[j] = V[i - 1][j];
                    V[i][j] = 0.0;
                }
            }
            d[i] = h;
        }
        for (int j = 0; j < n; ++j) {
            d[j] = V[j][j];
        }
        e[0] = 0.0;
    }

    private static void tred2(double[][] V, double[] d, double[] e) {
        int i;
        int n = V.length;
        System.arraycopy(V[n - 1], 0, d, 0, n);
        for (i = n - 1; i > 0; --i) {
            int k;
            double scale = 0.0;
            double h = 0.0;
            for (k = 0; k < i; ++k) {
                scale += Math.abs(d[k]);
            }
            if (scale == 0.0) {
                e[i] = d[i - 1];
                for (int j = 0; j < i; ++j) {
                    d[j] = V[i - 1][j];
                    V[i][j] = 0.0;
                    V[j][i] = 0.0;
                }
            } else {
                int j;
                int j2;
                for (k = 0; k < i; ++k) {
                    int n2 = k;
                    d[n2] = d[n2] / scale;
                    h += d[k] * d[k];
                }
                double f = d[i - 1];
                double g = Math.sqrt(h);
                if (f > 0.0) {
                    g = -g;
                }
                e[i] = scale * g;
                h -= f * g;
                d[i - 1] = f - g;
                for (j2 = 0; j2 < i; ++j2) {
                    e[j2] = 0.0;
                }
                for (j2 = 0; j2 < i; ++j2) {
                    V[j2][i] = f = d[j2];
                    g = e[j2] + V[j2][j2] * f;
                    for (int k2 = j2 + 1; k2 <= i - 1; ++k2) {
                        g += V[k2][j2] * d[k2];
                        int n3 = k2;
                        e[n3] = e[n3] + V[k2][j2] * f;
                    }
                    e[j2] = g;
                }
                f = 0.0;
                for (j2 = 0; j2 < i; ++j2) {
                    int n4 = j2;
                    e[n4] = e[n4] / h;
                    f += e[j2] * d[j2];
                }
                double hh = f / (h + h);
                for (j = 0; j < i; ++j) {
                    int n5 = j;
                    e[n5] = e[n5] - hh * d[j];
                }
                for (j = 0; j < i; ++j) {
                    f = d[j];
                    g = e[j];
                    for (int k3 = j; k3 <= i - 1; ++k3) {
                        double[] dArray = V[k3];
                        int n6 = j;
                        dArray[n6] = dArray[n6] - (f * e[k3] + g * d[k3]);
                    }
                    d[j] = V[i - 1][j];
                    V[i][j] = 0.0;
                }
            }
            d[i] = h;
        }
        for (i = 0; i < n - 1; ++i) {
            int k;
            V[n - 1][i] = V[i][i];
            V[i][i] = 1.0;
            double h = d[i + 1];
            if (h != 0.0) {
                for (k = 0; k <= i; ++k) {
                    d[k] = V[k][i + 1] / h;
                }
                for (int j = 0; j <= i; ++j) {
                    int k4;
                    double g = 0.0;
                    for (k4 = 0; k4 <= i; ++k4) {
                        g += V[k4][i + 1] * V[k4][j];
                    }
                    for (k4 = 0; k4 <= i; ++k4) {
                        double[] dArray = V[k4];
                        int n7 = j;
                        dArray[n7] = dArray[n7] - g * d[k4];
                    }
                }
            }
            for (k = 0; k <= i; ++k) {
                V[k][i + 1] = 0.0;
            }
        }
        for (int j = 0; j < n; ++j) {
            d[j] = V[n - 1][j];
            V[n - 1][j] = 0.0;
        }
        V[n - 1][n - 1] = 1.0;
        e[0] = 0.0;
    }

    private static void tql(double[] d, double[] e, int n) {
        for (int i = 1; i < n; ++i) {
            e[i - 1] = e[i];
        }
        e[n - 1] = 0.0;
        double f = 0.0;
        double tst1 = 0.0;
        for (int l = 0; l < n; ++l) {
            int m;
            tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
            for (m = l; m < n && !(Math.abs(e[m]) <= Math.EPSILON * tst1); ++m) {
            }
            if (m > l) {
                int iter = 0;
                do {
                    double c;
                    if (++iter >= 30) {
                        throw new RuntimeException("Too many iterations");
                    }
                    double g = d[l];
                    double p = (d[l + 1] - d[l]) / (2.0 * e[l]);
                    double r = Math.hypot(p, 1.0);
                    if (p < 0.0) {
                        r = -r;
                    }
                    d[l] = e[l] / (p + r);
                    d[l + 1] = e[l] * (p + r);
                    double dl1 = d[l + 1];
                    double h = g - d[l];
                    int i = l + 2;
                    while (i < n) {
                        int n2 = i++;
                        d[n2] = d[n2] - h;
                    }
                    f += h;
                    p = d[m];
                    double c2 = c = 1.0;
                    double c3 = c;
                    double el1 = e[l + 1];
                    double s = 0.0;
                    double s2 = 0.0;
                    for (int i2 = m - 1; i2 >= l; --i2) {
                        c3 = c2;
                        c2 = c;
                        s2 = s;
                        g = c * e[i2];
                        h = c * p;
                        r = Math.hypot(p, e[i2]);
                        e[i2 + 1] = s * r;
                        s = e[i2] / r;
                        c = p / r;
                        p = c * d[i2] - s * g;
                        d[i2 + 1] = h + s * (c * g + s * d[i2]);
                    }
                    p = -s * s2 * c3 * el1 * e[l] / dl1;
                    e[l] = s * p;
                    d[l] = c * p;
                } while (Math.abs(e[l]) > Math.EPSILON * tst1);
            }
            d[l] = d[l] + f;
            e[l] = 0.0;
        }
        for (int i = 0; i < n - 1; ++i) {
            int k = i;
            double p = d[i];
            for (int j = i + 1; j < n; ++j) {
                if (!(d[j] > p)) continue;
                k = j;
                p = d[j];
            }
            if (k == i) continue;
            d[k] = d[i];
            d[i] = p;
        }
    }

    private static void tql2(double[][] V, double[] d, double[] e, int n) {
        for (int i = 1; i < n; ++i) {
            e[i - 1] = e[i];
        }
        e[n - 1] = 0.0;
        double f = 0.0;
        double tst1 = 0.0;
        for (int l = 0; l < n; ++l) {
            int m;
            tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
            for (m = l; m < n && !(Math.abs(e[m]) <= Math.EPSILON * tst1); ++m) {
            }
            if (m > l) {
                int iter = 0;
                do {
                    double c;
                    if (++iter >= 30) {
                        throw new RuntimeException("Too many iterations");
                    }
                    double g = d[l];
                    double p = (d[l + 1] - d[l]) / (2.0 * e[l]);
                    double r = Math.hypot(p, 1.0);
                    if (p < 0.0) {
                        r = -r;
                    }
                    d[l] = e[l] / (p + r);
                    d[l + 1] = e[l] * (p + r);
                    double dl1 = d[l + 1];
                    double h = g - d[l];
                    int i = l + 2;
                    while (i < n) {
                        int n2 = i++;
                        d[n2] = d[n2] - h;
                    }
                    f += h;
                    p = d[m];
                    double c2 = c = 1.0;
                    double c3 = c;
                    double el1 = e[l + 1];
                    double s = 0.0;
                    double s2 = 0.0;
                    for (int i2 = m - 1; i2 >= l; --i2) {
                        c3 = c2;
                        c2 = c;
                        s2 = s;
                        g = c * e[i2];
                        h = c * p;
                        r = Math.hypot(p, e[i2]);
                        e[i2 + 1] = s * r;
                        s = e[i2] / r;
                        c = p / r;
                        p = c * d[i2] - s * g;
                        d[i2 + 1] = h + s * (c * g + s * d[i2]);
                        for (int k = 0; k < n; ++k) {
                            h = V[k][i2 + 1];
                            V[k][i2 + 1] = s * V[k][i2] + c * h;
                            V[k][i2] = c * V[k][i2] - s * h;
                        }
                    }
                    p = -s * s2 * c3 * el1 * e[l] / dl1;
                    e[l] = s * p;
                    d[l] = c * p;
                } while (Math.abs(e[l]) > Math.EPSILON * tst1);
            }
            d[l] = d[l] + f;
            e[l] = 0.0;
        }
        for (int i = 0; i < n - 1; ++i) {
            int j;
            int k = i;
            double p = d[i];
            for (j = i + 1; j < n; ++j) {
                if (!(d[j] > p)) continue;
                k = j;
                p = d[j];
            }
            if (k == i) continue;
            d[k] = d[i];
            d[i] = p;
            for (j = 0; j < n; ++j) {
                p = V[j][i];
                V[j][i] = V[j][k];
                V[j][k] = p;
            }
        }
    }

    private static double[] balance(double[][] A) {
        double sqrdx = Math.RADIX * Math.RADIX;
        int n = A.length;
        double[] scale = new double[n];
        for (int i = 0; i < n; ++i) {
            scale[i] = 1.0;
        }
        boolean done = false;
        while (!done) {
            done = true;
            for (int i = 0; i < n; ++i) {
                double r = 0.0;
                double c = 0.0;
                for (int j = 0; j < n; ++j) {
                    if (j == i) continue;
                    c += Math.abs(A[j][i]);
                    r += Math.abs(A[i][j]);
                }
                if (c == 0.0 || r == 0.0) continue;
                double g = r / (double)Math.RADIX;
                double f = 1.0;
                double s = c + r;
                while (c < g) {
                    f *= (double)Math.RADIX;
                    c *= sqrdx;
                }
                g = r * (double)Math.RADIX;
                while (c > g) {
                    f /= (double)Math.RADIX;
                    c /= sqrdx;
                }
                if (!((c + r) / f < 0.95 * s)) continue;
                done = false;
                g = 1.0 / f;
                int n2 = i;
                scale[n2] = scale[n2] * f;
                int j = 0;
                while (j < n) {
                    double[] dArray = A[i];
                    int n3 = j++;
                    dArray[n3] = dArray[n3] * g;
                }
                for (j = 0; j < n; ++j) {
                    double[] dArray = A[j];
                    int n4 = i;
                    dArray[n4] = dArray[n4] * f;
                }
            }
        }
        return scale;
    }

    private static void balbak(double[][] V, double[] scale) {
        int n = V.length;
        for (int i = 0; i < n; ++i) {
            int j = 0;
            while (j < n) {
                double[] dArray = V[i];
                int n2 = j++;
                dArray[n2] = dArray[n2] * scale[i];
            }
        }
    }

    private static int[] elmhes(double[][] A) {
        int n = A.length;
        int[] perm = new int[n];
        for (int m = 1; m < n - 1; ++m) {
            int j;
            double x = 0.0;
            int i = m;
            for (j = m; j < n; ++j) {
                if (!(Math.abs(A[j][m - 1]) > Math.abs(x))) continue;
                x = A[j][m - 1];
                i = j;
            }
            perm[m] = i;
            if (i != m) {
                double swap;
                for (j = m - 1; j < n; ++j) {
                    swap = A[i][j];
                    A[i][j] = A[m][j];
                    A[m][j] = swap;
                }
                for (j = 0; j < n; ++j) {
                    swap = A[j][i];
                    A[j][i] = A[j][m];
                    A[j][m] = swap;
                }
            }
            if (x == 0.0) continue;
            for (i = m + 1; i < n; ++i) {
                int j2;
                double y = A[i][m - 1];
                if (y == 0.0) continue;
                A[i][m - 1] = y /= x;
                for (j2 = m; j2 < n; ++j2) {
                    double[] dArray = A[i];
                    int n2 = j2;
                    dArray[n2] = dArray[n2] - y * A[m][j2];
                }
                for (j2 = 0; j2 < n; ++j2) {
                    double[] dArray = A[j2];
                    int n3 = m;
                    dArray[n3] = dArray[n3] + y * A[j2][i];
                }
            }
        }
        return perm;
    }

    private static void eltran(double[][] A, double[][] V, int[] perm) {
        int n = A.length;
        for (int mp = n - 2; mp > 0; --mp) {
            for (int k = mp + 1; k < n; ++k) {
                V[k][mp] = A[k][mp - 1];
            }
            int i = perm[mp];
            if (i == mp) continue;
            for (int j = mp; j < n; ++j) {
                V[mp][j] = V[i][j];
                V[i][j] = 0.0;
            }
            V[i][mp] = 1.0;
        }
    }

    private static void hqr(double[][] A, double[] d, double[] e) {
        int j;
        int i;
        int n = A.length;
        double r = 0.0;
        double q = 0.0;
        double p = 0.0;
        double anorm = 0.0;
        for (i = 0; i < n; ++i) {
            for (j = Math.max(i - 1, 0); j < n; ++j) {
                anorm += Math.abs(A[i][j]);
            }
        }
        int nn = n - 1;
        double t = 0.0;
        while (nn >= 0) {
            int l;
            int its = 0;
            do {
                int m;
                double z;
                double s;
                for (l = nn; l > 0; --l) {
                    s = Math.abs(A[l - 1][l - 1]) + Math.abs(A[l][l]);
                    if (s == 0.0) {
                        s = anorm;
                    }
                    if (!(Math.abs(A[l][l - 1]) <= Math.EPSILON * s)) continue;
                    A[l][l - 1] = 0.0;
                    break;
                }
                double x = A[nn][nn];
                if (l == nn) {
                    d[nn--] = x + t;
                    continue;
                }
                double y = A[nn - 1][nn - 1];
                double w = A[nn][nn - 1] * A[nn - 1][nn];
                if (l == nn - 1) {
                    p = 0.5 * (y - x);
                    q = p * p + w;
                    z = Math.sqrt(Math.abs(q));
                    x += t;
                    if (q >= 0.0) {
                        z = p + Math.copySign(z, p);
                        d[nn - 1] = d[nn] = x + z;
                        if (z != 0.0) {
                            d[nn] = x - w / z;
                        }
                    } else {
                        d[nn] = x + p;
                        e[nn] = -z;
                        d[nn - 1] = d[nn];
                        e[nn - 1] = -e[nn];
                    }
                    nn -= 2;
                    continue;
                }
                if (its == 30) {
                    throw new IllegalStateException("Too many iterations in hqr");
                }
                if (its == 10 || its == 20) {
                    t += x;
                    i = 0;
                    while (i < nn + 1) {
                        double[] dArray = A[i];
                        int n2 = i++;
                        dArray[n2] = dArray[n2] - x;
                    }
                    s = Math.abs(A[nn][nn - 1]) + Math.abs(A[nn - 1][nn - 2]);
                    y = x = 0.75 * s;
                    w = -0.4375 * s * s;
                }
                ++its;
                for (m = nn - 2; m >= l; --m) {
                    double v;
                    double u;
                    z = A[m][m];
                    r = x - z;
                    s = y - z;
                    p = (r * s - w) / A[m + 1][m] + A[m][m + 1];
                    q = A[m + 1][m + 1] - z - r - s;
                    r = A[m + 2][m + 1];
                    s = Math.abs(p) + Math.abs(q) + Math.abs(r);
                    if (m == l || (u = Math.abs(A[m][m - 1]) * (Math.abs(q /= s) + Math.abs(r /= s))) <= Math.EPSILON * (v = Math.abs(p /= s) * (Math.abs(A[m - 1][m - 1]) + Math.abs(z) + Math.abs(A[m + 1][m + 1])))) break;
                }
                for (i = m; i < nn - 1; ++i) {
                    A[i + 2][i] = 0.0;
                    if (i == m) continue;
                    A[i + 2][i - 1] = 0.0;
                }
                for (int k = m; k < nn; ++k) {
                    if (k != m) {
                        p = A[k][k - 1];
                        q = A[k + 1][k - 1];
                        r = 0.0;
                        if (k + 1 != nn) {
                            r = A[k + 2][k - 1];
                        }
                        if ((x = Math.abs(p) + Math.abs(q) + Math.abs(r)) != 0.0) {
                            p /= x;
                            q /= x;
                            r /= x;
                        }
                    }
                    if ((s = Math.copySign(Math.sqrt(p * p + q * q + r * r), p)) == 0.0) continue;
                    if (k == m) {
                        if (l != m) {
                            A[k][k - 1] = -A[k][k - 1];
                        }
                    } else {
                        A[k][k - 1] = -s * x;
                    }
                    x = (p += s) / s;
                    y = q / s;
                    z = r / s;
                    q /= p;
                    r /= p;
                    j = k;
                    while (j < nn + 1) {
                        p = A[k][j] + q * A[k + 1][j];
                        if (k + 1 != nn) {
                            double[] dArray = A[k + 2];
                            int n3 = j;
                            dArray[n3] = dArray[n3] - (p += r * A[k + 2][j]) * z;
                        }
                        double[] dArray = A[k + 1];
                        int n4 = j;
                        dArray[n4] = dArray[n4] - p * y;
                        double[] dArray2 = A[k];
                        int n5 = j++;
                        dArray2[n5] = dArray2[n5] - p * x;
                    }
                    int mmin = nn < k + 3 ? nn : k + 3;
                    for (i = l; i < mmin + 1; ++i) {
                        p = x * A[i][k] + y * A[i][k + 1];
                        if (k + 1 != nn) {
                            double[] dArray = A[i];
                            int n6 = k + 2;
                            dArray[n6] = dArray[n6] - (p += z * A[i][k + 2]) * r;
                        }
                        double[] dArray = A[i];
                        int n7 = k + 1;
                        dArray[n7] = dArray[n7] - p * q;
                        double[] dArray3 = A[i];
                        int n8 = k;
                        dArray3[n8] = dArray3[n8] - p;
                    }
                }
            } while (l + 1 < nn);
        }
    }

    private static void hqr2(double[][] A, double[][] V, double[] d, double[] e) {
        int k;
        int m;
        double w;
        double y;
        double x;
        int j;
        int i;
        int n = A.length;
        double z = 0.0;
        double s = 0.0;
        double r = 0.0;
        double q = 0.0;
        double p = 0.0;
        double anorm = 0.0;
        for (i = 0; i < n; ++i) {
            for (j = Math.max(i - 1, 0); j < n; ++j) {
                anorm += Math.abs(A[i][j]);
            }
        }
        int nn = n - 1;
        double t = 0.0;
        while (nn >= 0) {
            int l;
            int its = 0;
            do {
                for (l = nn; l > 0; --l) {
                    s = Math.abs(A[l - 1][l - 1]) + Math.abs(A[l][l]);
                    if (s == 0.0) {
                        s = anorm;
                    }
                    if (!(Math.abs(A[l][l - 1]) <= Math.EPSILON * s)) continue;
                    A[l][l - 1] = 0.0;
                    break;
                }
                x = A[nn][nn];
                if (l == nn) {
                    double d2 = x + t;
                    A[nn][nn] = d2;
                    d[nn] = d2;
                    --nn;
                    continue;
                }
                y = A[nn - 1][nn - 1];
                w = A[nn][nn - 1] * A[nn - 1][nn];
                if (l == nn - 1) {
                    p = 0.5 * (y - x);
                    q = p * p + w;
                    z = Math.sqrt(Math.abs(q));
                    A[nn][nn] = x += t;
                    A[nn - 1][nn - 1] = y + t;
                    if (q >= 0.0) {
                        z = p + Math.copySign(z, p);
                        d[nn - 1] = d[nn] = x + z;
                        if (z != 0.0) {
                            d[nn] = x - w / z;
                        }
                        x = A[nn][nn - 1];
                        s = Math.abs(x) + Math.abs(z);
                        p = x / s;
                        q = z / s;
                        r = Math.sqrt(p * p + q * q);
                        p /= r;
                        q /= r;
                        for (j = nn - 1; j < n; ++j) {
                            z = A[nn - 1][j];
                            A[nn - 1][j] = q * z + p * A[nn][j];
                            A[nn][j] = q * A[nn][j] - p * z;
                        }
                        for (i = 0; i <= nn; ++i) {
                            z = A[i][nn - 1];
                            A[i][nn - 1] = q * z + p * A[i][nn];
                            A[i][nn] = q * A[i][nn] - p * z;
                        }
                        for (i = 0; i < n; ++i) {
                            z = V[i][nn - 1];
                            V[i][nn - 1] = q * z + p * V[i][nn];
                            V[i][nn] = q * V[i][nn] - p * z;
                        }
                    } else {
                        d[nn] = x + p;
                        e[nn] = -z;
                        d[nn - 1] = d[nn];
                        e[nn - 1] = -e[nn];
                    }
                    nn -= 2;
                    continue;
                }
                if (its == 30) {
                    throw new IllegalArgumentException("Too many iterations in hqr");
                }
                if (its == 10 || its == 20) {
                    t += x;
                    i = 0;
                    while (i < nn + 1) {
                        double[] dArray = A[i];
                        int n2 = i++;
                        dArray[n2] = dArray[n2] - x;
                    }
                    s = Math.abs(A[nn][nn - 1]) + Math.abs(A[nn - 1][nn - 2]);
                    y = x = 0.75 * s;
                    w = -0.4375 * s * s;
                }
                ++its;
                for (m = nn - 2; m >= l; --m) {
                    double v;
                    double u;
                    z = A[m][m];
                    r = x - z;
                    s = y - z;
                    p = (r * s - w) / A[m + 1][m] + A[m][m + 1];
                    q = A[m + 1][m + 1] - z - r - s;
                    r = A[m + 2][m + 1];
                    s = Math.abs(p) + Math.abs(q) + Math.abs(r);
                    if (m == l || (u = Math.abs(A[m][m - 1]) * (Math.abs(q /= s) + Math.abs(r /= s))) <= Math.EPSILON * (v = Math.abs(p /= s) * (Math.abs(A[m - 1][m - 1]) + Math.abs(z) + Math.abs(A[m + 1][m + 1])))) break;
                }
                for (i = m; i < nn - 1; ++i) {
                    A[i + 2][i] = 0.0;
                    if (i == m) continue;
                    A[i + 2][i - 1] = 0.0;
                }
                for (k = m; k < nn; ++k) {
                    if (k != m) {
                        p = A[k][k - 1];
                        q = A[k + 1][k - 1];
                        r = 0.0;
                        if (k + 1 != nn) {
                            r = A[k + 2][k - 1];
                        }
                        if ((x = Math.abs(p) + Math.abs(q) + Math.abs(r)) != 0.0) {
                            p /= x;
                            q /= x;
                            r /= x;
                        }
                    }
                    if ((s = Math.copySign(Math.sqrt(p * p + q * q + r * r), p)) == 0.0) continue;
                    if (k == m) {
                        if (l != m) {
                            A[k][k - 1] = -A[k][k - 1];
                        }
                    } else {
                        A[k][k - 1] = -s * x;
                    }
                    x = (p += s) / s;
                    y = q / s;
                    z = r / s;
                    q /= p;
                    r /= p;
                    j = k;
                    while (j < n) {
                        p = A[k][j] + q * A[k + 1][j];
                        if (k + 1 != nn) {
                            double[] dArray = A[k + 2];
                            int n3 = j;
                            dArray[n3] = dArray[n3] - (p += r * A[k + 2][j]) * z;
                        }
                        double[] dArray = A[k + 1];
                        int n4 = j;
                        dArray[n4] = dArray[n4] - p * y;
                        double[] dArray2 = A[k];
                        int n5 = j++;
                        dArray2[n5] = dArray2[n5] - p * x;
                    }
                    int mmin = nn < k + 3 ? nn : k + 3;
                    for (i = 0; i < mmin + 1; ++i) {
                        p = x * A[i][k] + y * A[i][k + 1];
                        if (k + 1 != nn) {
                            double[] dArray = A[i];
                            int n6 = k + 2;
                            dArray[n6] = dArray[n6] - (p += z * A[i][k + 2]) * r;
                        }
                        double[] dArray = A[i];
                        int n7 = k + 1;
                        dArray[n7] = dArray[n7] - p * q;
                        double[] dArray3 = A[i];
                        int n8 = k;
                        dArray3[n8] = dArray3[n8] - p;
                    }
                    for (i = 0; i < n; ++i) {
                        p = x * V[i][k] + y * V[i][k + 1];
                        if (k + 1 != nn) {
                            double[] dArray = V[i];
                            int n9 = k + 2;
                            dArray[n9] = dArray[n9] - (p += z * V[i][k + 2]) * r;
                        }
                        double[] dArray = V[i];
                        int n10 = k + 1;
                        dArray[n10] = dArray[n10] - p * q;
                        double[] dArray4 = V[i];
                        int n11 = k;
                        dArray4[n11] = dArray4[n11] - p;
                    }
                }
            } while (l + 1 < nn);
        }
        if (anorm != 0.0) {
            for (nn = n - 1; nn >= 0; --nn) {
                Complex temp;
                p = d[nn];
                q = e[nn];
                int na = nn - 1;
                if (q == 0.0) {
                    m = nn;
                    A[nn][nn] = 1.0;
                    for (i = nn - 1; i >= 0; --i) {
                        w = A[i][i] - p;
                        r = 0.0;
                        for (j = m; j <= nn; ++j) {
                            r += A[i][j] * A[j][nn];
                        }
                        if (e[i] < 0.0) {
                            z = w;
                            s = r;
                            continue;
                        }
                        m = i;
                        if (e[i] == 0.0) {
                            t = w;
                            if (t == 0.0) {
                                t = Math.EPSILON * anorm;
                            }
                            A[i][nn] = -r / t;
                        } else {
                            x = A[i][i + 1];
                            y = A[i + 1][i];
                            q = Math.sqr(d[i] - p) + Math.sqr(e[i]);
                            A[i][nn] = t = (x * s - z * r) / q;
                            A[i + 1][nn] = Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z;
                        }
                        t = Math.abs(A[i][nn]);
                        if (!(Math.EPSILON * t * t > 1.0)) continue;
                        for (j = i; j <= nn; ++j) {
                            double[] dArray = A[j];
                            int n12 = nn;
                            dArray[n12] = dArray[n12] / t;
                        }
                    }
                    continue;
                }
                if (!(q < 0.0)) continue;
                m = na;
                if (Math.abs(A[nn][na]) > Math.abs(A[na][nn])) {
                    A[na][na] = q / A[nn][na];
                    A[na][nn] = -(A[nn][nn] - p) / A[nn][na];
                } else {
                    temp = EigenValueDecomposition.cdiv(0.0, -A[na][nn], A[na][na] - p, q);
                    A[na][na] = temp.re();
                    A[na][nn] = temp.im();
                }
                A[nn][na] = 0.0;
                A[nn][nn] = 1.0;
                for (i = nn - 2; i >= 0; --i) {
                    w = A[i][i] - p;
                    double sa = 0.0;
                    double ra = 0.0;
                    for (j = m; j <= nn; ++j) {
                        ra += A[i][j] * A[j][na];
                        sa += A[i][j] * A[j][nn];
                    }
                    if (e[i] < 0.0) {
                        z = w;
                        r = ra;
                        s = sa;
                    } else {
                        m = i;
                        if (e[i] == 0.0) {
                            temp = EigenValueDecomposition.cdiv(-ra, -sa, w, q);
                            A[i][na] = temp.re();
                            A[i][nn] = temp.im();
                        } else {
                            x = A[i][i + 1];
                            y = A[i + 1][i];
                            double vr = Math.sqr(d[i] - p) + Math.sqr(e[i]) - q * q;
                            double vi = 2.0 * q * (d[i] - p);
                            if (vr == 0.0 && vi == 0.0) {
                                vr = Math.EPSILON * anorm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
                            }
                            temp = EigenValueDecomposition.cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
                            A[i][na] = temp.re();
                            A[i][nn] = temp.im();
                            if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
                                A[i + 1][na] = (-ra - w * A[i][na] + q * A[i][nn]) / x;
                                A[i + 1][nn] = (-sa - w * A[i][nn] - q * A[i][na]) / x;
                            } else {
                                temp = EigenValueDecomposition.cdiv(-r - y * A[i][na], -s - y * A[i][nn], z, q);
                                A[i + 1][na] = temp.re();
                                A[i + 1][nn] = temp.im();
                            }
                        }
                    }
                    t = Math.max(Math.abs(A[i][na]), Math.abs(A[i][nn]));
                    if (!(Math.EPSILON * t * t > 1.0)) continue;
                    for (j = i; j <= nn; ++j) {
                        double[] dArray = A[j];
                        int n13 = na;
                        dArray[n13] = dArray[n13] / t;
                        double[] dArray5 = A[j];
                        int n14 = nn;
                        dArray5[n14] = dArray5[n14] / t;
                    }
                }
            }
            for (j = n - 1; j >= 0; --j) {
                for (i = 0; i < n; ++i) {
                    z = 0.0;
                    for (k = 0; k <= j; ++k) {
                        z += V[i][k] * A[k][j];
                    }
                    V[i][j] = z;
                }
            }
        }
    }

    private static Complex cdiv(double xr, double xi, double yr, double yi) {
        double cdivi;
        double cdivr;
        if (Math.abs(yr) > Math.abs(yi)) {
            double r = yi / yr;
            double d = yr + r * yi;
            cdivr = (xr + r * xi) / d;
            cdivi = (xi - r * xr) / d;
        } else {
            double r = yr / yi;
            double d = yi + r * yr;
            cdivr = (r * xr + xi) / d;
            cdivi = (r * xi - xr) / d;
        }
        return new Complex(cdivr, cdivi);
    }

    private static void sort(double[] d, double[] e) {
        int i = 0;
        int n = d.length;
        for (int j = 1; j < n; ++j) {
            double real = d[j];
            double img = e[j];
            for (i = j - 1; i >= 0 && !(d[i] >= d[j]); --i) {
                d[i + 1] = d[i];
                e[i + 1] = e[i];
            }
            d[i + 1] = real;
            e[i + 1] = img;
        }
    }

    private static void sort(double[] d, double[] e, double[][] V) {
        int i = 0;
        int n = d.length;
        double[] temp = new double[n];
        for (int j = 1; j < n; ++j) {
            int k;
            double real = d[j];
            double img = e[j];
            for (k = 0; k < n; ++k) {
                temp[k] = V[k][j];
            }
            for (i = j - 1; i >= 0 && !(d[i] >= d[j]); --i) {
                d[i + 1] = d[i];
                e[i + 1] = e[i];
                for (k = 0; k < n; ++k) {
                    V[k][i + 1] = V[k][i];
                }
            }
            d[i + 1] = real;
            e[i + 1] = img;
            for (k = 0; k < n; ++k) {
                V[k][i + 1] = temp[k];
            }
        }
    }
}

